#!/usr/bin/perl

# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

$CVS_Version_String = '$Id: SR_lib.pm,v 1.38 2004/05/03 21:19:34 tuerk Exp $';
$CVS_Name_String = '$Name: rel-1-4-01 $';

package SR_lib;

use strict;

use lib $ENV{"MASV_PERL_ROOT"};

use FileHandle;
use File::Path;
use Cwd;
use File::Basename;
use Parallel::ForkManager;

# get database specific stuff
BEGIN {
	use lib $ENV{"MASV_DATABASE_DESCRIPTION"};
	use MASV_db_desc;
}

use vars qw(@test_set @training_set @world_set @dev_set @all %speakerlist $htk_bin $source_scripts $base_dir 
			$source_wav_files $source_trans_dir $paramPool_dir $matlab_path $matlab_host 
			$host_list $host_infos $nice_factor $verbose_level $sv_systems_dir $sign @hosts %hostInfos $numHosts_target $database_description );

@test_set = ();
@training_set = ();
@world_set = ();
@dev_set = ();
@all = ();
%speakerlist = ();


foreach my $test_env_var ('MASV_PERL_HTKBIN','MASV_PERL_ROOT','MASV_BASEDIR','MASV_SOURCE_DATABASE', 'MASV_PARAMPOOL_DIR','MASV_MATLAB_HOME_DIR','MASV_MATLAB_HOST','MASV_HOSTLIST','MASV_HOST_INFOS', 'MASV_NICE_FACTOR','MASV_VERBOSE_LEVEL','MASV_DATABASE_DESCRIPTION','MASV_USE_LOCAL_MATLAB_CACHE') {
	die ("Environment variable $test_env_var is not set! Check your .bashrc/.profile or .tcshrc\n") if (!(defined($ENV{$test_env_var})));
}

# Globale Variablen
	$htk_bin = $ENV{"MASV_PERL_HTKBIN"}; #"${home_dir}bin/HTK-3.1/htk/bin.linux/";
	$source_scripts = $ENV{"MASV_PERL_ROOT"}; #"${home_dir}Scripts/";
	$base_dir = $ENV{"MASV_BASEDIR"}; #"${home_dir}Diss/HTK/"; 
	$source_wav_files = $ENV{"MASV_SOURCE_DATABASE"}; #"/raid/tera6/VERIDAT/VD/";
	$source_trans_dir = $ENV{"MASV_SOURCE_DATABASE"}; #"/raid/tera6/VERIDAT/VD/";
	$paramPool_dir = $ENV{"MASV_PARAMPOOL_DIR"}; #"/raid/tera6/VERIDAT/paramPool/";
	$matlab_path = $ENV{"MASV_MATLAB_HOME_DIR"}; #"${home_dir}matlab/";
	$matlab_host = $ENV{"MASV_MATLAB_HOST"}; #"linux52";
	$host_list = $ENV{"MASV_HOSTLIST"};  # /usr/local/lib/PARALLELHOSTS
	$host_infos = $ENV{"MASV_HOST_INFOS"}; ; # ${home_dir}/Infos/linux_cluster_mips
	$nice_factor = $ENV{"MASV_NICE_FACTOR"}; # 19
	$verbose_level = $ENV{"MASV_VERBOSE_LEVEL"}; # 1
	$database_description = $ENV{"MASV_DATABASE_DESCRIPTION"}; # ${home_dir}/Diss/HTK/database_desc/veridat_raw/
# 


$sv_systems_dir= "${base_dir}sv_systems/";
$sign = "/";


##
## parallel processing values


@hosts = &read_allowed_hosts($host_list);
%hostInfos = %{&read_host_infos($host_infos)};

$numHosts_target = 5;





# default speakers
&init_speaker_sets("revertToStandardSpeakerSet");



##################
# general routines
##################

sub open_file {
	my ($filemode, $filename) = @_;

	my $fh = new FileHandle;
	open ($fh, $filemode . $filename) ||  die ("Cannot open $filename");
	return $fh;
}


sub open_file_if_new {
	my ($filemode, $filename) = @_;
	
	if (-e $filename) {
		return -1;
	}
	else {
		my $fh = new FileHandle;
		open ($fh, $filemode . $filename) ||  die ("Cannot open $filename");
		return $fh;
	}
}


sub change_matlab_calc_script {

	my ($script_name, $result_id) = @_;
	my $fh;
	
	my @file_content;
	$fh = &SR_lib::open_file("<", "$script_name");
	@file_content = <$fh>;
	close $fh;
	
	
	my $line;
	
	my $matlab_result_name;
	for $line (@file_content) {
		if ($line =~ /matlab_result_name="(.*)";/) {
			$matlab_result_name = $1;
			last;
		}
	}
	
	
	$fh = &SR_lib::open_file(">", "$script_name");
	foreach $line (@file_content) {
		$line =~ s/'${matlab_result_name}[^']*'/'${matlab_result_name}${result_id}'/;
		print $fh "$line";
	}
	close $fh;
}


sub read_host_infos {
	
	my $host_infos_file = $_[0];
	
	my %hostInfoHash;
	my @content;
	my $return_ref;
	
	my $fh = open_file("<", "$host_infos_file");
		@content = <$fh>;
	close $fh;
	
	foreach my $line (@content) {
		if ($line =~ /^([a-z]+\d+)\s+(\d+\.\d+)/) {
			my $hostname = $1;
			my $speed = $2;
			$hostInfoHash{"$hostname"} = $speed;
		}
	}
	
	$return_ref = {%hostInfoHash};
	return $return_ref;
}


sub read_allowed_hosts {
	
	my $allowed_hosts_list = $_[0];
	
	my @host_array;
	my @content;
	
	my $fh = open_file("<", "$allowed_hosts_list");
		@content = <$fh>;
	close $fh;
	
	my $host_counter = 0;
	foreach my $line (@content) {
		chomp($line);
		if ($line =~ /^[a-z]+\d+$/) {
			$host_array[$host_counter] = $line;
			$host_counter++;
		}
	}
	return @host_array;
}

sub test_containedInArray {
	my $testitem = $_[0];
	my $array_ref = $_[1];
	
	foreach my $item (@$array_ref) {
		if ($testitem eq $item) {
			return 1;
		}
	}
	return 0;
}


sub check_and_create_dir {
	my $dir_path = $_[0];

	if (!opendir(DIR, $dir_path))                  
	{
		if ($dir_path) {
			mkpath($dir_path)  || die "cannot make dir: $dir_path";
		}
	}
	closedir DIR;

}

sub create_force_alignment_templates {
	my $template_mlf = $_[0];
	my $S_list_file = $_[1];
	my $force_alignment_mlf = $_[2];
	my $S_list_allowed = $_[3];
	my $frame_duration = $_[4];
	my @template = ();
	my @allowed_array = ();
	
	my $fh = open_file("<", "$template_mlf");
		@template = <$fh>;
	close $fh;
	
	$fh = open_file("<", "$S_list_allowed");
		@allowed_array = <$fh>;
	close $fh;
	chomp(@allowed_array);
	
	my $fh_S_list = open_file(">", "$S_list_file");
	my $fh_fa_mlf = open_file(">", "$force_alignment_mlf");

	print $fh_fa_mlf $template[0];
	
	my $current_rec;
	my $current_rec_full_path;
	my $current_path;
	my $current_part;
	my $current_line;
	my $allowed_flag = 0;
	
	my @temp_path_fields;
	my @temp_path;
	
	my @rec_fields;
	my @temp_fields;
	
	my $begin_frame;
	my $end_frame;
	
	for (my $counter=1;$counter<=($#template - 1);$counter++) {
		$current_line = $template[$counter];
		if ($current_line =~ /"(.*)"/) {
			$current_rec_full_path = $1;
			$current_rec_full_path =~ s/\.rec/\.param/;
			if (&test_containedInArray($current_rec_full_path, \@allowed_array)) {
				$allowed_flag = 1;
				@temp_path_fields = split(/\//, $current_rec_full_path);
				$current_rec = $temp_path_fields[-1];

				@temp_path = split(/$current_rec/, $current_rec_full_path);
				$current_path = $temp_path[0];

				$current_part = 1;
			} 
			else {
				$allowed_flag = 0;
			}
		}
		elsif (($allowed_flag) && ($current_line =~ /^\d+\s+\d+/)) {
			@temp_fields = split (/\s+/,$current_line);
			@rec_fields = split (/\./,$current_rec);
			$begin_frame = int(($temp_fields[0] / $frame_duration)+0.5);
			$end_frame = int(($temp_fields[1] / $frame_duration)+0.5) - 1;
			print $fh_S_list "${current_path}${rec_fields[0]}_part${current_part}.param=${current_rec_full_path}", "[${begin_frame},${end_frame}]\n";
			
			print $fh_fa_mlf '"', "${current_path}${rec_fields[0]}_part${current_part}.lab", '"', "\n";
			print $fh_fa_mlf $temp_fields[2] , "\n";
			print $fh_fa_mlf ".\n";
			
			$current_part += 1;
		}
	
	}

	close $fh_S_list;
	close $fh_fa_mlf;
}


sub join_single_word_mlf {
	my $mlf_file = $_[0];
	my $time_offset = 0;
	my $internal_time_offset = 0; # fuer Woerter, die in mehrere Label expandiert werden
	
	my $begin_time;
	my $end_time;
	my @line_fields;
	
	
	my $fh = open_file("<", "$mlf_file");
	my @file_content = <$fh>;
	close $fh;
	
	$fh = open_file(">", "$mlf_file");
	print $fh $file_content[0];
	
	for (my $counter=1;$counter<=($#file_content - 1);$counter++) {
		if ($file_content[$counter + 1] =~ /part1\.rec/) {
			print $fh ".\n";
			$time_offset = 0;
			$internal_time_offset = 0;
			next;
		}
		if ($file_content[$counter] =~ s/_part1\.rec/\.rec/) {
			print $fh $file_content[$counter];
			next;
		}
		if ($file_content[$counter] =~ /^\d+\s+\d+/) {
			@line_fields = split (/\s+/,$file_content[$counter]);
			$begin_time = $line_fields[0];
			$end_time = $line_fields[1];
			$line_fields[0] = $begin_time + $time_offset - $internal_time_offset;
			$line_fields[1] = $end_time + $time_offset - $internal_time_offset;
			print $fh join(' ', @line_fields) , "\n";
			$time_offset = $line_fields[1];
			$internal_time_offset = $end_time;
			next;
		}
		if ($file_content[$counter] =~ /^\./) {
			$internal_time_offset = 0;
		}
		
	}
	print $fh ".\n";
	close $fh;
	
}

sub split_fileList {
	my $file = $_[0];
	my $numParts = int($_[1]);
	
	my @file_content;
	my $partList_name;
	my $part;
	my $part_counter;
	my @temp;
	
	
	# field-seperator neu setzen
	my $temp_field_seperator = $";
	$" = "";
		
	my $fh = open_file("<", "$file");
	@file_content = <$fh>;
	close $fh;
	
	my $block_length = int(scalar(@file_content) / $numParts);
	my $rest_length = scalar(@file_content) % $numParts;
	my $last_length = $rest_length + $block_length;
	
	my ($filename, $path) = fileparse($file);
	my $splitList_dir = &get_distList_dir($path . $sign . $filename);

	check_and_create_dir($splitList_dir);

	print STDERR "SR_lib::split_fileList: splitting file $filename into $numParts parts.\n";
	
	for ($part_counter = 1; $part_counter < $numParts; $part_counter++) {
		$partList_name = &get_distList_name($part_counter);
		$part = "$splitList_dir" . $partList_name;
		$fh = open_file(">", "$part");
		@temp = splice(@file_content, 0, $block_length);
		print $fh "@temp";
		close $fh;
	}
	
	$partList_name = &get_distList_name($part_counter);
	$part = "$splitList_dir" . $partList_name;
	$fh = open_file(">", "$part");
	@temp = splice(@file_content, 0, $last_length);
	print $fh "@temp";
	close $fh;	
	
	$" = $temp_field_seperator;
}

sub get_distList_dir {
	my $full_path = $_[0];
	
	my $splitList_dir = "${full_path}_splitLists${sign}";
	
	return $splitList_dir;
}

sub get_distList_name {
	my $part = $_[0];

	$part = "part${part}";
	
	return $part;
}



sub split_arrayList {
	my $arrayref = $_[0];
	my $numParts = int($_[1]);
	
	my @temp_array = @$arrayref;
	
	my @return_array;
	my $part_counter;		

	
	my $block_length = int(scalar(@temp_array) / $numParts);
	my $rest_length = scalar(@temp_array) % $numParts;
	my $last_length = $rest_length + $block_length;
	
	if ($block_length == 0) {
		$part_counter = 0;		
		for ($part_counter = 0; $part_counter < ($last_length); $part_counter++) {
			$return_array[$part_counter] = [ splice(@temp_array, 0, 1) ];
		}
	}
	else {
		for ($part_counter = 0; $part_counter < ($numParts - 1); $part_counter++) {
			$return_array[$part_counter] = [ splice(@temp_array, 0, $block_length) ];
		}
		$return_array[$part_counter] = [ splice(@temp_array, 0, $last_length) ];
		$part_counter++;
	}
	return ($part_counter, @return_array);
	
}


sub sort_and_uniq {
	my $arrayref = $_[0];
	
	@$arrayref=sort(@$arrayref);

	my $prev_item = @$arrayref[0];	
	for (my $item_counter=1; $item_counter < scalar(@$arrayref); $item_counter++) {
		if ($prev_item eq @$arrayref[$item_counter]) {
			splice(@$arrayref, $item_counter, 1);
			$item_counter--;
		}
		$prev_item = @$arrayref[$item_counter];

	}

}


sub sorting_num { 
	my $retval;

	if ($a > $b) { 
		$retval = 1;
	} 
	elsif ($a == $b) {
		$retval = 0;
	} 
	else { 
		$retval = -1;
	}
	$retval; 
}



sub par_get_available_hosts {

	my %hostHash;
	
	my @selectedHosts;
	$selectedHosts[0]='';
	
	# check, which hosts are alive	
	
	my $ls_dir = $ENV{"HOME"};

	print STDERR "SR_lib::par_get_available_hosts: checking hosts in hostlist\n";
	
	my $host_counter = 0;
	foreach my $host (@hosts) {
		my $ssh_stat;
		my $ping_stat = system("ping -c 1 -w 3 $host >/dev/null");
		if ($ping_stat == 0) {
			$ssh_stat = system("ssh -x $host ls $ls_dir >/dev/null");
			if ($ssh_stat == 0) {
				my $load_string = `ssh -x $host uptime`;
				$load_string =~ /(\d+\.\d+),\s+(\d+\.\d+)/;
				my $short_time_load = $1;
				my $mid_time_load = $2;
				my $load = ($short_time_load + $mid_time_load)/2;
				my $speed = (defined($hostInfos{"$host"})) ? $hostInfos{"$host"} : 900;
				
				$load = ( ( $load + 0.6 ) / $speed ) + ( 0.4 / 700 ); # 60% of the job is dependent on speed of proc., 40 % of job is I/O
				
				if (defined($hostHash{"$load"})) {
					$hostHash{"$load"} = $hostHash{"$load"} . "," . $host;
				}
				else {
					$hostHash{"$load"} = $host;
				}
				if ($verbose_level >= 2) { 
					print STDERR "SR_lib::par_get_available_hosts: $host is alive;\n";
					print STDERR "                                 current mean load: $load\n";
				}	
			}
		}
	}
	
	# choose best ones
	my @loadList = sort sorting_num (keys(%hostHash));
	
	# select from this list from the top
	$host_counter = 0;
JUMP:foreach my $load (@loadList) {
		my $host_group = $hostHash{"$load"};
		my @single_hosts = split( ',', $host_group);
		for my $host (@single_hosts) {
			push(@selectedHosts, $host);
			if ($verbose_level >= 1) { 
				print STDERR "SR_lib::par_get_available_hosts: adding $host to hostlist. (current load $load)\n";
			}
			$host_counter++;
			if ($host_counter == $numHosts_target) {
				last JUMP;
			}
		}
	}
	
	my $hostsNum = $host_counter;
	if (($hostsNum != $numHosts_target) && ($hostsNum != 0)) {
		print STDERR "SR_lib::par_get_available_hosts: only $hostsNum hosts are available.\n";
	}
	elsif ($hostsNum == 0) {
		print STDERR "No hosts available, reverting to localhost...\n";
		$selectedHosts[1]=$ENV{"HOSTNAME"};
		$hostsNum = 1;
	}
	
	return ($hostsNum, @selectedHosts);

}

sub processConfigurations{

	my ($testModelConf, $crossTestConf,$worldListConf);

	$testModelConf = $_[0];
	
	if ($_[1]) {
		$crossTestConf = $_[1];
	}
	if ($_[2]) {
		$worldListConf = $_[2];
	}

	my $testModelArrRef;
	my $testModelSessions;
	my $crossTestList;
	my $crossTestArrRef;
	my $crossTestSessions;
	my $worldModelArrRef;
	my $worldModelSessions;
 	
 	my @temp;	
 	
 	@temp = split(/\//, $testModelConf);
	$testModelArrRef = &get_speakerArrRef($temp[0]);
 	$testModelSessions = &MASV_db_desc::get_SessionsListName($temp[1]);
 	
 	if ($crossTestConf) {
	 	@temp = split(/\//, $crossTestConf);
 		$crossTestList = $temp[0];
		$crossTestArrRef = &get_speakerArrRef($temp[0]);
 		$crossTestSessions = &MASV_db_desc::get_SessionsListName($temp[1]);
 	}
 	else {
 		$crossTestArrRef = [()];
 		$crossTestSessions = "";
	} 		

 	if ($worldListConf) {
 		@temp = split(/\//, $worldListConf);
		$worldModelArrRef = &get_speakerArrRef($temp[0]);
 		$worldModelSessions = &MASV_db_desc::get_SessionsListName($temp[1]);
 	}
	else {
	 	$worldModelArrRef = [()];
 		$worldModelSessions = "";
 	}

 	return($testModelArrRef, $testModelSessions, 
 		$crossTestArrRef, $crossTestSessions,
 		$worldModelArrRef, $worldModelSessions);

	
	sub get_speakerArrRef {
		my $listidentifier = $_[0];
		my $arrayref;
		
		if ((length($listidentifier) == 4) && (substr($listidentifier,0,1) eq "0") ){
			$arrayref = [($listidentifier)];
		}
		else {
			if (!(defined($speakerlist{$listidentifier}))) {
				die ("SR_lib: error in processConfigurations; no speakerlist $listidentifier defined!\n");
			}
			$arrayref = [@{$speakerlist{$listidentifier}}];
		}
		return $arrayref;
	}
 		
}


sub getLabels {
  my ($source_dir, $speaker_dir , $ses_dir , $base_src_file) = @_;
  my $prompt_labels_ref;
  my $ses_labels_ref;

  ($prompt_labels_ref, $ses_labels_ref) = &MASV_db_desc::read_db_label("${source_dir}${SR_lib::sign}${speaker_dir}${SR_lib::sign}${ses_dir}${SR_lib::sign}${base_src_file}.$MASV_db_desc::label_file_ext");
  return ($prompt_labels_ref, $ses_labels_ref);
}


sub check_for_noises {
	my ($source_dir, $speaker_dir , $ses_dir , $base_src_file) = @_;
	my $label_file_handle;
	my @label_file_content;
	my $return_value = 1; # 1=free of noise labels , 0=noise labels contained
	
	
	my ($prompt_labels_ref, $ses_labels_ref) = &SR_lib::getLabels($source_dir, $speaker_dir , $ses_dir , $base_src_file);
		
	my $theLabelStr = join(",", @{$ses_labels_ref});
	foreach my $theNoise (@MASV_db_desc::NoisesArray) {
		if (($theLabelStr =~ /$theNoise/) ) {
			$return_value = 0;
			last;			# leave further filtering, we found an item
		}
	}
	
	
	return $return_value;
	
}


sub read_trainingInfoFile {
	my $path = $_[0];

	my $file_handle = &SR_lib::open_file("<", "${path}/training.info");
	my @file_content = <$file_handle>;
	close $file_handle;
	chomp($file_content[0]);
	return $file_content[0]; # erste Zeile
}



sub write_trainingInfoFile {
	my $path = $_[0];
	my $sessionList = $_[1];
	
	my $file_handle = &SR_lib::open_file(">", "${path}/training.info");
	print $file_handle $sessionList , "\n";
	close $file_handle;
	
}



sub run_and_check_command {
		my $command = $_[0];
		my $log_handle = $_[1];
		
		my $error_state = 0;
		
		if ($log_handle) {
			print $log_handle "executing $command \n\n";
		}
		
		my $pm = new Parallel::ForkManager(1);
		
		$pm->run_on_finish(
			sub { my ($pid, $exit_code) = @_;
				  $error_state = $exit_code;
			}
		);
		
		my $pid = $pm->start;
		
		if ($pid != 0) {
			$error_state = system($command);
			
			if ($error_state) {
				$error_state = 11;
			}
		}
		$pm->finish($error_state);

		$pm->wait_all_children;

		if ($error_state) {
			if ($log_handle) {
				print $log_handle "Error while executing $command\n";
				close $log_handle;
			}
			else {
				print STDERR "Error while executing $command\n";
			}
			exit 1;
		}
}


sub read_pool_config {

	my $sv_system_name = $_[0];
	
	my $sv_system_full_path;
	my $pool_config_full_path;
	my $pool_config_handle;
	my @poolConfigContent;
	
	my $hash_ref;
		
	$sv_system_full_path = $SR_lib::sv_systems_dir . $sv_system_name;

	# read speaker set
	$pool_config_full_path = $sv_system_full_path . ${SR_lib::sign} . "links" . ${SR_lib::sign} . "currentParamPool";
	if (!(-l $pool_config_full_path )) {
		print STDERR "SR_lib.pm error: no currentParamPool link found!\n";
		exit 1;
	}
	$pool_config_full_path .= ${SR_lib::sign} . "configs";
	# search first config C_ ...
	opendir INDIR, $pool_config_full_path;
	my @configs_list = grep(/^C_/, readdir (INDIR));
	closedir INDIR;
	
	if ((scalar @configs_list) == 0) {
		print STDERR "SR_lib.pm error: no config in current paramPool found!\n";
		exit 1;
	}

	$pool_config_full_path .= ${SR_lib::sign} . $configs_list[0];
	
	
	
	$pool_config_handle = &open_file("<", $pool_config_full_path);
	@poolConfigContent = <$pool_config_handle>;
	close $pool_config_handle;
	
	foreach (@poolConfigContent) {
		my @items=split(/=/, $_);
		my $parameter = $items[0];
		my $value = $items[1];
		$parameter =~ s/\s+//g;
		$value =~ s/\s+//g;
		if ( ($parameter) && ($value) ) {
			${$hash_ref}{$parameter}=$value;
		}
	}
	
	return $hash_ref;
	
	
}


sub init_speaker_sets {

	my $sv_system_name = $_[0];
	
	my $sv_system_full_path;
	my $spkset_full_path;
	my $spkset_handle;
	my @spksetContent;
	
	if ($sv_system_name eq "revertToStandardSpeakerSet") {
		$spkset_full_path = $base_dir . "templates" . ${SR_lib::sign} . "speaker_sets" . ${SR_lib::sign} ."standard";
	}
	else {
		
		$sv_system_full_path = $SR_lib::sv_systems_dir . $sv_system_name;
	
		# read speaker set
		$spkset_full_path = $sv_system_full_path . ${SR_lib::sign} . "links" . ${SR_lib::sign} . "currentSpeakerSet";
		if (!(-l $spkset_full_path )) {
			print STDERR "SR_lib.pm warning: no currentSpeakerSet link found, reverting to standard set.\n";
			$spkset_full_path = $sv_system_full_path . ${SR_lib::sign} . "speaker_sets" . ${SR_lib::sign} . "standard";
			if (!(-e $spkset_full_path)) {
				print STDERR "SR_lib.pm warning: no standard speaker set found, reverting to standard set in templates/speaker_sets/!\n";
				$spkset_full_path = $base_dir . "templates" . ${SR_lib::sign} . "speaker_sets" . ${SR_lib::sign} . "standard";
				if (!(-e $spkset_full_path)) {
					print STDERR "SR_lib.pm: no speaker set found, there's even no standard set in templates/speaker_sets/!\n";
					exit 1;
				}
			}
		}
	
	}
	
	
	$spkset_handle = &open_file("<", $spkset_full_path);
	@spksetContent = <$spkset_handle>;
	close $spkset_handle;
	
	foreach (@spksetContent) {
		eval($_);
		unless ($@ eq "") {
			die "SR_lib.pm: error in speaker set file $spkset_full_path :\nline $_ \nerror: $@ \n\n";
		}
	}
	
	
	
	@test_set = @{$speakerlist{test_set}};
	@training_set = @{$speakerlist{training_set}};
	@world_set = @{$speakerlist{world_set}};
	@dev_set = @{$speakerlist{dev_set}};
	
	$speakerlist{all_tt} = [@training_set, @test_set];
	&SR_lib::sort_and_uniq($speakerlist{all_tt});
	$speakerlist{all_test} = [@test_set];
	$speakerlist{all_training} = [@training_set];
	$speakerlist{all_hnorm} = [@training_set, @dev_set];
	&SR_lib::sort_and_uniq($speakerlist{all_hnorm});
	
	@all = ();
	push(@all, @test_set, @training_set, @world_set, @dev_set);
	$speakerlist{all} = [@all];
	&SR_lib::sort_and_uniq($speakerlist{all});

}


sub create_multispeaker_list {
	my $sv_system_name = $_[0];
	my $config_arr_ref = $_[1];
	
	my ($exec_command, $raw_result);
	
	my $sv_system_path = $sv_systems_dir . $sv_system_name;
	
	my $temp_dir = cwd;
	chdir $sv_system_path;

	$exec_command = "rm -f tmp/current_multispk_list";
	$raw_result = `$exec_command`;
	$exec_command = "touch tmp/current_multispk_list";
	$raw_result = `$exec_command`;
	
	
	foreach my $single_conf (@{$config_arr_ref}) {
		my ($SpkArrRef, $Sessions) 
		 = 
		 &SR_lib::processConfigurations($single_conf);
		my @temp_dirs = @{$SpkArrRef};
		foreach my $current_dir (@temp_dirs) {
			$exec_command = "cat ${current_dir}/$Sessions >> tmp/current_multispk_list";
			$raw_result = `$exec_command`;
		}
	}
	
	chdir $temp_dir;
}


sub get_filename_multispeaker_list {
	
	my $sv_system_name = $_[0];
	
	my $sv_system_path = $sv_systems_dir . $sv_system_name;
	
	return ($sv_system_path . $sign . "tmp" . $sign . "current_multispk_list");
}


sub get_no_entries_multispeaker_list {

	my $sv_system_name = $_[0];

	my ($exec_command, $raw_result, @content, $no_entries);


	my $sv_system_path = $sv_systems_dir . $sv_system_name;

	my $temp_dir = cwd;
	chdir $sv_system_path;
	
	$exec_command = "wc -l tmp/current_multispk_list";
	$raw_result = `$exec_command`;
	@content = split('\s+', $raw_result);
	$no_entries = $content[1];
	
	$exec_command = "rm -f tmp/current_multispk_list";
	$raw_result = `$exec_command`;

	chdir $temp_dir;
	
	return $no_entries;
	
}


sub remove_multispeaker_list {
	
	my $sv_system_name = $_[0];

	my ($exec_command, $raw_result);

	my $sv_system_path = $sv_systems_dir . $sv_system_name;

	my $temp_dir = cwd;
	chdir $sv_system_path;

	$exec_command = "rm -f tmp/current_multispk_list";
	$raw_result = `$exec_command`;

	chdir $temp_dir;
}


sub read_hmm_prototype {
		my $file = $_[0];
		my $file_handle;
		my @file_content;
		my $num_of_states;
		
		$file_handle = &SR_lib::open_file("<", "$file");
		@file_content = <$file_handle>;
		close $file_handle; 
		
		foreach my $line (@file_content) {
			if ($line =~ /<NumStates>\s*(\d+)/) {
				# num of states
				$num_of_states = $1;
				last;
			}		
		}
		
		return $num_of_states;

}



sub create_prototype_from_template {
		my $path = $_[0];
		my $new_prototype = $_[1];
		my $template_prototype = $_[2];
		my $file_handle;
		my @file_content;
		
		
		$file_handle = &SR_lib::open_file("<", "$path$SR_lib::sign$template_prototype");
		@file_content = <$file_handle>;
		close $file_handle; 
		
		
		grep { s/~h\s+"$template_prototype"/~h "$new_prototype"/ } @file_content;
		$" = "";	
		$file_handle = &SR_lib::open_file(">", "$path$SR_lib::sign$new_prototype");
		print $file_handle "@file_content";
		close $file_handle; 
}


1;

