#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: create_MASV_sv_system.pl,v 1.5 2004/03/07 10:52:58 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';

$|=1; #Force buffer flush on STDOUT

use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;


use vars qw(%label_StatisticHash
			%label_training_StatisticHash
			%label_test_StatisticHash
			%label_StatisticHash_speaker
			%label_training_StatisticHash_speaker
			%label_test_StatisticHash_speaker
			%training_sessions_array_refs
			%evaluate_sessions_array_refs
);

###########################

###########################
# Default values for options
my $help = 0;
my $Identify = 0;
my @typearray = ();
my @envarray =();
my @speakerarray = ();
my @speakertestarray = ();
my @filterNoisesArray=();
my $makeMLFs = '';
my $createStatisticFile = '';
my $skipInterruptedSessions ='';
my $create_only_script_files = '';

my %training_sessions_array_refs;
my %evaluate_sessions_array_refs;

my @selected_sessions_array;
my @temp_join_sessions_array;

my $mlf_realized_handle;
my $mlf_prompted_handle;

my $statistic_handle;
my $speaker_statistic_handle;
###########################

%label_StatisticHash = ();
%label_training_StatisticHash = ();
%label_test_StatisticHash = ();
%label_StatisticHash_speaker = ();
%label_training_StatisticHash_speaker = ();
%label_test_StatisticHash_speaker = ();



# Processing Options

GetOptions('t|typesession=s' => \@typearray, 's|speakers=s' => \@speakerarray,
	       'e|environment=s' => \@envarray,
           'm|makeMLFs' => \$makeMLFs, 'f|filterNoises=s' => \@filterNoisesArray,
           'c|createStatisticFile' => \$createStatisticFile,
           'i|skipInterruptedSessions' => \$skipInterruptedSessions,
		   'onlyScripts=s' => \$create_only_script_files,
           'help|?' => \$help,
           'version' => \$Identify);

if ($Identify) {
	printf "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if ((@ARGV != 2) || ($help)) {
   pod2usage(1);
   exit 1;
 }

my $sv_system_name = $ARGV[0];
my $poolname = $ARGV[1];
###########################

my $source_param_dir = "${SR_lib::paramPool_dir}${poolname}${SR_lib::sign}";
my $link_param_dir = "${SR_lib::paramPool_dir}${poolname}";
my $sv_system_path = "${SR_lib::sv_systems_dir}${sv_system_name}${SR_lib::sign}";
my $log_path = "${sv_system_path}log${SR_lib::sign}";
my $link_name = "${sv_system_path}links/currentParamPool";


unless (-e $source_param_dir) {
	die("\nThere's no pool $source_param_dir \n\n");
}


if (!@typearray) {
	@typearray = ('P[1-7]');
}

if (!@envarray) {
	@envarray =('all');
}


# generate list of session types
@typearray = split(/,/,join(',',@typearray));

# generate list of speakers
@speakerarray = split(/,/,join(',',@speakerarray));

foreach my $speakerlist_id (@speakerarray) {
   push (@speakertestarray, @{$SR_lib::speakerlist{$speakerlist_id}});
}
if (!@speakertestarray) {
   @speakertestarray = ('all');
}

# generate list of filters for Labels
@filterNoisesArray = split(',' ,join(',',@filterNoisesArray)); 


# generate list of test/training sessions
@envarray = split(/,/,join(',',@envarray));


foreach my $envlist (@envarray) {
   $training_sessions_array_refs{$envlist} = $MASV_db_desc::training_sessions_lists{$envlist};
   $evaluate_sessions_array_refs{$envlist} = $MASV_db_desc::evaluate_sessions_lists{$envlist};
}


foreach my $envlist (@envarray) {
	push(@selected_sessions_array, @{$training_sessions_array_refs{$envlist}}, @{$evaluate_sessions_array_refs{$envlist}});
}
my @all_sessions = @MASV_db_desc::all_sessions;


opendir SRCDIR, $source_param_dir;
my @speaker_dirs = grep(/^\d{4}$/, readdir (SRCDIR));
closedir SRCDIR;
@speaker_dirs = sort(@speaker_dirs);

#check Paths
&SR_lib::check_and_create_dir("${SR_lib::matlab_path}calculation_scripts/temp_scripts");

&SR_lib::check_and_create_dir($sv_system_path);
&SR_lib::check_and_create_dir("${sv_system_path}tmp");
&SR_lib::check_and_create_dir("${sv_system_path}protoconfs");

&SR_lib::check_and_create_dir("${sv_system_path}expConfigs");
&SR_lib::check_and_create_dir("${sv_system_path}scripts");
&SR_lib::check_and_create_dir("${sv_system_path}links");
&SR_lib::check_and_create_dir("${sv_system_path}speaker_sets");
&SR_lib::check_and_create_dir("${sv_system_path}speaker_set_lists");
&SR_lib::check_and_create_dir("${sv_system_path}feature_sets");

&SR_lib::check_and_create_dir($log_path);

my $log_handle = &SR_lib::open_file(">>", "$log_path" . "log.txt");
print $log_handle "\n\n";
print $log_handle scalar localtime, "\n";
print $log_handle "Running create_MASV_experiment.pl .....\n";
print $log_handle "------------------------------------------\n";
print $log_handle "Parameters:\n";
print $log_handle "sv_system_name           : $sv_system_name\n";
print $log_handle "poolname                 : $poolname\n";
print $log_handle "Options\n";
print $log_handle "typesessions             : @typearray\n";
print $log_handle "speakers                 : @speakerarray\n";
print $log_handle "environment              : @envarray\n";
print $log_handle "filterNoisesArray        : @filterNoisesArray\n";
print $log_handle "makeMLFs                 : $makeMLFs\n";
print $log_handle "createStatisticFile      : $createStatisticFile\n";
print $log_handle "skipInterruptedSessions  : $skipInterruptedSessions\n\n";
print $log_handle "onlyScripts              : $create_only_script_files\n\n";
print $log_handle "cvs_version              : $CVS_Version_String\n";
print $log_handle "cvs_name                 : $CVS_Name_String\n";


if ($makeMLFs) {
	if (!($create_only_script_files)) {
		$mlf_realized_handle = &SR_lib::open_file(">", "$sv_system_path" . "mlf_realized.mlf");
	}
	else {
		$mlf_realized_handle = &SR_lib::open_file(">", "$sv_system_path" . "mlf_realized_for_${create_only_script_files}.mlf");
	}
	print $mlf_realized_handle "#!MLF!#\n";
}


my $progress_value_total = scalar(@speaker_dirs);
my $progress_counter = 0;
print "\n";

foreach my $speaker_allowed (@speakertestarray) {
	if (!($speaker_allowed eq "all") && !(&SR_lib::test_containedInArray($speaker_allowed, \@speaker_dirs)) ) {
		print "$speaker_allowed not available in pool $source_param_dir , update necessary ?\n";
	}
}


my $use_speaker = 0;
my $first_selected_speaker = '';

SPKLOOP: foreach my $speaker_dir (@speaker_dirs) {

	$progress_counter++;
	print "Current speaker: $speaker_dir |";
	my $progress_display_border = int ($progress_counter / $progress_value_total * 10);
	for (my $progress_display_counter = 0; $progress_display_counter < $progress_display_border; $progress_display_counter++) {
		print "*";
	}
	for (my $progress_display_counter = $progress_display_border; $progress_display_counter <= 10; $progress_display_counter++) {
		print " ";
	}
	print "|\r";

	
	$use_speaker = 0;
	
	if (&SR_lib::test_containedInArray("all", \@speakertestarray)) {
		$use_speaker = 1;
	}
	else {
		$use_speaker = &SR_lib::test_containedInArray($speaker_dir, \@speakertestarray);
	}
	
	next SPKLOOP if (!($use_speaker));
	
	
	
	$first_selected_speaker = ($first_selected_speaker) ? $first_selected_speaker : $speaker_dir;
	ENVLIST: foreach my $envlist (@envarray) {
		#check Path
		my $out_speaker_path=$sv_system_path.$speaker_dir;
		&SR_lib::check_and_create_dir($out_speaker_path);
		
			
		my $testlist_handle;
		my $traininglist_handle;
		my $completelist_handle;
		
		
		if ($create_only_script_files) { # check if S_list file already exists -> do not overwrite
			$testlist_handle = &SR_lib::open_file_if_new(">", "$out_speaker_path${SR_lib::sign}S_list_${envlist}_evaluate");
			if ($testlist_handle == -1)  {
				next ENVLIST;
			}
			$traininglist_handle = &SR_lib::open_file_if_new(">", "$out_speaker_path${SR_lib::sign}S_list_${envlist}_training");
			if ($traininglist_handle == -1)  {
				close $testlist_handle;
				next ENVLIST;
			}
			$completelist_handle = &SR_lib::open_file_if_new(">", "$out_speaker_path${SR_lib::sign}S_list_${envlist}");
			if  ($completelist_handle == -1) {
				close $testlist_handle;
				close $traininglist_handle;
				next ENVLIST;
			}
		}
		else {
			$testlist_handle = &SR_lib::open_file(">", "$out_speaker_path${SR_lib::sign}S_list_${envlist}_evaluate");
			$traininglist_handle = &SR_lib::open_file(">", "$out_speaker_path${SR_lib::sign}S_list_${envlist}_training");
			$completelist_handle = &SR_lib::open_file(">", "$out_speaker_path${SR_lib::sign}S_list_${envlist}");
		}

		if (($createStatisticFile) && ($skipInterruptedSessions || ($speaker_dir eq $first_selected_speaker))) {
			$speaker_statistic_handle = &SR_lib::open_file(">", "$out_speaker_path${SR_lib::sign}prompt_statistic_${envlist}.txt");
		}

		
		opendir SRCDIR, "$source_param_dir${SR_lib::sign}$speaker_dir";
		my @ses_dirs = grep ( /\d{2}/o, readdir (SRCDIR));
		closedir SRCDIR;
		
		foreach my $ses_dir (@ses_dirs) {
		
			opendir SRCDIR, "${source_param_dir}${SR_lib::sign}${speaker_dir}${SR_lib::sign}${ses_dir}";
			my @src_files = grep(/param$/o, readdir(SRCDIR));
			close SRCDIR;
			
			foreach my $src_file (@src_files) {
				foreach my $session_type (@typearray) {
					if ($src_file =~ /(.*$session_type)\.param$/) {
						my $base_src_file = $1;
						
						if ((&SR_lib::check_for_noises(${SR_lib::source_trans_dir}, ${speaker_dir}, ${ses_dir}, ${base_src_file})) || !($skipInterruptedSessions)) {
							if (&SR_lib::test_containedInArray($ses_dir, $evaluate_sessions_array_refs{$envlist})) {
								print $testlist_handle "$link_name" . ${SR_lib::sign} . "$speaker_dir" ."$SR_lib::sign" . "$ses_dir" . "$SR_lib::sign" . "$base_src_file" . ".param\n";
							}
						 
							if (&SR_lib::test_containedInArray($ses_dir, $training_sessions_array_refs{$envlist})) {
								print $traininglist_handle "$link_name" . ${SR_lib::sign} . "$speaker_dir" ."$SR_lib::sign" . "$ses_dir" . "$SR_lib::sign" . "$base_src_file" . ".param\n";
							}
							if (&SR_lib::test_containedInArray($ses_dir, $training_sessions_array_refs{$envlist}) || &SR_lib::test_containedInArray($ses_dir, $evaluate_sessions_array_refs{$envlist})) {
								print $completelist_handle "$link_name" . ${SR_lib::sign} . "$speaker_dir" ."$SR_lib::sign" . "$ses_dir" . "$SR_lib::sign" . "$base_src_file" . ".param\n";
							}
							
							my ($prompt_labels_ref, $ses_labels_ref) = &SR_lib::getLabels($SR_lib::source_trans_dir, $speaker_dir , $ses_dir , $base_src_file);
							&count_prompted_words($prompt_labels_ref, $training_sessions_array_refs{$envlist}, $evaluate_sessions_array_refs{$envlist}, $ses_dir);
							
							if ($makeMLFs && ($envlist eq $envarray[0])) { # mlf nur erstellen, wenn im ersten Durchlauf von foreach $envlist....
								&writeMLFs($SR_lib::source_trans_dir, $speaker_dir , $ses_dir , $base_src_file, $prompt_labels_ref, $ses_labels_ref);
							}
						
						}
						else
						{
						  print "skipped ${speaker_dir}${SR_lib::sign}${ses_dir}${SR_lib::sign}${base_src_file}; noise or interruptions contained. \n";
						  print $log_handle "skipped ${speaker_dir}${SR_lib::sign}${ses_dir}${SR_lib::sign}${base_src_file}; noise or interruptions contained. \n";
						
						}
					}	
				}
			} # end foreach $src_file (@src_files)
			
		} # end foreach $ses_dir (@ses_dirs)
		
		if (($createStatisticFile) && ($skipInterruptedSessions || ($speaker_dir eq $first_selected_speaker))) {
			&writespeakerStatisticFile();
			close $speaker_statistic_handle;
			
			# label-Hashes zurcksetzen fr neuen Durchgang von foreach $envlist
			%label_StatisticHash_speaker = ();
			%label_training_StatisticHash_speaker = ();
			%label_test_StatisticHash_speaker = ();
		
		}
		
		
		close $testlist_handle;
		close $traininglist_handle;
		close $completelist_handle;
		if (($createStatisticFile) && (($speaker_dir eq $first_selected_speaker))) {
			$statistic_handle = &SR_lib::open_file(">", "$sv_system_path" . "general_prompt_statistic_${envlist}.txt");
			print $statistic_handle ("environment: $envlist\n");
			&writeGlobalStatisticFile();
			close $statistic_handle;
		}
   } # end foreach $envlist


} # end foreach $speaker_dir (@speaker_dirs) 



close $log_handle;




#############################
# globales gepromptetes MLF
#############################
if ($makeMLFs) {
	if (!($create_only_script_files)) {
		$mlf_prompted_handle = &SR_lib::open_file(">", "$sv_system_path" . "mlf_prompted_expanded.mlf");
	}
	else {
		$mlf_prompted_handle = &SR_lib::open_file(">", "$sv_system_path" . "mlf_prompted_expanded_for_${create_only_script_files}.mlf");
	}
	print $mlf_prompted_handle "#!MLF!#\n";

    my $speaker_dir=$speaker_dirs[0];
    opendir SRCDIR, "${source_param_dir}${SR_lib::sign}$speaker_dir";
    my @ses_dirs = grep ( /\d{2}/o, readdir (SRCDIR));
    closedir SRCDIR;
    foreach my $ses_dir (@ses_dirs) {
		opendir SRCDIR, "${source_param_dir}${SR_lib::sign}${speaker_dir}${SR_lib::sign}${ses_dir}";
		my @src_files = grep(/param$/o, readdir(SRCDIR));
		close SRCDIR;
		foreach my $src_file (@src_files) {
			foreach my $session_type (@typearray) {
				if ($src_file =~ /(.*$session_type)\.param$/) {
					my $base_src_file = $1;
					&writePromptedMLF($SR_lib::source_trans_dir, $speaker_dir , $ses_dir , $base_src_file);
				}
			}
		}
    }
  	close $mlf_prompted_handle;
  	
	close $mlf_realized_handle;
	
	if (!($create_only_script_files)) {
		my $mlf_gmm_handle = &SR_lib::open_file(">", "$sv_system_path" . "mlf_gmm.mlf");
		print $mlf_gmm_handle "#!MLF!#\n";
		print $mlf_gmm_handle "\"*/*/????????.lab\"\n";
		print $mlf_gmm_handle "gmmstate\n";
		print $mlf_gmm_handle ".\n";
		close $mlf_gmm_handle;
	}
}

if (!($create_only_script_files)) {
	my $exec_string = "ln -s ${link_param_dir} ${link_name}";
	system($exec_string);
	
	$exec_string = "cp ${SR_lib::base_dir}templates/speaker_sets/standard ${sv_system_path}speaker_sets/";
	system($exec_string);
	
	$exec_string = "ln -s ${sv_system_path}speaker_sets/standard ${sv_system_path}links/currentSpeakerSet";
	system($exec_string);
	
	$exec_string = "cp ${SR_lib::base_dir}templates/protoconfs/* ${sv_system_path}protoconfs/";
	system($exec_string);
	
	$exec_string = "cp ${SR_lib::base_dir}templates/expConfigs/* ${sv_system_path}expConfigs/";
	system($exec_string);
}

exit 0;
 
 
 

sub writeGlobalStatisticFile {
  	print $statistic_handle ("total set\n");
  	foreach my $word (keys(%label_StatisticHash)) {
  		print $statistic_handle ("${word}: $label_StatisticHash{$word}\n");
  	}
  	print $statistic_handle "\n";
  	print $statistic_handle ("training set + evaluate set:\n");
  	foreach my $word (keys(%label_training_StatisticHash)) {
  		my $sum = $label_training_StatisticHash{$word} + $label_test_StatisticHash{$word};
  		print $statistic_handle ("${word}: $sum\n");
  	}
	print $statistic_handle "\n";  	
  	print $statistic_handle ("training set:\n");
  	foreach my $word (keys(%label_training_StatisticHash)) {
  		print $statistic_handle ("${word}: $label_training_StatisticHash{$word}\n");
  	}
  	print $statistic_handle "\n";
  	print $statistic_handle ("evaluate set:\n");
  	foreach my $word (keys(%label_test_StatisticHash)) {
  		print $statistic_handle ("${word}: $label_test_StatisticHash{$word}\n");
  	}
  	%label_StatisticHash = ();
	%label_training_StatisticHash = ();
	%label_test_StatisticHash = ();

}


sub writespeakerStatisticFile {
  	print $speaker_statistic_handle ("total set\n");
  	foreach my $word (keys(%label_StatisticHash_speaker)) {
  		print $speaker_statistic_handle ("${word}: $label_StatisticHash_speaker{$word}\n");
  	}
  	print $speaker_statistic_handle "\n";
  	print $speaker_statistic_handle ("training set + evaluate set:\n");
  	foreach my $word (keys(%label_training_StatisticHash_speaker)) {
  		my $sum = $label_training_StatisticHash_speaker{$word} + (defined($label_test_StatisticHash_speaker{$word}) ? 				$label_test_StatisticHash_speaker{$word} : 0);
  		print $speaker_statistic_handle ("${word}: $sum\n");
  	}
	print $speaker_statistic_handle "\n";  	
  	print $speaker_statistic_handle ("training set:\n");
  	foreach my $word (keys(%label_training_StatisticHash_speaker)) {
  		print $speaker_statistic_handle ("${word}: $label_training_StatisticHash_speaker{$word}\n");
  	}
  	print $speaker_statistic_handle "\n";
  	print $speaker_statistic_handle ("evaluate set:\n");
  	foreach my $word (keys(%label_test_StatisticHash_speaker)) {
  		print $speaker_statistic_handle ("${word}: $label_test_StatisticHash_speaker{$word}\n");
  	}
}


sub count_prompted_words {
#############################################
# individuelle Statistik erstellen
#############################################
my ($prompt_labels_ref,  $training_ref, $evaluate_ref, $ses_dir) = @_;

 foreach (@$prompt_labels_ref) {
    if ($createStatisticFile) {
      my $promptString = $_;
      foreach my $word (@{&MASV_db_desc::convert_prompt_string($promptString)}) {
		$label_StatisticHash_speaker{$word}++;						
		$label_StatisticHash{$word}++;						
      }
      if (&SR_lib::test_containedInArray($ses_dir, $evaluate_ref)) {
		foreach my $word (@{&MASV_db_desc::convert_prompt_string($promptString)}) {
		  $label_test_StatisticHash_speaker{$word}++;							
		  $label_test_StatisticHash{$word}++;							
		}
      }
      if (&SR_lib::test_containedInArray($ses_dir, $training_ref)) {
		foreach my $word (@{&MASV_db_desc::convert_prompt_string($promptString)}) {
		  $label_training_StatisticHash_speaker{$word}++;							
		  $label_training_StatisticHash{$word}++;							
		}						
      }
    }
  } # end of foreach (@$prompt_labels_ref)

}


sub writeMLFs {
  my ($source_dir, $speaker_dir , $ses_dir , $base_src_file, $prompt_labels_ref, $ses_labels_ref) = @_;
  my $theFilter;
  my $theLabel;
  my $numberstring;
  my $session_type = substr($base_src_file,6,2);


  #####################################
  # Realisierte Labels speichern
  #####################################
  print $mlf_realized_handle ("\"*${SR_lib::sign}${speaker_dir}${SR_lib::sign}${ses_dir}${SR_lib::sign}${base_src_file}.lab\"\n");
  print $mlf_realized_handle "silStart\n";				
  foreach my $theLabel (@$ses_labels_ref) {
    my $filterFlag = 0;
    foreach my $theFilter (@filterNoisesArray) {
      if (($theLabel =~ /\[(.*)\]/) && ( ($theFilter eq 'all') || ($theFilter eq $1) )) {
	    $filterFlag = 1;
	    last;			# leave further filtering, we found an item
      }			    	
    }
    if ($filterFlag == 0) {
      print $mlf_realized_handle ( &MASV_db_desc::convert_label_string($theLabel) . "\n");
    }

  }
  print $mlf_realized_handle "silEnd\n";
  print $mlf_realized_handle ".\n";

}






sub writePromptedMLF {
	my ($source_dir, $speaker_dir , $ses_dir , $base_src_file) = @_;
	my $prompt_labels_ref;
	my $ses_labels_ref;
	my $theLabel;
	my $session_type = substr($base_src_file,6,2);
	
	($prompt_labels_ref, $ses_labels_ref) = &SR_lib::getLabels(${source_dir},${speaker_dir},${ses_dir},${base_src_file});
	
	print $mlf_prompted_handle ("\"*${SR_lib::sign}*${SR_lib::sign}${ses_dir}${SR_lib::sign}??????${session_type}.lab\"\n");
	print $mlf_prompted_handle ("silStart\n");
	my $i = 1;
	foreach (@$prompt_labels_ref) {
	
		my $numberstring = join("\n", @{&MASV_db_desc::convert_prompt_string($_)});
		print $mlf_prompted_handle ( $numberstring . "\n");
		if ($i <= ((scalar @$prompt_labels_ref) - 1) ) {
			print $mlf_prompted_handle ("sp\n");
		}	
		$i++
	}	# end of foreach (@$prompt_labels_ref)
	print $mlf_prompted_handle "silEnd\n";
	print $mlf_prompted_handle ".\n";
	
}



__END__

=head1 NAME

create_MASV_experiment.pl  - create a sv_system directory using the database described 
                             in the shell environment variable $MASV_DATABASE_DESCRIPTION.

=head1 SYNOPSIS

create_MASV_experiment.pl [options] sv_system_name poolname

Creates a directory <sv_system_name> in the sv_system directory;
signal files are used from pool <poolname>.

 Options:

 -t | typesession = s     specify type of recordings.
                          Default: 'P[1-7]' (all VeriDat triple numbers)
                            
 -s | speakers = s        specify set of speakers.
                          Default: speakers of set "@all" from the template file
                          "<ENV::MASV_BASEDIR>templates/speaker_sets/standard". Add additional
                          sets here. 
                            
 -e | environment = s     specify environment set.
                          Default: all.
                          Examples for possible values (from VeriDat definition): 
                          FixedQuiet (training:01,09,13,12; evaluate:03,05,17)
                          all (training:01,02,03,04;
                               evaluate: 05,06,07,08,09,10,11,12,13,14,15,16,17,18,19,20)
                          Make your own definitions in "MASV_db_desc.pm" in the hashes 
                          "%training_sessions_lists" and "%evaluate_sessions_lists".

 -m | makeMLFs            create two MLF files, one with prompted text, one 
                          with transcription of speaker's utterance.
                          default: not set
                            
 -f | filterNoises = s    only used, when MLF files are generated;
                          filters noise labels from the labels.
                          default: not set
                            
 -c | createStatisticFile 
                          create main statistic file, showing distribution of 
                          labels in test and training set.
                          Does not consider skipped recordings from individual 
                          subjects (see option -i).
                            
 -i | skipInterruptedSessions   
                          skip Sessions containing noise labels
                          (defined in "MASV_db_desc.pm").
  
 -onlyScripts = s         do not create new SV system, only create script files
                          in existing SV system. Existing script files are not 
                          overwritten.
                          Two MLF files (for prompted text and the transcriptions)
                          are generated with the tag "for_<onlyScripts>" in the file
                          name.

 -? | help                display this message.
 
=cut

