#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#####################################################

my $CVS_Version_String = '$Id: run_MASV_experiment.pl,v 1.17 2004/05/04 09:06:30 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;


my $exit_state = 0;

###########################
# Default values for options
my $help = 0;
my $Identify = 0;

my $minVar = "";
my $makeCohortNorm=0;
my $num_iterations_world="";
my $num_iterations_model="";
my $hmm_list = "";
my $dict = "";
my $mixtures= "";
my @teeModels = ();
my $numbernetWorldTest = "";
my $mlf_training = "";
my $mlf_test = "";
my @trainingModelConf = "";
my @HViteTrainedModelConf = "";
my @HViteWorldConf = "";
my @HViteCrossTestConf = "";
my $seed_world = "";
my $seed_models = "";
my @world_list = "";
my @world_test_list = "";
my $protoDir = "";
my @hmmBaseDirWorld = "";
my @hmmBaseDirModel = "";
my $hmm_version_model_test = "";
my $hmm_version_world_test = "";
my $hmm_version_cohort_test = "";
my $parallelComputation = 0;
my $createdScript = "";
my $FA_id = "";
my $doTestsOnly = 0;
my $doHNorm = "";
my $useGMM = "";
my $useHEAdapt = "";
my $startMatlab = 0;
my $pathMatlabResults = "";
my $autoClean_FA_Files = 0;



my $createdScript_temp;
my $doTestsOnly_temp;
my $startMatlab_temp;

###########################

my $minVarSwitch;
my $parallelSwitch;
my $useHEAdaptSwitch;


# Processing Options

# flat
GetOptions('cs|createScript=s' => \$createdScript_temp,
           'doTestsOnly' => \$doTestsOnly_temp,
		   'sm|startMatlab' => \$startMatlab_temp,
           'help|?' => \$help,
           'version' => \$Identify);

if ($Identify) {
	print STDOUT "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if ((@ARGV != 2) || ($help)) {
	pod2usage(1);
   exit 1;
}

if ($SR_lib::verbose_level) {print STDOUT "Running $0 ...\n";}

my $expConfig = $ARGV[0];
my $sv_system_name = $ARGV[1];

my $sv_system_path = "${SR_lib::sv_systems_dir}${sv_system_name}$SR_lib::sign";
my $log_path = "${sv_system_path}log$SR_lib::sign";
my $experimentConfig_path = "${sv_system_path}expConfigs${SR_lib::sign}";


unless (open(CONFFILE, "${experimentConfig_path}$expConfig") ) {
	die "experiment config file $expConfig not found!\n\n";
}
my @confContent = <CONFFILE>;
close CONFFILE;

foreach (@confContent) {
	eval($_);
	unless ($@ eq "") {
		die "error in config file $expConfig :\nline $_ \nerror: $@ \n\n";
	}
}

$createdScript = $createdScript_temp if ($createdScript_temp);
$doTestsOnly = $doTestsOnly_temp if ($doTestsOnly_temp);
$startMatlab = $startMatlab_temp if ($startMatlab_temp);

# normalize to 1 or 0
$autoClean_FA_Files = ($autoClean_FA_Files) ? 1 : 0;
$startMatlab = ($startMatlab) ? 1 : 0;


@teeModels = split(',',join(',',@teeModels));
@trainingModelConf = split(',',join(',',@trainingModelConf));
@HViteTrainedModelConf = split(',',join(',',@HViteTrainedModelConf));
@HViteWorldConf = split(',',join(',',@HViteWorldConf));
@HViteCrossTestConf = split(',',join(',',@HViteCrossTestConf));
@world_list = split(',',join(',',@world_list));
@world_test_list = split(',',join(',',@world_test_list));
@hmmBaseDirWorld = split(',',join(',',@hmmBaseDirWorld));
@hmmBaseDirModel = split(',',join(',',@hmmBaseDirModel));

if (!($doTestsOnly )) {
	if( !((scalar(@trainingModelConf) == scalar(@HViteTrainedModelConf)) &&
		(scalar(@trainingModelConf) == scalar(@HViteCrossTestConf)) &&
		(scalar(@HViteWorldConf) == scalar(@world_list)) &&
		(scalar(@HViteWorldConf) == scalar(@world_test_list)) &&
		(scalar(@HViteWorldConf) == scalar(@hmmBaseDirWorld)) &&
		(scalar(@trainingModelConf) == scalar(@hmmBaseDirModel)) )) 
	{
		print STDERR  'scalar(@trainingModelConf) ' . scalar(@trainingModelConf) . "\n";
		print STDERR  'scalar(@HViteTrainedModelConf) ' . scalar(@HViteTrainedModelConf) . "\n";
		print STDERR  'scalar(@HViteCrossTestConf) ' . scalar(@HViteCrossTestConf) . "\n";
		print STDERR  'scalar(@HViteWorldConf) ' . scalar(@HViteWorldConf) . "\n";
		print STDERR  'scalar(@world_list) ' . scalar(@world_list) . "\n";
		print STDERR  'scalar(@world_test_list) ' . scalar(@world_test_list) . "\n";
		print STDERR  'scalar(@hmmBaseDirWorld) ' . scalar(@hmmBaseDirWorld) . "\n";
		print STDERR  'scalar(@trainingModelConf) ' . scalar(@trainingModelConf) . "\n";
		print STDERR  'scalar(@hmmBaseDirModel) ' . scalar(@hmmBaseDirModel) . "\n";
		
		die "Length of comma separated lists must be eqal!\n";
	}
}
else {
	if( !((scalar(@HViteTrainedModelConf) == scalar(@HViteCrossTestConf)) &&
		(scalar(@HViteWorldConf) == scalar(@hmmBaseDirWorld)) &&
		(scalar(@HViteTrainedModelConf) == scalar(@hmmBaseDirModel)) )) 
	{
		print STDERR  'scalar(@HViteTrainedModelConf) ' . scalar(@HViteTrainedModelConf) . "\n";
		print STDERR  'scalar(@HViteCrossTestConf) ' . scalar(@HViteCrossTestConf) . "\n";
		print STDERR  'scalar(@hmmBaseDirModel) ' . scalar(@hmmBaseDirModel) . "\n";
		print STDERR  'scalar(@HViteWorldConf) ' . scalar(@HViteWorldConf) . "\n";
		print STDERR  'scalar(@hmmBaseDirWorld) ' . scalar(@hmmBaseDirWorld) . "\n";

		die "Length of comma separated lists must be eqal!\n";
	}
}
my $numSplitsModel = scalar(@HViteTrainedModelConf) - 1;
my $numSplitsWorld = scalar(@HViteWorldConf) - 1;

if ($FA_id eq "") {
	if ($numSplitsModel == 1) {
		$FA_id = $hmmBaseDirModel[0];
	}
	else {
		die "please set a file id (option id|FA_id)\n";
	}
}



$minVarSwitch = ($minVar eq "") ? "" : "-v $minVar";
$parallelSwitch = ($parallelComputation) ? "-p" : "";
$useHEAdaptSwitch = ($useHEAdapt && $seed_models) ? "-useHEAdapt" : "";
if (($makeCohortNorm) && ($hmm_version_cohort_test eq "")) {
	die "Version of cohort model not set!\n";
}


if ( ($doTestsOnly) &&  ($hmm_version_model_test eq "")  ) {
	die "Version of speaker model not set!\n";
}



&SR_lib::check_and_create_dir($log_path);

my $exp_Identifier;

$exp_Identifier = $createdScript;

my $exp_log = "experiment_${exp_Identifier}.log";

my $log_handle = &SR_lib::open_file(">", "$log_path" . "run_experiment_${exp_Identifier}.log");

print $log_handle "\n\n";
print $log_handle scalar localtime, "\n";
print $log_handle "Running run_MASV_experiment.pl .....\n";
print $log_handle "---------------------------------------\n";
print $log_handle "Parameters:\n";
print $log_handle "sv_system             : $sv_system_path\n";
print $log_handle "Options:\n";
print $log_handle "mlf_training           : $mlf_training\n";
print $log_handle "mlf_test               : $mlf_test\n";
print $log_handle "mixtures               : $mixtures\n";
print $log_handle "hmm_list               : $hmm_list\n";
print $log_handle "minVar                 : $minVar\n";
print $log_handle "parallelComputation    : $parallelComputation\n";
print $log_handle "createScript           : $createdScript\n";
print $log_handle "makeCohortNorm         : $makeCohortNorm\n";
print $log_handle "num_iterations_world   : $num_iterations_world\n";
print $log_handle "num_iterations_model   : $num_iterations_model\n";
print $log_handle "useTee                 : @teeModels\n";
print $log_handle "trainingModelConf      : @trainingModelConf\n";
print $log_handle "HViteTrainedModelConf  : @HViteTrainedModelConf\n";
print $log_handle "HViteCrossConf         : @HViteCrossTestConf\n";
print $log_handle "HViteWorldConf         : @HViteWorldConf\n";
print $log_handle "seed_world             : $seed_world\n";
print $log_handle "seed_models            : $seed_models\n";
print $log_handle "world_list             : @world_list\n";
print $log_handle "world_test_list        : @world_test_list\n";
print $log_handle "numbernetWorldTest     : $numbernetWorldTest\n";
print $log_handle "protoDir               : $protoDir\n";
print $log_handle "hmmBaseDirWorld        : @hmmBaseDirWorld\n";
print $log_handle "hmmBaseDirModel        : @hmmBaseDirModel\n";
print $log_handle "hmm_version_model_test : $hmm_version_model_test\n";
print $log_handle "hmm_version_world_test : $hmm_version_world_test\n";
print $log_handle "hmm_version_cohort_test: $hmm_version_cohort_test\n";
print $log_handle "FA_id                  : $FA_id\n";
print $log_handle "doTestsOnly            : $doTestsOnly\n";
print $log_handle "doHNorm                : $doHNorm\n";
print $log_handle "useGMM                 : $useGMM\n";
print $log_handle "useHEAdapt             : $useHEAdapt\n";
print $log_handle "startMatlab            : $startMatlab\n";
print $log_handle "pathMatlabResults      : $pathMatlabResults\n";


my $script_handle;


$script_handle = &SR_lib::open_file(">", "$createdScript");
print $script_handle "#!/bin/bash\n";
print $script_handle "# generated automatically by run_MASV_experiment.pl\n";
print $script_handle "current_path=\$PWD;\n";
print $script_handle "script_name=\$0;\n\n";
print $script_handle "# warning: change SV_SYSTEM_NAME also in matlab script!!\n"; 
print $script_handle "SV_SYSTEM_NAME=\"${sv_system_name}\";\n\n"; 
print $script_handle "MATLAB_CALC_PATH=\${MASV_MATLAB_HOME_DIR}calculation_scripts/\${SV_SYSTEM_NAME}_calculation_script_for_${createdScript}/\n";

print $script_handle "matlab_startupfile=\"calculation_scripts/\${SV_SYSTEM_NAME}_calculation_script_for_${createdScript}/startup.m\";\n";
print $script_handle "matlab_result_name=\"${FA_id}\";\n\n";

print $script_handle "EXP_LOG_FILE=\"${exp_log}\";\n\n";
print $script_handle "if test \$script_name == \"-bash\"; then\n\n";
print $script_handle "\techo \"Don't use 'source <script>' or '. <script>'! Use 'bash script'\";\n";
print $script_handle "\techo;\n";
print $script_handle "\treturn 1;\n\n";
print $script_handle "fi\n\n";
print $script_handle "do_world_part=1;\n";
print $script_handle "do_model_part=1;\n";
print $script_handle "do_test_model_part=1;\n";
print $script_handle "do_test_world_part=1;\n";
print $script_handle "do_archive_models=1;\n";
print $script_handle "do_matlab_part=${startMatlab};\n";
print $script_handle "do_clean_part=${autoClean_FA_Files};\n\n";

print $script_handle "message2log.pl -v -logfile \$EXP_LOG_FILE \"starting experiment $exp_Identifier\" \$SV_SYSTEM_NAME || exit; \n\n";

if ($useGMM) {
	&do_GMM_System();
}
else {
	&do_HMM_System();
}


print $script_handle "message2log.pl -v -logfile \$EXP_LOG_FILE \"finished experiment $exp_Identifier\" \$SV_SYSTEM_NAME || exit;\n";

print $script_handle "\n\n\n\n\n\n### Data from config file\n<<run_MASV_experiment_EOF\n";
foreach (@confContent) {
	print $script_handle "\t$_";
}
print $script_handle "run_MASV_experiment_EOF\n";


close $script_handle;



if ($pathMatlabResults) {
	&SR_lib::check_and_create_dir($pathMatlabResults);
}


close $log_handle;

chmod(0755, $createdScript);

exit 0;





sub do_HMM_System {
	#################
	# training world model
	#################
	#####
	my ($exec_string, $exec_string1, $exec_string2, $exec_string3);

	if (!($doTestsOnly)) {

		print $script_handle "if test \$do_world_part == 1; then\n";
		print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"training world model(s)\" \$SV_SYSTEM_NAME || exit; \n\n";
		print $log_handle "running HCompV for world model\n";

		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			print $script_handle "\t## World model training\n";
			if ($seed_world) {
				print $script_handle "\t## Seeding world model\n";
				$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE \$SV_SYSTEM_NAME \'mkdir -p world/${hmmBaseDirWorld[$index]}/hmm.0\' || exit;\nrun_HTKCommand.pl -logfile \$EXP_LOG_FILE \$SV_SYSTEM_NAME \'cp ${seed_world}/* world/${hmmBaseDirWorld[$index]}/hmm.0/\' || exit";
				print $script_handle "\t$exec_string\n";
			}
			else {
				$exec_string = "run_HCompV.pl -logfile \$EXP_LOG_FILE -protoDir=$protoDir -hmmBaseDir=$hmmBaseDirWorld[$index] $minVarSwitch $parallelSwitch -w -wl=$world_list[$index] \$SV_SYSTEM_NAME  || exit";
				print $script_handle "\t$exec_string\n";
			}
			if ($mixtures > 1) {
				$exec_string = "echo \"ME $mixtures  {*.state[2-40].mix}\" > $sv_system_path/tmp/temp_HHEd_config  || exit";
				print $script_handle "\t$exec_string\n";
				$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE \$SV_SYSTEM_NAME \'${SR_lib::htk_bin}HHEd -C ../../configs/global_HHEd_config -M world/${hmmBaseDirWorld[$index]}/hmm.0 -d world/${hmmBaseDirWorld[$index]}/hmm.0 $sv_system_path/tmp/temp_HHEd_config ../../hmm_lists/$hmm_list \'  || exit";
				print $script_handle "\t$exec_string\n";
			}
		}
		print $script_handle "\n\n";
	}
	
	######
	if (!($doTestsOnly)) {
		if (!($seed_world)) {
			foreach my $teeModel (@teeModels) {
				print $log_handle "fixing tee model\'s $teeModel entries in transition matrix\n";
				for(my $index=0; $index<=$numSplitsWorld; $index++) {
					$exec_string = "change_entry_transitions_to_tee.pl \$SV_SYSTEM_NAME $hmmBaseDirWorld[$index] $teeModel  || exit;";
					print $script_handle "\t## fix tee model $teeModel\n";
					print $script_handle "\t$exec_string\n\n";
				}
			}
		}
	}
	
	######
	if (!($doTestsOnly)) {
		print $log_handle "training and testing world models\n";
		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			$exec_string = "run_itWorldModelTraining.pl -logfile \$EXP_LOG_FILE  -hmmBaseDir=$hmmBaseDirWorld[$index] $parallelSwitch -wc=$world_list[$index] -m $mlf_training -h $hmm_list -tc=$world_test_list[$index] -dict $dict -n $numbernetWorldTest  \$SV_SYSTEM_NAME 1 $num_iterations_world || exit;";
			print $script_handle "\t## train and find best world model\n";
			print $script_handle "\t$exec_string\n\n";
		}
		print $script_handle "fi # end do_world_part\n\n\n";
	}
	######
	if ( !($doTestsOnly) && ($hmm_version_world_test eq "") ) {
		print $log_handle "getting best world model\n";
		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			$exec_string1 = "BEST_WORLD_MODEL=`";
			$exec_string2 = "get_bestWorldModel.pl \$SV_SYSTEM_NAME $hmmBaseDirWorld[$index]";
			$exec_string3 = "` || exit;";
			print $script_handle $exec_string1 . $exec_string2 . $exec_string3 ."\n\n\n";
		}
	}
	else {
		print $log_handle "use world model version $hmm_version_world_test\n";
		$exec_string = "BEST_WORLD_MODEL=$hmm_version_world_test \#fixed value";
		print $script_handle $exec_string ."\n\n\n";
		
	}
	######
	print $log_handle "template for forced alignment\n";

	print $script_handle "if test \$do_test_world_part == 1; then\n";
	print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"testing world model(s)\" \$SV_SYSTEM_NAME || exit; \n\n";
	
	for(my $index=0; $index<=$numSplitsWorld; $index++) {
		$exec_string1 = "run_HVite.pl -logfile \$EXP_LOG_FILE -outputWordMLF -hmmBaseDir=$hmmBaseDirWorld[$index] $parallelSwitch -w -wc=$HViteWorldConf[$index] -m $mlf_test -h $hmm_list -dict $dict -o=\'force_aligned_${FA_id}_FAtemplate_\' ";
		if ($index>0) {$exec_string1 .= "-a ";}
		$exec_string2 = "\$SV_SYSTEM_NAME";
		$exec_string3 = " $hmm_version_world_test";
		print $script_handle "\t## Template for forced alignment from world model\n";
		print $script_handle "\t" . $exec_string1 . $exec_string2 . " \$BEST_WORLD_MODEL" . " || exit;\n\n";
	}
	######
	print $log_handle "world model test\n";
	for(my $index=0; $index<=$numSplitsWorld; $index++) {
		$exec_string1 = "run_HVite.pl -logfile \$EXP_LOG_FILE -hmmBaseDir=$hmmBaseDirWorld[$index] $parallelSwitch -w -wc=$HViteWorldConf[$index] -m $mlf_test -h $hmm_list -dict $dict -o=\'force_aligned_${FA_id}_\' ";
		if ($index>0) {$exec_string1 .= "-a ";}
		$exec_string2 = "\$SV_SYSTEM_NAME";
		$exec_string3 = " $hmm_version_world_test";
		print $script_handle "\t## Test with world model\n";
		print $script_handle "\t" . $exec_string1 . $exec_string2 . " \$BEST_WORLD_MODEL" . " || exit;\n\n\n\n";
	}
	
	if ($doHNorm) {
		######
		print $log_handle "HNorm template for forced alignment\n";
		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			my $hnorm_conf = $HViteWorldConf[$index];
			$hnorm_conf =~ s/[^\/]*/all_hnorm/;
			$exec_string1 = "run_HVite.pl -logfile \$EXP_LOG_FILE -outputWordMLF -hmmBaseDir=$hmmBaseDirWorld[$index] $parallelSwitch -w -wc=$hnorm_conf -m $mlf_test -h $hmm_list -dict $dict -o=\'force_aligned_hnorm_${FA_id}_FAtemplate_\' ";
			if ($index>0) {$exec_string1 .= "-a ";}
			$exec_string2 = "\$SV_SYSTEM_NAME";
			$exec_string3 = " $hmm_version_world_test";
			print $script_handle "\t## HNorm template for forced alignment from world model\n";
			print $script_handle "\t" . $exec_string1 . $exec_string2 . " \$BEST_WORLD_MODEL" . " || exit;\n\n";
		}
		######
		print $log_handle "HNorm world model test\n";
		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			my $hnorm_conf = $HViteWorldConf[$index];
			$hnorm_conf =~ s/[^\/]*/all_hnorm/;
			$exec_string1 = "run_HVite.pl -logfile \$EXP_LOG_FILE -hmmBaseDir=$hmmBaseDirWorld[$index] $parallelSwitch -w -wc=$hnorm_conf -m $mlf_test -h $hmm_list -dict $dict -o=\'force_aligned_hnorm_${FA_id}_\' ";
			if ($index>0) {$exec_string1 .= "-a ";}
			$exec_string2 = "\$SV_SYSTEM_NAME";
			$exec_string3 = " $hmm_version_world_test";
			print $script_handle "\t## HNorm test with world model\n";
			print $script_handle "\t" . $exec_string1 . $exec_string2 . " \$BEST_WORLD_MODEL" . " || exit;\n\n\n\n";
		}
		
	}
	print $script_handle "fi # end do_test_world_part \n\n\n";
	
	#################
	# training models
	#################
	if ( !($doTestsOnly) && ($hmm_version_model_test eq "") ) {
		$hmm_version_model_test = "\$BEST_WORLD_MODEL";
		$num_iterations_model = "\$BEST_WORLD_MODEL";
	}
	
	#####
	if (!($doTestsOnly)) {
		print $script_handle "if test \$do_model_part == 1; then\n";
		print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"training speaker models\" \$SV_SYSTEM_NAME || exit; \n\n";
		if ($seed_models) {
			for(my $index=0; $index<=$numSplitsModel ; $index++) {
				$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=$trainingModelConf[$index] \$SV_SYSTEM_NAME \'mkdir -p  ${hmmBaseDirModel[$index]}/hmm.0\' || exit;";
				print $script_handle "## Building base for models\n";
				print $script_handle "\t$exec_string\n";
			}
			for(my $index=0; $index<=$numSplitsModel ; $index++) {
				my $model_source;
				
				if ($seed_models eq "bestWorld") {
					$model_source = "../world/${hmmBaseDirWorld[$index]}/hmm.";
					$model_source .= "\$BEST_WORLD_MODEL";
				}
				else {
					$model_source = "../${seed_models}";
				}
				$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=$trainingModelConf[$index] \$SV_SYSTEM_NAME \"cp ${model_source}"; 
				$exec_string .= "/* ./${hmmBaseDirModel[$index]}/hmm.0\" || exit";
				print $script_handle "\t" . $exec_string . "\n";

				if ($mixtures > 1) {
					$exec_string = "echo \"ME $mixtures  {*.state[2-40].mix}\" > $sv_system_path/tmp/temp_HHEd_config  || exit";
					print $script_handle "\t" . $exec_string . "\n";
					$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=$trainingModelConf[$index]  \$SV_SYSTEM_NAME \'${SR_lib::htk_bin}HHEd -C ../../configs/global_HHEd_config -M ${hmmBaseDirModel[$index]}/hmm.0 -d ${hmmBaseDirModel[$index]}/hmm.0 $sv_system_path/tmp/temp_HHEd_config ../../hmm_lists/$hmm_list \'  || exit";
					print $script_handle "\t" . $exec_string . "\n";
				}
				print $script_handle "\n\n\n";
			}
		}
		else { # start models from scratch
			print $log_handle "running HCompV for spk models\n";
			print $script_handle "# build up models from scratch\n";
			for(my $index=0; $index<=$numSplitsModel ; $index++) {
				$exec_string = "run_HCompV.pl -logfile \$EXP_LOG_FILE -protoDir=$protoDir -hmmBaseDir=$hmmBaseDirModel[$index] $minVarSwitch $parallelSwitch -tc=$trainingModelConf[$index] \$SV_SYSTEM_NAME  || exit";
				print $script_handle "\t$exec_string\n";
				if ($mixtures > 1) {
					$exec_string = "echo \"ME $mixtures  {*.state[2-40].mix}\" > $sv_system_path/tmp/temp_HHEd_config  || exit";
					print $script_handle "\t$exec_string\n";
					$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=$trainingModelConf[$index]  \$SV_SYSTEM_NAME \'${SR_lib::htk_bin}HHEd -C ../../configs/global_HHEd_config -M ${hmmBaseDirModel[$index]}/hmm.0 -d ${hmmBaseDirModel[$index]}/hmm.0 $sv_system_path/tmp/temp_HHEd_config ../../hmm_lists/$hmm_list \'  || exit";
					print $script_handle "\t$exec_string\n";
				}
				print $script_handle "\n\n\n";
			}
		}
		
		
	}
	#####
	
	if (!($doTestsOnly) && ($makeCohortNorm) ) {
		print $log_handle "creating base for cohort models\n";
		#####
		for(my $index=0; $index<=$numSplitsModel; $index++) {
			$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=world_set/aa \$SV_SYSTEM_NAME \'mkdir -p $hmmBaseDirModel[$index]\' || exit;";
			print $script_handle "\t## Building base for models\n";
			print $script_handle "\t$exec_string\n";
		}
				
		#####
		for(my $index=0; $index<=$numSplitsModel; $index++) {
			$exec_string1 = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=world_set/aa \$SV_SYSTEM_NAME \"cp ../world/${hmmBaseDirWorld[$index]}/hmm."; 
			$exec_string2 = "$hmm_version_world_test";
			$exec_string3 = "/* ./${hmmBaseDirModel[$index]}/hmm.0\"  || exit;";
			print "\t" . $script_handle . $exec_string1 . "\$BEST_WORLD_MODEL" . $exec_string3 . "\n\n\n";
		}
	}
	
	if (!($doTestsOnly)) {
		print $log_handle "running HERest for models\n";
		for(my $index=0; $index<=$numSplitsModel ; $index++) {
			$exec_string = "run_HERest.pl -logfile \$EXP_LOG_FILE $useHEAdaptSwitch -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -tc=$trainingModelConf[$index] -m $mlf_training -h $hmm_list \$SV_SYSTEM_NAME $num_iterations_model || exit;";
			print $script_handle "\t## training\n";
			print $script_handle "\t$exec_string\n\n";
		}
	}
	
	if ( !($doTestsOnly) && ($makeCohortNorm) ) {
		for(my $index=0; $index<=$numSplitsModel ; $index++) {
			print $log_handle "running HERest for cohort models\n";
			$exec_string = "run_HERest.pl -logfile \$EXP_LOG_FILE $useHEAdaptSwitch -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -tc=world_set/aa -m $mlf_training -h $hmm_list \$SV_SYSTEM_NAME $num_iterations_model || exit;";
			print $script_handle "\t$exec_string\n\n";
		}
	}
	if (!($doTestsOnly)) {
		print $script_handle "fi # end do_model_part \n\n\n";
	}
	
	#################
	# running HVite
	#################
	
	#####
	print $log_handle "running HVite for models\n";

	print $script_handle "if test \$do_test_model_part == 1; then \n";
	print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"testing speaker models\" \$SV_SYSTEM_NAME || exit; \n\n";

	
	for(my $index=0; $index<=$numSplitsModel ; $index++) {
		$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -o=\'force_aligned_${FA_id}_\' -t=\'force_aligned_${FA_id}_FAtemplate_client_x_world.mlf\' -tc=$HViteTrainedModelConf[$index] -m $mlf_test -h $hmm_list -dict $dict ";
		if ($index>0) {$exec_string .= "-a ";}
		$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_model_test || exit;";
		print $script_handle "\t## Test self x self\n";
		print $script_handle "\t$exec_string\n\n";
	}
	######
	print $log_handle "running HVite imp against model\n";
	for(my $index=0; $index<=$numSplitsModel ; $index++) {
		$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -x -o=\'force_aligned_${FA_id}_\' -t=\'force_aligned_${FA_id}_FAtemplate_client_x_world.mlf\' -tc=$HViteTrainedModelConf[$index] -xc=$HViteCrossTestConf[$index] -m $mlf_test -h $hmm_list -dict $dict ";
		if ($index>0) {$exec_string .= "-a ";}
		$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_model_test || exit;";
		print $script_handle "\t## test cross x self\n";
		print $script_handle "\t$exec_string\n\n";
	}

	if ($doHNorm) {
		print $log_handle "HNorm running HVite for models\n";
		for(my $index=0; $index<=$numSplitsModel ; $index++) {
			$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -o=\'force_aligned_hnorm_${FA_id}_\' -t=\'force_aligned_hnorm_${FA_id}_FAtemplate_client_x_world.mlf\' -tc=$HViteTrainedModelConf[$index] -m $mlf_test -h $hmm_list -dict $dict ";
			if ($index>0) {$exec_string .= "-a ";}
			$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_model_test || exit;";
			print $script_handle "\t## HNorm test self x self\n";
			print $script_handle "\t$exec_string\n\n";
		}
		######
		print $log_handle "HNorm running HVite imp against model\n";
		for(my $index=0; $index<=$numSplitsModel ; $index++) {
			my $hnorm_conf = $HViteCrossTestConf[$index];
			$hnorm_conf =~ s/[^\/]*/dev_set/;
			$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -x -o=\'force_aligned_hnorm_${FA_id}_\' -t=\'force_aligned_hnorm_${FA_id}_FAtemplate_client_x_world.mlf\' -tc=$HViteTrainedModelConf[$index] -xc=$hnorm_conf -m $mlf_test -h $hmm_list -dict $dict ";
			if ($index>0) {$exec_string .= "-a ";}
			$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_model_test || exit;";
			print $script_handle "\t## HNorm test cross x self\n";
			print $script_handle "\t$exec_string\n\n";
		}
		
		
	}
	#####
	if ($makeCohortNorm) {
		
		print $log_handle "running HVite model and imp spks against cohort models\n";
		for(my $index=0; $index<=$numSplitsModel ; $index++) {
			$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -c -o=\'force_aligned_${FA_id}_\' -t=\'force_aligned_${FA_id}_FAtemplate_client_x_world.mlf\' -tc=world_set/aa -xc=all/aa  -m $mlf_test -h $hmm_list -dict $dict ";
			if ($index>0) {$exec_string .= "-a ";}
			$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_cohort_test || exit;";
			print $script_handle "\t## test cohort x self\n";
			print $script_handle "\t$exec_string\n\n";
		}
	}

	print $script_handle "fi # end do_test_model_part \n\n\n";
	
	&do_archive_models();
	
	if ($startMatlab) {
		&do_matlabCalculation();
	}
	
	
	&do_autoClean();

	
}






sub do_GMM_System {
	#################
	# training world model
	#################
	#####
	my ($exec_string, $exec_string1, $exec_string2, $exec_string3);

	if (!($doTestsOnly)) {

		print $script_handle "if test \$do_world_part == 1; then\n";
		print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"training world model(s)\" \$SV_SYSTEM_NAME || exit; \n\n";
		print $log_handle "building base/running HCompV for world model\n";
		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			print $script_handle "\t## World model training\n";
			
			if ($seed_world) {
				print $script_handle "\t## Seeding world model\n";
				$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE \$SV_SYSTEM_NAME \'mkdir -p world/${hmmBaseDirWorld[$index]}/hmm.0\' || exit;\nrun_HTKCommand.pl -logfile \$EXP_LOG_FILE \$SV_SYSTEM_NAME \'cp ${seed_world}/* world/${hmmBaseDirWorld[$index]}/hmm.0/\' || exit;";
				print $script_handle "\t$exec_string\n\n";
			}
			else {
				$exec_string = "run_HCompV.pl -logfile \$EXP_LOG_FILE -protoDir=$protoDir -useGMM -hmmBaseDir=$hmmBaseDirWorld[$index] $minVarSwitch $parallelSwitch -w -wl=$world_list[$index] \$SV_SYSTEM_NAME  || exit";
				print $script_handle "\t$exec_string\n";
				if ($mixtures > 1) {
					$exec_string = "echo \"ME $mixtures  {*.state[2].mix}\" > $sv_system_path/tmp/temp_HHEd_config  || exit";
					print $script_handle "\t$exec_string\n\n";
					$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE \$SV_SYSTEM_NAME \'${SR_lib::htk_bin}HHEd -C ../../configs/global_HHEd_config -M world/${hmmBaseDirWorld[$index]}/hmm.0 -d world/${hmmBaseDirWorld[$index]}/hmm.0 $sv_system_path/tmp/temp_HHEd_config ../../hmm_lists/$hmm_list \'  || exit";
					print $script_handle "\t$exec_string\n\n";
				}
			}
		}
	}
	
	
	if (!($doTestsOnly)) {
		print $log_handle "training and testing world models\n";
		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			$exec_string = "run_itWorldModelTraining.pl -logfile \$EXP_LOG_FILE -useGMM  -hmmBaseDir=$hmmBaseDirWorld[$index] $parallelSwitch -wc=$world_list[$index] -m $mlf_training -h $hmm_list -tc=$world_test_list[$index] -dict $dict  \$SV_SYSTEM_NAME 1 $num_iterations_world || exit;";
			print $script_handle "\t## train and find best world model\n";
			print $script_handle "\t$exec_string\n\n";
		}
		print $script_handle "fi # end do_world_part\n\n\n";
	}

	######


	
	######
	if ( !($doTestsOnly) && ($hmm_version_world_test eq "") ) {
		print $log_handle "getting best world model\n";
		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			$exec_string1 = "BEST_WORLD_MODEL=`";
			$exec_string2 = "get_bestWorldModel_GMM.pl \$SV_SYSTEM_NAME $hmmBaseDirWorld[$index]";
			$exec_string3 = "` || exit;";
			print $script_handle $exec_string1 . $exec_string2 . $exec_string3 ."\n\n\n";
		}
	}
	else {
		print $log_handle "use world model version $hmm_version_world_test\n";
		$exec_string = "BEST_WORLD_MODEL=$hmm_version_world_test \#fixed value";
		print $script_handle $exec_string ."\n\n\n";
		
	}

	######
	print $log_handle "world model test\n";
	print $script_handle "if test \$do_test_world_part == 1; then\n";
	print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"testing world model(s)\" \$SV_SYSTEM_NAME || exit; \n\n";

	for(my $index=0; $index<=$numSplitsWorld; $index++) {
		$exec_string1 = "run_HVite.pl -logfile \$EXP_LOG_FILE -useGMM -hmmBaseDir=$hmmBaseDirWorld[$index] $parallelSwitch -w -wc=$HViteWorldConf[$index] -m $mlf_test -h $hmm_list -dict $dict -o=\'force_aligned_${FA_id}_\'  ";
		if ($index>0) {$exec_string1 .= "-a ";}
		$exec_string2 = "\$SV_SYSTEM_NAME";
		$exec_string3 = " $hmm_version_world_test  || exit;";
		print $script_handle "\t## Test with world model\n";
		print $script_handle "\t" . $exec_string1 . $exec_string2 . " \$BEST_WORLD_MODEL" . " || exit;\n\n\n\n";
	}
	
	if ($doHNorm) {
		print $log_handle "HNorm world model test\n";
		for(my $index=0; $index<=$numSplitsWorld; $index++) {
			my $hnorm_conf = $HViteWorldConf[$index];
			$hnorm_conf =~ s/[^\/]*/all_hnorm/;	
			$exec_string1 = "run_HVite.pl -logfile \$EXP_LOG_FILE -useGMM -hmmBaseDir=$hmmBaseDirWorld[$index] $parallelSwitch -w -wc=$hnorm_conf -m $mlf_test -h $hmm_list -dict $dict -o=\'force_aligned_hnorm_${FA_id}_\'  ";
			if ($index>0) {$exec_string1 .= "-a ";}
			$exec_string2 = "\$SV_SYSTEM_NAME";
			$exec_string3 = " $hmm_version_world_test  || exit;";
			print $script_handle "\t## Test with world model\n";
			print $script_handle "\t" . $exec_string1 . $exec_string2 . " \$BEST_WORLD_MODEL" . " || exit;\n\n\n\n";
		}
		
	}
	print $script_handle "fi # end do_test_world_part \n\n\n";
	
	#################
	# training models
	#################
	
	if ( !($doTestsOnly) && ($hmm_version_model_test eq "") ) {
		$hmm_version_model_test = "\$BEST_WORLD_MODEL";
		$num_iterations_model = "\$BEST_WORLD_MODEL";
	}
	
	if (!($doTestsOnly)) {

		print $script_handle "if test \$do_model_part == 1; then\n";
		print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"training speaker models\" \$SV_SYSTEM_NAME || exit; \n\n";
		
		if ($seed_models) {
			print $script_handle "\t## Building base for models\n";
			for(my $index=0; $index<=$numSplitsModel ; $index++) {
				$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=$trainingModelConf[$index] \$SV_SYSTEM_NAME \'mkdir -p  ${hmmBaseDirModel[$index]}/hmm.0\' || exit;";
				print $script_handle "\t$exec_string\n";
			}
			for(my $index=0; $index<=$numSplitsModel ; $index++) {
				my $model_source;
				
				if ($seed_models eq "bestWorld") {
					$model_source = "../world/${hmmBaseDirWorld[$index]}/hmm.";
					$model_source .= "\$BEST_WORLD_MODEL";
				}
				else {
					$model_source = "../${seed_models}";
				}
				$exec_string1 = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=$trainingModelConf[$index] \$SV_SYSTEM_NAME \"cp ${model_source}"; 
				$exec_string2 = "/* ./${hmmBaseDirModel[$index]}/hmm.0\" || exit;";
				print  $script_handle "\t" . $exec_string1 . $exec_string2 . "\n\n\n";
			}
			
		}
		else { # start models from scratch
			print $log_handle "running HCompV for spk models\n";
			print $script_handle "# build up models from scratch\n";
			for(my $index=0; $index<=$numSplitsModel; $index++) {
				$exec_string = "run_HCompV.pl -logfile \$EXP_LOG_FILE -protoDir=$protoDir -useGMM -hmmBaseDir=$hmmBaseDirModel[$index] $minVarSwitch $parallelSwitch -tc=$trainingModelConf[$index] \$SV_SYSTEM_NAME  || exit";
				print $script_handle "\t$exec_string\n";
				if ($mixtures > 1) {
					$exec_string = "echo \"ME $mixtures  {*.state[2].mix}\" > $sv_system_path/tmp/temp_HHEd_config  || exit";
					print $script_handle "\t$exec_string\n";
					$exec_string = "run_HTKCommand.pl -logfile \$EXP_LOG_FILE -i -tc=$trainingModelConf[$index] \$SV_SYSTEM_NAME \'${SR_lib::htk_bin}HHEd -C ../../../configs/global_HHEd_config -M ${hmmBaseDirModel[$index]}/hmm.0 -d ${hmmBaseDirModel[$index]}/hmm.0 $sv_system_path/tmp/temp_HHEd_config ../../../hmm_lists/$hmm_list \'  || exit";
					print $script_handle "\t$exec_string\n";
				}
				print $script_handle "\n\n\n";
			}
			
		}
	}
	
	#####	
	if (!($doTestsOnly)) {
		print $log_handle "running HERest for models\n";
		for(my $index=0; $index<=$numSplitsModel ; $index++) {
			$exec_string = "run_HERest.pl -logfile \$EXP_LOG_FILE $useHEAdaptSwitch -useGMM -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -tc=$trainingModelConf[$index] -m $mlf_training -h $hmm_list \$SV_SYSTEM_NAME $num_iterations_model || exit;";
			print $script_handle "\t## training\n";
			print $script_handle "\t$exec_string\n\n";
		}

		print $script_handle "fi # end do_model_part \n\n\n";
	}
	
	
	#################
	# running HVite
	#################
	
	#####
	print $log_handle "running HVite for models\n";

	print $script_handle "if test \$do_test_model_part == 1; then \n";
	print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"testing speaker models\" \$SV_SYSTEM_NAME || exit; \n\n";

	
	for(my $index=0; $index<=$numSplitsModel ; $index++) {
		$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -useGMM -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -o=\'force_aligned_${FA_id}_\'  -tc=$HViteTrainedModelConf[$index] -m $mlf_test -h $hmm_list -dict $dict ";
		if ($index>0) {$exec_string .= "-a ";}
		$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_model_test || exit;";
		print $script_handle "\t## Test self x self\n";
		print $script_handle "\t$exec_string\n\n";
	}
	######
	print $log_handle "running HVite imp against model\n";
	for(my $index=0; $index<=$numSplitsModel ; $index++) {
		$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -useGMM -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -x -o=\'force_aligned_${FA_id}_\' -tc=$HViteTrainedModelConf[$index] -xc=$HViteCrossTestConf[$index] -m $mlf_test -h $hmm_list -dict $dict ";
		if ($index>0) {$exec_string .= "-a ";}
		$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_model_test || exit;";
		print $script_handle "\t## test cross x self\n";
		print $script_handle "\t$exec_string\n\n";
	}
	#####

	if ($doHNorm) {
		
		print $log_handle "HNorm running HVite for models\n";
		for(my $index=0; $index<=$numSplitsModel ; $index++) {
			$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -useGMM -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -o=\'force_aligned_hnorm_${FA_id}_\'  -tc=$HViteTrainedModelConf[$index] -m $mlf_test -h $hmm_list -dict $dict ";
			if ($index>0) {$exec_string .= "-a ";}
			$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_model_test || exit;";
			print $script_handle "\t## HNorm test self x self\n";
			print $script_handle "\t$exec_string\n\n";
		}
		######
		print $log_handle "HNorm running HVite imp against model\n";
		for(my $index=0; $index<=$numSplitsModel ; $index++) {
			my $hnorm_conf = $HViteCrossTestConf[$index];
			$hnorm_conf =~ s/[^\/]*/dev_set/;
			$exec_string = "run_HVite.pl -logfile \$EXP_LOG_FILE -useGMM -hmmBaseDir=$hmmBaseDirModel[$index] $parallelSwitch -x -o=\'force_aligned_hnorm_${FA_id}_\' -tc=$HViteTrainedModelConf[$index] -xc=$hnorm_conf -m $mlf_test -h $hmm_list -dict $dict ";
			if ($index>0) {$exec_string .= "-a ";}
			$exec_string .= "\$SV_SYSTEM_NAME $hmm_version_model_test || exit;";
			print $script_handle "\t## HNorm test cross x self\n";
			print $script_handle "\t$exec_string\n\n";
		}
		
	}

	print $script_handle "fi # end do_test_model_part \n\n\n";
	
	&do_archive_models();
	
	if ($startMatlab) {
		&do_matlabCalculation();
	}
	
	
	&do_autoClean();

}



sub do_matlabCalculation {

	my $matlab_handle;
	my $exec_string;
	
	print $script_handle "\n\n\n# Generation of matlab script\n#####\n";
	print $script_handle "if test \$do_matlab_part == 1; then\n";
	print $script_handle "mkdir -p \${MATLAB_CALC_PATH}\n";
	print $script_handle "cat > \${MATLAB_CALC_PATH}startup.m <<-'matlab_file_gen_EOF'\n";

	print $script_handle "\t\%automatically created from  ${createdScript}\n\n";
	print $script_handle "\tcurrent_path = pwd;\n";
	print $script_handle "\tcd ../..\n";
	print $script_handle "\tstartup\n";
	print $script_handle "\tcd(current_path);\n";
	print $script_handle "\tstartScript=read_experiment_details;\n";
	print $script_handle "\tpath_for_results = \'$pathMatlabResults\';\n";
	print $script_handle "\tFA_id = \'$FA_id\';\n";
	print $script_handle "\tsv_system_name = \'$sv_system_name\';\n";
	print $script_handle "\tmodel_archive=read_model_archive(path_for_results,sv_system_name,FA_id);\n";
	print $script_handle "\trs=calc_FR_FA(sv_system_name,FA_id,0,0,1,\'simple world\','without pauses\',\'per frame\',[\'force_aligned_\' FA_id \'_\']);\n";
	print $script_handle "\trs.expDetails.script = startScript;\n";
	print $script_handle "\trs.expDetails.model_archive = model_archive;\n";
	print $script_handle "\trs.model_info = ext_model_info_w_HVite_info(rs.model_info, sv_system_name,[\'force_aligned_\' FA_id \'_\']);\n";
	if ($pathMatlabResults) {
		print $script_handle "\tsave_result(rs,path_for_results);\n";
	}
	else {
		print $script_handle "\tsave_result(rs);\n";
	}
	print $script_handle "\tsr=calc_score_result(rs);\n";
	print $script_handle "\tsr=add_part_to_score_result(rs,sr,\'det\',\'DET with same gender impostors\',\'same gender imp\',0,rs.model_info.trained_models,{\'same gender\'},{\'without training\'},{\'all\'});\n";

	print $script_handle "\tsr.expDetails.script = startScript;\n";
	if ($pathMatlabResults) {
		print $script_handle "\tsave_score_result(sr,path_for_results);\n";
	}
	else {
		print $script_handle "\tsave_score_result(sr);\n";
	}
	if ($doHNorm) {
		print $script_handle "\trs_norming=calc_FR_FA(sv_system_name,FA_id,0,0,1,\'simple world\','without pauses\',\'per frame\',[\'force_aligned_hnorm_\' FA_id \'_\']);\n";
		print $script_handle "\trs=do_hnorm(rs, \'$doHNorm\', rs_norming);\n";
		print $script_handle "\trs=rename_result(rs, [\'hnormed_\' FA_id ]);\n";
		if ($pathMatlabResults) {
			print $script_handle "\tsave_result(rs,path_for_results);\n";
		}
		else {
			print $script_handle "\tsave_result(rs);\n";
		}
		print $script_handle "\tsr=calc_score_result(rs);\n";
		print $script_handle "\tsr=add_part_to_score_result(rs,sr,\'det\',\'DET with same gender impostors\',\'same gender imp\',0,rs.model_info.trained_models,{\'same gender\'},{\'without training\'},{\'all\'});\n";
		print $script_handle "\tsr.expDetails.script = startScript;\n";
		if ($pathMatlabResults) {
			print $script_handle "\tsave_score_result(sr,path_for_results);\n";
		}
		else {
			print $script_handle "\tsave_score_result(sr);\n";
		}
		
	}
	print $script_handle "\texit;\n";
	
	print $script_handle "matlab_file_gen_EOF\n\n\n";
	
	
	print $script_handle "fi # end do_matlab_part\n####\n\n";
	print $script_handle "\n\n## starting matlab\n";
	print $script_handle "if test \$do_matlab_part == 1; then\n";
	print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"starting matlab\" \$SV_SYSTEM_NAME || exit;\n\n";
	print $script_handle "\trm -f \${MATLAB_CALC_PATH}*.sh;\n";
	print $script_handle "\tcp \${current_path}/\${script_name} \${MATLAB_CALC_PATH};\n";
	print $script_handle "\tresult=`ssh -x \${MASV_MATLAB_HOST} uptime` || MASV_MATLAB_HOST=\$HOSTNAME\n";
	print $script_handle "\tssh -x \${MASV_MATLAB_HOST} \"cd \${MATLAB_CALC_PATH}; run_matlab.sh \$SV_SYSTEM_NAME $FA_id ;\"\n";
	
	print $script_handle "fi # end do_matlab_part\n\n\n";
	
}

sub do_autoClean {
	
	my $exec_string;
	
	$exec_string = "make_clean_MASV_experiment.pl -cleanAll -expConfig $expConfig \$SV_SYSTEM_NAME || exit;";
	print $script_handle "if test \$do_clean_part == 1; then\n";
	print $script_handle "\n\t## clean up\n";
	print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"cleaning up\" \$SV_SYSTEM_NAME || exit;\n";
	print $script_handle "\t$exec_string\n\n";
	print $script_handle "fi # end do_clean_part\n\n\n";
	
}

sub do_archive_models {
	
	my $exec_string;
	
	$exec_string = "archive_experiment.pl -logfile \$EXP_LOG_FILE -archive_directory=$pathMatlabResults -archive_name=\${SV_SYSTEM_NAME}-${FA_id}.tar.bzip2  \$SV_SYSTEM_NAME force_aligned_${FA_id}_ || exit;";
	print $script_handle "\n\n\n";
	print $script_handle "if test \$do_archive_models == 1; then\n";
	print $script_handle "\n\t## archive\n";
	print $script_handle "\tmessage2log.pl -v -logfile \$EXP_LOG_FILE \"archiving models to \" \$SV_SYSTEM_NAME || exit;\n";
	print $script_handle "\t$exec_string\n\n";
	print $script_handle "fi # end do_archive_models\n\n\n";
	
}





__END__

=head1 NAME

run_MASV_experiment.pl  - create complete training and test sequence for 
                            an experiment

=head1 SYNOPSIS

run_MASV_experiment.pl [options] expConfig  sv_system_name

Creates a complete shell script for training and test of 
the experiment of <sv_system_name> with given options. 
The file <expConfig> from the SV system directory 
"expConfig/" contains the settings.
Creates necessary shell (and Matlab) scripts for execution later on.

 Options:
 

 -cs | createScript=s     create shell script with given name;

 
 -doTestsOnly             perform only HVite tests for spk models with world, self and
                          impostor test.
						  Default: setting from expConfig,

 -sm|startMatlab          start Matlab performance calculation after tests.
						  Default: setting from expConfig,
 
 -version                 print version information.

 -? | help                display this message.
 
=cut

