function score_struct = calc_single_model_scores(sv_system_name, ...
									full_calc_flag, ...
									full_info_flag, ...
									llhsum_type, ...
									score_type, ...
									mlf_file_name)
%
% function score_struct = calc_single_model_scores
%                         (sv_system_name,
%                         full_calc_flag,
%                         full_info_flag,
%                         llhsum_type,
%                         score_type,
%                         mlf_file_name)
%
%
% builds a result_struct containing the FR and FA values for the sv_system
% 'sv_system_name'. The result_name is used when searching for an existing .mat-file
% in the results directory. If none is found, the result_struct is calculated. 
%
% result_struct
% -------------
% result_struct.FR_struct   a FR_struct cell array (no. of trained models)
% result_struct.FA_struct   a FA_struct cell array (no. of trained models)
% result_struct.exp_name    the name of the sv_system
% result_struct.result_name the name of the result
%
% FR_struct
% ---------
% FR_struct.data.borders         a vector with normalized llh values, where a
%                                test was done. Length is No. of tests.
% FR_struct.data.rec_info        info string for all test, containing model, session and 
%                                recording.
% 
% FR_struct.model_name           name of the model/speaker
%
% if calculated with flag 'full_info_flag' set:
% FR_struct.data.mlf_data      array of mlf_value structs, size: length of borders vector.
%
% mlf_value
% ---------
% mlf_value.start_vector         vector of all starting time of all phonemes of the force
%                                alignment
% mlf_value.end_time             end point of last phoneme
% mlf_value.phoneme_vector       cell vector with the phoneme labels
%     
%  length of the  vectors: no. of labels in the recording
%
%
% FA_struct
% ---------
% FA_struct.data                 see FR_struct (including mlf_value extension)
% 
% CVS_Version_String = '$Id: calc_single_model_scores.m,v 1.9 2004/02/13 13:21:12 tuerk Exp $';
% CVS_Name_String = '$Name: rel-1-4-01 $';

% ###########################################################
%
% This file is part of the matlab scripts of the MASV System.
% MASV = Munich Automatic Speaker Verification
%
% Copyright 2002-2003, Ulrich Trk
% Institute of Phonetics and Speech Communication
% University of Munich
% tuerk@phonetik.uni-muenchen.de
%
%
%   MASV is free software; you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation; either version 2 of the License, or
%   (at your option) any later version.
%
%   MASV is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
%
%   You should have received a copy of the GNU General Public License
%   along with MASV; if not, write to the Free Software
%   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
%
% ###########################################################



	
	SR_lib;
	[sets] = read_spk_set(sv_system_name);
	
	if (~(strcmp(score_type,'simple sum')) & (~(strcmp(score_type,'per frame')))),
		error('Wrong type for score_type; use ''simple sum'' or ''per frame''!');
	end

	
	% initialize score_struct

	sv_system_path=strcat(basepath, sv_system_name);

	mlf_file = strcat(sv_system_path, filesep, mlf_file_name);						
	temp_llh_struct=calc_model_values(mlf_file,llhsum_type,full_calc_flag,full_info_flag);

	if (strcmp(score_type,'simple sum')),
		temp_score_array = [temp_llh_struct.data.simple_llh_sum];
	end
	if (strcmp(score_type,'per frame')),
		temp_score_array = [temp_llh_struct.data.llh_per_frame];
	end
	
	temp_rec_array = char({temp_llh_struct.data.file});
	

	[temp_struct, index_sort]=count_cdf_with_borders(temp_score_array,temp_rec_array);
	score_struct = temp_struct;
	
	if (full_info_flag)
		score_struct.mlf_data=struct('start_vector',{temp_llh_struct.data.start_vector}, 'end_time', {temp_llh_struct.data.end_time}, 'phoneme_vector',{temp_llh_struct.data.phoneme_vector} , 'llh_values',{temp_llh_struct.data.llh_vector});
		score_struct.mlf_data=score_struct.mlf_data(index_sort);
	end	


	score_struct.info.llhsum_type = llhsum_type;
	score_struct.info.score_type = score_type;
	             
	score_struct.info.paramPool_config = read_sv_system_paramPool_Info(sv_system_name);
	score_struct.info.sets = sets;
	score_struct.info.spk_struct_array = read_spk_info;



	

	score_struct.exp_name = sv_system_name;
	
	indizes = findstr(mlf_file_name, filesep);
	if (~(isempty(indizes))),
		mlf_file_name(indizes)='_';
	end
	score_struct.result_name = mlf_file_name;
	score_struct.full_name = [sv_system_name '-' mlf_file_name];


