#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: run_HERest.pl,v 1.24 2004/04/15 14:11:45 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

use Parallel::ForkManager;

my ($trainingModelArrRef, $trainingModelSessionsLists); 

my $exit_state = 0;
my $local_exit_code = 0;
my @ssh_stat = [];

my $nice_factor = $SR_lib::nice_factor;
$nice_factor = $SR_lib::nice_factor; # make compiler happy, no warning

my $start_time = scalar localtime;

###########################
# Default values for options
my $buildWorldModel=0;
my $help = 0;
my $log_file = '';
my $Identify = 0;
my $minVar = 0.0001;
my $hmm_list = "HMM_number.list";
my $mlf = "mlf_prompted.mlf";
my $trainingModelConf = "training_set/at";
my $trainingWorldConf = "world_set/aa";
my $hmmBaseDir = '';
my $macros = '';
my $extraOptions = '';
my $distributedClient = 0;
my $dist_iteration_counter=0;
my $totalHostsNum=0;
my $onStartup = '';
my $onFinish = '';
my $prefixCommand = '';
my $parallelComputation = 0;
my $useGMM = 0;
my $useHEAdapt = 0;
###########################
my $extraOptionsString;
my $useGMMString;
my $macrosString;
my $useHEAdaptString;

my ($hostsNum, @hostList);
my $realHostsNum;
my @exe_models;

my $prompt_HTK_Command_String = ($SR_lib::verbose_level > 1) ? "-A" : "";

# Processing Options

# flat
GetOptions('w|world' => \$buildWorldModel,
		   'useGMM' => \$useGMM,
		   'v|varFloor=f' => \$minVar,
	       'h|hmm_list=s' => \$hmm_list,
	       'm|mlf=s' => \$mlf,
	       'tc|trainingModelConf=s' => \$trainingModelConf,
	       'wl|world_list=s' => \$trainingWorldConf,
	       'hmmBaseDir=s' => \$hmmBaseDir,
		   'macros=s' => \$macros,
		   'onStartup=s' => \$onStartup,
		   'onFinish=s' => \$onFinish,
		   'prefixCommand=s' => \$prefixCommand,
		   'xo|extraOptions=s' => \$extraOptions,
           'dist=s' => \$distributedClient,
           'dist_iteration_counter=s' => \$dist_iteration_counter,
           'totalHostsNum=i' => \$totalHostsNum,
           'p|parallelComputation' => \$parallelComputation,
		   'a|useHEAdapt' => \$useHEAdapt,
           'help|?' => \$help,
		   'logfile=s' => \$log_file,
           'version' => \$Identify);

if ($Identify) {
	print STDOUT "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if (((@ARGV < 2) || (@ARGV > 3)) || ($help)) {
   pod2usage(1);
   exit 1;
}

if ($SR_lib::verbose_level) {print STDOUT "Running $0 ...\n";}

if ($distributedClient && $parallelComputation) {
	print STDERR "Error: Use only the -p|parallelComputation option!\n\n";
	exit 1;
}

if ($buildWorldModel && $useHEAdapt) {
	print STDERR "Warning: Option \"useHEAdapt\" works only when training non-world models!\n\n";
}	

my $first_iteration_no = 1;
my $sv_system_name = $ARGV[0];
my $num_iterations = $ARGV[1];

&SR_lib::init_speaker_sets($sv_system_name);


if (@ARGV == 3) {
	$first_iteration_no = $ARGV[2];
}

$extraOptionsString = ($extraOptions) ? "-xo=\'$extraOptions\'"  :  "";
$useGMMString = ($useGMM) ? "-useGMM"  :  "";

$prefixCommand =~ s/[;\s]*$/;/ if ($prefixCommand);

my $prefixCommandString = ($prefixCommand) ? "-prefixCommand=\'$prefixCommand\'" : "";


$macrosString = ($macros) ? "-macros $macros" : "";
$useHEAdaptString = ($useHEAdapt) ? "-useHEAdapt"  :  "";


if (!($hmmBaseDir)) {
	$hmmBaseDir = "hmm";
}



my $hmmlist_path = "${SR_lib::base_dir}hmm_lists$SR_lib::sign";
my $sv_system_path = "${SR_lib::sv_systems_dir}$sv_system_name$SR_lib::sign";
my $log_path = "${sv_system_path}log$SR_lib::sign";
$log_file = ($log_file) ? $log_file : "log.txt";
my $world_path = "${sv_system_path}world$SR_lib::sign";

my $global_HERest_config="${SR_lib::base_dir}configs${SR_lib::sign}global_HERest_config";



if ($parallelComputation) {
	($hostsNum, @hostList) = &SR_lib::par_get_available_hosts;
}

if (! $distributedClient) {
	if ($onStartup) {
		if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
		$local_exit_code = system($onStartup);
	}
}



if (! $buildWorldModel) {
# seperate Modelle
	($trainingModelArrRef, $trainingModelSessionsLists) = &SR_lib::processConfigurations($trainingModelConf);

	my @models = @{$trainingModelArrRef};


	if ($parallelComputation) {
		# Wieviele hosts machen Sinn?
		my @dummy;
		($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@models, $hostsNum);
	}
	elsif ($distributedClient) {
		# VP-Liste auf bentigte Zahl aufteilen
		my ($dummy, @models_lists) = &SR_lib::split_arrayList(\@models, $totalHostsNum);
		@exe_models = @{$models_lists[$distributedClient - 1]};
	}
	else { 
		#lokal ausgefhrt, alles lassen, wie es ist
  		@exe_models = @models;
	}

	if ($parallelComputation) {
		my $pm = new Parallel::ForkManager($realHostsNum);
		
		$pm->run_on_finish(
			sub { my ($pid, $exit_code, $ident) = @_;
				  $ssh_stat[$ident] = $exit_code;
			}
		);

		
		for (my $host_counter=1; $host_counter <= $realHostsNum; $host_counter++) {
			$pm->start($host_counter) and next;
			
			if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
			if ($onStartup) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
				$local_exit_code = system("ssh -x $hostList[$host_counter] \"$onStartup\"");
			}
			my $exec_string = "ssh -x $hostList[$host_counter]  /usr/bin/nice -n $nice_factor \"run_HERest.pl -dist=$host_counter  $useGMMString $useHEAdaptString -totalHostsNum=$realHostsNum -tc=$trainingModelConf -h $hmm_list -m $mlf -v=$minVar -hmmBaseDir=$hmmBaseDir $macrosString $extraOptionsString $prefixCommandString $sv_system_name $num_iterations $first_iteration_no \"";
			if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
			$local_exit_code += system($exec_string);
			if ($onFinish) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
				$local_exit_code += system("ssh -x $hostList[$host_counter] \"$onFinish\"");
			}
			if ($local_exit_code) {
				$local_exit_code = 11;
			}
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
			if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}
			$pm->finish($local_exit_code);
	
		}
	  
		$pm->wait_all_children;
		if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
		if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
		
		for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
			if ($SR_lib::verbose_level) { print STDOUT  "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
			$exit_state += $ssh_stat[$host_counter];
		}
		
		
		if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}


	}
	else 
	{
		
		for (my $iteration_counter = $first_iteration_no; $iteration_counter <= $num_iterations+$first_iteration_no-1; $iteration_counter++) {
		
			foreach my $vp_dir (@exe_models) {
				my $hmmpath = $sv_system_path.$vp_dir.$SR_lib::sign.$hmmBaseDir.$SR_lib::sign;
				my $hmmpathin=$hmmpath."hmm.".($iteration_counter-1);
				my $hmmpathout=$hmmpath."hmm.".($iteration_counter);
				my $macrosin=$hmmpathin;
				my $macrosOptionString = ($macros) ? "-H ${macrosin}/$macros" : "";
				#$trainingBinary = ($useHEAdapt) ? "HEAdapt -j 15.0 -H ${hmmpath}HEAdapt_dummy.mmf" : "HERest";
				my $trainingBinary = ($useHEAdapt) ? "${SR_lib::htk_bin}HEAdapt -j 15.0" : "${SR_lib::htk_bin}HERest";
				
				&SR_lib::check_and_create_dir($hmmpathout);
			
				my $exec_string="$prefixCommand $trainingBinary -C $global_HERest_config $prompt_HTK_Command_String -M $hmmpathout $macrosOptionString -d $hmmpathin -v $minVar -S $sv_system_path$vp_dir/$trainingModelSessionsLists  -I ${sv_system_path}$mlf $extraOptions ${hmmlist_path}$hmm_list";
			    #print ("$exec_string \n");
			    $exit_state = system ($exec_string);
				if ($exit_state) {
					print STDERR "Command $exec_string failed\n";
					exit 1;
				}
				if ($iteration_counter == $first_iteration_no) {
					&SR_lib::write_trainingInfoFile($hmmpath, $trainingModelSessionsLists);
				}

			
			}
		}

	}
	  
}
else # World Model -> gemeinsames Modell fr alle Trainingsdaten
{

	if ($parallelComputation) {

		&SR_lib::create_multispeaker_list($sv_system_name, [$trainingWorldConf]);
		my $multispeaker_list = &SR_lib::get_filename_multispeaker_list($sv_system_name);
		my ($worldTrainingArrRef, $worldTrainingSessionsLists)	= &SR_lib::processConfigurations($trainingWorldConf);

		# split the list to equal parts (num of parts is given by $hostsNum)
		&SR_lib::split_fileList($multispeaker_list, $hostsNum);
		for (my $iteration_counter = $first_iteration_no; $iteration_counter <= $num_iterations+$first_iteration_no-1; $iteration_counter++) {
			my $pm = new Parallel::ForkManager($hostsNum);
			
			$pm->run_on_finish(
				sub { my ($pid, $exit_code, $ident) = @_;
					  $ssh_stat[$ident] = $exit_code;
				}
			);

			my $world_hmmpathin=$world_path.$hmmBaseDir.$SR_lib::sign."hmm.".($iteration_counter-1);
			my $world_hmmpathout=$world_path.$hmmBaseDir.$SR_lib::sign."hmm.".($iteration_counter);
			my $macrosin=$world_hmmpathin;
			my $macrosOptionString = ($macros) ? "-H ${macrosin}/$macros" : "";
			&SR_lib::check_and_create_dir($world_hmmpathout);
			
			for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
				$pm->start($host_counter) and next;
				
				if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
				if ($onStartup) {
					if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
					$local_exit_code = system("ssh -x $hostList[$host_counter] \"$onStartup\"");
				}
				my $exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HERest.pl -dist=$host_counter -dist_iteration_counter=$iteration_counter  $useGMMString  -totalHostsNum=$hostsNum -w -wl $trainingWorldConf -h $hmm_list -m $mlf -v=$minVar -hmmBaseDir=$hmmBaseDir $macrosString $extraOptionsString $prefixCommandString $sv_system_name $num_iterations $first_iteration_no \"";
				if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
				$local_exit_code += system($exec_string);
				if ($onFinish) {
					if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
					$local_exit_code += system("ssh -x $hostList[$host_counter] \"$onFinish\"");
				}
				if ($local_exit_code) {
					$local_exit_code = 11;
				}
				
				if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
				if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}
	
				$pm->finish($local_exit_code);
		
			}
		    
			$pm->wait_all_children;
			if ($SR_lib::verbose_level) { print STDOUT ("All processes in this iteration step completed.\n");}
			my $join_stat=system ("HERest -C $global_HERest_config -p 0 $prompt_HTK_Command_String -M $world_hmmpathout $macrosOptionString -d $world_hmmpathin -v $minVar -I ${sv_system_path}$mlf  ${hmmlist_path}$hmm_list $world_hmmpathout/*.acc");					
			if ($join_stat) {
				exit 1;
			}
			if ($SR_lib::verbose_level) { print STDOUT ("All *.acc joined.\n");}
			unlink <${world_hmmpathout}${SR_lib::sign}*.acc>;
		} # end for ($iteration_counter)
		&SR_lib::write_trainingInfoFile(($world_path.$hmmBaseDir), $trainingWorldConf);
		if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
	
		if ($SR_lib::verbose_level) { print STDOUT ("Deleting split lists...\n");}
		my $distList_dir = &SR_lib::get_distList_dir($multispeaker_list);	
		my $rm_status = system("rm -r $distList_dir");	
		if ($rm_status != 0) {
			print STDERR "Error while deleting split lists in $sv_system_path . \n";
		}
		&SR_lib::remove_multispeaker_list($sv_system_name);
 
		if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
		
		for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
			$exit_state += $ssh_stat[$host_counter];
		}
		if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}
		

	}
	else 
	{
			if ($distributedClient) {
					my $multispeaker_list = &SR_lib::get_filename_multispeaker_list($sv_system_name);
					my $exe_script_path = &SR_lib::get_distList_dir($multispeaker_list);
					my $exe_script_file = &SR_lib::get_distList_name($distributedClient);

					my $world_hmmpathin=$world_path.$hmmBaseDir.$SR_lib::sign."hmm.".($dist_iteration_counter-1);
					my $world_hmmpathout=$world_path.$hmmBaseDir.$SR_lib::sign."hmm.".($dist_iteration_counter);
					my $macrosin=$world_hmmpathin;
					my $macrosOptionString = ($macros) ? "-H ${macrosin}/$macros" : "";
				
					$exit_state = system ("$prefixCommand ${SR_lib::htk_bin}HERest -C $global_HERest_config -p $distributedClient $prompt_HTK_Command_String -M $world_hmmpathout $macrosOptionString -d $world_hmmpathin -v $minVar -S ${exe_script_path}$exe_script_file  -I ${sv_system_path}$mlf $extraOptions ${hmmlist_path}$hmm_list");			
			}
			else
			{
				&SR_lib::create_multispeaker_list($sv_system_name, [$trainingWorldConf]);
				my ($worldTrainingArrRef, $worldTrainingSessionsLists) = &SR_lib::processConfigurations($trainingWorldConf);

				my $multispeaker_list = &SR_lib::get_filename_multispeaker_list($sv_system_name);

				for (my $iteration_counter = $first_iteration_no; $iteration_counter <= $num_iterations+$first_iteration_no-1; $iteration_counter++) {
					my $world_hmmpathin=$world_path.$hmmBaseDir.$SR_lib::sign."hmm.".($iteration_counter-1);
					my $world_hmmpathout=$world_path.$hmmBaseDir.$SR_lib::sign."hmm.".($iteration_counter);
					my $macrosin=$world_hmmpathin;
					my $macrosOptionString = ($macros) ? "-H ${macrosin}/$macros" : "";
					&SR_lib::check_and_create_dir($world_hmmpathout);
				
					$exit_state = system ("$prefixCommand ${SR_lib::htk_bin}HERest -C $global_HERest_config $prompt_HTK_Command_String -M $world_hmmpathout $macrosOptionString -d $world_hmmpathin -v $minVar -S $multispeaker_list  -I ${sv_system_path}$mlf $extraOptions ${hmmlist_path}$hmm_list");
					if ($exit_state) {
						print STDERR "Command failed\n";
						exit 1;
					}
				} # end for ($iteration_counter)
				&SR_lib::remove_multispeaker_list($sv_system_name);

				&SR_lib::write_trainingInfoFile(($world_path.$hmmBaseDir), $trainingWorldConf);
				
			}
		}
				


} # end if (! $buildWorldModel) {


if (! $distributedClient) {
	if ($onFinish) {
		if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
		$local_exit_code = system($onFinish);
	}
}



if (! $distributedClient) {

	&SR_lib::check_and_create_dir($log_path);
	
	my $log_handle = &SR_lib::open_file(">>", "$log_path" . "$log_file");
	print $log_handle "\n\n";
	print $log_handle "$start_time \n";
	print $log_handle "Running run_HERest.pl .....\n";
	print $log_handle "---------------------------\n";
	print $log_handle "Parameters:\n";
	print $log_handle "sv_system        : $sv_system_path\n";
	print $log_handle "num_iterations    : $num_iterations\n";
	print $log_handle "first_iteration_no: $first_iteration_no\n";
	print $log_handle "Options:\n";
	print $log_handle "useGMM            : $useGMM\n";
	print $log_handle "useHEAdapt        : $useHEAdapt\n";
	print $log_handle "varFloor          : $minVar\n";
	print $log_handle "mlf               : $mlf\n";
	print $log_handle "hmm_list          : $hmm_list\n";
	print $log_handle "buildWorld        : $buildWorldModel\n";
	if ($buildWorldModel) {
		print $log_handle "wl                : $trainingWorldConf\n";
	} else {
		print $log_handle "tc                : $trainingModelConf\n";
	}
	print $log_handle "hmmBaseDir        : $hmmBaseDir\n";
	print $log_handle "macros            : $macros\n";
	print $log_handle "onStartup         : $onStartup\n";
	print $log_handle "onFinish          : $onFinish\n";
	print $log_handle "prefixCommand     : $prefixCommand\n";
	print $log_handle "extraOptions      : $extraOptions\n";
	print $log_handle "logfile           : $log_file\n";
	print $log_handle "cvs_version       : $CVS_Version_String\n";
	print $log_handle "cvs_name          : $CVS_Name_String\n";
	print $log_handle "Finishing... ",scalar localtime, "\n";
	
	if ($exit_state) {
		print $log_handle "\n Command failed !!!\n";
	}
	
	
	close $log_handle;
	
}

exit $exit_state;




__END__

=head1 NAME

run_HERest.pl  - run HERest for a SV system

=head1 SYNOPSIS

run_HERest.pl [options] sv_system_name
                        num_iterations [first_iteration_no]

Run HERest for the SV system <sv_system_name>. Performs <num_iterations> iterations, 
starting by default with model version 0. 
Optional parameter <first_iteration_no> gives number of the 
first model version to be built when starting the iteration loop. Default value is 
therefore 1.

 Options:
 
 -w | world                make a single (world) model 
                          
 -wl | world_list          configuration for world flat start
                           default is 'world_set/aa'
                          
 -tc | trainingModelConf   use configuration for normal estimation step
                           default is 'all_training/at'
                           see "sub processConfigurations" in SR_lib.pm
 
 -m | mlf                  specify mlf file, default is "mlf_prompted.mlf"
                          
 -h | hmm_list             specify hmm list, default is "HMM_number.list"
                          
 -v | minVar               minimum value for the elements of the covariance matrix
                           of the model. Default value is 0.001

 -hmmBaseDir               build models in given directory in 
                           speaker resp. world directory
                           default is "hmm"

 -macros                   use given file for HMM macros; it must reside in the directory
                           of the HMM models

 -useGMM                   adapt behaviour for using a single state model with multiple
                           mixtures (GMM).
 
 -useHEAdapt               use adapting scheme of HEAdapt with MAP procedure instead of  
                           Baum-Welch reestimation; does not work with -w.
                           MAP adaption factor is fixed to 15.0.
                           Works here only with patched version of HEAdapt.
 
 -xo | extraOptions        pass extra options to HERest

 -onStartup=s              command to perform before running on parallel host

 -onFinish=s               command to perform after running on parallel host

 -prefixCommand=s          command to perform before running HERest

 -p | parallelComputation  use several hosts for running HERest
                           (useful for normal and world training)

 -logfile                  set logfile; default is log/log.txt.

 -? | help                 display this message.
 
=cut

