#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: switch_MASV_spk_data_set.pl,v 1.6 2004/03/31 17:38:57 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use strict;

use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

my ($hmmsToAdapt, $help, $Identify, $sv_system_name, $poolname, $spkset, $pool_full_path, $sv_system_full_path,
	$spkset_full_path, $links_full_path, $exec_string, $suffix, $counter);

###########################
# Default values for options
$hmmsToAdapt="";
$help = 0;
$Identify = 0;
###########################


# Processing Options

GetOptions('hmmsToAdapt=s' => \$hmmsToAdapt,
           'help|?' => \$help,
           'version' => \$Identify);

if ($Identify) {
	printf "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

my $hmmsDir;
my $paramType;

if ( $hmmsToAdapt ) {
	my @tempItems = split('/', $hmmsToAdapt);
	if (scalar @tempItems != 2) {
		print STDERR "option hmmsToAdapt: use a slash to seperate hmm_dir and param_type!\n\n";
		exit 1;
	}
	$hmmsDir = $tempItems[0];
	$paramType = $tempItems[1];
}


if ( ((@ARGV != 3) && (@ARGV != 2) ) || ($help)) {
   pod2usage(1);
   exit 1;
}

$sv_system_name = $ARGV[0];
$spkset = $ARGV[1];
if (@ARGV == 3) {
	$poolname = $ARGV[2];
	$pool_full_path = ${SR_lib::paramPool_dir} . $poolname;
}

# read speaker set
&SR_lib::init_speaker_sets($sv_system_name);


$sv_system_full_path = ${SR_lib::sv_systems_dir} . $sv_system_name;
$spkset_full_path = $sv_system_full_path . ${SR_lib::sign} . "speaker_sets" . ${SR_lib::sign} . $spkset;

$links_full_path = $sv_system_full_path . ${SR_lib::sign} . "links";


#check Paths
&SR_lib::check_and_create_dir($links_full_path);

# create main link
if (@ARGV == 3) {
	unlink <${links_full_path}${SR_lib::sign}currentParamPool>;
	$exec_string = "ln -s $pool_full_path ${links_full_path}${SR_lib::sign}currentParamPool";
	&SR_lib::run_and_check_command($exec_string, 0);
}

unlink <${links_full_path}${SR_lib::sign}currentSpeakerSet>;
$exec_string = "ln -s $spkset_full_path ${links_full_path}${SR_lib::sign}currentSpeakerSet";
&SR_lib::run_and_check_command($exec_string, 0);


if ( $hmmsToAdapt ) {
	
	my $hmm_dir_full_path = $sv_system_full_path . ${SR_lib::sign} . $hmmsDir;
	opendir SRCDIR, $hmm_dir_full_path;
	my @hmms = grep(!/^\./, readdir(SRCDIR)); #Forget about . files
	closedir SRCDIR;

	foreach my $hmm_file (@hmms) {
		&change_hmm_param_type( ($hmm_dir_full_path . ${SR_lib::sign} . $hmm_file), $paramType);
	}
}

exit 0;


sub change_hmm_param_type {

	my $hmm_file = $_[0];
	my $param_type = $_[1];
	
	my $fh = &SR_lib::open_file("<", $hmm_file);
	my @hmm = <$fh>;
	close ($fh);
	
	$fh = &SR_lib::open_file(">", $hmm_file);
	
	foreach my $line (@hmm) {
		$line =~ s/<(USER|MFCC|LPC|LPCEPSTRA|LPDELCEP|IREFC|FBANK|MELSPEC)[^>]*>/<$param_type>/;
		print $fh $line;		
	}
	close ($fh);
	
}

sub create_individual_links {

	foreach my $spk_set (keys(%SR_lib::speakerlist)) {
		if ($spk_set eq "world_set") {
			$suffix = "W";
		}
		elsif ($spk_set eq "dev_set") {
			$suffix = "D";
		}
		elsif ($spk_set eq "training_set") {
			$suffix = "M";
		}
		elsif ($spk_set eq "test_set") {
			$suffix = "I";
		}
		else {
			next;
		}
		
		$counter = 1;
		foreach my $speaker (@{$SR_lib::speakerlist{$spk_set}}) {
			$exec_string = "ln -s ${pool_full_path}${SR_lib::sign}$speaker ${links_full_path}${SR_lib::sign}currentParamPool_" . $suffix . sprintf("%.3d",$counter);
			&SR_lib::run_and_check_command($exec_string, 0);
			$counter++;
		}
	}
	
}



__END__

=head1 NAME

switch_MASV_spk_data_set.pl  - change speaker set configuration and parameter pool.

=head1 SYNOPSIS

switch_MASV_spk_data_set.pl [options]  sv_system_name  speaker_set_name  [poolname]

Creates a new link structure in the SV system directory "links/" which
points to the given speaker set and (optionally) to the given parameterpool. 

 Options:
                          
 -hmmsToAdapt=s           change parameter type of hmms (typically prototype hmms);
                          the argument contains two values seperated by a slash ('/'):
                          path to directory with hmms and new parameter type.
                          e.g. "proto_1G_gmm/USER_D_A"
 
 -? | help                display this message.
 
=cut

