/**************************************************************************
***************************************************************************

Modul           : audio.c
Titel           : Several audiofunctions on sun
Autor           : F. Schiel
Datum/Aenderung : 15.02.95 / 27.02.95

Beschreibung:

Anzulinkende Module	:
laudio,lm

Subroutinen:
sparc10_audio		: ausgabe auf audio() device an Sparc 10 
sparc_audio		: ausgabe 16 bit linear auf audio() device an 
                          Sparc 1, 2 oder 10.
sparc_audio_ulaw	: ausgabe 8 bit ulaw auf audio() 
                          device an Sparc 1, 2 oder 10.
sparc_audio_alaw	: ausgabe 8 bit alaw auf audio() 
                          device an Sparc 1, 2 oder 10.

*/
#include <stdio.h>                  
#include <errno.h>                  
#include <ctype.h>                  
#include <stdlib.h>                 
#include <fcntl.h>                  
#include <sys/types.h>              
#include <sys/file.h>               
#include <sys/stat.h>               
#include <sys/param.h>              
#include <sys/signal.h>             
                                    
#include <stropts.h>                
#include <sys/ioctl.h>              
                                    
#include <multimedia/libaudio.h>    
#include <multimedia/audio_device.h>
#if defined(__SVR4) || defined(__svr4__)
#include <multimedia/audio_encode.h>
#define AUDIO_DEV_AMD 0
#define AUDIO_DEV_SPEAKERBOX 1
#else
#include <multimedia/ulaw2linear.h>
#endif
# include "ipkclib.h"

/* DEFINES, only used within this module ********************************/
 

/* TYPE DEFINITIONS, only used within this module  **********************/
                                                                          
                                                                          
/* GLOBAL VARIABLES (avoid, if possible !) ******************************/
                                                                          
                                                                          
/* FUNCTION PROTOTYPES, capsuled in this module *************************/


/*----------------------------------------------------------------------
Name            : sparc10_audio
Module		: audio.c
Title           : gibt short array auf device audio() aus

Description:
schreibt anz werte in samples auf das pseudodevice /dev/audio linear, 16 
Bit mit rate Hz Abtastrate. Je nach dem Flag port wird das Signal auf
Speaker (0) oder Headphone Jack (1) ausgegeben.

Parameters:
samples		: array mit signaldaten in SUN format
anz		: anzahl der werte in samples
rate		: abtastrate in HZ
volume		: lautst"arke im Bereich 0.0 - 1.0
port		: 0 : ausgabe auf speaker, 1 : ausgabe auf headphone jack

Return-Value    : 1 schreiben erfolgreich
		  0 Fehler aufgetreten
-------------------------------------------------------------------------*/
short sparc10_audio(short *samples,long anz,long rate,double volume,
               unsigned int port)
{ 
	char *audio_dev = "/dev/audio";
	int err;
	int audio_fd = -1;	    /* file descriptor for audio device */
	Audio_hdr audio_hdr;
        long pld,werr;

/* check of parameters							*/
    if(rate < 0 || anz < 0)
    {
	fprintf(stderr,"sparc10_audio: sample rate or count is negative: %d %d\n",
		rate,anz);
	return(0);
    }
    if(volume < 0.0 || volume > 1.0)
    {
	fprintf(stderr,"sparc10_audio: volume is not within range 0.0-1.0\n");
	volume = 0.8;
    }
    if(port == 0)
      port = AUDIO_SPEAKER;
    else if(port == 1)
      port = AUDIO_HEADPHONE;
    else
    {
	fprintf(stderr,"sparc10_audio: port flag incorrect: %d\n",port);
	return(0);
    }

/* default audio device "offnen						*/
	audio_fd = open(audio_dev, O_WRONLY | O_NDELAY);
	if ((audio_fd < 0) && (errno == EBUSY)) 
	{
		fprintf(stderr, "sparc10_audio: %s is busy\n",audio_dev);
		return(0);
	}
	if (audio_fd < 0) {
		fprintf(stderr, "sparc10_audio: error opening audio device %s\n",audio_dev);
		return(0);
	}

/* Output Configuration in Audio Header setzen:
   sample rate: rate Hz   
   precision  : 16 Bit
   encoding   : linear							*/
    audio_hdr.sample_rate = rate;
    audio_hdr.samples_per_unit = 1;
    audio_hdr.bytes_per_unit = 2;
    audio_hdr.channels = 1;
    audio_hdr.encoding = AUDIO_ENCODING_LINEAR;
    audio_hdr.data_size = AUDIO_UNKNOWN_SIZE;

/* Audio Header auf device setzen					*/
    if(audio_set_play_config(audio_fd,&audio_hdr) != AUDIO_SUCCESS)
    {
	fprintf(stderr,"sparc10_audio: could not set configuration in audio device %s\n",audio_dev);
        (void) close(audio_fd);			/* close output */
	return(0);
    }

/* volume setzen							*/
    err = audio_set_play_gain(audio_fd,&volume);
    if (err != AUDIO_SUCCESS)
	fprintf(stderr,"sparc10_audio: couldn't set Volume: %f\n",volume);

/* port setzen								*/
    err = audio_set_play_port(audio_fd,&port);
    if (err != AUDIO_SUCCESS)
	fprintf(stderr,"sparc10_audio: couldn't set output port\n");

/* Daten aus samples auf device schreiben				*/
  for( werr=0,pld=0 ; pld < anz*2; )
    {
    werr = write(audio_fd, (char *)samples + pld, anz*2 - pld);
    if ( werr < 0 ) 
      {
      switch( errno )
        {
        case EAGAIN:
          continue;
        default:
          fprintf(stderr, "sparc10_audio error: %s\n",strerror(errno));
          (void) close(audio_fd);                       /* close output */
          return(0);
        }
      }
    pld += werr;
    }

/* audio device schliessen						*/
    (void) close(audio_fd);			/* close output */

    return(1);


} /* end subroutine : sparc10_audio */
/*----------------------------------------------------------------------
Name            : sparc_audio
Module		: audio.c
Title           : gibt short array auf device audio() aus

Description:
schreibt anz werte in samples auf das pseudodevice /dev/audio aus.
Entweder linear, 16 Bit mit rate Hz Abtastrate (Sparc 10 speakerbox)
oder mit ulaw 8 Bit mit 8 kHz (Sparc 1 oder 2). 
Je nach dem Flag port wird das Signal auf
Speaker (0) oder Headphone Jack (1) oder Lineout Jack (2) ausgegeben.
BEACHTE: Bei Ausgabe an Sparc 1 oder 2 koennen nur Vielfache von
8 kHz Abtastrate ausgegeben werden. Diese werden ohne Filterung
auf 8 kHz downgesampelt.

Parameters:
samples		: array mit signaldaten in SUN format
anz		: anzahl der werte in samples
rate		: abtastrate in HZ
volume		: lautst"arke im Bereich 0.0 - 1.0
port		: 0 : ausgabe auf speaker, 1 : ausgabe auf headphone jack
                  2 : ausgabe auf line out (nur Sparc 10 speakerbox)

Return-Value    : 1 schreiben erfolgreich
		  0 Fehler aufgetreten
-------------------------------------------------------------------------*/
short sparc_audio(short *samples,long anz,long rate,double volume,
               unsigned int port)
{ 
	char *audio_dev = "/dev/audio";
 	unsigned char *newsamples;
	int err,downsamp;
	int audio_fd = -1;	    /* file descriptor for audio device */
#if defined(__svr4__) || defined(__SVR4)
	struct audio_device audev;
#endif
	int device;
        long newanz,i;
	Audio_hdr audio_hdr;
        long pld,werr;

/* check of parameters							*/
    if(rate < 0 || anz < 0)
    {
	fprintf(stderr,"sparc_audio: sample rate or count is negative: %d %d\n",
		rate,anz);
	return(0);
    }
    if(volume < 0.0 || volume > 1.0)
    {
	fprintf(stderr,"sparc_audio: volume is not within range 0.0-1.0\n");
	volume = 0.8;
    }

/* default audio device "offnen, falls nicht sofort verfuegbar abbrechen */
	audio_fd = open(audio_dev, O_WRONLY | O_NDELAY);
	if ((audio_fd < 0) && (errno == EBUSY)) 
	{
		fprintf(stderr, "sparc_audio: %s is busy\n",audio_dev);
		return(0);
	}
	if (audio_fd < 0) {
		fprintf(stderr, "sparc_audio: error opening audio device %s\n",audio_dev);
		return(0);
	}

/* Check for type of audio device: Sparc 1 or 2 or Sparc 10 */
#if defined(__svr4__) || defined(__SVR4)
    if(ioctl(audio_fd,AUDIO_GETDEV,&audev))
#else
    if(ioctl(audio_fd,AUDIO_GETDEV,&device))
#endif
    { 
      fprintf(stderr, "sparc_audio: error checking type of audio device %s\n",
              audio_dev);
      (void) close(audio_fd);			/* close output */
      return(0);
    }
#if defined(__svr4__) || defined(__SVR4)
    device = (strncmp(audev.name,"SUNW,am79c30",12)!=0);
#endif
/* Output Configuration in Audio Header setzen:
   If Sparc 10 speakerbox:
   sample rate: rate Hz   
   precision  : 16 Bit
   encoding   : linear							
   If Sparc 1 or 2 audioamd:
   sample rate: 8000 Hz 
   precision  : 8 Bit  
   encoding   : ulaw  						*/
    if(device == AUDIO_DEV_SPEAKERBOX)
    {
      audio_hdr.sample_rate = rate;
      audio_hdr.samples_per_unit = 1;
      audio_hdr.bytes_per_unit = 2;
      audio_hdr.channels = 1;
      audio_hdr.encoding = AUDIO_ENCODING_LINEAR;
      audio_hdr.data_size = AUDIO_UNKNOWN_SIZE;
    }
    else if (device == AUDIO_DEV_AMD)
    {                                             
      if((rate % 8000) != 0)
      { 
        fprintf(stderr, "sparc_audio: amd device can play only multiple of 8000 Hz rate %s\n",
	audio_dev);
        (void) close(audio_fd);			/* close output */
        return(0);
      }
      downsamp = rate/8000;
      newanz = anz/downsamp;
      if((newsamples = (unsigned char *)malloc(newanz)) == NULL)
      { 
      fprintf(stderr, "sparc_audio: not enough memory\n");
      (void) close(audio_fd);			/* close output */
      return(0);
      }
/* covert 16 bit, 16 kHz linear to 8 bit, 8 kHz ulaw */
      for(i=0;i<newanz;i++) newsamples[i] = audio_s2u(samples[i*downsamp]);
      audio_hdr.sample_rate = 8000;               
      audio_hdr.samples_per_unit = 1;             
      audio_hdr.bytes_per_unit = 1;               
      audio_hdr.channels = 1;                     
      audio_hdr.encoding = AUDIO_ENCODING_ULAW; 
      audio_hdr.data_size = AUDIO_UNKNOWN_SIZE;   
    }                                             
    else
    { 
      fprintf(stderr, "sparc_audio: unknown type of audio device %s (%d)\n",
	audio_dev,device);
      (void) close(audio_fd);			/* close output */
      return(0);
    }

/* Audio Header auf device setzen					*/
    if(audio_set_play_config(audio_fd,&audio_hdr) != AUDIO_SUCCESS)
    {
      fprintf(stderr,"sparc_audio: could not set configuration in audio device %s\n",audio_dev);
      (void) close(audio_fd);			/* close output */
      if(device == AUDIO_DEV_AMD) free(newsamples);
      return(0);
    }

/* volume setzen							*/
    err = audio_set_play_gain(audio_fd,&volume);
    if (err != AUDIO_SUCCESS)
	fprintf(stderr,"sparc_audio: couldn't set Volume: %f\n",volume);

/* port setzen								*/
    if(port == 0)
      port = AUDIO_SPEAKER;
    else if(port == 1)
      port = AUDIO_HEADPHONE;
    else if(port == 2)
    {
      if(device == AUDIO_DEV_SPEAKERBOX)
        port = AUDIO_LINE_OUT;
      else 
      { 
        fprintf(stderr,"sparc_audio_ulaw: cannot use line out - using speaker\n");
        port = AUDIO_SPEAKER;  
      } 
    }
    else
    {
	fprintf(stderr,"sparc_audio: port flag incorrect: %d\n",port);
	return(0);
    }
    err = audio_set_play_port(audio_fd,&port);
    if (err != AUDIO_SUCCESS)
	fprintf(stderr,"sparc_audio: couldn't set output port\n");

/* Daten aus samples auf device schreiben				*/
    if(device == AUDIO_DEV_SPEAKERBOX)
      {
      for( werr=0,pld=0 ; pld < anz*2; )
        {
        werr = write(audio_fd, (char *)samples + pld, anz*2 - pld);
        if ( werr < 0 ) 
          {
          switch( errno )
            {
            case EAGAIN:
              continue;
            default:
              fprintf(stderr, "sparc_audio error: %s\n",strerror(errno));
              (void) close(audio_fd);                       /* close output */
              return(0);
            }
          }
        pld += werr;
        }
      }
    else
      {
      for( werr=0,pld=0 ; pld < newanz; )
        {
        werr = write(audio_fd, (char *)newsamples + pld, newanz - pld);
        if ( werr < 0 ) 
          {
          switch( errno )
            {
            case EAGAIN:
              continue;
            default:
              fprintf(stderr, "sparc10_audio error: %s\n",strerror(errno));
              free(newsamples);
              (void) close(audio_fd);                       /* close output */
              return(0);
            }
          }
        pld += werr;
        }
      free(newsamples);
      }


/* audio device schliessen						*/
    (void) close(audio_fd);			/* close output */
 
    return(1);


} /* end subroutine : sparc_audio */
/*----------------------------------------------------------------------
Name            : sparc_audio_ulaw
Module		: audio.c
Title           : gibt char array ulaw auf device audio() aus

Description:
schreibt anz chars aus samples auf das pseudodevice /dev/audio aus.
Mit ulaw coding 8 Bit, 8 kHz (Sparc 1, 2 oder 10). 
Je nach dem Flag port wird das Signal auf
Speaker (0) oder Headphone Jack (1) oder Lineout Jack (2) ausgegeben.
BEACHTE: Bei Ausgabe an Sparc 1 oder 2 koennen nur Vielfache von
8 kHz Abtastrate ausgegeben werden. Diese werden ohne Filterung
auf 8 kHz downgesampelt. Desgleichen ist eine Ausgabe auf line out nicht 
moeglich.

Parameters:
samples		: array mit signaldaten 8 bit ulaw
anz		: anzahl der werte in samples
volume		: lautst"arke im Bereich 0.0 - 1.0
port		: 0 : ausgabe auf speaker, 1 : ausgabe auf headphone jack
                  2 : ausgabe auf line out (nur Sparc 10 speakerbox)

Return-Value    : 1 schreiben erfolgreich
		  0 Fehler aufgetreten
-------------------------------------------------------------------------*/
short sparc_audio_ulaw(unsigned char *samples,long anz,double volume,
               unsigned int port)
{ 
	char *audio_dev = "/dev/audio";
	int err;
	int audio_fd = -1;	    /* file descriptor for audio device */
#if defined(__svr4__) || defined(__SVR4)
	struct audio_device audev;
#endif
	int device;
        long i;
	Audio_hdr audio_hdr;
        long pld,werr;

/* check of parameters							*/
    if(anz < 0)
    {
	fprintf(stderr,"sparc_audio_ulaw: sample count is negative: %d %d\n",
		anz);
	return(0);
    }
    if(volume < 0.0 || volume > 1.0)
    {
	fprintf(stderr,"sparc_audio_ulaw: volume is not within range 0.0-1.0\n");
	volume = 0.8;
    }

/* default audio device "offnen, falls nicht sofort verfuegbar abbrechen */
	audio_fd = open(audio_dev, O_WRONLY | O_NDELAY);
	if ((audio_fd < 0) && (errno == EBUSY)) 
	{
		fprintf(stderr, "sparc_audio_ulaw: %s is busy\n",audio_dev);
		return(0);
	}
	if (audio_fd < 0) {
		fprintf(stderr, "sparc_audio_ulaw: error opening audio device %s\n",audio_dev);
		return(0);
	}

/* Check for type of audio device: Sparc 1 or 2 or Sparc 10 */
#if defined(__svr4__) || defined(__SVR4)
    if(ioctl(audio_fd,AUDIO_GETDEV,&audev))
#else
    if(ioctl(audio_fd,AUDIO_GETDEV,&device))
#endif
    { 
      fprintf(stderr, "sparc_audio_ulaw: error checking type of audio device %s\n",
              audio_dev);
      (void) close(audio_fd);			/* close output */
      return(0);
    }
#if defined(__svr4__) || defined(__SVR4)
    device = (strncmp(audev.name,"SUNW,am79c30",12)!=0);
#endif
/* Output Configuration in Audio Header setzen */
    audio_hdr.sample_rate = 8000;
    audio_hdr.samples_per_unit = 1;
    audio_hdr.bytes_per_unit = 1;
    audio_hdr.channels = 1;
    audio_hdr.encoding = AUDIO_ENCODING_ULAW;
    audio_hdr.data_size = AUDIO_UNKNOWN_SIZE;

/* Audio Header auf device setzen					*/
    if(audio_set_play_config(audio_fd,&audio_hdr) != AUDIO_SUCCESS)
    {
      fprintf(stderr,"sparc_audio_ulaw: could not set configuration in audio device %s\n",audio_dev);
      (void) close(audio_fd);			/* close output */
      return(0);
    }

/* volume setzen							*/
    err = audio_set_play_gain(audio_fd,&volume);
    if (err != AUDIO_SUCCESS)
	fprintf(stderr,"sparc_audio_ulaw: couldn't set Volume: %f\n",volume);

/* port setzen								*/
    if(port == 0)
      port = AUDIO_SPEAKER;
    else if(port == 1)
      port = AUDIO_HEADPHONE;
    else if(port == 2)
    {
      if(device == AUDIO_DEV_SPEAKERBOX)
        port = AUDIO_LINE_OUT;
      else 
      { 
        fprintf(stderr,"sparc_audio_ulaw: cannot use line out - using speaker\n");
        port = AUDIO_SPEAKER;  
      } 
    }
    else
    {
	fprintf(stderr,"sparc_audio_ulaw: port flag incorrect: %d\n",port);
	return(0);
    }
    err = audio_set_play_port(audio_fd,&port);
    if (err != AUDIO_SUCCESS)
	fprintf(stderr,"sparc_audio: couldn't set output port\n");

/* Daten aus samples auf device schreiben				*/
  for( werr=0,pld=0 ; pld < anz*2; )
    {
    werr = write(audio_fd, (char *)samples + pld, anz*2 - pld);
    if ( werr < 0 ) 
      {
      switch( errno )
        {
        case EAGAIN:
          continue;
        default:
          fprintf(stderr, "sparc10_audio error: %s\n",strerror(errno));
          (void) close(audio_fd);                       /* close output */
          return(0);
        }
      }
    pld += werr;
    }


/* audio device schliessen						*/
    (void) close(audio_fd);			/* close output */
 
    return(1);


} /* end subroutine : sparc_audio_ulaw */
/*----------------------------------------------------------------------
Name            : sparc_audio_alaw
Module		: audio.c
Title           : gibt char array alaw auf device audio() aus

Description:
schreibt anz chars aus samples auf das pseudodevice /dev/audio aus.
Mit alaw coding 8 Bit, 8 kHz (Sparc 1, 2 oder 10). 
Je nach dem Flag port wird das Signal auf
Speaker (0) oder Headphone Jack (1) oder Lineout Jack (2) ausgegeben.
BEACHTE: Bei Ausgabe an Sparc 1 oder 2 koennen nur Vielfache von
8 kHz Abtastrate ausgegeben werden. Diese werden ohne Filterung
auf 8 kHz downgesampelt. Desgleichen ist eine Ausgabe auf line out nicht 
moeglich.

Parameters:
samples		: array mit signaldaten 8 bit ulaw
anz		: anzahl der werte in samples
volume		: lautst"arke im Bereich 0.0 - 1.0
port		: 0 : ausgabe auf speaker, 1 : ausgabe auf headphone jack
                  2 : ausgabe auf line out (nur Sparc 10 speakerbox)

Return-Value    : 1 schreiben erfolgreich
		  0 Fehler aufgetreten
-------------------------------------------------------------------------*/
short sparc_audio_alaw(unsigned char *samples,long anz,double volume,
               unsigned int port)
{ 
	char *audio_dev = "/dev/audio";
	int err;
	int audio_fd = -1;	    /* file descriptor for audio device */
#if defined(__svr4__) || defined(__SVR4)
	struct audio_device audev;
#endif
	int device;
        long i;
	Audio_hdr audio_hdr;
        long pld,werr;

/* check of parameters							*/
    if(anz < 0)
    {
	fprintf(stderr,"sparc_audio_alaw: sample count is negative: %d %d\n",
		anz);
	return(0);
    }
    if(volume < 0.0 || volume > 1.0)
    {
	fprintf(stderr,"sparc_audio_alaw: volume is not within range 0.0-1.0\n");
	volume = 0.8;
    }

/* default audio device "offnen, falls nicht sofort verfuegbar abbrechen */
	audio_fd = open(audio_dev, O_WRONLY | O_NDELAY);
	if ((audio_fd < 0) && (errno == EBUSY)) 
	{
		fprintf(stderr, "sparc_audio_alaw: %s is busy\n",audio_dev);
		return(0);
	}
	if (audio_fd < 0) {
		fprintf(stderr, "sparc_audio_alaw: error opening audio device %s\n",audio_dev);
		return(0);
	}

/* Check for type of audio device: Sparc 1 or 2 or Sparc 10 */
#if defined(__svr4__) || defined(__SVR4)
    if(ioctl(audio_fd,AUDIO_GETDEV,&audev))
#else
    if(ioctl(audio_fd,AUDIO_GETDEV,&device))
#endif
    { 
      fprintf(stderr, "sparc_audio_alaw: error checking type of audio device %s\n",
              audio_dev);
      (void) close(audio_fd);			/* close output */
      return(0);
    }
#if defined(__svr4__) || defined(__SVR4)
    device = (strncmp(audev.name,"SUNW,am79c30",12)!=0);
#endif
/* Output Configuration in Audio Header setzen */
    audio_hdr.sample_rate = 8000;
    audio_hdr.samples_per_unit = 1;
    audio_hdr.bytes_per_unit = 1;
    audio_hdr.channels = 1;
    audio_hdr.encoding = AUDIO_ENCODING_ALAW;
    audio_hdr.data_size = AUDIO_UNKNOWN_SIZE;

/* Audio Header auf device setzen					*/
    if(audio_set_play_config(audio_fd,&audio_hdr) != AUDIO_SUCCESS)
    {
      fprintf(stderr,"sparc_audio_alaw: could not set configuration in audio device %s\n",audio_dev);
      (void) close(audio_fd);			/* close output */
      return(0);
    }

/* volume setzen							*/
    err = audio_set_play_gain(audio_fd,&volume);
    if (err != AUDIO_SUCCESS)
	fprintf(stderr,"sparc_audio_alaw: couldn't set Volume: %f\n",volume);

/* port setzen								*/
    if(port == 0)
      port = AUDIO_SPEAKER;
    else if(port == 1)
      port = AUDIO_HEADPHONE;
    else if(port == 2)
    {
      if(device == AUDIO_DEV_SPEAKERBOX)
        port = AUDIO_LINE_OUT;
      else 
      { 
        fprintf(stderr,"sparc_audio_alaw: cannot use line out - using speaker\n");
        port = AUDIO_SPEAKER;  
      } 
    }
    else
    {
	fprintf(stderr,"sparc_audio_alaw: port flag incorrect: %d\n",port);
	return(0);
    }
    err = audio_set_play_port(audio_fd,&port);
    if (err != AUDIO_SUCCESS)
	fprintf(stderr,"sparc_audio_alaw: couldn't set output port\n");

/* Daten aus samples auf device schreiben				*/
  for( werr=0,pld=0 ; pld < anz*2; )
    {
    werr = write(audio_fd, (char *)samples + pld, anz*2 - pld);
    if ( werr < 0 ) 
      {
      switch( errno )
        {
        case EAGAIN:
          continue;
        default:
          fprintf(stderr, "sparc10_audio error: %s\n",strerror(errno));
          (void) close(audio_fd);                       /* close output */
          return(0);
        }
      }
    pld += werr;
    }


/* audio device schliessen						*/
    (void) close(audio_fd);			/* close output */
 
    return(1);


} /* end subroutine : sparc_audio_alaw */
