/***********************************************************************
************************************************************************

Module		: hear_stf-2.1
Title		: playing stereo files to audio

Author		: F. Schiel
Datum/Aenderung : 21.02.95 / 21.07.95

Description	: 
Takes all filenames from command line that are not icsiargs
and plays them to a sparc or LINUX audio device.

Usage:
hear_stf [v=yes][port=0][swap=no][rate=<rate>]
         [length=<secs>] c=r|l|b file1 [file2 ...]

v=yes : prints infos about each file to stdout (default=no)
c=r|l|b : plays right channel (r), left channel (L) od both (b, default)
port=0|1 : 0 = speaker (default), 1 = headphone
vol=0.0-1.0 : SUNOS: audio volume (default 0.8)
              LINUX: audio volume factor
rate=16000 : sampling rate (default 16000)
swap=yes : swap the short samples (default no)
length=<seconds> : outputs only seconds from beginning

Linking modules or libraries:
ipkclib,audio,m

Contained functions:
			: 

*************************************************************************/
# include <stdio.h>
# include <ipkclib.h>

/* DEFINES, only used within this module ********************************/
# define VOLUME 0.80      /* Default Laustaerke 0.0 ... 1.0 */
# define PORT 0           /* Default Ausgabeport: 0 = speaker */
# define RATE 16000       /* Default sampling rate */


/* TYPE DEFINITIONS, only used within this module  **********************/


/* GLOBAL VARIABLES (avoid, if possible !) ******************************/


/* FUNCTION PROTOTYPES, capsuled in this module *************************/

main(int argc,const char **argv)
{
	char 	filenam[40],buff,
		*appname;
	char 	*channel,*charpoint;
	short   *data, *dataptr,*playdata;
	int	i,k,ret;
        int 	verb = 0;
 	int	type = 0;
        int     port = PORT;
        int     swap = 0;
	long	rate = RATE;
        long	sample,samples,bytes;
	float	length = 0.;
	double	volume = VOLUME;
	FILE	*file;
	icsiargtab_t args[] =
        {
          { 0, "2.0 : Playing all files from command line as stereo files to audio", ARG_DESC},
          { "v","verbose mode", ARG_BOOL, &verb },
          { "swap","swap short samples", ARG_BOOL, &swap },
	  { "c","channel (r|l|b)", ARG_STR, &channel, ARG_REQ },
	  { "port","audio port (0:speaker, 1:headphone, 2:lineout)", ARG_INT, &port },
	  { "rate","sampling rate", ARG_LONG, &rate },
	  { "vol","audio volume (0.0 - 1.0)", ARG_DOUBLE, &volume },
	  { "length","seconds of played signal", ARG_FLOAT, &length },
	  {0,0,0}
	};

  /* Test for icsiargs if any */
  icsiargs(args,&argc,&argv,&appname);
  if(verb) printargs(stdout,appname,args);
  if(argc < 1)
    {
    fprintf(stderr,"usage: %% %s [v=yes][port=0][vol=0.8][swap=no]\n\t\t[length=<seconds>] c=r|l|b file1 [file2 ...]\n",appname);
    exit(-1);
    }
#ifdef LINUX
  if(port != 0)
    {
    fprintf(stderr,"%s: other ports than 0 not implemented on LINUX\n",appname);
    port = 0;
    }
#endif 
  for(k=0;k<argc;k++)
  {
    strcpy(filenam,argv[k]);
    if(verb) printf("%s: File = %s\n",appname,filenam);
    if((file = fopen(filenam,"r")) == NULL)
    {
	fprintf(stderr,"%s: cannot open file %s\n",appname,filenam);
	perror(appname);
        continue;
    }
    /* check for size */
    fseek(file,0,2);
    bytes = ftell(file);
    samples = bytes/4;
    rewind(file);
    if(length)
    {
      if(length > (float)samples/rate)
        fprintf(stderr,"%s: not enough samples in file %s, playing all\n",
                appname,filenam);
      else
        samples = (int)(length*rate);
    }
    /* get memory for block wise read */
    if(((data = (short *)malloc(bytes)) == NULL)     ||
       ((playdata = (short *)malloc(bytes/2)) == NULL)  )
    {
      fprintf(stderr,"%s: not enough memory to read data in file %s\n",
              appname,filenam);
      perror(appname);
      fclose(file);
      continue;
    }
    if(verb) printf("%s: number of samples = %ld (%.2f sec)\n",
                    appname,samples,(float)samples/(float)rate);
    /* read speech data as a block (faster) */
    if(fread(data,sizeof(long),samples,file) != samples)
    {                                                                    
      fprintf(stderr,"%s: could not read data in file %s\n",  
              appname,filenam);                                          
      perror(appname);                                                   
      free(data);
      free(playdata);
      fclose(file);
      continue;
    }                                                                    
    fclose(file);
    switch (*channel)
    {
       case 'r':
       if(verb) printf("%s: playing right channel only\n",appname);
       dataptr = data + 1;
       for(i=0;i<samples;i++)
       {
         /* swap samples if necessary */
         if(swap) shortvms2sun(dataptr);
         playdata[i] = *dataptr;
         dataptr += 2;
       }
       break;
       case 'l':
       if(verb) printf("%s: playing left channel only\n",appname);
       dataptr = data;
       for(i=0;i<samples;i++)
       {
         /* swap samples if necessary */
         if(swap) shortvms2sun(dataptr);
         playdata[i] = *dataptr;
         dataptr += 2;
       }
       break;
       case 'b':
       if(verb) printf("%s: playing both channels\n",appname);
       dataptr = data;
       for(i=0;i<samples;i++)
       {
         /* swap samples if necessary */
         if (swap) shortvms2sun(dataptr);
         sample = (long)(*dataptr);
         dataptr++;
         /* swap samples if necessary */
         if (swap) shortvms2sun(dataptr);
         sample += (long)(*dataptr);
         sample /= 2; 
         playdata[i] = (short)sample;
         dataptr++;
       }
       break;
       default:
       fprintf(stderr,"%s: error - unknown option c = %s\n",appname,channel);
       exit(-1);
    }
    free(data);
    /* play data to sparc audio device */
#ifdef SUNOS
    if(verb) printf("%s: invoking sparc_audio\n", appname);
    if(!sparc_audio(playdata,samples,rate,volume,port))
    {
      fprintf(stderr,"%s: cannot use sparc audio device\n",appname);
      exit(-1);
    }
#endif 
#ifdef LINUX
    if(verb) printf("%s: invoking linux_audio\n", appname);
    if(linux_audio(playdata,2*samples,DSP_TYPE_LINEAR_16,rate,volume)
       != DSP_NO_ERROR)
    {
      fprintf(stderr,"%s: could not send data to linux audio card (error %d)\n",
                                                   appname,ret);
      exit(-1);
    }
#endif
    free(playdata);
  } /* ende schleife ueber argumente */
  return(0);
}
