/***********************************************************************
************************************************************************

Module          : raw2phondat-2.2
Title           : transform a raw sound file into a PhonDat sound file

Author          : F. Schiel
Date/revision   : 28.02.95 / 05.08.96

Ported to Linux.
05.05.95 : Changed parameters ortho and cano into possible file names.
           If they are filenames, these files are opened and the
           strings are read from these files instead of the command line.
26.06.95 : Fixed some errors.
27.12.95 : flag$init set default to 32149 (ILS init file)
05.08.96 : Fixed wrong calculation of abs_ampl under SUNOS

Description     : 
Reads a 16 bit raw sound file from stdin, builds a PhonDat header
with standard values or given option at stdout, calculates the
necessary values and appends the sound data to the header at stdout.
The result is a full compatible PhonDat file (version 1 or 2).
See man page for usage.

Link modules and libraries:
ipkclib

Contained functions:
                      : 

*************************************************************************/
# ifndef FILE
# include <stdio.h>
# endif
# include <ipkclib.h>
# include <time.h>

/* DEFINES, only used within this module ********************************/
# define VERSION 1
# define RATE 16000 
# define BITS 16

/* TYPE DEFINITIONS, only used within this module  **********************/


/* GLOBAL VARIABLES (avoid, if possible !) ******************************/


/* FUNCTION PROTOTYPES, capsuled in this module *************************/


main(int argc,const char **argv)
{
	char 	chr,*sex,sexdef[2] = "M",*appname,
                *spid,*filnam,spiddef[2] = {'\0','\0'},
		*ortho = NULL, *cano = NULL;
        char 	*chrptr1,*chrptr2;
	short	sample,swapsample,rest;
 	short	absampl = 0;
	int	i,j,k;
 	int	version = VERSION;
        int 	verb = 0, bits = BITS, words = 0,day = 0,month = 0,year = 0;
	long	bytes,err,skip = 0,rate = RATE;
	long	samplecount = 0; 
	FILE	*fpbuff,*fppar;
	icsiargtab_t args[] =
        {
          { 0, "2.0 : transforms a raw sound file into a Phondat soundfile, "
            "\n  usage: cat rawfile | raw2phondat [options] > phondatfile", ARG_DESC},
          { "v","verbose mode", ARG_BOOL, &verb },
          { "spid","PhonDat speaker ID (2 chars)", ARG_STR, &spid },
          { "rate","PhonDat sampling rate", ARG_LONG, &rate },
          { "skip","# of bytes skipped from input", ARG_LONG, &skip },
          { "filename","PhonDat file name", ARG_STR, &filnam },
          { "day","PhonDat recording day", ARG_INT, &day},
          { "month","PhonDat recording month", ARG_INT, &month},
          { "year","PhonDat recording year", ARG_INT, &year},
          { "sex","PhonDat sex of speaker", ARG_STR, &sex},
          { "version","PhonDat header version", ARG_INT, &version },
          { "bits","PhonDat resolution in bit", ARG_INT, &bits},
          { "words","PhonDat number of words", ARG_INT, &words},
          { "ortho","PhonDat 2 orthography (or file)", ARG_STR, &ortho},
          { "cano","PhonDat 2 canonical form (or file)", ARG_STR, &cano},
	  {0,0,0}
	};
 	Phon_header_2 header;
	time_t tim;
	struct tm *kal;

  tim = time(NULL);
  if(tim != -1) 
    kal = localtime(&tim);
  else
    kal = NULL;
  if(kal != NULL) 
  {
    day = kal->tm_mday;
    month = (kal->tm_mon+1);
    year = (1900+kal->tm_year);
  }
  spid = spiddef;
  sex = sexdef;
  filnam = spiddef;
  /* Test for icsiargs if any */
  icsiargs(args,&argc,&argv,&appname);
  if(verb) printargs(stderr,appname,args);
  /* Fill PhonDat header with life ... */
  header.sprk[0] = spid[0];
  header.sprk[1] = spid[1];
  header.swdh = 0;
  header.isf = rate;
  strncpy(header.ifl,filnam,31);
  header.ifl[31] = '\0';
  header.day = (char)day;
  header.month = (char)month;
  header.year = (short)year;
  header.sex = sex[0];
  header.version = (char)version;
  header.adc_bits = bits - 12;
  header.words = words;
  header.wdh = 0;
  header.flagtype = -32000;
  header.flaginit = 32149;
  if((header.version == 1) && ((cano != NULL) || (ortho != NULL)))
  {
    fprintf(stderr,"%s: assuming PhonDat header version 2 since arguments \n     ortho and/or cano are not empty - hope that's ok\n",appname);
    header.version = 2;
  }
  if((header.version == 2) && (ortho == NULL))
    fprintf(stderr,"%s: orthography is empty - hope that's ok\n",appname);
  if((header.version == 2) && (cano == NULL))
    fprintf(stderr,"%s: canonical form is empty - hope that's ok\n",appname);
  if(skip)
  {
    err = 0;
    for(i=0;i<skip;i++)
    {
      err += fread(&chr,sizeof(char),1,stdin);
    }
    if(err != skip)
    {
      fprintf(stderr,"%s: could not read %d bytes from stdin to skip\n",
              appname,skip);
      perror(appname);
      exit(-1);
    }
  if(verb) fprintf(stderr,"%s: skipped %d bytes\n",appname,skip);
  }
  /* open buffer for read (we have to count first !) */
  if((fpbuff = tmpfile()) == NULL)
  {                                                                      
    fprintf(stderr,"%s: could not open temporary file\n", appname);       
    perror(appname);
    exit(-1);                                                            
  }                                                                      
  /* copy samples to temporary file, look for maximum and count them */
  while(fread(&sample,sizeof(short),1,stdin) == 1)
  {
#ifdef SUNOS
    chrptr1 = (char *)&sample;
    chrptr2 = (char *)&swapsample;
    chrptr2++;
    *chrptr2 = *chrptr1;
    chrptr1++;
    chrptr2--;
    *chrptr2 = *chrptr1;
    if(swapsample > absampl) absampl = swapsample;
    if(swapsample < -absampl) absampl = -swapsample;
#endif
#ifdef LINUX
    if(sample > absampl) absampl = sample;
    if(sample < -absampl) absampl = -sample;
#endif
    if(fwrite(&sample,sizeof(short),1,fpbuff) != 1)
    {                                                                      
      fprintf(stderr,"%s: could not write to temporary file\n", appname);       
      perror(appname);
      fclose(fpbuff);
      exit(-1);                                                            
    }                                                                      
    samplecount++;
  } /* of while reading samples from stdin */
  if(verb) fprintf(stderr,"%s: writing PhonDat format %d\n",
                   appname,header.version);
  header.abs_ampl = absampl;
  if(verb) fprintf(stderr,"%s: absampl=%d, # of samples=%ld\n",
                   appname,header.abs_ampl,samplecount);
  bytes = 2 * samplecount;
  header.nspbk = bytes / 512;
  rest = bytes % 512;
  /* size of data block should be multiple of 1 block (512) */
  if(rest != 0) header.nspbk++;
  if(verb) fprintf(stderr,"%s: number of speech blocks (512) = %d\n",
                   appname,header.nspbk);
  /* check parameters ortho und cano for being filenames */
  if((fppar = fopen(ortho,"r")) != NULL)
  {
    free(ortho);
    fseek(fppar,0,SEEK_END);
    bytes = ftell(fppar)-1;
    fseek(fppar,0,SEEK_SET);
    if((ortho = (char *)calloc(bytes+1,sizeof(char))) == NULL)
    {
      fprintf(stderr,"%s: out of memory (ortho)\n",appname);
      exit(-1);
    }
    fread(ortho,bytes,sizeof(char),fppar);
    for(i=0;i<bytes;i++)
      if(ortho[i] == '\n')
        for(j=i;j<bytes;j++) ortho[j] = ortho[j+1];
  }
  if((fppar = fopen(cano,"r")) != NULL)
  {
    free(cano);
    fseek(fppar,0,SEEK_END);
    bytes = ftell(fppar)-1;
    fseek(fppar,0,SEEK_SET);
    if((cano = (char *)calloc(bytes+1,sizeof(char))) == NULL)
    {
      fprintf(stderr,"%s: out of memory (cano)\n",appname);
      exit(-1);
    }
    fread(cano,bytes,sizeof(char),fppar);
    for(i=0;i<bytes;i++)
      if(cano[i] == '\n')
        for(j=i;j<bytes;j++) cano[j] = cano[j+1];
  }
  /* now we are ready to write ... */
#ifdef LINUX
  if(!write_header_vms(stdout,&header,ortho,cano))
  {
    fprintf(stderr,"%s: cannot write header to stdout\n",appname);
    fclose(fpbuff);
    exit(-1);
  }
#endif
#ifdef SUNOS
  if(!write_header_sun(stdout,&header,ortho,cano))
  {
    fprintf(stderr,"%s: cannot write header to stdout\n",appname);
    fclose(fpbuff);
    exit(-1);
  }
#endif
  fseek(fpbuff,0,0);
  /* copy samples from temporary file to stdout */
  samplecount = 0;
  while(fread(&sample,sizeof(short),1,fpbuff) == 1)
  {
    if(fwrite(&sample,sizeof(short),1,stdout) != 1)
    {                                                                      
      fprintf(stderr,"%s: could not write to stdout\n", appname);       
      perror(appname);
      fclose(fpbuff);
      exit(-1);                                                            
    }                                                                      
    samplecount++;
  } 
  if(verb) fprintf(stderr,"%s: written %ld bytes\n",appname,
                   samplecount*2);
  fclose(fpbuff);
  /* fill up last block with zeros */
  chr = '\0';
  if(rest != 0) rest = 512 - rest;
  if(verb) fprintf(stderr,"%s: fill up with %d zeros\n",appname,rest);
  for(i=0;i<rest;i++)
  {
    if(fwrite(&chr,sizeof(char),1,stdout) != 1)
    {
    fprintf(stderr,"%s: could not write to stdout\n", appname);
    perror(appname);
    exit(-1);
    } 
  }
  free(ortho);
  free(cano);
  return(0);
}
