/***********************************************************************
************************************************************************

Module          : nist2phondat-1.1
Title           : transforms a NIST sound file into a PhonDat sound file

Author          : F. Schiel
Date/revision   : 28.06.95 / 19.07.95


Description     : 
Reads a soundfile with NIST header from file nistfile, extracts the 
necessary
information form the header, checks whether the sound format is 
compatibel to PhonDat, prompts for additional information or
gives a warning if there 's information missing (depends on
option settings, see there), creates a PhonDat 1 header in file
phonfile and writes the sound samples to that file.
The result is a full compatible PhonDat file (version 1).
See man page for usage.
Errors are reported to stderr.

Link modules and libraries:
ipkclib

Contained functions:
                      : 

*************************************************************************/
# ifndef FILE
# include <stdio.h>
# endif
# include <ipkclib.h>
# include <time.h>

/* DEFINES, only used within this module ********************************/
# define VERSION 1
# define RATE 16000 
# define BITS 16
# define REQ_OBJECTS 4

/* TYPE DEFINITIONS, only used within this module  **********************/


/* GLOBAL VARIABLES (avoid, if possible !) ******************************/


/* FUNCTION PROTOTYPES, capsuled in this module *************************/


main(int argc,const char **argv)
{
	char 	chr,*sex,sexdef[2] = {'\0','\0'},*appname,
                *spid,*nistfilnam,*phonfilnam,spiddef[2] = {'\0','\0'},
		*ortho = NULL, *cano = NULL,*charpoint;
        char	buff[256],object[256],type[256],value[256],
                months[5];
	short	sample,rest,channel;
 	short	absampl = 0, swap = 0;
	int	i,k;
 	int	version = VERSION,readnistheader = 0, req_objects = 0;
        int 	verb = 0, bits = 0, words = 0,day = 0,month = 0,year = 0;
	long	bytes,err,skip = 0,rate = RATE;
	long	headercount = 0, samplecount = 0,nistheadersize; 
	FILE	*fpnist,*fpphon;
	icsiargtab_t args[] =
        {
          { 0, "1.0 : transforms a NIST sound file into a Phondat soundfile, "
            "\n  usage: nist2phondat nistfile=<nistf> phonfile=<phonf> [options]", ARG_DESC},
          { "v","verbose mode", ARG_BOOL, &verb },
          { "spid","PhonDat speaker ID (2 chars)", ARG_STR, &spid },
          { "nistfile","NIST file name", ARG_STR, &nistfilnam, ARG_REQ},
          { "phonfile","PhonDat file name", ARG_STR, &phonfilnam, ARG_REQ},
          { "day","PhonDat recording day", ARG_INT, &day},
          { "month","PhonDat recording month", ARG_INT, &month},
          { "year","PhonDat recording year", ARG_INT, &year},
          { "sex","PhonDat sex of speaker", ARG_STR, &sex},
          { "bits","PhonDat resolution in bit", ARG_INT, &bits},
          { "words","PhonDat number of words", ARG_INT, &words},
	  {0,0,0}
	};
 	Phon_header_2 header;
  spid = spiddef;
  sex = sexdef;
  header.abs_ampl = 0;
  header.isf = RATE;
  /* Test for icsiargs if any */
  icsiargs(args,&argc,&argv,&appname);
  if(verb) printargs(stderr,appname,args);
  /* check and open inputfile */
  if((fpnist = fopen(nistfilnam,"r")) == NULL)
  {
    fprintf(stderr,"%s: error - could not open nistfile %s\n",
            appname,nistfilnam);
    exit(-1);
  }
  /* check and open outputfile */
  if((fpphon = fopen(phonfilnam,"w")) == NULL)
  {
    fprintf(stderr,"%s: error - could not open phonfile %s\n",
            appname,phonfilnam);
    exit(-1);
  }
  /* read first two lines from input and check for NIST */
  if(fgets(buff,256,fpnist) == NULL)
  {
    fprintf(stderr,"%s: error - could not read first line \n",
            appname);
    exit(-1);
  }
  headercount += strlen(buff);
  if(strncmp(buff,"NIST_1A",7) != 0)
  {
    fprintf(stderr,"%s: error - not a NIST/SPHERE file %s\n",
            appname,nistfilnam);
    exit(-1);
  }
  if(fgets(buff,256,fpnist) == NULL)
  {
    fprintf(stderr,"%s: error - could not read line\n",
            appname);
    exit(-1);
  }
  headercount += strlen(buff);
  sscanf(buff,"%ld",&nistheadersize);
  if(verb) fprintf(stderr,"%s: NIST/SPHERE header size = %ld\n",
                   appname,nistheadersize);
  /* read header and process known entries. 
  Note that the entries 'samples_rate', 'sample_count', 'sample_n_bytes',
  'channel_count' are required for transformation. If one of these 
  are missing an error is issued.
  Also note that if the following entries are present, they have to be
  set to the following values. If not, an error is issued.
  'recording_date'		: Format DD-MMM-YYYY
  'speaker_sex'			: one of M,F,m,f,Male,Female,male,female
  'sample_byte_format'		: 10 (Motorola)  or 01 (Intel)
  'sample_sig_bits'		: 14 - 16
  'sample_coding'		: pcm      			*/
  while(fgets(buff,256,fpnist) != NULL)
  {
    headercount += strlen(buff);
    if(strncmp(buff,"end_head",8) == 0)
    {
      readnistheader = 1;
      break;
    }
    if(*buff == ';') continue;
    sscanf(buff,"%s %s %s",object,type,value);
    if(verb) fprintf(stderr,"%s: %s %s %s\n",appname,object,type,value);

    /* First, test for known objects; if required objects are found,
    increase req_objects by 1 */
    if(strncmp(object,"sample_rate",11) == 0)
    {
      req_objects++;
      sscanf(value,"%ld",&(header.isf));
      if (verb) fprintf(stderr,"%s: found sampling rate = %ld\n",appname,
                        header.isf);
    }
    else
    if(strncmp(object,"sample_n_bytes",14) == 0)
    {
      if (verb) fprintf(stderr,"%s: found number of bytes = %s\n",appname,
                        object);
      req_objects++;
      if(strncmp(value,"2",1) != 0)
      {
        fprintf(stderr,"%s: error - sample_n_bytes = %s\n",
            appname,value);
        exit(-1);
      }
    }
    else
    if(strncmp(object,"sample_count",12) == 0)
    {
      req_objects++;
      sscanf(value,"%ld",&samplecount);
      if (verb) fprintf(stderr,"%s: found sample_count = %ld\n",appname,
                        samplecount);
      bytes = 2 * samplecount;
      header.nspbk = bytes / 512; 
      rest = bytes % 512;
      /* size of data block should be multiple of 1 block (512) */
      if(rest != 0) header.nspbk++;
      if(verb) fprintf(stderr,"%s: number of speech blocks (512) = %d\n",
                   appname,header.nspbk);
    }
    else
    if(strncmp(object,"channel_count",13) == 0)
    {
      req_objects++;
      sscanf(value,"%hd",&channel); 
      if(strcmp(value,"1") != 0)
      {
        fprintf(stderr,"%s: error - NIST/SPHERE file contains more or less than one channel\n",
            appname);
        exit(-1);
      }
    }
    /* Second, test for optional objects that require certain
    values, if they are present */
    else
    if(strncmp(object,"speaker_sex",11) == 0)
    {
      if(sex[0] != '\0')
        fprintf(stderr,"%s: warning - sex from line %s overides sex from input %s\n",
                appname,sex,value);
      else
      {
        if((strcmp(value,"m") == 0) || (strcmp(value,"M") == 0)
         ||(strcmp(value,"Male") == 0) || (strcmp(value,"male") == 0))
          sex[0] = 'M';
        else if((strcmp(value,"f") == 0) || (strcmp(value,"F") == 0)
         ||(strcmp(value,"Female") == 0) || (strcmp(value,"female") == 0))
          sex[0] = 'W';
        else
        {
          fprintf(stderr,"%s: warning - speaker_sex = %s, using '\0' for output\n",
            appname,value);
          exit(-1);
        }
      }
    }
    else
    if(strncmp(object,"sample_byte_format",18) == 0)
    {
      if (verb) fprintf(stderr,"%s: found sample_byte_format = %s\n",appname,
                        value);
      if(strcmp(value,"10") == 0) swap = 1;
      else if(strcmp(value,"01") == 0) swap = 0;
      else
      {
        fprintf(stderr,"%s: warning - sample_byte_format = %s, asuming '01' (Intel)\n",
            appname,value);
        swap = 0;
      }
    }
    else
    if(strncmp(object,"sample_sig_bits",15) == 0)
    {
      if(bits != 0)
        fprintf(stderr,"%s: warning - bits from line %d overides bits from input %s\n",
                appname,bits,value);
      else
      {
        sscanf(value,"%d",&bits);
        if (verb) fprintf(stderr,"%s: found adc_bits = %d\n",appname, bits);
      }
    }
    else
    if(strncmp(object,"sample_coding",13) == 0)
    {
      if(strncmp(value,"pcm",3) != 0)
      {
        fprintf(stderr,"%s: error - sample_coding = %s (NOT pcm)\n",
            appname,value);
        exit(-1);
      }
    }
    /* Third, test for optional entries */
    else
    if((strncmp(object,"speaker_id",10) == 0))
    {
      if(spid[0] != '\0')
        fprintf(stderr,"%s: warning - spkr_id from line %s overides spkr_id from input %s\n",
                appname,spid,value);
      else
      {
        if(strlen(value) > 2)
          fprintf(stderr,"%s: warning - speaker_id longer than 2 chars (%s), using first 2 chars\n",
            appname,value);
        else if (strlen(value) == 1)
          fprintf(stderr,"%s: warning - speaker_id has only one char (%s)\n",
            appname,value);
        spid[0] = value[0];
        spid[1] = value[1];
      }
    }
    else
    if(strncmp(object,"recording_date",14) == 0)
    {
      if(month != 0)
        fprintf(stderr,"%s: warning - found recording_date = %s\nbut using given options: %d-%d-%d\n",
                appname,value,day,month,year);
      else
      {
        i = 0;
        while(value[i] != '\0')  
        {
          if((value[i] == '-')||(value[i] == '.')) value[i] = ' ';        
          i++;
        }  
        if(sscanf(value,"%d %s %d",&day,months,&year) != 3)
        {
          fprintf(stderr,"%s: error - could not process recording_date = %s\n",
                appname,value);
        }
        else
        {
          for(i=0;i<strlen(months);i++) months[i] = tolower(months[i]);
          if(strcmp(months,"jan") == 0) month = 1;
          else if(strncmp(months,"feb",3) == 0) month = 2;
          else if(strncmp(months,"mar",3) == 0) month = 3;
          else if(strncmp(months,"apr",3) == 0) month = 4;
          else if(strncmp(months,"may",3) == 0) month = 5;
          else if(strncmp(months,"jun",3) == 0) month = 6;
          else if(strncmp(months,"jul",3) == 0) month = 7;
          else if(strncmp(months,"aug",3) == 0) month = 8;
          else if(strncmp(months,"sep",3) == 0) month = 9;
          else if(strncmp(months,"oct",3) == 0) month = 10;
          else if(strncmp(months,"nov",3) == 0) month = 11;
          else if(strncmp(months,"dec",3) == 0) month = 12;
          else if(isdigit(months[0])) sscanf(months,"%d",&month);
          else
          {
            fprintf(stderr,"%s: error - could not process recording_date = %s\n",
                appname,value);
          }
        }
      }
    }
    else
    if(strncmp(object,"sample_max",10) == 0)
    {
      sscanf(value,"%d",&absampl);
      if(absampl < 0) absampl = -absampl;
      if(header.abs_ampl < absampl) header.abs_ampl = absampl;
      if(verb) fprintf(stderr,"%s: found sample_max = %d\n",
            appname,header.abs_ampl);
    }
    if(strncmp(object,"sample_min",10) == 0)
    {
      sscanf(value,"%d",&absampl);
      if(absampl < 0) absampl = -absampl;
      if(header.abs_ampl < absampl) header.abs_ampl = absampl;
      if(verb) fprintf(stderr,"%s: found sample_min = %d\n",
            appname,header.abs_ampl);
    }
    /* leave rest of entries untouched */
    else ;

  } /* end of while over header */
  if(verb) fprintf(stderr,"%s: read %d bytes from header (%d)\n",
          appname,headercount,nistheadersize);
  for(i=0;i<nistheadersize-headercount;i++) fgetc(fpnist);
  if(!readnistheader)
  {
    fprintf(stderr,"%s: could not read NIST/SPHERE header\n",
            appname);
    exit(-1);
  }
  if(req_objects != REQ_OBJECTS)
  {
    fprintf(stderr,"%s: error - NIST/SPHERE header has not the required entries\n",
            appname);
    exit(-1);
  }
  if(verb) fprintf(stderr,"%s: using date: %d-%d-%d\n",
                appname,day,month,year);

  /* Fill PhonDat header with life ... */
  header.sprk[0] = spid[0];
  header.sprk[1] = spid[1];
  header.swdh = 0;
  strncpy(header.ifl,phonfilnam,31);
  header.ifl[31] = '\0';
  header.day = (char)day;
  header.month = (char)month;
  header.year = (short)year;
  header.version = (char)VERSION;
  header.words = words;
  header.wdh = 0;
  header.adc_bits = bits - 12;
  header.sex = sex[0];
  header.flagtype = -32000;
  header.flaginit = 0;
  if(verb) fprintf(stderr,"%s: writing PhonDat format %d\n",
                   appname,header.version);
  /* now we are ready to write ... */
#ifdef LINUX
  if(!write_header_vms(fpphon,&header,ortho,cano))
  {
    fprintf(stderr,"%s: cannot write header to file %s\n",
    appname,phonfilnam);
    exit(-1);
  }
#endif  
#ifdef SUNOS 
  if(!write_header_sun(fpphon,&header,ortho,cano))
  {
    fprintf(stderr,"%s: cannot write header to file %s\n",
    appname,phonfilnam);
    exit(-1);
  }
#endif  
  /* copy samples */
  samplecount = 0;
  if(verb && swap) fprintf(stderr,"%s: swapping\n", appname);
  while(fread(&sample,sizeof(short),1,fpnist) == 1)
  {
    /* swap samples if necessary */
    if(swap)
    {
    charpoint = (char *)(&sample);
    chr = charpoint[0];
    charpoint[0] = charpoint[1];
    charpoint[1] = chr;
    }
    if(fwrite(&sample,sizeof(short),1,fpphon) != 1)
    {                                                                      
      fprintf(stderr,"%s: could not write to file %s\n", 
      appname,phonfilnam);       
      perror(appname);
      exit(-1);                                                            
    }                                                                      
    samplecount++;
  } 
  if(verb) fprintf(stderr,"%s: written %ld bytes\n",appname,
                   samplecount*2);
  if(samplecount != bytes/2) 
    fprintf(stderr,"%s: warning - counted samples incorrect (%ld)\n",
            appname,samplecount);
  /* fill up last block with zeros */
  chr = '\0';
  if(rest != 0) rest = 512 - rest;
  if(verb) fprintf(stderr,"%s: fill up with %d zeros\n",appname,rest);
  for(i=0;i<rest;i++)
  {
    if(fwrite(&chr,sizeof(char),1,fpphon) != 1)
    {
    fprintf(stderr,"%s: could not write to file %s\n", 
    appname,phonfilnam);
    perror(appname);
    exit(-1);
    } 
  }
  return(0);
}
