/***********************************************************************
************************************************************************

Module          : phondat2nist-1.1
Title           : transforms a PhonDat sound file into a NIST sound file

Author          : F. Schiel
Date/revision   : 17.07.95 / 05.04.96


Description     : 
Reads a soundfile with PhonDat 1 or PhonDat 2 format from file 
phonfile, extracts the necessary
information form the header, checks whether the sound format is 
compatibel to PhonDat, prompts for additional information or
gives a warning if there 's information missing (depends on
option settings, see there), creates a NIST compatible header in file
nistfile and writes the sound samples to that file.
The result is a full compatible NIST file.
See man page for usage.
Errors are reported to stderr.

History:
1.1	: NIST header size is overwritten by higher value, if amount
          of PhonDat Header Data doesn't fit into default NIST header
          A warning is printed to stderr.
          The NIST utterance id is the input file name without 
          trailing path and without suffix (eg. /tmp/eggd4590.16
          -> eggd4590)

Link modules and libraries:
ipkclib

Contained functions:
                      : 

*************************************************************************/
# ifndef FILE
# include <stdio.h>
# endif
# include <ipkclib.h>
# include <string.h>
# include <time.h>

/* DEFINES, only used within this module ********************************/
# define NISTHEADERSIZE 1024 	/* default nist header size */

/* TYPE DEFINITIONS, only used within this module  **********************/


/* GLOBAL VARIABLES (avoid, if possible !) ******************************/


/* FUNCTION PROTOTYPES, capsuled in this module *************************/


main(int argc,const char **argv)
{
	char 	chr,*cptr,*sex,sexdef[2] = {'\0','\0'},*appname,
                *spid,*nistfilnam,*phonfilnam,spiddef[2] = {'\0','\0'},
		*ortho = NULL, *cano = NULL,*charpoint,
		*database_id = NULL;
        char	buff[256],object[256],type[256],value[256],
                months[5];
	short	sample,rest,channel;
 	short	absampl = 0; 
	int	swap = 0;
	int	i,k;
 	int	readnistheader = 0, req_objects = 0;
        int 	verb = 0, bits = 0, words = 0,day = 0,month = 0,year = 0;
	long	bytes,err,skip = 0;
	long	headercount = 0, samplecount = 0,nistheadersize = NISTHEADERSIZE; 
	FILE	*fpnist,*fpphon;
	icsiargtab_t args[] =
        {
          { 0, "1.1 : transforms a PhonDat file into a NIST soundfile, "
            "\n  usage: phondat2nist nistfile=<nistf> phonfile=<phonf> [options]", ARG_DESC},
          { "v","verbose mode", ARG_BOOL, &verb },
          { "nistfile","NIST file name", ARG_STR, &nistfilnam, ARG_REQ},
          { "phonfile","PhonDat file name", ARG_STR, &phonfilnam, ARG_REQ},
          { "database_id","Name of Corpus", ARG_STR, &database_id},
          { "nistheadersize","size of NIST Header in bytes", ARG_INT, &nistheadersize},
          { "swap","swapping short samples", ARG_BOOL, &swap},
	  {0,0,0}
	};
 	Phon_header_2 *header;
  spid = spiddef;
  sex = sexdef;
  /* Test for icsiargs if any */
  icsiargs(args,&argc,&argv,&appname);
  if(verb) printargs(stderr,appname,args);
  /* check and open inputfile */
  if((fpnist = fopen(nistfilnam,"w")) == NULL)
  {
    fprintf(stderr,"%s: error - could not open nistfile %s\n",
            appname,nistfilnam);
    exit(-1);
  }
  /* check and open outputfile */
  if((fpphon = fopen(phonfilnam,"r")) == NULL)
  {
    fprintf(stderr,"%s: error - could not open phonfile %s\n",
            appname,phonfilnam);
    exit(-1);
  }
  /* read the phondat header */
#ifdef SUNOS 
  if((header = read_header_sun(fpphon,&ortho,&cano)) == NULL)
  {
    fprintf(stderr,"%s: error - could not read phondat header from %s\n",
            appname,phonfilnam);
    exit(-1);
  }
#endif  
#ifdef LINUX
  if((header = read_header_vms(fpphon,&ortho,&cano)) == NULL)
  {
    fprintf(stderr,"%s: error - could not read phondat header from %s\n",
            appname,phonfilnam);
    exit(-1);
  }
#endif  
  /* check header sizes:
     if the PhonDat header exceeds nistheadersize (default: 1024),
     the nistheadersize is set to the PhonDat headersize to 
     garantie a successful writing.
     Note that in that case the parameter given from command line
     is overwritten */
  if ((header->anz_header*512) > nistheadersize)
  {
    nistheadersize = header->anz_header*512;
    fprintf(stderr,"%s: %s - set nist header size to %ld\n",appname,
            nistfilnam,nistheadersize);
  }
  /* creating NIST Header */
  /* required entries */
  sprintf(buff,"NIST_1A\n   %d\n",nistheadersize);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  if (database_id != NULL)
  {
   sprintf(buff,"database_id -s%d %s\n",strlen(database_id),database_id);
   headercount += strlen(buff);
   fprintf(fpnist,"%s",buff);
  }
  /* delete suffix */
  if((cptr = strrchr(phonfilnam,'.')) != NULL) *cptr = '\0';
  /* delete trailing path */
  if((cptr = strrchr(phonfilnam,'/')) != NULL) strcpy(phonfilnam,cptr+1);
  sprintf(buff,"utterance_id -s%d %s\n",strlen(phonfilnam),phonfilnam);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"sample_count -i %ld\n",header->nspbk*256);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"sample_n_bytes -i 2\n");
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"channel_count -i 1\n");
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  /* optional entries */
  sprintf(buff,"sample_coding -s3 pcm\n");
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  if(swap)
   sprintf(buff,"sample_byte_format -s2 10\n");
  else
   sprintf(buff,"sample_byte_format -s2 01\n");
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"sample_rate -i %ld\n",header->isf);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"sample_sig_bits -i 16\n");
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"sample_max -i %d\n",header->abs_ampl);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"recording_date -s10 %.2d-%.2d-%.4d\n",
          header->day,header->month,header->year);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"speaker_id -s2 %c%c\n",
          header->sprk[0],header->sprk[1]);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"speaker_sex -s1 %c\n",
          header->sex);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  sprintf(buff,"words -i %d\n",header->words);
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  /* entries special for phondat format 2 */
  if(header->version == 2)
  {
   if((ortho == NULL) || (cano == NULL))
   {
    fprintf(stderr,"%s: error - could not read ortho or cano from PhonDat file version 2\n",
            appname);
    exit(-1);
   }
   sprintf(buff,"orthography -s%d ",strlen(ortho));
   headercount += strlen(buff);
   fprintf(fpnist,"%s",buff);
   headercount += strlen(ortho) + 1;
   fprintf(fpnist,"%s\n",ortho);
   sprintf(buff,"canonic_transcript -s%d ",strlen(cano));
   headercount += strlen(buff);
   fprintf(fpnist,"%s",buff);
   headercount += strlen(cano) + 1;
   fprintf(fpnist,"%s\n",cano);
  }
  sprintf(buff,"end_head\n");
  headercount += strlen(buff);
  fprintf(fpnist,"%s",buff);
  if(verb) printf("%s: written %d bytes to NIST header\n",
                  appname,headercount);
  if(headercount > nistheadersize)
  {
    fprintf(stderr,"%s: error - Header Size exceeding nistheadersize (%d)\n",
            appname,nistheadersize);
    exit(-1);
  }
  /* fill up with ' ' to nistheadersize bytes */
  for(i=0;i<nistheadersize-headercount;i++) fprintf(fpnist," ");

  /* copy samples */
  samplecount = 0;
  fseek(fpphon,header->anz_header*512,SEEK_SET);
  if(verb && swap) fprintf(stderr,"%s: swapping\n", appname);
  while(fread(&sample,sizeof(short),1,fpphon) == 1)
  {
    /* swap samples if necessary */
    if(swap)
    {
    charpoint = (char *)(&sample);
    chr = charpoint[0];
    charpoint[0] = charpoint[1];
    charpoint[1] = chr;
    }
    if(fwrite(&sample,sizeof(short),1,fpnist) != 1)
    {                                                                      
      fprintf(stderr,"%s: could not write to file %s\n", 
      appname,nistfilnam);       
      perror(appname);
      exit(-1);                                                            
    }                                                                      
    samplecount++;
  } 
  if(verb) fprintf(stderr,"%s: written %ld bytes\n",appname,
                   samplecount*2);
  if(samplecount != header->nspbk*256) 
    fprintf(stderr,"%s: warning - counted samples incorrect (%ld != %ld)\n",
            appname,samplecount,header->nspbk*256);
  return(0);
}
