#!/usr/bin/perl

# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: calc_size_MASV_experiment.pl,v 1.3 2004/02/17 09:43:08 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use strict;

use lib $ENV{"MASV_PERL_ROOT"};
use SR_lib;

use File::Copy;
use File::Path;
use Cwd;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

my $exit_state = 0;

my ($minVar, $makeCohortNorm, $num_iterations_world, $num_iterations_model, $hmm_list, $dict,
	$doHNorm, $useHEAdapt, @teeModels, $numbernetWorldTest, $mlf_training, $mlf_test, @trainingModelConf,
	@HViteTrainedModelConf, @HViteWorldConf, @HViteCrossTestConf, $seed_world, $seed_models,
	@world_list, @world_test_list, $protoDir, @hmmBaseDirWorld, @hmmBaseDirModel,
	$hmm_version_model_test, $hmm_version_world_test, $hmm_version_cohort_test,
	$parallelComputation, $createdScript, $FA_id, $startMatlab, $pathMatlabResults,
	$autoClean_FA_Files, $useGMM, $mixtures, $doTestsOnly);

###########################
# Default values for options
my $help = 0;
my $Identify = 0;

@world_test_list = "dev_set/S_list_all";
$num_iterations_world = 10;
@HViteTrainedModelConf = "all_training/aa";
@HViteCrossTestConf = "all_test/aa";
$num_iterations_model = 1;

my $expConfig = "";

my @world_test_list_temp;
my $num_iterations_world_temp;
my @HViteTrainedModelConf_temp;
my @HViteCrossTestConf_temp;
my $num_iterations_model_temp;

my $data_amount;
my $data_sum;

my $data_amount_model = 0.1; # per iteration, in MB
my $data_amount_world_tests = 0.00143; # per iteration, per test, in MB
my $data_amount_model_tests = 0.0005831; # per iteration, per test, in MB
###########################
# Processing Options
GetOptions('expConfig=s' => \$expConfig,
           'wt|world_test_list=s' => \@world_test_list_temp,
		   'iw|num_iterations_world=i' => \$num_iterations_world_temp,
		   'hvtc|HViteTrainedModelConf=s' => \@HViteTrainedModelConf_temp,
		   'hvxc|HViteCrossConf=s' => \@HViteCrossTestConf_temp,
		   'im|num_iterations_model=i' => \$num_iterations_model_temp,
		   'help|?' => \$help,
		   'version' => \$Identify);

if ($Identify) {
	printf "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}


if ((@ARGV != 1) || ($help)) {
   pod2usage(1);
   exit 1;
}

my $sv_system_name = $ARGV[0];

&SR_lib::init_speaker_sets($sv_system_name);

my $sv_system_path = "${SR_lib::sv_systems_dir}$sv_system_name${SR_lib::sign}";
my $experimentConfig_path = "${sv_system_path}expConfigs${SR_lib::sign}";


if ($expConfig) {
	unless (open(CONFFILE, "${experimentConfig_path}$expConfig") ) {
		die "experiment config file $expConfig not found!\n\n";
	}
	my @confContent = <CONFFILE>;
	close CONFFILE;
	
	foreach (@confContent) {
		eval($_);
		unless ($@ eq "") {
			die "error in config file $expConfig :\nline $_ \nerror: $@ \n\n";
		}
	}
}

@world_test_list = @world_test_list_temp if (@world_test_list_temp);
$num_iterations_world = $num_iterations_world_temp if ($num_iterations_world_temp);
@HViteTrainedModelConf = @HViteTrainedModelConf_temp if (@HViteTrainedModelConf_temp);;
@HViteCrossTestConf = @HViteCrossTestConf_temp if (@HViteCrossTestConf_temp);
$num_iterations_model = $num_iterations_model_temp if ($num_iterations_model_temp);

@world_test_list = split(/,/,join(',',@world_test_list));
@HViteTrainedModelConf = split(/,/,join(',',@HViteTrainedModelConf));
@HViteCrossTestConf = split(/,/,join(',',@HViteCrossTestConf));

$num_iterations_world++;
$num_iterations_model++;

my @testModelSessions = ();
my @spk_dirs;
my ($exec_command, @raw_result, $raw_result, @content, $free_space);
my ($numSpkModels, $numImpSpks) = (0, 0);

foreach my $single_model_conf (@HViteTrainedModelConf) {
	my ($testModelArrRef, $single_testModelSessions) 
	 = 
	 &SR_lib::processConfigurations($single_model_conf);
	 
	@spk_dirs = @{$testModelArrRef};
	$numSpkModels += scalar(@spk_dirs);
	
	push(@testModelSessions, $single_testModelSessions);
}

foreach my $single_model_conf (@HViteCrossTestConf) {
	my ($crossTestModelArrRef, $crossTestModelSessions) 
	 = 
	 &SR_lib::processConfigurations($single_model_conf);
	my @imp_dirs = @{$crossTestModelArrRef};
	$numImpSpks += scalar(@imp_dirs);
}



# data amount for testing world model
&SR_lib::create_multispeaker_list($sv_system_name, \@world_test_list);

my $no_of_world_tests = &SR_lib::get_no_entries_multispeaker_list($sv_system_name);

&SR_lib::remove_multispeaker_list($sv_system_name);



my $temp_dir = cwd;
chdir $sv_system_path;
chdir $spk_dirs[0];

my $no_of_model_tests = 0;
foreach my $single_testModelSessions (@testModelSessions) {
	$exec_command = "wc -l $single_testModelSessions";
	$raw_result = `$exec_command`;
	@content = split('\s+', $raw_result);
	$no_of_model_tests += $content[1];
}

chdir $temp_dir;

$data_sum = 0;
print "\nOverview of required data amount:\n";
print "-------------------------\n";
$data_amount = $num_iterations_world * $data_amount_model;
$data_sum += $data_amount;
printf ("world model:  %.1f MB\n" ,$data_amount);

$data_amount = ($num_iterations_world - 1) * $no_of_world_tests * $data_amount_world_tests;
$data_sum += $data_amount;
printf ("world model tests:  %.1f MB\n" ,$data_amount);

$data_amount = $num_iterations_model * $data_amount_model * $numSpkModels;
$data_sum += $data_amount;
printf ("spk models:  %.1f MB\n" ,$data_amount);

$data_amount = $no_of_model_tests * $numSpkModels * ($numImpSpks + 3) * $data_amount_model_tests;
$data_sum += $data_amount;
printf ("spk model tests:  %.1f MB\n" ,$data_amount);

$data_amount = $data_amount * 0.5 + 2; # matlab results
$data_sum += $data_amount;
printf ("matlab results:  %.1f MB\n", $data_amount);

print "----------------------------------\n";
printf ("total:  %.1f MB\n\n", $data_sum);


$exec_command = "df -m ${SR_lib::sv_systems_dir}";
@raw_result = `$exec_command`;
@content = split('\s+', $raw_result[1]);
$free_space = $content[3];

if ($free_space > $data_sum + 10) {
	print "required space is available.\n\n";
}
else {
	print "not enough disk space (missing " . int($data_sum - $free_space) . "MBs)\n\n";
	$exit_state = 1;
}

exit $exit_state;




__END__

=head1 NAME

calc_size_MASV_experiment.pl  - prints the required data amount
                                      for an experiment.

=head1 SYNOPSIS

calc_size_MASV_experiment.pl [-expConfig | other options] sv_system_name

Calculate the required data amount for an experiment. Parameters can be given
either seperately or with the help of an experiment config file.
The path to the file <expConfig> is given relative to the directory "expConfig/"
of the SV system.

 Options:
 
 -expConfig=s          load config file from the SV system 
                          directory "expConfig/". The settings can be 
                          overwritten from the command line options.
 
 -wt | world_test_list=s  configuration for testing the world model with 
                          "run_worldModelTest.pl".
                          Default is 'dev_set/aa'
                          
 -iw|num_iterations_world=s 
                          training iterations for world model.

 -im|num_iterations_model=s 
                          training iterations for speaker models.

 -hvtc|HViteTrainedModelConf=s
                          configuration when testing speaker models;
						  default is "all_training/aa".
						  
 -hvxc|HViteCrossConf=s   configuration when testing with impostor speakers;
						  default is "all_test/aa".

                          
 -version                 print version information.

 -? | help                display this message.
 
=cut

