#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: run_HVite.pl,v 1.37 2004/05/04 09:06:14 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

use Parallel::ForkManager;

my $exit_state = 0;
my $local_exit_code = 0;
my @ssh_stat = [];

my $nice_factor = $SR_lib::nice_factor;
$nice_factor = $SR_lib::nice_factor; # make compiler happy, no warning

my $start_time = scalar localtime;

###########################
# Default values for options
my $help = 0;
my $log_file = '';
my $Identify = 0;
my $outputmlf = "default value";
my $dict_name = "generaldict";
my $crossTest = 0;
my $worldTest = 0;
my $worldModelTest = 0;
my $cohortTest = 0;
my @singleTestFiles = ();
my $testModelConf = "training_set/at";
my $crossTestConf = "test_set/aa";
my $worldListConf = "all_tt/aa";
my $worldModelTestConf = "dev_set/aa";
my $hmm_list = "HMM_number.list";
my $mlf = "mlf_prompted.mlf";
my $templateMLF = "";
my $appendToMLF = 0;
my $onStartup = '';
my $onFinish = '';
my $prefixCommand = '';
my $redirectCommand = '';
my $hmmBaseDir = '';
my $outputWordMLF = 0;
my $extraOptions = '';
my $distributedClient = 0;
my $totalHostsNum=0;
my $parallelComputation = 0;
my $useGMM = 0;
###########################
my $extraOptionsString;
my $appendToMLFString;
my $useGMMString;
my $outputWordMLFString;
my $templateMLFString;

my ($hostsNum, @hostList);
my ($realHostsNum, @dummy);
my (@exe_models, @exe_cross_models);

my $prompt_HTK_Command_String = ($SR_lib::verbose_level > 1) ? "-A" : "";


# Processing Options
GetOptions('o|outputMLF=s' => \$outputmlf, 
		   'useGMM' => \$useGMM,
           'd|dict=s' => \$dict_name,
	       'x|crossTest' => \$crossTest,
	       'w|worldTest' => \$worldTest,
	       'wmt|worldModelTest' => \$worldModelTest,
		   'c|cohortTest' => \$cohortTest,
	       's|singleTestFiles=s' => \@singleTestFiles,
	       'h|hmm_list=s' => \$hmm_list,
	       'm|mlf=s' => \$mlf,
	       't|templateMLF=s' => \$templateMLF,
	       'a|appendToMLF' => \$appendToMLF,
	       'wm|outputWordMLF' => \$outputWordMLF,
	       'tc|testModelConf=s' => \$testModelConf,
	       'xc|crossTestConf=s' => \$crossTestConf,
	       'wc|worldListConf=s' => \$worldListConf,
	       'wmtc|worldModelTestConf=s' => \$worldModelTestConf,
	       'hmmBaseDir=s' => \$hmmBaseDir,
		   'onStartup=s' => \$onStartup,
		   'onFinish=s' => \$onFinish,
		   'prefixCommand=s' => \$prefixCommand,
		   'redirectCommand=s' => \$redirectCommand,
		   'xo|extraOptions=s' => \$extraOptions,
           'dist=s' => \$distributedClient,
           'totalHostsNum=i' => \$totalHostsNum,
           'p|parallelComputation' => \$parallelComputation,
	       'help|?' => \$help,
		   'logfile=s' => \$log_file,
           'version' => \$Identify);

if ($Identify) {
	print STDOUT "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}


if ((@ARGV != 2) || ($help)) {
	pod2usage(1);
   exit 1;
}

if ($SR_lib::verbose_level) {print STDOUT "Running $0 ...\n";}


if (($crossTest +  $worldTest + $worldModelTest + $cohortTest) > 1) {
   die ("You can use only one of these options: cross test, world test, world model test or cohort test.");
}

 if ($distributedClient && $parallelComputation) {
 	print STDERR "Use only the -p|parallelComputation option!\n\n";
 	exit 1;
 }

my $sv_system_name = $ARGV[0];
my $hmm_version = $ARGV[1];

&SR_lib::init_speaker_sets($sv_system_name);

$prefixCommand =~ s/[;\s]*$/;/ if ($prefixCommand);
my $prefixCommandString = ($prefixCommand) ? "-prefixCommand=\'$prefixCommand\'" : "";

my $redirectCommandString = ($redirectCommand) ? "-redirectCommand=\'$redirectCommand\'" : "";

$extraOptionsString = ($extraOptions) ? "-xo=\'$extraOptions\'" : "";
$appendToMLFString = ($appendToMLF) ? "-a" : "";
$outputWordMLFString = ($outputWordMLF) ? "-wm" : "";
$templateMLFString = ($templateMLF) ? "-t $templateMLF" : "";

$useGMMString = ($useGMM) ? "-useGMM"  :  "";
if ($useGMM) {
	$hmm_list = "HMM_number_gmm.list";
	$mlf = "mlf_gmm.mlf";
}



if (!($hmmBaseDir)) {
	$hmmBaseDir = "hmm";
}

my %pool_config = %{&SR_lib::read_pool_config($sv_system_name)};
my $frame_duration = $pool_config{"TARGETRATE"};

my ($testModelArrRef, $testModelSessions, 
 $crossTestArrRef, $crossTestSessions,
 $worldModelArrRef, $worldModelSessions) 
 = 
 &SR_lib::processConfigurations($testModelConf, $crossTestConf, $worldListConf);


# list names for info files
my $testModelList = ${[split("/",$testModelConf)]}[0];
my $crossTestList = ${[split("/",$crossTestConf)]}[0];
my $worldModelList = ${[split("/",$worldListConf)]}[0];

# generate list of testFiles
my @singleTestFiles = split(/,/,join(',',@singleTestFiles));


my $hmmlist_path = "${SR_lib::base_dir}hmm_lists$SR_lib::sign";
my $dict_path = "${SR_lib::base_dir}syntax$SR_lib::sign";
my $sv_system_path = "${SR_lib::sv_systems_dir}${sv_system_name}$SR_lib::sign";
my $log_path = "${sv_system_path}log$SR_lib::sign";
$log_file = ($log_file) ? $log_file : "log.txt";

if ($outputmlf eq "default value") {
	$outputmlf = "force_aligned_";
} else {
	$outputmlf = "$outputmlf";
}

if (! $distributedClient) {
	if ($onStartup) {
		if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
		$local_exit_code = system($onStartup);
	}
}


#######################
#######################
if ((!$crossTest) && (!$worldTest) && (!$worldModelTest) && (!$cohortTest)) {

  my @models = @{$testModelArrRef};

  if ($parallelComputation) {
	($hostsNum, @hostList) = &SR_lib::par_get_available_hosts;
	# Wieviele hosts machen Sinn?
	($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@models, $hostsNum);
  }
  elsif ($distributedClient) {
		# VP-Liste auf bentigte Zahl aufteilen
		my ($dummy, @models_lists) = &SR_lib::split_arrayList(\@models, $totalHostsNum);
		@exe_models = @{$models_lists[$distributedClient - 1]};
  }
  else
  { #lokal ausgefhrt, alles lassen, wie es ist
  		@exe_models = @models;
  }

  if ($parallelComputation) {
		my $pm = new Parallel::ForkManager($realHostsNum);

		$pm->run_on_finish(
			sub { my ($pid, $exit_code, $ident) = @_;
				  $ssh_stat[$ident] = $exit_code;
			}
		);
		
		for (my $host_counter=1; $host_counter <= $realHostsNum; $host_counter++) {
			$pm->start($host_counter) and next;
			my $exec_string;
			if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
			if ($onStartup) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
				$local_exit_code = system("ssh -x $hostList[$host_counter] \"$onStartup\"");
			}
			if (!@singleTestFiles) {
				$exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum  $useGMMString -o $outputmlf $appendToMLFString $templateMLFString -d $dict_name -h $hmm_list -m $mlf -tc $testModelConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString $sv_system_name $hmm_version \"";
			} else {
				$exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum  $useGMMString -o $outputmlf $appendToMLFString $templateMLFString -d $dict_name -h $hmm_list -m $mlf -tc $testModelConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString -s " . join(',', @singleTestFiles) . " $sv_system_name $hmm_version \"";
			}
			if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
			$local_exit_code += system($exec_string);
			if ($onFinish) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
				$local_exit_code += system("ssh -x $hostList[$host_counter] \"$onFinish\"");
			}
			if ($local_exit_code) {
				$local_exit_code = 11;
			}
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
			if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}

			$pm->finish($local_exit_code);
	
		}
	  
		$pm->wait_all_children;
		if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
		if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
		
		for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
			$exit_state += $ssh_stat[$host_counter];
		}
			
		if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}


  }
  else 
  {
		my $temp_mlf;
		my $final_mlf;
		my $force_alignment_mlf;
		my $S_list_file;
		my ($exec_string1, $exec_string_S, $exec_string2, $exec_string);
		my $current_redirectCommand;
		
		foreach my $model (@exe_models) {
					$temp_mlf = "${sv_system_path}${model}/temp_force_aligned_client_x_model.mlf";
					$final_mlf = "${sv_system_path}${model}/${outputmlf}client_x_model.mlf";
					if ($templateMLF) {
						$force_alignment_mlf = "${sv_system_path}${model}/template_single_words.mlf";
						$S_list_file = "${sv_system_path}$model/S_list_template_single_words";
						&SR_lib::create_force_alignment_templates("${sv_system_path}$model/$templateMLF", $S_list_file, $force_alignment_mlf,"${sv_system_path}$model/${testModelSessions}",$frame_duration);
					}
					else {
						$force_alignment_mlf = "${sv_system_path}$mlf";
						$S_list_file = "${sv_system_path}$model/${testModelSessions}";
					}
					$exec_string1="$prefixCommand ${SR_lib::htk_bin}HVite $prompt_HTK_Command_String -a -d ${sv_system_path}$model/$hmmBaseDir/hmm.${hmm_version} -i $temp_mlf  -m -o W -I  $force_alignment_mlf ";
					$exec_string_S=" -S $S_list_file";
					$exec_string2=" $extraOptions ${dict_path}$dict_name ${hmmlist_path}$hmm_list";
					
					if ($redirectCommand) {
						$current_redirectCommand = $redirectCommand;
						$current_redirectCommand =~ s/%sv_system_path%/$sv_system_path/;
						$current_redirectCommand =~ s/%model_dir%/$model/;
						$current_redirectCommand =~ s/%state_seq_file_id%/$model/;
						$current_redirectCommand =~ s/%test_type%/normal/;
						$current_redirectCommand =~ s/%S_list_file%/$S_list_file/;
					}
					
					if (!@singleTestFiles) {
						$exec_string  = $exec_string1 . $exec_string_S . $exec_string2 . " " . $current_redirectCommand;
					}
					else
					{
						$exec_string  = $exec_string1 . $exec_string2 . " " . join(' ', @singleTestFiles) . " " . $current_redirectCommand;
					}
					#print "$exec_string \n";
					$exit_state = system ($exec_string);
					if ($exit_state) {
						print STDERR "Command $exec_string failed\n";
						exit 1;
					}
					
					if ($templateMLF) {
						&SR_lib::join_single_word_mlf($temp_mlf);
						unlink "$force_alignment_mlf";
						unlink "$S_list_file";
					}
					
					if ((-e $final_mlf) && ($appendToMLF))
					{
						$exec_string  = "sed --silent -e '{1!p;}' $temp_mlf >> $final_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
						$exec_string = "rm $temp_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
					}
					else
					{
						$exec_string = "mv $temp_mlf  $final_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
					}
					#print ("$exec_string \n");
			
		} # end foreach $model ...
  }

  if (! $distributedClient) {
	  my $trainingSessions = &SR_lib::read_trainingInfoFile("${sv_system_path}" . $models[0] . "/$hmmBaseDir");
	  &write_info_file("${outputmlf}client_x_model.info", $testModelList, $hmm_version, $hmmBaseDir, $trainingSessions, $testModelSessions,$appendToMLF);
  }

}


######################
######################
if ($crossTest) {

  my @models = @{$testModelArrRef};
  my @cross_models = @{$crossTestArrRef};
  
  if ($parallelComputation) {
		($hostsNum, @hostList) = &SR_lib::par_get_available_hosts;
		# Wieviele hosts machen Sinn?
		if (scalar(@models) > scalar(@cross_models)) {
			# grere Liste aufteilen
			($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@models, $hostsNum);
		}
		else
		{
			($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@cross_models, $hostsNum);	
		}
  }
  elsif ($distributedClient) {
		# VP-Liste auf bentigte Zahl aufteilen
		if (scalar(@models) > scalar(@cross_models)) {
			my ($dummy, @models_lists) = &SR_lib::split_arrayList(\@models, $totalHostsNum);
			@exe_models = @{$models_lists[$distributedClient - 1]};
			@exe_cross_models = @cross_models;
		}
		else
		{
			my ($dummy, @cross_models_lists) = &SR_lib::split_arrayList(\@cross_models, $totalHostsNum);	
			@exe_cross_models = @{$cross_models_lists[$distributedClient - 1]};
			@exe_models = @models;
		}
  }
  else
  { #lokal ausgefhrt, alles lassen, wie es ist
	  @exe_models = @models;
	  @exe_cross_models = @cross_models;
  }
	if ($parallelComputation) {
		my $pm = new Parallel::ForkManager($realHostsNum);
		
		$pm->run_on_finish(
			sub { my ($pid, $exit_code, $ident) = @_;
				  $ssh_stat[$ident] = $exit_code;
			}
		);
		for (my $host_counter=1; $host_counter <= $realHostsNum; $host_counter++) {
			$pm->start($host_counter) and next;
			
			my $exec_string;
			
			if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
			if ($onStartup) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
				$local_exit_code = system("ssh -x $hostList[$host_counter] \"$onStartup\"");
			}
			if (!@singleTestFiles) {
				$exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum $useGMMString -o $outputmlf $appendToMLFString $templateMLFString -d $dict_name -x -h $hmm_list -m $mlf -tc $testModelConf -xc $crossTestConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString $sv_system_name $hmm_version \"";
			} else {
				$exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum $useGMMString -o $outputmlf $appendToMLFString $templateMLFString -d $dict_name -x -h $hmm_list -m $mlf -tc $testModelConf -xc $crossTestConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString -s " . join(',', @singleTestFiles) . " $sv_system_name $hmm_version \"";				
			}
			if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
			$local_exit_code += system($exec_string);
			if ($onFinish) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
				$local_exit_code += system("ssh -x $hostList[$host_counter] \"$onFinish\"");
			}
			if ($local_exit_code) {
				$local_exit_code = 11;
			}
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
			if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}
	
			$pm->finish($local_exit_code);
	
		}
	  
		$pm->wait_all_children;
		if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
		if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
		
		for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
			$exit_state += $ssh_stat[$host_counter];
		}
	
		if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}
	
	
	}
	else 
	{
		my $temp_mlf;
		my $final_mlf;
		my $force_alignment_mlf;
		my $S_list_file;
		my ($exec_string1, $exec_string_S, $exec_string2, $exec_string);
		my $current_redirectCommand;
	
		foreach my $model (@exe_models) {
			foreach my $cross_model (@exe_cross_models) {
					next if ($cross_model eq $model);
					$temp_mlf = "${sv_system_path}${model}/temp_force_aligned_${cross_model}_x_model.mlf";
					$final_mlf = "${sv_system_path}${model}/${outputmlf}${cross_model}_x_model.mlf";
					if ($templateMLF) {
						$force_alignment_mlf = "${sv_system_path}${model}/template_${cross_model}_single_words.mlf";
						$S_list_file = "${sv_system_path}$model/S_list_template_${cross_model}_single_words";
						&SR_lib::create_force_alignment_templates("${sv_system_path}$cross_model/$templateMLF", $S_list_file, $force_alignment_mlf,"${sv_system_path}$cross_model/$crossTestSessions",$frame_duration);
					}
					else {
						$force_alignment_mlf = "${sv_system_path}$mlf";
						$S_list_file = "${sv_system_path}$cross_model/$crossTestSessions";
					}
					$exec_string1="$prefixCommand ${SR_lib::htk_bin}HVite $prompt_HTK_Command_String -a -d ${sv_system_path}$model/$hmmBaseDir/hmm.${hmm_version} -i $temp_mlf -m -o W -I  $force_alignment_mlf ";
					$exec_string_S=" -S $S_list_file";
					$exec_string2=" $extraOptions ${dict_path}$dict_name ${hmmlist_path}$hmm_list";

					if ($redirectCommand) {
						$current_redirectCommand = $redirectCommand;
						$current_redirectCommand =~ s/%sv_system_path%/$sv_system_path/;
						$current_redirectCommand =~ s/%model_dir%/${model}_${cross_model}/;
						$current_redirectCommand =~ s/%state_seq_file_id%/$cross_model/;
						$current_redirectCommand =~ s/%test_type%/$model/;
						$current_redirectCommand =~ s/%S_list_file%/$S_list_file/;
					}

					if (!@singleTestFiles) {
						$exec_string  = $exec_string1 . $exec_string_S . $exec_string2 . " " . $current_redirectCommand;
					}
					else
					{
						$exec_string  = $exec_string1 . $exec_string2 . " " . join(' ', @singleTestFiles) . " " . $current_redirectCommand;
					}
					$exit_state = system ($exec_string);
					if ($exit_state) {
						print STDERR "Command $exec_string failed\n";
						exit 1;
					}
					
					if ($templateMLF) {
						&SR_lib::join_single_word_mlf($temp_mlf);
						unlink "$force_alignment_mlf";
						unlink "$S_list_file";
					}
					
					if ((-e $final_mlf) && ($appendToMLF))
					{
						$exec_string = "sed --silent -e '{1!p;}' $temp_mlf >> $final_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
						$exec_string = "rm $temp_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
					}
					else
					{
						$exec_string = "mv $temp_mlf  $final_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
					}
					#print ("$exec_string \n");
			
			}
		} # end foreach $model ...
	}
	
	if (! $distributedClient) {
		my $trainingSessions = "ignored here";
		&write_info_file("${outputmlf}imp_x_model.info", $testModelList, $hmm_version, $hmmBaseDir, $trainingSessions, $crossTestList, $crossTestSessions,$appendToMLF);
	}

}

###########################
###########################
if ($worldTest) {

  my @models = @{$worldModelArrRef};
  
  if ($parallelComputation) {
	($hostsNum, @hostList) = &SR_lib::par_get_available_hosts;
	# Wieviele hosts machen Sinn?
	($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@models, $hostsNum);
  }
  elsif ($distributedClient) {
	   # VP-Liste auf bentigte Zahl aufteilen
	   my ($dummy, @models_lists) = &SR_lib::split_arrayList(\@models, $totalHostsNum);
	   @exe_models = @{$models_lists[$distributedClient - 1]};
  }
  else
  { #lokal ausgefhrt, alles lassen, wie es ist
	  @exe_models = @models;
  }

  if ($parallelComputation) {
		my $pm = new Parallel::ForkManager($realHostsNum);
		
		$pm->run_on_finish(
			sub { my ($pid, $exit_code, $ident) = @_;
				  $ssh_stat[$ident] = $exit_code;
			}
		);
		
		for (my $host_counter=1; $host_counter <= $realHostsNum; $host_counter++) {
		  $pm->start($host_counter) and next;
		  
		  my $exec_string;
		  if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
			if ($onStartup) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
				$local_exit_code = system("ssh -x $hostList[$host_counter] \"$onStartup\"");
			}
		  if (!@singleTestFiles) {
			  $exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum $useGMMString -o $outputmlf $appendToMLFString $outputWordMLFString -d $dict_name -w -h $hmm_list -m $mlf -wc $worldListConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString $sv_system_name $hmm_version \"";
		  } else {
			  $exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum $useGMMString -o $outputmlf $appendToMLFString $outputWordMLFString -d $dict_name -w -h $hmm_list -m $mlf -wc $worldListConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString -s " . join(',', @singleTestFiles) . " $sv_system_name $hmm_version \"";
		  }
		  if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
		  $local_exit_code += system($exec_string);
			if ($onFinish) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
				$local_exit_code += system("ssh -x $hostList[$host_counter] \"$onFinish\"");
			}
		  if ($local_exit_code) {
			  $local_exit_code = 11;
		  }
		  if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
		  if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}
		
		$pm->finish($local_exit_code);
			
		}
	  
	  $pm->wait_all_children;
	  if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
	  if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
		
	  for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
			if ($SR_lib::verbose_level) { print STDOUT  "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
			$exit_state += $ssh_stat[$host_counter];
	  }

	  if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}


  }
  else 
  {
		my $temp_mlf;
		my $final_mlf;
		my $format_string;
		my ($exec_string1, $exec_string_S, $exec_string2, $exec_string);
		my $S_list_file;
		my $current_redirectCommand;

		foreach my $model (@exe_models) {

			$S_list_file = "${sv_system_path}$model/${worldModelSessions}";
			$temp_mlf = "${sv_system_path}${model}/temp_force_aligned_client_x_world.mlf";
			$final_mlf = "${sv_system_path}${model}/${outputmlf}client_x_world.mlf";
			$format_string = ($outputWordMLF) ? "" : "-m -o W";
			$exec_string1="$prefixCommand ${SR_lib::htk_bin}HVite $prompt_HTK_Command_String -a -d ${sv_system_path}world/${hmmBaseDir}/hmm.${hmm_version} -i $temp_mlf $format_string -I  ${sv_system_path}$mlf";
			$exec_string_S=" -S $S_list_file";
			$exec_string2=" $extraOptions ${dict_path}$dict_name ${hmmlist_path}$hmm_list";

			if ($redirectCommand) {
				$current_redirectCommand = $redirectCommand;
				$current_redirectCommand =~ s/%sv_system_path%/$sv_system_path/;
				$current_redirectCommand =~ s/%model_dir%/world/;
				$current_redirectCommand =~ s/%state_seq_file_id%/$model/;
				$current_redirectCommand =~ s/%test_type%/world/;
				$current_redirectCommand =~ s/%S_list_file%/$S_list_file/;
			}

			if (!@singleTestFiles) {
				$exec_string  = $exec_string1 . $exec_string_S . $exec_string2 . " " . $current_redirectCommand;
			}
			else
			{
				$exec_string  = $exec_string1 . $exec_string2 . " " . join(' ', @singleTestFiles) . " " . $current_redirectCommand;
			}
			$exit_state = system ($exec_string);
			if ($exit_state) {
				print STDERR "Command $exec_string failed\n";
				exit 1;
			}
			if ((-e $final_mlf) && ($appendToMLF))
			{
				$exec_string = "sed --silent -e '{1!p;}' $temp_mlf >> $final_mlf";
				$exit_state = system ($exec_string);
				if ($exit_state) {
					print STDERR "Command $exec_string failed\n";
					exit 1;
				}
				$exec_string = "rm $temp_mlf";
				$exit_state = system ($exec_string);
				if ($exit_state) {
					print STDERR "Command $exec_string failed\n";
					exit 1;
				}
			}
			else
			{
				$exec_string = "mv $temp_mlf  $final_mlf";
				$exit_state = system ($exec_string);
				if ($exit_state) {
					print STDERR "Command $exec_string failed\n";
					exit 1;
				}
			}
			#print ("$exec_string \n");
			
		} # end foreach $model ...
  }
  if (! $distributedClient) {
	  my $worldTrainingSessions = &SR_lib::read_trainingInfoFile("${sv_system_path}world/$hmmBaseDir");
	  &write_info_file("${outputmlf}client_x_world.info", $worldModelList, $hmm_version, $hmmBaseDir, $worldTrainingSessions, $worldModelSessions,$appendToMLF);
  }


}


###########################
###########################
if ($worldModelTest) {

  my @exe_testSpks;
  my ($worldModelTestArrRef, $worldModelTestSessions) =  &SR_lib::processConfigurations($worldModelTestConf);
  
  my @testSpks = @{$worldModelTestArrRef};
  
  if ($parallelComputation) {
	($hostsNum, @hostList) = &SR_lib::par_get_available_hosts;
	# Wieviele hosts machen Sinn?
	($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@testSpks, $hostsNum);
  }
  elsif ($distributedClient) {
	   # VP-Liste auf bentigte Zahl aufteilen
	   my ($dummy, @spks_lists) = &SR_lib::split_arrayList(\@testSpks, $totalHostsNum);
	   @exe_testSpks = @{$spks_lists[$distributedClient - 1]};
  }
  else
  { #lokal ausgefhrt, alles lassen, wie es ist
	  @exe_testSpks = @testSpks;
  }

  if ($parallelComputation) {
	  	my $pm = new Parallel::ForkManager($realHostsNum);

		$pm->run_on_finish(
			sub { my ($pid, $exit_code, $ident) = @_;
				  $ssh_stat[$ident] = $exit_code;
			}
		);

		for (my $host_counter=1; $host_counter <= $realHostsNum; $host_counter++) {
		  $pm->start($host_counter) and next;
		  
		  my $exec_string;
		  if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
			if ($onStartup) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
				$local_exit_code = system("ssh -x $hostList[$host_counter] \"$onStartup\"");
			}
		  if (!@singleTestFiles) {
			  $exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum $useGMMString -o $outputmlf $appendToMLFString $outputWordMLFString -d $dict_name -wmt -h $hmm_list -m $mlf -wmtc $worldModelTestConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString $sv_system_name $hmm_version \"";
		  } else {
			  $exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum $useGMMString -o $outputmlf $appendToMLFString $outputWordMLFString -d $dict_name -wmt -h $hmm_list -m $mlf -wmtc $worldModelTestConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString -s " . join(',', @singleTestFiles) . " $sv_system_name $hmm_version \"";
		  }
		  if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
		  $local_exit_code += system($exec_string);
			if ($onFinish) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
				$local_exit_code += system("ssh -x $hostList[$host_counter] \"$onFinish\"");
			}
		  if ($local_exit_code) {
			  $local_exit_code = 11;
		  }
		  if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
		  if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}
		
		$pm->finish($local_exit_code);
			
		}
	  
	  $pm->wait_all_children;
	  if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
	  if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
		
	  for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
			$exit_state += $ssh_stat[$host_counter];
	  }



  }
  else 
  {
		my $final_mlf;
		my ($exec_string1, $exec_string_S, $exec_string2, $exec_string);
		my $S_list_file;
		my $current_redirectCommand;

		foreach my $spk (@exe_testSpks) {

			$final_mlf = "${sv_system_path}${spk}/${outputmlf}";
			$S_list_file = "${sv_system_path}$spk/${worldModelTestSessions}";
			$exec_string1="$prefixCommand ${SR_lib::htk_bin}HVite $prompt_HTK_Command_String -d ${sv_system_path}world/${hmmBaseDir}/hmm.${hmm_version} -i $final_mlf ";

			$exec_string_S=" -S $S_list_file";

			$exec_string2=" $extraOptions ${dict_path}$dict_name ${hmmlist_path}$hmm_list";

			if ($redirectCommand) {
				$current_redirectCommand = $redirectCommand;
				$current_redirectCommand =~ s/%sv_system_path%/$sv_system_path/;
				$current_redirectCommand =~ s/%model_dir%/world/;
				$current_redirectCommand =~ s/%state_seq_file_id%/$spk/;
				$current_redirectCommand =~ s/%test_type%/wmt/;
				$current_redirectCommand =~ s/%S_list_file%/$S_list_file/;
			}

			if (!@singleTestFiles) {
				$exec_string  = $exec_string1 . $exec_string_S . $exec_string2 . " " . $current_redirectCommand;
			}
			else
			{
				$exec_string  = $exec_string1 . $exec_string2 . " " . join(' ', @singleTestFiles) . " " . $current_redirectCommand;
			}
			# print $exec_string , "\n";
			$exit_state = system ($exec_string);
			if ($exit_state) {
				print STDERR "Command $exec_string failed\n";
				exit 1;
			}			
		} # end foreach $spk ...
  } # end if ($parallelComputation) {
  if (! $distributedClient) {
		if ($SR_lib::verbose_level) { print STDOUT "Joining mlfs ...\n";}

	  	my $exec_string;
		my $final_mlf = "${sv_system_path}world/${outputmlf}";
		my $processed_first_part;
		if (! $appendToMLF ) {
			unlink "$final_mlf";
			$processed_first_part = 0;
		}
		else {
			$processed_first_part = 1;
		}
		foreach my $spk (@testSpks) {
			my $spk_mlf = "${sv_system_path}${spk}/${outputmlf}";
			if ($processed_first_part)
			{	
				# append remaining parts to final_mlf
				$exec_string = "sed --silent -e '{1!p;}' $spk_mlf >> $final_mlf";
				$exit_state = system ($exec_string);
				if ($exit_state) {
					print STDERR "Command $exec_string failed\n";
					exit 1;
				}
				$exec_string = "rm $spk_mlf";
				$exit_state = system ($exec_string);
				if ($exit_state) {
					print STDERR "Command $exec_string failed\n";
					exit 1;
				}
			}
			else
			{
				# generate first part of final_mlf
				$exec_string = "mv $spk_mlf  $final_mlf";
				$exit_state = system ($exec_string);
				if ($exit_state) {
					print STDERR "Command $exec_string failed\n";
					exit 1;
				}
				$processed_first_part = 1;
			}
		}
	  if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}

  }


}


######################
######################
if ($cohortTest) {

  my @models = @{$testModelArrRef};
  my @cross_models = @{$crossTestArrRef};
  
  if ($parallelComputation) {
		($hostsNum, @hostList) = &SR_lib::par_get_available_hosts;
		# Wieviele hosts machen Sinn?
		if (scalar(@models) > scalar(@cross_models)) {
			# grere Liste aufteilen
			($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@models, $hostsNum);
		}
		else
		{
			($realHostsNum, @dummy) = &SR_lib::split_arrayList(\@cross_models, $hostsNum);	
		}
  }
  elsif ($distributedClient) {
		# VP-Liste auf bentigte Zahl aufteilen
		if (scalar(@models) > scalar(@cross_models)) {
			my ($dummy, @models_lists) = &SR_lib::split_arrayList(\@models, $totalHostsNum);
			@exe_models = @{$models_lists[$distributedClient - 1]};
			@exe_cross_models = @cross_models;
		}
		else
		{
			my ($dummy, @cross_models_lists) = &SR_lib::split_arrayList(\@cross_models, $totalHostsNum);	
			@exe_cross_models = @{$cross_models_lists[$distributedClient - 1]};
			@exe_models = @models;
		}
  }
  else
  { #lokal ausgefhrt, alles lassen, wie es ist
	  @exe_models = @models;
	  @exe_cross_models = @cross_models;
  }
	if ($parallelComputation) {
		my $pm = new Parallel::ForkManager($realHostsNum);
		
		$pm->run_on_finish(
			sub { my ($pid, $exit_code, $ident) = @_;
				  $ssh_stat[$ident] = $exit_code;
			}
		);

		for (my $host_counter=1; $host_counter <= $realHostsNum; $host_counter++) {
			$pm->start($host_counter) and next;
			
			my $exec_string;
			if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
			if ($onStartup) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onStartup\" ...\n");}
				$local_exit_code = system("ssh -x $hostList[$host_counter] \"$onStartup\"");
			}
			if (!@singleTestFiles) {
				$exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum $useGMMString -o $outputmlf $appendToMLFString  $templateMLFString -d $dict_name -c -h $hmm_list -m $mlf -tc $testModelConf -xc $crossTestConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString $sv_system_name $hmm_version \"";
			} else {
				$exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HVite.pl -dist=$host_counter -totalHostsNum=$realHostsNum $useGMMString -o $outputmlf $appendToMLFString $templateMLFString -d $dict_name -c -h $hmm_list -m $mlf -tc $testModelConf -xc $crossTestConf -hmmBaseDir=$hmmBaseDir $extraOptionsString $prefixCommandString $redirectCommandString -s " . join(',', @singleTestFiles) . " $sv_system_name $hmm_version \"";				
			}
			if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
			$local_exit_code += system($exec_string);
			if ($onFinish) {
				if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
				$local_exit_code += system("ssh -x $hostList[$host_counter] \"$onFinish\"");
			}
			if ($local_exit_code) {
				$local_exit_code = 11;
			}
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
			if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}
	
			$pm->finish($local_exit_code);
	
		}
	  
		$pm->wait_all_children;
		if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
		if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
		
		for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
			if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
			$exit_state += $ssh_stat[$host_counter];
		}
	
		if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}
	
	
	}
	else 
	{
		my $temp_mlf;
		my $final_mlf;
		my $force_alignment_mlf;
		my $S_list_file;
		my ($exec_string1, $exec_string_S, $exec_string2, $exec_string);
		my $current_redirectCommand;
	
		foreach my $model (@exe_models) {
			foreach my $cross_model (@exe_cross_models) {
					$temp_mlf = "${sv_system_path}${model}/temp_force_aligned_cohort_${cross_model}_x_model.mlf";
					$final_mlf = "${sv_system_path}${model}/${outputmlf}cohort_${cross_model}_x_model.mlf";
					if ($templateMLF) {
						$force_alignment_mlf = "${sv_system_path}${model}/template_${cross_model}_single_words.mlf";
						$S_list_file = "${sv_system_path}$model/S_list_template_${cross_model}_single_words";
						&SR_lib::create_force_alignment_templates("${sv_system_path}$cross_model/$templateMLF", $S_list_file, $force_alignment_mlf,"${sv_system_path}$cross_model/$crossTestSessions",$frame_duration);
					}
					else {
						$force_alignment_mlf = "${sv_system_path}$mlf";
						$S_list_file = "${sv_system_path}$cross_model/$crossTestSessions";
					}
	
					$exec_string1="$prefixCommand ${SR_lib::htk_bin}HVite $prompt_HTK_Command_String -a -d ${sv_system_path}$model/$hmmBaseDir/hmm.${hmm_version} -i $temp_mlf -m -o W -I  $force_alignment_mlf";
					$exec_string_S=" -S $S_list_file";
					$exec_string2=" $extraOptions ${dict_path}$dict_name ${hmmlist_path}$hmm_list";

					if ($redirectCommand) {
						$current_redirectCommand = $redirectCommand;
						$current_redirectCommand =~ s/%sv_system_path%/$sv_system_path/;
						$current_redirectCommand =~ s/%model_dir%/${model}_${cross_model}/;
						$current_redirectCommand =~ s/%state_seq_file_id%/$cross_model/;
						$current_redirectCommand =~ s/%test_type%/$model/;
						$current_redirectCommand =~ s/%S_list_file%/$S_list_file/;
					}


					if (!@singleTestFiles) {
						$exec_string  = $exec_string1 . $exec_string_S . $exec_string2 . " " . $current_redirectCommand;
					}
					else
					{
						$exec_string  = $exec_string1 . $exec_string2 . " " . join(' ', @singleTestFiles) . " " . $current_redirectCommand;
					}
					$exit_state = system ($exec_string);
					if ($exit_state) {
						print STDERR "Command $exec_string failed\n";
						exit 1;
					}

					if ($templateMLF) {
						&SR_lib::join_single_word_mlf($temp_mlf);
						unlink "$force_alignment_mlf";
						unlink "$S_list_file";
					}

					if ((-e $final_mlf) && ($appendToMLF))
					{
						$exec_string = "sed --silent -e '{1!p;}' $temp_mlf >> $final_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
						$exec_string = "rm temp_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
					}
					else
					{
						$exec_string = "mv $temp_mlf  $final_mlf";
						$exit_state = system ($exec_string);
						if ($exit_state) {
							print STDERR "Command $exec_string failed\n";
							exit 1;
						}
					}
					#print ("$exec_string \n");
			
			}
		} # end foreach $model ...
	}
	
	if (! $distributedClient) {
		my $trainingSessions = "ignored here";
		&write_info_file("${outputmlf}cohort_x_model.info", $testModelList, $hmm_version, $hmmBaseDir, $trainingSessions, $crossTestList, $crossTestSessions,$appendToMLF);
	}

}

if (! $distributedClient) {
	if ($onFinish) {
		if ($SR_lib::verbose_level) { print STDOUT ("Executing \"$onFinish\" ...\n");}
		$local_exit_code = system($onFinish);
	}
}


if (! $distributedClient) {

	&SR_lib::check_and_create_dir($log_path);
	
	my $log_handle = &SR_lib::open_file(">>", "$log_path" . "$log_file");
	print $log_handle "\n\n";
	print $log_handle "$start_time \n";
	print $log_handle "Running run_HVite.pl .....\n";
	print $log_handle "---------------------------\n";
	print $log_handle "Parameters:\n";
	print $log_handle "sv_system_name   : $sv_system_name\n";
	print $log_handle "hmm_version       : $hmm_version\n";
	print $log_handle "Options:\n";
	print $log_handle "useGMM            : $useGMM\n";
	print $log_handle "outputmlf         : $outputmlf\n";
	print $log_handle "appendToMLF       : $appendToMLF\n";
	print $log_handle "outputWordMLF     : $outputWordMLF\n";
	print $log_handle "dict_name         : $dict_name\n";
	print $log_handle "crossTest         : $crossTest\n";
	if (($crossTest) || ($cohortTest)) {
		print $log_handle "crossTestConf     : $crossTestConf\n";
	}
	print $log_handle "worldTest         : $worldTest\n";
	if ($worldTest) {
		print $log_handle "worldListConf     : $worldListConf\n";
	}
	if (!$worldTest) {
		print $log_handle "testModelConf     : $testModelConf\n";
	}
	print $log_handle "worldModelTest    : $worldModelTest\n";
	print $log_handle "cohortTest        : $cohortTest\n";
	print $log_handle "mlf               : $mlf\n";
	print $log_handle "templateMLF       : $templateMLF\n";
	print $log_handle "hmm_list          : $hmm_list\n";
	print $log_handle "hmmBaseDir        : $hmmBaseDir\n";
	print $log_handle "onStartup         : $onStartup\n";
	print $log_handle "onFinish          : $onFinish\n";
	print $log_handle "prefixCommand     : $prefixCommand\n";
	print $log_handle "redirectCommand   : $redirectCommand\n";
	print $log_handle "extraOptions      : $extraOptions\n";
	print $log_handle "singleTestFiles   : @singleTestFiles\n";
	print $log_handle "p                 : $parallelComputation\n";
	print $log_handle "logfile           : $log_file\n";
	print $log_handle "cvs_version       : $CVS_Version_String\n";
	print $log_handle "cvs_name          : $CVS_Name_String\n";
	print $log_handle "finishing ...";
	print $log_handle scalar localtime, "\n";
	if ($exit_state) {
		print $log_handle "\n Command failed !!!\n";
	}
	
	if (! $worldModelTest ) {
		&write_general_info_file ("${outputmlf}general_HVite.info", $mlf, $dict_name, $hmm_list);
	}

	close $log_handle;
	
}


exit $exit_state;



sub write_info_file {
	
	my $infopath;
	my $outputinfo;
	my $ModelList;
	my $ModelVersion;
	my $hmmBaseDir;
	my $trainingSessions;
	my $CrossList;
	my $Sessions;
	my $appendFlag;
	my $info_handle;
	
	if (scalar(@_) == 7) {# no cross or cohort mode
		($outputinfo, $ModelList, $ModelVersion, $hmmBaseDir, $trainingSessions, $Sessions, $appendFlag) = @_;
		$CrossList = "";
	}
	else {
		($outputinfo, $ModelList, $ModelVersion, $hmmBaseDir, $trainingSessions, $CrossList, $Sessions, $appendFlag) = @_;
	}

	$infopath = "${sv_system_path}${SR_lib::sign}info${SR_lib::sign}";
	&SR_lib::check_and_create_dir($infopath);
	
	
	if (!($appendFlag)) {
		$info_handle = &SR_lib::open_file(">", "$infopath" . "$outputinfo");
		if (scalar(@_) == 7) {# no cross or cohort mode
			print $info_handle "${ModelList}\n";
			print $info_handle "$Sessions,${ModelVersion},${hmmBaseDir},${trainingSessions}\n";
		}
		else {
			print $info_handle "${ModelList}\n";
			print $info_handle "${CrossList}\n";
			print $info_handle "$Sessions,${ModelVersion},${hmmBaseDir}\n";
		}
		
	}
	else {
		$info_handle = &SR_lib::open_file("<", "$infopath" . "$outputinfo");
		my @temp_content = <$info_handle>;
		close $info_handle;
		$info_handle = &SR_lib::open_file(">", "$infopath" . "$outputinfo");
		for (my $counter = 0; $counter <= $#temp_content; $counter++) {
			print $info_handle $temp_content[$counter];
		}
		
		if (scalar(@_) == 7) {# no cross or cohort mode
			print $info_handle "${Sessions},${ModelVersion},${hmmBaseDir},${trainingSessions}\n";
		}
		else {
			print $info_handle "${Sessions},${ModelVersion},${hmmBaseDir}\n";
		}
	}

	close $info_handle;

}



sub write_general_info_file {
	
	my $infopath;
	my $outputinfo;
	my $mlf_file;
	my $dict_name;
	my $hmm_list;

	my $info_handle;
	
	($outputinfo, $mlf_file, $dict_name, $hmm_list) = @_;

	$infopath = "${sv_system_path}${SR_lib::sign}info${SR_lib::sign}";
	&SR_lib::check_and_create_dir($infopath);
	
	
	$info_handle = &SR_lib::open_file(">", "$infopath" . "$outputinfo");
	print $info_handle "$mlf_file\n";
	print $info_handle "$dict_name\n";
	print $info_handle "$hmm_list\n";
	
	close $info_handle;

}


__END__

=head1 NAME

run_HVite.pl  - run HVite for a SV system

=head1 SYNOPSIS

run_HVite.pl [options] sv_system_name hmm_version


Run HVite for the SV system <sv_system_name>. Uses HMM models 
of the version <hmm_version>.

 Options:
 
 -o | outputMLF=s         use given name for output MLF file; default is 
                            - "force_aligned_client_x_model.mlf" for normal operation
                            - "force_aligned_0000_x_model.mlf" for impostor test
                              with 0000 replaced by the speaker ID
                            - "force_aligned_client_x_world.mlf" for world test
                            - "force_aligned_cohort_0000_x_model.mlf" for cohort test

 -wm | outputWordMLF      use word level for output MLF file. Useful when using the MLF
                          later on as template for force alignment and single tee models
                          can occur. Note that "words" have to defined by a normal model
                          and a tee model (e.g. "izwanzig", using "zwanzig" and "sp")

 -a | appendToMLF         append HVite output MLF to existing MLF with same name.
                          If the MLF file does not exist, it will be created.
                          Default value: not set

 -t | templateMLF=s       use the given MLF file as a template for force alignment with
                          contraints both on label sequence and time sequence.
                          Note: normal forced alignment enforces only the label sequence.
                          The -m option is ignored, but session conf. of the -tc option
                          is used instead, all recordings that also accur in the training
                          data are used.

 -d | dict=s              use given dictionary from the directory "syntax".
                          Default value: "generaldict"
 
 -s | singleTestFiles=s   perform only test with given test files (comma seperated list).
 
 -m | mlf=s               specify MLF file, default is "mlf_prompted.mlf"
                          
 -h | hmm_list=s          specify HMM list, default is "HMM_number.list"

 -x | crossTest           perform test of all test sentences against all models;
                          impostor test needs the options -xc and -tc !
 
 -xc | crossTestConf=s    configuration for impostor test
                          default is 'test_set/aa' i.e. all recs of speakers 
                          of "test_set" with sessions from the "S_list_all" lists are tested.
                          
 -w | worldTest           perform test of all test sentences against world model.
                          
 -wc | worldListConf=s    used configuration for world test
                          default is 'all_tt/aa' (all speakers from training and test set).
                          
 -wmt | worldModelTest    perform test of world model, write results in a single mlf;
                          this option is normally used by run_worldModelTest.pl.
                          
 -wmtc | worldModelTestConf=s   configuration to specify speakers for world model test;
                                default is 'dev_set/aa';
                                see "sub processConfigurations" in SR_lib.pm;
                                this option is normally used by run_worldModelTest.pl.
                          
 -tc | trainingModelConf=s      configuration to specify trained models;
                                default is 'training_set/at'
                                see "sub processConfigurations" in SR_lib.pm

 -c | cohortTest          perform tests of cohort speakers on given models;
                          use -tc to specify models to be tested ("all_tt/aa" e.g.),
                          use -xc to specify test speakers ("world_set/aa" e.g.);
                            session specification given by -xc is used
                            (like option -x)

 -hmmBaseDir=s            use models in given directory in 
                          speaker resp. world directory;
                          default is "hmm"

 -useGMM                  adapt behaviour for using a single state model with 
                          multiple mixtures (GMM).
 
 -onStartup=s             command to perform before running on parallel host

 -onFinish=s              command to perform after running on parallel host

 -prefixCommand=s         command to perform before running HVite

 -redirectCommand=s       command to perform directly after HVite (useful for
                          redirections of stdout of HVite when using trace levels e.g.).
                          The commands can use the following 'variable' names:
                           - %sv_system_path%    = path to sv system
                           - %model_dir%         = currently tested model (can also be 'world')
                           - %state_seq_file_id% = further identifier (spk or cross spk
                           - %test_type%         = identifier of the five different test types
                                                 ('normal', cross_spk, 'world', cohort_spk, 'wmt')
                           - %S_list_file%       = name of S_list file

 -xo | extraOptions =s    pass extra options to HVite

 -p|parallelComputation   use several hosts for running HVite

 -logfile                 set logfile; default is log/log.txt.

 -? | help                display this message.
 
=cut

