#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: run_itWorldModelTraining.pl,v 1.5 2004/03/22 08:20:13 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

my $start_time = scalar localtime;

###########################
# Default values for options
my $help = 0;
my $log_file = '';
my $Identify = 0;
my $dict_name = "generaldict";
my $hmm_list = "HMM_number.list";
my $wdnet = "numbernet";
my $world_conf = "world_set/aa";
my $test_conf = "dev_set/aa";
my $hmmBaseDir = 'hmm';
my $extraOptions = '';
my $macros = '';
my $mlf = "mlf_prompted.mlf";
my $onStartup = '';
my $onFinish = '';
my $prefixCommand = '';
my $parallelComputation = 0;
my $useGMM = 0;
my $epsilon = 0.1;
my $epsilon_factor = 0.0;
###########################


# Processing Options

GetOptions('useGMM' => \$useGMM,
           'd|dict=s' => \$dict_name,
	       'h|hmm_list=s' => \$hmm_list,
	       'm|mlf=s' => \$mlf,
		   'wc|world_conf=s' => \$world_conf,
		   'tc|test_conf=s' => \$test_conf,
		   'n|wdnet=s' => \$wdnet,
	       'hmmBaseDir=s' => \$hmmBaseDir,
		   'xo|extraOptions=s' => \$extraOptions,
	       'macros=s' => \$macros,
		   'onStartup=s' => \$onStartup,
		   'onFinish=s' => \$onFinish,
		   'prefixCommand=s' => \$prefixCommand,
           'p|parallelComputation' => \$parallelComputation,
           'epsilon=f' => \$epsilon,
           'epsilon_factor=f' => \$epsilon_factor,
	       'help|?' => \$help,
		   'logfile=s' => \$log_file,
           'version' => \$Identify);

if ($Identify) {
	print STDOUT "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if ((@ARGV != 3) || ($help)) {
	pod2usage(1);
	exit 1;
}

if ($SR_lib::verbose_level) {print STDOUT "Running $0 ...\n";}



my $sv_system_name = $ARGV[0];
my $start_version = $ARGV[1];
my $num_iterations_world_max = $ARGV[2];

$log_file = ($log_file) ? $log_file : "log.txt";

my $macrosOptionString = ($macros) ? "-macros $macros" : "";
my $extraOptionsString = ($extraOptions) ? "-xo=\'$extraOptions\'" : ""; 

$prefixCommand =~ s/[;\s]*$/;/ if ($prefixCommand);
my $prefixCommandString = ($prefixCommand) ? "-prefixCommand=\'$prefixCommand\'" : "";

my $parallelSwitch = ($parallelComputation) ? "-p" : "";
my $useGMMString = ($useGMM) ? "-useGMM"  :  "";
my $onStartupString = ($onStartup) ? "-onStartup=\"$onStartup\"" : "";
my $onFinishString = ($onFinish) ? "-onFinish=\"$onFinish\"" : "";
my $epsilonString = ($epsilon == 0.1) ? "" : "-epsilon=$epsilon";
my $epsilonFactorString = ($epsilon_factor == 0) ? "" : "-epsilon_factor=$epsilon_factor";

my $sv_system_path = "${SR_lib::sv_systems_dir}${sv_system_name}$SR_lib::sign";
my $log_path = "${sv_system_path}log$SR_lib::sign";


my $current_best_model_version;
my $exec_string;

for my $current_version ( $start_version .. ($num_iterations_world_max + $start_version - 1) ) {

		$exec_string = "run_HERest.pl -logfile $log_file $useGMMString $extraOptionsString $macrosOptionString -hmmBaseDir=$hmmBaseDir $parallelSwitch $prefixCommandString $onStartupString $onFinishString -w -wl=$world_conf -m $mlf -h $hmm_list $sv_system_name 1 $current_version";
		system($exec_string);
		
		
		$exec_string = "run_worldModelTest.pl -logfile $log_file $useGMMString $extraOptionsString $macrosOptionString $parallelSwitch $prefixCommandString $onStartupString $onFinishString -l $test_conf -m $mlf  -n $wdnet -h $hmm_list -dict $dict_name -hmmBaseDir=$hmmBaseDir $sv_system_name $current_version  0";
		system($exec_string);
		
		
		if ($useGMM) {
			$exec_string = "get_bestWorldModel_GMM.pl $epsilonString $epsilonFactorString $sv_system_name $hmmBaseDir";
		}
		else {
			$exec_string = "get_bestWorldModel.pl $sv_system_name $hmmBaseDir";
		}
		
		open(PIPE, "$exec_string |");
		$current_best_model_version = <PIPE>;
		close(PIPE);
		chomp($current_best_model_version);

		
		if ($current_best_model_version < $current_version ) {
			last;
		}

}

if ($SR_lib::verbose_level) {print STDOUT "Finishing $0 .\n";}

&SR_lib::check_and_create_dir($log_path);

my $log_handle = &SR_lib::open_file(">>", "$log_path" . "$log_file");
print $log_handle "\n\n";
print $log_handle "$start_time \n";
print $log_handle "Running run_itWorldModelTraining.pl .....\n";
print $log_handle "-----------------------------------------\n";
print $log_handle "Parameters:\n";
print $log_handle "sv_system_name      : $sv_system_name\n";
print $log_handle "start_version       : $start_version\n";
print $log_handle "num_its_world_max   : $num_iterations_world_max\n";
print $log_handle "Options:\n";
print $log_handle "useGMM              : $useGMM\n";
print $log_handle "world_conf          : $world_conf\n";
print $log_handle "test_conf           : $test_conf\n";
print $log_handle "dict_name           : $dict_name\n";
print $log_handle "wdnet               : $wdnet\n";
print $log_handle "mlf                 : $mlf\n";
print $log_handle "hmm_list            : $hmm_list\n";
print $log_handle "hmmBaseDir          : $hmmBaseDir\n";
print $log_handle "extraOptions        : $extraOptions\n";
print $log_handle "parallelComputation : $parallelComputation\n";
print $log_handle "onStartup           : $onStartup\n";
print $log_handle "onFinish            : $onFinish\n";
print $log_handle "prefixCommand       : $prefixCommand\n";
print $log_handle "logfile             : $log_file\n";
print $log_handle "macros              : $macros\n";
print $log_handle "cvs_version         : $CVS_Version_String\n";
print $log_handle "cvs_name            : $CVS_Name_String\n";
print $log_handle "Finishing... ",scalar localtime, "\n";
close $log_handle;
	


exit 0;





__END__

=head1 NAME

run_itWorldModelTraining.pl  - train world model and test after each it.

=head1 SYNOPSIS

run_itWorldModelTraining.pl [options]  sv_system_name  start_hmm_version  max_iterations


Train world model with run_HERest.pl of  SV system <sv_system_name> and perform a
test after each iteration with run_worldModelTest.pl. It starts with HMM models of 
the version <start_hmm_version> and performs and maximum <max_iterations>. 

 Options:

 -wc | world_conf=s        training configuration. Default is "world_set/aa".
 
 -tc | test_conf=s         test configuration. Default is "dev_set/aa".
 
 -d | dict=s               use given dictionary from the directory "syntax".
                           default is "generaldict".
						   
 -m | mlf                  specify MLF file, default is "mlf_prompted.mlf"
                           
 -h | hmm_list             specify HMM list, default is "HMM_number.list"

 -hmmBaseDir               use world models in given directory;
                           default is "hmm"

 -macros                   use given file for HMM macros; it must reside in the directory
                           of the hmm models.

 -n | wdnet                specify word network from the directory "syntax",
                           default is "numbernet".

 -xo | extraOptions        pass extra options to HVite and HERest

 -useGMM                   adapt behaviour for using a single state model with multiple
                           mixtures (GMM).
 
 -p | parallelComputation  use several hosts for running HVite

 -onStartup=s              command to perform before running on parallel host

 -onFinish=s               command to perform after running on parallel host

 -prefixCommand=s          command to perform before running HERest, HVite

 -epsilon=f                minimum absolute step size when determining the 
                           best world model version;
                           default: 0.1

 -epsilon_factor=f         calculate step size when determining the
                           the best world model version based on first value of
                           the llhs scaled with given factor; 
                           default: 0

 -logfile                  set logfile; default is log/log.txt.

 -? | help                 display this message.
 
=cut

