#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: run_worldModelTest.pl,v 1.22 2004/05/03 11:03:53 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

my $start_time = scalar localtime;

###########################
# Default values for options
my $help = 0;
my $log_file = '';
my $Identify = 0;
my $outputmlf = "default value";
my $dict_name = "generaldict";
my $hmm_list = "HMM_number.list";
my $wdnet = "numbernet";
my $test_conf = "dev_set/aa";
my $hmmBaseDir = '';
my $extraOptions = '';
my $macros = '';
my $mlf = "mlf_prompted.mlf";
my $onStartup = '';
my $onFinish = '';
my $prefixCommand = '';
my $parallelComputation = 0;
my $useGMM = 0;
###########################

my $macrosOptionString;

my $used_outputmlf;

my $prompt_HTK_Command_String = ($SR_lib::verbose_level > 1) ? "-A" : "";


# Processing Options

GetOptions('o|outputMLF=s' => \$outputmlf, 
		   'useGMM' => \$useGMM,
           'd|dict=s' => \$dict_name,
	       'h|hmm_list=s' => \$hmm_list,
	       'm|mlf=s' => \$mlf,
		   'l|list=s' => \$test_conf,
		   'n|wdnet=s' => \$wdnet,
	       'hmmBaseDir=s' => \$hmmBaseDir,
		   'xo|extraOptions=s' => \$extraOptions,
	       'macros=s' => \$macros,
           'p|parallelComputation' => \$parallelComputation,
		   'onStartup=s' => \$onStartup,
		   'onFinish=s' => \$onFinish,
		   'prefixCommand=s' => \$prefixCommand,
	       'help|?' => \$help,
		   'logfile=s' => \$log_file,
           'version' => \$Identify);

if ($Identify) {
	print STDOUT "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}


if (((@ARGV != 2) && (@ARGV != 3)) || ($help)) {
	pod2usage(1);
	exit 1;
}

if ($SR_lib::verbose_level) {print STDOUT "Running $0 ...\n";}

my $sv_system_name = $ARGV[0];
my $hmm_version = $ARGV[1];
my $num_of_iterations;

&SR_lib::init_speaker_sets($sv_system_name);


if (@ARGV == 3) {
	$num_of_iterations = $ARGV[2];
} else {
	$num_of_iterations = 0;
}

if (!($hmmBaseDir)) {
	$hmmBaseDir = "hmm";
}

my %pool_config = %{&SR_lib::read_pool_config($sv_system_name)};
my $frame_duration = $pool_config{"TARGETRATE"};

my ($testModelArrRef, $testModelSessions) = &SR_lib::processConfigurations($test_conf);


my $hmmlist_path = "${SR_lib::base_dir}hmm_lists$SR_lib::sign";
my $dict_path = "${SR_lib::base_dir}syntax$SR_lib::sign";
my $sv_system_path = "${SR_lib::sv_systems_dir}${sv_system_name}$SR_lib::sign";
my $log_path = "${sv_system_path}log$SR_lib::sign";
$log_file = ($log_file) ? $log_file : "log.txt";

my $parallelComputation_String = ($parallelComputation) ? "-p" : "";
my $onStartupString = ($onStartup) ? "-onStartup=\"$onStartup\"" : "";
my $onFinishString = ($onFinish) ? "-onFinish=\"$onFinish\"" : "";

$prefixCommand =~ s/[;\s]*$/;/ if ($prefixCommand);
my $prefixCommandString = ($prefixCommand) ? "-prefixCommand=\'$prefixCommand\'" : "";

for(my $current_hmm_version = $hmm_version; $current_hmm_version <= $hmm_version + $num_of_iterations; $current_hmm_version++) {

	if ($outputmlf eq "default value") {
		$used_outputmlf = "test_${hmmBaseDir}_${current_hmm_version}.mlf";
	} else {
		$used_outputmlf = "${outputmlf}_${current_hmm_version}.mlf";
	}
	
	my $macrosin="${sv_system_path}world/${hmmBaseDir}/hmm.${current_hmm_version}";
	$macrosOptionString = ($macros) ? "-H ${macrosin}/$macros" : "";
	my ($exec_string, $extraOptionsString);
	
	if ($useGMM) {
		$extraOptionsString = "-xo=\'" . $extraOptions . " $macrosOptionString -m -a " . "-I ${sv_system_path}$mlf\'"; 
		$exec_string="run_HVite.pl -logfile $log_file -wmt -wmtc=$test_conf $extraOptionsString -hmmBaseDir ${hmmBaseDir} -outputMLF=${used_outputmlf} -m $mlf -dict=$dict_name -hmm_list=$hmm_list $parallelComputation_String $prefixCommandString $onStartupString $onFinishString $sv_system_name ${current_hmm_version}";
	}
	else {
		$extraOptionsString = "-xo=\'" . $extraOptions . " -w ${dict_path}$wdnet $macrosOptionString -m -o W\'";
		$exec_string="run_HVite.pl -logfile $log_file -wmt -wmtc=$test_conf $extraOptionsString -hmmBaseDir ${hmmBaseDir} -outputMLF=${used_outputmlf}  -dict=$dict_name -hmm_list=$hmm_list $parallelComputation_String $prefixCommandString $onStartupString $onFinishString $sv_system_name ${current_hmm_version}";
	}
	
	if ($SR_lib::verbose_level) { print STDOUT ("$exec_string \n");}
	system ($exec_string);
	
	if ($useGMM) {
		$exec_string = "calc_GMM_world_llh.pl -fd $frame_duration ${sv_system_path}world/${used_outputmlf} > ${sv_system_path}world/calc_GMM_world_llh_${used_outputmlf}.txt";
	}
	else {
		$exec_string = "${SR_lib::htk_bin}HResults -p -I ${sv_system_path}$mlf ${hmmlist_path}$hmm_list ${sv_system_path}world/${used_outputmlf} > ${sv_system_path}world/hresults_${used_outputmlf}.txt";
	}
	if ($SR_lib::verbose_level) { print STDOUT ("$exec_string \n");}
	system ($exec_string);

}


&SR_lib::check_and_create_dir($log_path);

my $log_handle = &SR_lib::open_file(">>", "$log_path" . "$log_file");
print $log_handle "\n\n";
print $log_handle "$start_time \n";
print $log_handle "Running run_worldModelTest.pl .....\n";
print $log_handle "---------------------------\n";
print $log_handle "Parameters:\n";
print $log_handle "sv_system_path     : $sv_system_path\n";
print $log_handle "hmm_version         : $hmm_version\n";
print $log_handle "num_of_iterations   : $num_of_iterations\n";
print $log_handle "Options:\n";
print $log_handle "useGMM              : $useGMM\n";
print $log_handle "S_list              : $test_conf\n";
print $log_handle "outputmlf           : $outputmlf\n";
print $log_handle "dict_name           : $dict_name\n";
print $log_handle "wdnet               : $wdnet\n";
print $log_handle "mlf                 : $mlf\n";
print $log_handle "hmm_list            : $hmm_list\n";
print $log_handle "hmmBaseDir          : $hmmBaseDir\n";
print $log_handle "extraOptions        : $extraOptions\n";
print $log_handle "parallelComputation : $parallelComputation\n";
print $log_handle "onStartup           : $onStartup\n";
print $log_handle "onFinish            : $onFinish\n";
print $log_handle "prefixCommand       : $prefixCommand\n";
print $log_handle "macros              : $macros\n";
print $log_handle "logfile             : $log_file\n";
print $log_handle "cvs_version         : $CVS_Version_String\n";
print $log_handle "cvs_name            : $CVS_Name_String\n";
print $log_handle "Finishing... ",scalar localtime, "\n";
close $log_handle;
	


exit 0;





__END__

=head1 NAME

run_worldModelTest.pl  - run tests on world model with development speakers

=head1 SYNOPSIS

run_worldModelTest.pl [options] sv_system_name start_hmm_version [iterations = 0]


Run HVite for the SV system <sv_system_name> with given world models in normal
speech recognition mode. It uses HMM models of the version <hmm_version>. 
Test samples are taken by default from the recording configuration 'dev_set/aa'.

 Options:

 -l | list=s               recording configuration. Default is "dev_set/aa".
 
 -o | outputMLF=s          use given name for output MLF file; default is 
                            - "test_<hmmBaseDir>_<model_version>.mlf
                           
 -d | dict=s               use given dictionary from the directory "syntax".
                           default is "generaldict".
						   
 -m | mlf                  specify MLF file, default is "mlf_prompted.mlf"
                           
 -h | hmm_list             specify HMM list, default is "HMM_number.list"

 -hmmBaseDir               use models in given directory;
                           default is "hmm"

 -macros                   use given file for HMM macros; it must reside in the directory
                           of the hmm models.

 -n | wdnet                specify word network from the directory "syntax",
                           default is "numbernet"; ignored when using GMMs.

 -xo | extraOptions        pass extra options to HVite

 -useGMM                   adapt behaviour for using a single state model with multiple
                           mixtures (GMM).
 
 -p | parallelComputation  use several hosts for running HVite

 -onStartup=s              command to perform before running on parallel host

 -onFinish=s               command to perform after running on parallel host

 -prefixCommand=s          command to perform before running HVite

 -logfile                  set logfile; default is log/log.txt.

 -? | help                 display this message.
 
=cut

