#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: run_HTKCommand.pl,v 1.14 2004/02/17 09:43:10 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Cwd;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

my $exit_state = 0;

my $start_time = scalar localtime;

###########################
# Default values for options
my $help = 0;
my $log_file = '';
my $Identify = 0;
my $iterateOnVPs = '';
my $testModelConf = "all_training/aa";
###########################

# Processing Options
GetOptions('i|iterateOnVPs' => \$iterateOnVPs,
	       'tc|testModelConf=s' => \$testModelConf,
		   'help|?' => \$help,
		   'logfile=s' => \$log_file,
           'version' => \$Identify);

if ($Identify) {
	print STDOUT "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if ((@ARGV <= 1) || ($help)) {
   pod2usage(1);
   exit 1;
}

if ($SR_lib::verbose_level) {print STDOUT "Running $0 ...\n";}

my $sv_system_name = shift(@ARGV);
&SR_lib::init_speaker_sets($sv_system_name);


my @command_string = @ARGV;
my $command_line = join(' ', @command_string);

my ($testModelArrRef, $testModelSessions) 
 = 
 &SR_lib::processConfigurations($testModelConf);


my $sv_system_path = "${SR_lib::sv_systems_dir}$sv_system_name${SR_lib::sign}";
my $log_path = "${sv_system_path}log${SR_lib::sign}";
$log_file = ($log_file) ? $log_file : "log.txt";



if ($iterateOnVPs) {
  
	my $temp_dir = cwd;
	
	my @spk_dirs = @{$testModelArrRef};
	
	
	foreach my $spk_dir (@spk_dirs) {
		chdir $sv_system_path;
		chdir $spk_dir;
		my $exec_command = $command_line;
		$exec_command =~ s/%spk_dir%/$spk_dir/; 
		if ($SR_lib::verbose_level) { print STDOUT "spk " . $spk_dir . ": " . $exec_command , "\n";}
		$exit_state = system("$exec_command");
		#if ($exit_state) {
		#	print "Command $exec_command failed\n";
		#	exit 1;
		#}
	
	}
	chdir $temp_dir;
}
else {
	my $temp_dir = cwd;
	chdir $sv_system_path;
	
	$exit_state = system("$command_line");
	
	chdir $temp_dir;
}



&SR_lib::check_and_create_dir($log_path);

my $log_handle = &SR_lib::open_file(">>", "$log_path" . "$log_file");
print $log_handle "\n\n";
print $log_handle "$start_time \n";
print $log_handle "Running run_HTKCommand.pl .....\n";
print $log_handle "---------------------------\n";
print $log_handle "Parameters:\n";
print $log_handle "sv_system_path   : $sv_system_path\n";
print $log_handle "command line      : $command_line\n";
print $log_handle "Options:\n";
print $log_handle "iterateOnVPs      : $iterateOnVPs\n";
print $log_handle "testModelConf     : $testModelConf\n";
print $log_handle "logfile           : $log_file\n";
print $log_handle "cvs_version       : $CVS_Version_String\n";
print $log_handle "cvs_name          : $CVS_Name_String\n";
print $log_handle "finishing... ", scalar localtime, "\n";

if ($exit_state) {
	print $log_handle "\n Command failed !!!\n";
}

close $log_handle;


exit $exit_state;




__END__

=head1 NAME

run_HTKCommand.pl  - run an arbitrary command for an SV system

=head1 SYNOPSIS

run_HTKCommand.pl [options] sv_system_name commandstring

Runs an arbitrary (not necessarily HTK) command (given by 
<commandstring>) for the SV system <sv_system_name>. 
Useful to log the commands in the SV system log file.
Also used for applying commands to a bunch of speaker directories.
The command string can contain '%spk_dir%' which will expand to the
current speaker directory when using the -i option.


 Options:
 
 -i | iterateOnVPs        perform the command in each speakers's directory

 -tc | trainingModelConf  configuration to specify the speaker directories;
                          default is 'all_training/aa'
                          see "sub processConfigurations" in SR_lib.pm.
                          The script file part of the configuration 
                          (part after the slash) is ignored here.
                          
 -logfile                 set logfile; default is log/log.txt.

 -? | help                display this message.
 
=cut


