
/***********************************************************************
************************************************************************

Module          : preprocess
Title           : vorverarbeitung von zeitsignalen

Author          : F. Schiel
Date/revision   : 16.12.93 / 27.08.93

Description     : 
Vorverarbeitung von Zeitsignal in Frames.

Link modules and libraries:
ipkclib
m

Contained functions:
preprocesslpc     :  vorverarbeitung
autom             : autoregressive analyse
preem		  : Hochpassfilter
windw             : Hamming window
a2cep             : berechnet cepstrum aus LPC

*************************************************************************/
# include <stdio.h>
# include <math.h>
# include "ipkclib.h"

/* DEFINES, only used within this module ********************************/
# define NORMSIGNAL 32768.	/* normierungsfaktor des zeitsignals	*/
# define PREEMP 0.9                     /* Filter-Koeff. der preemphase */  
# define X_PRE 0.0                      /* letzter Filterwert in preem()*/  

/* TYPE DEFINITIONS, only used within this module  **********************/


/* GLOBAL VARIABLES (avoid, if possible !) ******************************/


/* FUNCTION PROTOTYPES, capsuled in this module *************************/
static int readwin(float h[]);


/*----------------------------------------------------------------------
Name            : preprocesslpc
Module          : preprocesslpc
Title           : f"uhrt cepstrale merkmalsextraktion durch

Description:
"Ubergeben wird ein Array signal mit einer Sprach"au"serung mit anz       
Abtastwerten (Signal).                        
Alle framestep Samples wird ein Hammingwindow auf das Signal                  
multipliziert, eine Hoehenanhebung durchgefuehrt, 
die LPC-Koeffizienten berechnet und aus diesen wiederum 
Cepstralkoeffizienten (Merkmalsvektor). 
Die Merkmalsvektoren werden in   
einem Array zur"uckgeliefert. Die Anzahl der berechneten                 
Merkmalsvektoren ist featureanz. Die Dimension der Vektoren berechnet sich zu 
lpcorder + 1 (Energiedichte im ersten Element). 
Im ersten Element ist die Energiedichte, ab dem zweiten Element LPCORDER 
Cepstral-Koeffizienten.
Ist die Energie in einem Window exact gleich 0, so wird f"ur alle 
Merkmale 0 zur"uckgeliefert und bei iolog(0) == 0 
eine warning auf stderr geschrieben.

Parameters:
signal          : array von signalwerten
sampleanz	: anzahl der Abtastwerte in signal
featureanz      : anzahl der berechneten merkmalsvektoren  
lpcorder        : ordnung des lpc-filters (ergebnis hat lpcorder + 1 
		  elemente pro merkmalsvektor)  
framestep       : abstand der merkmalsvektoren in samples 
windowlength    : Fenstergr"o"se des Hammingwindows in samples  

Return-Value	: pointer auf array von featureanz merkmalsvektoren
                  mit dimensionalitaet lpcorder + 1
		  NULL bei Fehler
-------------------------------------------------------------------------*/
float *preprocesslpc(short *signal,long sampleanz,long *featureanz,
                   short lpcorder,short framestep,short windowlength)
{
	long flg,i,j,k;
	float *features,*fsig,*featureptr,*window,
              *fsignal,*a,
	      *alpha,*rc,*r;
	float x_pre,energie;
	 
if(iolog(0)) printf("preprocesslpc: started\n");

/* Speicher fuer konstante Vektoren bereitstellen */
if((window = (float *)calloc(windowlength,sizeof(float))) == NULL)
  {
  fprintf(stderr,"preprocesslpc: out of memory\n");
  perror("preprocesslpclpc");
  return(NULL);
  } 
if((fsignal = (float *)calloc(windowlength,sizeof(float))) == NULL)
  {
  fprintf(stderr,"preprocesslpc: out of memory\n");
  perror("preprocesslpclpc");
  return(NULL);
  } 
if((a = (float *)calloc(lpcorder,sizeof(float))) == NULL)
  {
  fprintf(stderr,"preprocesslpc: out of memory\n");
  perror("preprocesslpclpc");
  return(NULL);
  } 
if((alpha = (float *)calloc(lpcorder+1,sizeof(float))) == NULL)
  {
  fprintf(stderr,"preprocesslpc: out of memory\n");
  perror("preprocesslpclpc");
  return(NULL);
  } 
if((rc = (float *)calloc(lpcorder,sizeof(float))) == NULL)
  {
  fprintf(stderr,"preprocesslpc: out of memory\n");
  perror("preprocesslpclpc");
  return(NULL);
  } 
if((r = (float *)calloc(lpcorder+1,sizeof(float))) == NULL)
  {
  fprintf(stderr,"preprocesslpc: out of memory\n");
  perror("preprocesslpclpc");
  return(NULL);
  } 

/* initialisiere window */
flg = -1;

/* berechne anzahl der merkmalsvektoren = anzahl der fenster, die in 
   das zeitsignal gerade noch passen. Die Position des ersten frames liegt
   bei windowlength/2, die des letzten muss vor sampleanz - windowlength/2
   liegen 								*/
*featureanz = ((sampleanz - windowlength) / framestep ) + 1;


if(iolog(0)) printf("preprocesslpc: featureanz = %d\n",*featureanz);

/* speicher f"ur merkmalsvektoren anfordern				*/
if((features = (float *)calloc(*featureanz,(lpcorder+2)*sizeof(float))) == NULL)
{
  fprintf(stderr,"preprocesslpc: can't allocate memory \n");
  perror("preprocesslpc");
  return(NULL);
}

/* speicher f"ur float-vektor zur filterung des ganzen signals anfordern	*/
if((fsig = (float *)calloc(sampleanz,sizeof(float))) == NULL)
{
  fprintf(stderr,"preprocesslpc: can't allocate memory \n");
  perror("preprocesslpc");
  return(NULL);
}

if(iolog(0)) printf("preprocesslpc: memory allocated for features and fsig\n");

/* gesamtes sprachsignal filtern (differenzierung zur anhebung der oberen
   frequenzen zum ausgleich der abstrahlung)				*/
/* kopieren nach float-feld und normierung auf +/- 1.0			*/
for(i=0;i<sampleanz;i++) fsig[i] = (float)signal[i] / NORMSIGNAL;
if(iolog(0)) 
{
printf("preprocesslpc: signal before filtering :\n");
for(j=0;j<8;j++)
    printf("%.2e ",fsig[j]);
printf("\n");
}

/* preemphasis  filter 							*/
x_pre = X_PRE;
preem(sampleanz,fsig,&x_pre,PREEMP);
if(iolog(0)) 
{
printf("preprocesslpc: signal after filtering:\n");
for(j=0;j<8;j++)
    printf("%.2e ",fsig[j]);
printf("\n");
}

/*schleife ueber sprachsignal */
featureptr = features;
for(i=0,k=0;k < *featureanz;i+=framestep,k++)
{
/* kopiere segment nach floatfeld und berechne  energiedichte		*/
	energie = 0.;
	for(j=0;j<windowlength;j++)
	{
		fsignal[j]=fsig[i+j];
		energie += signal[i+j] * signal[i+j];
	}
	*featureptr++ = sqrt((double)energie) / windowlength;

/* test ob energie im Window 0 ist. In diesem Fall hat die Berechnung
   von Merkmalen keinen Sinn. Es werden Nullwerte f"ur diesen Frame
   zur"uckgeliefert							*/
        if(energie == 0.)
	{
	   if(iolog(0)) fprintf(stderr,
	   "preprocesslpc: warning: found frame %d with zero energie\n",k);
	   for(j=0;j<lpcorder;j++) *featureptr++ = 0.;
	   continue;
	}

if(iolog(0)) printf("preprocesslpc: copied segment [%d]\n",k);
if(iolog(0)) printf("preprocesslpc: energie in segment [%d] = %e\n",k,*(featureptr-1));
if(iolog(0)) 
{
printf("preprocesslpc: segment [%d]:\n",k);
for(j=0;j<8;j++)
    printf("%.2e ",fsignal[j]);
printf("\n");
}

/* windowing */
	windw(windowlength,fsignal,window,&flg);

if(iolog(0)) printf("preprocesslpc: windowed segment [%d]:\n",k);
if(iolog(0)) 
{
for(j=0;j<8;j++)
    printf("%.2e ",fsignal[j]);
printf("\n");
}

/* LPC coefficients */
	autom(windowlength,fsignal,lpcorder,a,alpha,rc,r);

if(iolog(0)) printf("preprocesslpc: calculated lpc coeff.:\n");
if(iolog(0)) 
{
for(j=0;j<lpcorder;j++)
    printf("%.2e ",a[j]);
printf("\n");
}


/* Cepstral coefficients berechnen 					 */
	a2cep(a,lpcorder,featureptr);

if(iolog(0)) printf("preprocesslpc: calculated cepstral coeff.:\n");
if(iolog(0)) 
{
for(j=0;j<lpcorder;j++)
    printf("%.2e ",featureptr[j]);
printf("\n");
}


/* featurepointer auf naechsten vektor stellen */
	featureptr += lpcorder;

}/*ende schleife ueber sprachsignal */

/* speicher f"ur vektoren freigeben					*/
cfree((char *)fsig);
cfree((char *)window);
cfree((char *)fsignal);
cfree((char *)a);
cfree((char *)alpha);
cfree((char *)rc);
cfree((char *)r);

if(iolog(0)) printf("preprocesslpc: done\n");

return(features);

} /* end subroutine : preprocesslpc  */

/*----------------------------------------------------------------------
Name            : windw
Module          : util1.c
Title           : Hamming window

Description:
Berechnet Window Ausschnitt aus Sprachsignal. Die Funktion mu"s zur 
Initialisierung mit *flg = -1 aufgerufen werden (dabei wird die Windown-
Funktion berechnet und in h abgelegt). Dabei wird *flg zu 1 gesetzt.
Mit *flg = 1 wird das Window aus h ab dem pointer x f"ur n Werte auf das 
Signal multipliziert. Vorsicht, dabei wird also der Inhalt ver"andert, 
daher nicht auf das Orginal-Signal anwenden (es sei denn, die Windows 
"uberlappen sich nicht).


Parameters:
n               : Windowl"ange in samples
flg		: pointer auf flag: -1 initialisierung
				     1 windowing
x		: pointer auf array mit signal
h		: pointer auf array mit window 
                : 

Return-Value    : void
-------------------------------------------------------------------------*/
void windw (short n,float *x,float *h,long *flg)
{
     float          c1=0.54,
                    c2=0.46,
                    pi=3.14159265;
	register float *x_akt,*h_akt;
	register int	i;
     double         constant;
     int            j,k,nv2,np1,nm1;

     np1 = n+1;
     nm1 = n-1;
     nv2 = np1/2;
     constant = (pi+pi)/(double)nm1;
    
     if (*flg == -1)	/* calculate hamming window */
     { for (i=0; i<nv2; i++)
       h[i] = c1 - c2*cos((double)i*constant);
       *flg = 1;
     }

     if (*flg == -2)	/* read window funktion out of file */
     {
	i = readwin(h);
	if (i != nv2){
	   printf("wrong window funktion: has %d elements, should have %d !\n",
		i,nv2);
	   exit(1);
	}
       *flg = 1;
     }

     x_akt = x; h_akt = h;
     if(*flg)	
     { for (i=0; i<nv2; i++)
       *x_akt++ *= *h_akt++;

       k=nm1-i;
       h_akt = &h[k];
       for ( ; i<n; i++)
	         *x_akt++  *= *h_akt--;
     }
} /* end of routine WINDW.C *******************************************/

/*----------------------------------------------------------------------
Name            : readwin
Module          : preprocess.c
Title           : reads the values of an window funktion out of a file 

Description:
reads the values of an arbitrary window funktion out of a file named
'window.dat'. The file format must have the following conditions:
-	comment lines must begin with 'c' or 'C' or '*'. They are printed out at
	standard output.
-	data lines must include the window value in '=', e.g.
	" H(  65)=0.89721680E-02= H(  96)"
The function returns the number of matched window values.	

Parameters:
                : 
                : 

Return-Value    : 
-------------------------------------------------------------------------*/
static int readwin(float h[])
{
int	ii = 0,
	len;
char	*start,*end;
char zeile[80],
	file[30],
	temp[25];
FILE *winfil;


	winfil = fopen ("window.dat","r");
	if (winfil == NULL) {perror("can't open window file");exit(1);}
	while (fgets(zeile,79,winfil) != NULL){
		if (strlen(zeile)< 10) continue;
		if (zeile[0]=='c'||zeile[0]=='C'||zeile[0]=='.'||zeile[0]=='*')
			printf("%s",zeile);
		else {	start = (char *) strchr(zeile,'=') + 1;
			end = (char *) strrchr(zeile,'=');
			if (!start || !end){
				printf("error in file format: (%s)\n",zeile);
				exit(1);
			}
			strncpy(temp,start,(int)(end-start));
			h[ii++] = atof(temp);
		}
	} /* end while */
	fclose(winfil);
	return (ii);
} /* end of routine READWIN.C *******************************************/

/*----------------------------------------------------------------------
Name            : preem
Module          : util1.c
Title           : Hochpassfilter

Description:
das "ubergebene (gefensterte) signalsegment wird mit Hochpass gefiltert.

Parameters:
n		: anzahl der werte im signalsegent
x               : pointer auf signalsegment
x_pre		: ?
pre		: ?

Return-Value    : void 
-------------------------------------------------------------------------*/
void preem(short n,float *x,float *x_pre,float pre)
{
	register int	i;
	register float	*x_akt,xh;

	x_akt = x;
	for (i=0; i<n; i++)
	{
		xh = *x_akt;
		*x_akt++  -= *x_pre * pre;
		*x_pre = xh;
	}
} /* End of routine PREEM.C *******************************************/

/*----------------------------------------------------------------------
Name            : autom
Module          : util1.c
Title           : autoregressive analyse

Description:
berechnet die KKoeffizienten des inversen LPC-Filters f"ur ein gegebenes
Signal-Segment

Parameters:
n               : anzahl der Werte im Signal-Segment
m		: Ordnung des Filters
x		: Signal-Segment, array mit n Werten
a		: inverse Filter-Koeffizienten
alpha		: vector of energy terms for each iterat
rc		: vector of reflection coefficients     
r		: vector of autokorrelation             

Return-Value    : void
-------------------------------------------------------------------------*/
void autom(short n,float *x,short m,float *a,float *alpha,float *rc,float *r)
{
     int       mp1,i,idx,
               minc,mh;
     float     alpmin,
		*r_akt;		/* output pointer	*/
     register  float 	s=0.,*data1,*data2;


/*do autocorrelation, mp1 coefficients                                 */
{
register int	k,np;	/* indices		*/

     r_akt = r;
     mp1 = m+1;
     for (k=0; k<mp1; k++) 
     { s = 0.; data2 = x + k;
       for (np = k, data1 = x; np < n; np++) s += *data1++ * *data2++;
       *r_akt++ = s;
     }
}
/*do autoregressive analysis                                          */
{
register int	ip;	/* indices		*/
register float aib,aip;

     a[0]     = 1.;
     alpha[0] = r[0];
     if (!m || r[0] <= 0 || m == 1){ perror("autom: invalid parameter"); exit(0);}
     rc[0]    = -r[1]/r[0];
     a[1]     = rc[0];
     alpha[1] = r[0]+r[1]*rc[0];

    r_akt = &r[2];
    for (minc=2; minc<=m; minc++)
     { s=0.;
       alpmin=alpha[minc-1];
	data1 = r_akt++; data2 = a; 
        for (ip=1; ip<=minc; ip++)   s  += *data1-- * *data2++;
       s /= -alpmin;
       mh     = minc/2; 
	 data1 = &a[minc]; data2 = a;
         for (ip = 1; ip <= mh; ip++)
	 { aip = *++data2;
           aib = *--data1;
           *data2 = aip + s * aib;
           *data1 = aib + s * aip;
         }
       a[minc]    = s;
       alpha[minc]=alpmin-alpmin * s * s;
       rc[minc-1] = s;
     }
}
} /* end of routine AUTOM.C ********************************************/

/*----------------------------------------------------------------------
Name            : a2cep 
Module          : util1.c
Title           : berechnet cepstrum aus LPC

Description:

Parameters:
sa              : array von LPC-Koeffizienten
m               : anzahl der LPC-Koeffizienten in sa
alpha		: vector of energy terms for each iterat
cep		: m+1 Cepstral-Koeffizienten

Return-Value    : void
-------------------------------------------------------------------------*/
void a2cep(float *sa,short m,float *cep)
{
int	l;			/* Laufindices		*/
register float	sum;		/* Zwischenvariable	*/
register int	j,jb;		/* Laufindices		*/
register float	*sa_akt,*cep_akt;	/* aktuelle Adressen	*/

	cep_akt = cep;		/* Initialisierung	*/
	sa_akt = sa;

	*cep_akt++ = -*++sa_akt;

	for (l = 2; l <= m ; l++){
	   sum = l * sa[l];
	   cep_akt = &cep[l-1];
	   sa_akt = sa;
	   for (j = 1,jb = l - 1; j < l ; j++, jb--)
		sum += *++sa_akt * *--cep_akt * jb;	/*  !!! (jb-1)  */
	   cep[l-1] = -sum / l;
	}

	return;
} /* ende subroutine a2cep */

