
/***********************************************************************
************************************************************************

Module		: DATA
Title		: reading data from PhonDat files version 2

Author		: F. Schiel
Date/revision   : 16.12.93 / 15.11.93

Description	: 
contains function for reading and converting data from PhonDat files 
version 2. You need an ANSI C compiler to use this module. As the data 
formats differ on various operating systems, you must choose the approriate 
function for your system.

Link modules and libraries:
header

Contained functions:
read_data_v2_vms	: reads PhonDat data under VMS operating system
                          (Header version 2 only !)
read_data_v2_sun	: reads PhonDat data under SUN operating system
                          (Header version 2 only !)
read_data_sun		: reads PhonDat data under SUN operating system

*************************************************************************/
# include <stdio.h>
# include "ipkclib.h"

/* DEFINES, only used within this module ********************************/


/* TYPE DEFINITIONS, only used within this module  **********************/


/* GLOBAL VARIABLES (avoid, if possible !) ******************************/


/* FUNCTION PROTOTYPES, capsuled in this module *************************/


/*----------------------------------------------------------------------
Name            : read_data_v2_vms
Module		: DATA
Title           : reads PhonDat data under VMS operating system

Description:
reads speech data from a PhonDat data file version 2 under a VMS operating 
system. the data is stored in dynamic array of signed short, the memory of
this array can be freed by using cfree() to the returned pointer. before
invoking this function you should read the header information of the data
file by the function read_header_v2_vms() in module phondat_header and 
give the pointer to the header structure Phon_header_2 to this function. the
functions returns the pointer to the speech data and the number of read
samples, if an error occurs it writes an error message to stderr and
returns NULL. 

Parameters:
header		: pointer to header structure version 2 
samples		: pointer to # of read samples 
file		: pointer to PhonDat file

Return-Value    : pointer to speech data
		  Null if error occurs
-------------------------------------------------------------------------*/
short *read_data_v2_vms(Phon_header_2 *header,int *samples,FILE *file)
{ 
	short *data;

/* check if header structure is present					*/
    if(header == NULL)
    {
	fprintf(stderr,"read_data_v2_vms: no header structure present\n");
	return(NULL);
    }

/* check version in header structure					*/
    if(header->version != 2)
    {
	fprintf(stderr,"read_data_v2_vms: no header structure version 2\n");
	return(NULL);
    }

/* read # of header blocks (512 byte), skip file pointer to first data	*/
    if(fseek(file,header->anz_header*512,0) == EOF)
    {
	fprintf(stderr,"read_data_v2_vms: error operating with filepointer\n");
	perror("read_data_v2_vms");
	return(NULL);
    }

printf("read_data_v2_vms: file pointer set to %d\n",ftell(file));

/* allocate memory for speech data					*/
    *samples = header->nspbk*256;

printf("read_data_v2_vms: number of data samples: %d\n",*samples);

    if((data = (short *)calloc(*samples,sizeof(short))) == NULL)
    {
	fprintf(stderr,
	"read_data_v2_vms: cannot allocate memory for speech data\n");
	perror("read_data_v2_vms");
	return(NULL);
    }

/* read speech data							*/
    if(fread(data,sizeof(short),*samples,file) != *samples)
    {
	fprintf(stderr,
	"read_data_v2_vms: error reading speech data from file\n");
	perror("read_data_v2_vms");
	cfree((char *)data);
	return(NULL);
    }

    return(data);

} /* end subroutine : read_data_v2_vms */

/*----------------------------------------------------------------------
Name            : read_data_v2_sun
Module		: DATA
Title           : reads PhonDat data under SUN operating system

Description:
reads speech data from a PhonDat data file version 2 under a SUN operating 
system. the data is stored in dynamic array of signed short, the memory of
this array can be freed by using cfree() to the returned pointer. before
invoking this function you should read the header information of the data
file by the function read_header_v2_sun() in module phondat_header and give the
pointer to the header structure Phon_header_2 to this function. the
functions returns the pointer to the speech data and the number of read
samples, if an error occurs it writes an error message to stderr and
returns NULL. 

Parameters:
header		: pointer to header structure version 2 
samples		: pointer to # of read samples 
file		: pointer to PhonDat file

Return-Value    : pointer to speech data
		  Null if error occurs
-------------------------------------------------------------------------*/
short *read_data_v2_sun(Phon_header_2 *header,int *samples,FILE *file)
{ 
	char buff,*charpoint;
	short *data;
	int i;

/* check if header structure is present					*/
    if(header == NULL)
    {
	fprintf(stderr,"read_data_v2_sun: no header structure present\n");
	return(NULL);
    }

/* check version in header structure					*/
    if(header->version != 2)
    {
	fprintf(stderr,"read_data_v2_sun: no header structure version 2\n");
	return(NULL);
    }

/* read # of header blocks (512 byte), skip file pointer to first data	*/
    if(fseek(file,header->anz_header*512,0) == EOF)
    {
	fprintf(stderr,"read_data_v2_sun: error operating with filepointer\n");
	perror("read_data_v2_sun");
	return(NULL);
    }

/* allocate memory for speech data					*/
    *samples = header->nspbk*256;

    if((data = (short *)calloc(*samples,sizeof(short))) == NULL)
    {
	fprintf(stderr,
	"read_data_v2_sun: cannot allocate memory for speech data\n");
	perror("read_data_v2_sun");
	return(NULL);
    }

/* read speech data							*/
    if(fread(data,sizeof(short),*samples,file) != *samples)
    {
	fprintf(stderr,
	"read_data_v2_sun: error reading speech data from file\n");
	perror("read_data_v2_sun");
	cfree((char *)data);
	return(NULL);
    }

/* conversion to sun: change higher and lower byte in each sample	*/
    charpoint = (char *)data;
    for(i=0;i<*samples;i++)
    {
	buff = charpoint[0];
	charpoint[0] = charpoint[1];
	charpoint[1] = buff;
	charpoint += 2;
    }	

    return(data);

} /* end subroutine : read_data_v2_sun */

/*----------------------------------------------------------------------
Name            : read_data_sun
Module		: DATA
Title           : reads PhonDat data under SUN operating system

Description:
reads speech data from a PhonDat data file version 1 or 2 under a SUN operating 
system. The data is stored in dynamic array of signed short, the memory of
this array can be freed by using cfree() to the returned pointer. before
invoking this function you should read the header information of the data
file by the function read_header_sun() in module phondat_header and give the
pointer to the header structure Phon_header_2 to this function. the
functions returns the pointer to the speech data and the number of read
samples, if an error occurs it writes an error message to stderr and
returns NULL. 

Parameters:
header		: pointer to header structure version 2 
samples		: pointer to # of read samples 
file		: pointer to PhonDat file

Return-Value    : pointer to speech data
		  Null if error occurs
-------------------------------------------------------------------------*/
short *read_data_sun(Phon_header_2 *header,int *samples,FILE *file)
{ 
	char buff,*charpoint;
	short *data;
	int i;

/* check if header structure is present					*/
    if(header == NULL)
    {
	fprintf(stderr,"read_data_sun: no header structure present\n");
	return(NULL);
    }

/* check version in header structure and skip the header blocks	*/
    if(header->version == 1)
      fseek(file,512,0);
    if(header->version == 2)
      fseek(file,header->anz_header*512,0);

/* allocate memory for speech data					*/
    *samples = header->nspbk*256;

    if((data = (short *)calloc(*samples,sizeof(short))) == NULL)
    {
	fprintf(stderr,
	"read_data_sun: cannot allocate memory for speech data\n");
	perror("read_data_sun");
	return(NULL);
    }

/* read speech data							*/
    if(fread(data,sizeof(short),*samples,file) != *samples)
    {
	fprintf(stderr,
	"read_data_sun: error reading speech data from file\n");
	perror("read_data_sun");
	cfree((char *)data);
	return(NULL);
    }

/* conversion to sun: change higher and lower byte in each sample	*/
    charpoint = (char *)data;
    for(i=0;i<*samples;i++)
    {
	buff = charpoint[0];
	charpoint[0] = charpoint[1];
	charpoint[1] = buff;
	charpoint += 2;
    }	

    return(data);

} /* end subroutine : read_data_sun */

/*----------------------------------------------------------------------
Name            : read_data_vms
Module		: DATA
Title           : reads PhonDat data under VMS operating system

Description:
reads speech data from a PhonDat data file version 1 or 2 under a VMS operating 
system. The data is stored in dynamic array of signed short, the memory of
this array can be freed by using cfree() to the returned pointer. before
invoking this function you should read the header information of the data
file by the function read_header_sun() in module phondat_header and give the
pointer to the header structure Phon_header_2 to this function. the
functions returns the pointer to the speech data and the number of read
samples, if an error occurs it writes an error message to stderr and
returns NULL. 

Parameters:
header		: pointer to header structure version 2 
samples		: pointer to # of read samples 
file		: pointer to PhonDat file

Return-Value    : pointer to speech data
		  Null if error occurs
-------------------------------------------------------------------------*/
short *read_data_vms(Phon_header_2 *header,int *samples,FILE *file)
{ 
	char buff,*charpoint;
	short *data;
	int i;

/* check if header structure is present					*/
    if(header == NULL)
    {
	fprintf(stderr,"read_data_vms: no header structure present\n");
	return(NULL);
    }

/* check version in header structure and skip the header blocks	*/
    if(header->version == 1)
      fseek(file,512,0);
    if(header->version == 2)
      fseek(file,header->anz_header*512,0);

/* allocate memory for speech data					*/
    *samples = header->nspbk*256;

    if((data = (short *)calloc(*samples,sizeof(short))) == NULL)
    {
	fprintf(stderr,
	"read_data_vms: cannot allocate memory for speech data\n");
	perror("read_data_vms");
	return(NULL);
    }

/* read speech data							*/
    if(fread(data,sizeof(short),*samples,file) != *samples)
    {
	fprintf(stderr,
	"read_data_vms: error reading speech data from file\n");
	perror("read_data_vms");
	cfree((char *)data);
	return(NULL);
    }

    return(data);

} /* end subroutine : read_data_vms */
