
/*=========================================================

module		: matrix.c
title		: matrix - functions
written by	: B. Plannerer
date		: . . 90 / 21.06.94

description	: matrix - functions :
		    + genmat   : allocate memory for a matrix
		    + delmat   : free allocated memory
		    + getmat   : enter matrix by keyboard
		    + prmat    : show matrix on screen
		    + writemat : write matrix to file
		    + readmat  : read matrix from file
		    + ascwmat  : write matrix to ascii - file
		    + ascrmat  : read matrix form ascii - file
		    + gau      : Gauss - algorithm
		    + invdet   : compute inv(a) and det(a)

needed modules	:   

called functions:

=========================================================*/


#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <ipkclib.h>

/*"numerisch Null": bei IEEE-DOUBLE groesser als 2.2E-308 bzw. 2 hoch -1022*/
#ifndef ZERO
#define ZERO 1.0E-30
#endif

/*kleinster Pivot-Betrag bezogen auf Zeilennorm*/
/*fuer GAUSS-ALG.*/
#ifndef MINPIV
#define MINPIV 1.0E-10
#endif


/*-------------------------------------------------------*/
/*void prmat (matrix *) 				 */
/*Ausgabe einer Matrix auf Standardausgabe		 */
/*-------------------------------------------------------*/
void prmat (matrix * m) {
    long i,j ;
    printf ("\n") ;
    for (i = 0 ; i < m->z ; i++) {
	for (j = 0 ; j < m->s ; j++) {
	    /*Matrixelement ausgeben*/
	    printf ("%.10le\t",m->m[i][j]) ;
	    }/*for j*/
	printf ("\n") ;
	}/*for i*/
    printf ("\n") ;
    }/*Ende prmat*/



/*-------------------------------------------------------*/
/*matrix * genmat (long,long)				 */
/*erzeugt "matrix"					 */
/*-------------------------------------------------------*/
matrix * genmat (long z , long s) {
	long i;
	long zmsiz ;
	long zsiz ;
        long size;
	char * b_ptr ;
	matrix * m_ptr ;
	zsiz = z*sizeof(double*) ; /*Platzbedarf fuer Pointer  auf Elemente*/
	zmsiz = s*sizeof(double) ; /*Platzbedarf fuer eine Zeile */
	/*Platz schaffen fuer Matrix*/
        size = sizeof(matrix) + zsiz + z*zmsiz;
        /* wenn die Zahl der Zeilen (pointer) gerade ist, stimmt das
           alignment der doubles auf manchen Prozessoren (z.B. Sun)
           nicht mehr (pointer nicht durch 8 teilbar). Deshalb werden
           4 Byte mehr angefordert und die Pointer richtig plaziert (s.u.) */
        if((z % 2) == 0) size += 4;
	if( (b_ptr = calloc((size_t)size,sizeof(char)) ) == NULL) {
	    printf ("OUT OF MEMORY ERROR (%s,%d)\n",__FILE__,__LINE__) ;
	    return (NULL) ;
	    }
	/*Pointer setzen*/
	m_ptr = (matrix *)b_ptr ;
	b_ptr += sizeof (matrix) ;
	m_ptr->m = (double **)b_ptr ;
	b_ptr += zsiz ;
        if((z % 2) == 0) b_ptr += 4;
	for (i = 0 ; i < z ; i++) {
	    m_ptr->m[i] = (double*)b_ptr ;
	    b_ptr += zmsiz ;
	    }
	m_ptr->z = z ;
	m_ptr->s = s ;
	return (m_ptr) ;
	}/*Ende genmat*/



/*-------------------------------------------------------*/
/*void delmat (matrix ** )				 */
/*loescht "matrix"					 */
/*-------------------------------------------------------*/
void delmat (matrix ** m) {
    free (*m) ;
    *m = (matrix *)NULL ;
    }/*Ende delmat*/



/*-------------------------------------------------------*/
/*int writemat (FILE * , matrix *)			 */
/*schreibt "matrix" auf Platte				 */
/*-------------------------------------------------------*/
int writemat (FILE * fp , matrix * m ) {
    long i ;
    /*Struktur schreiben*/
    if ( (fwrite (m,sizeof(matrix),1,fp)) != 1 ) {
	/*Struktur konnte nicht geschrieben werden*/
	printf ("ERROR WRITING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
	return (-1) ;
	}/* if < 1 */
    /*Zeilen schreiben*/
    /*Es wird zeilenweise geschrieben, da evtl. Zeilentausch vorgenommen wurde*/
    for (i = 0 ; i < m->z ; i++) {
	if ((long)(fwrite(m->m[i],sizeof(double),(size_t)m->s,fp)) != m->s) {
	    printf ("ERROR WRITING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
	    return (-1) ;
	    }
	}/*for i*/
    return (0) ;
    }/*Ende writemat*/



/*-------------------------------------------------------*/
/*int ascwmat (FILE * , matrix *)			 */
/*schreibt "matrix" im ASCII-Format auf Platte		 */
/*-------------------------------------------------------*/
int ascwmat (FILE *fp, matrix *m) {
    long i,j ;
    /*Struktur schreiben*/
    if (fprintf(fp,"%ld\n",m->z) < 0) {
	printf ("ERROR WRITING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
	return (-1) ;
	}
    if (fprintf(fp,"%ld\n",m->s) < 0) {
	printf ("ERROR WRITING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
	return (-1) ;
	}
    /*Elemente schreiben*/
    for (i = 0 ; i < m->z ; i++) {
	for (j = 0 ; j < m->s ; j++) {
	    if (fprintf(fp,"%.10le\n",m->m[i][j]) < 0) {
		printf ("ERROR WRITING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
		return (-1) ;
		}
	    }/*for j*/
	}/*for i*/
    return (0) ;
    }/*Ende ascwmwat*/



/*-------------------------------------------------------*/
/*matrix * readmat (FILE *)				 */
/*liest "matrix" von Platte				 */
/*-------------------------------------------------------*/
matrix * readmat (FILE * fp) {
    long i ;
    long zmsiz ;
    long zsiz ;
    long esiz ;
    matrix * m ;
    char * b_ptr ;
    /*Platz fuer Struktur*/
    if ( (b_ptr = calloc(1,sizeof(matrix)) ) == NULL) {
	printf ("OUT OF MEMORY ERROR (%s,%d)\n",__FILE__,__LINE__) ;
	return (NULL) ;
	}
    /*Struktur lesen*/
    if ( (fread(b_ptr,sizeof(matrix),1,fp)) != 1) {
	printf ("ERROR READING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
	free (b_ptr) ;
	return (NULL) ;
	}
    m = (matrix *)b_ptr ;
    zsiz = m->z*sizeof(double*) ; /*Platz fuer Pointer*/
    zmsiz = m->s*sizeof(double) ; /*Platz fuer eine Zeile*/
    esiz = m->z*zmsiz ; 	  /*Platz fuer alle Elemente*/
    /*Platz fuer Matrixelemente und Zeilenpointer */
    if ((b_ptr = realloc(b_ptr,(size_t)(sizeof(matrix)+zsiz+esiz)))==NULL) {
	printf ("OUT OF MEMORY ERROR (%s,%d)\n",__FILE__,__LINE__) ;
	return (NULL) ;
	}
     m = (matrix *) b_ptr ;
     b_ptr += sizeof (matrix) ;
     m->m = (double **)b_ptr ;
     b_ptr += zsiz ;
    /*Matrixelemente einlesen*/
	if ((long)fread(b_ptr,sizeof(char),(size_t)esiz,fp) != esiz) {
	    printf ("ERROR READING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
	    delmat (&m) ;
	    return (NULL) ;
	    }
     /*Zeilenpointer setzen*/
	for (i = 0 ; i < m->z ; i++) {
	    m->m[i] = (double*)b_ptr ;
	    b_ptr += zmsiz ;
	    }
    return (m) ;
    }/*Ende readmat*/



/*-------------------------------------------------------*/
/*matrix * ascrmat (FILE *)				 */
/*liest "matrix" von Platte				 */
/*-------------------------------------------------------*/
matrix * ascrmat (FILE *fp) {
    long i,j ;
    char * b_ptr ;
    matrix * m ;
    long zsiz ;
    long zmsiz ;
    /*Platz fuer Struktur*/
    if ( (b_ptr = calloc(1,sizeof(matrix)) ) == NULL) {
	printf ("OUT OF MEMORY ERROR (%s,%d)\n",__FILE__,__LINE__) ;
	return (NULL) ;
	}
    m = (matrix*)b_ptr ;
    /*Struktur lesen*/
    if (fscanf(fp,"%ld",&m->z)!=1) {
	printf ("ERROR READING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
	free (b_ptr) ;
	return (NULL) ;
	}
    if (fscanf(fp,"%ld",&m->s)!=1) {
	printf ("ERROR READING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
	free (b_ptr) ;
	return (NULL) ;
	}
    zsiz = m->z*sizeof(double*) ; /*Platz fuer Pointer*/
    zmsiz = m->s*sizeof(double) ; /*Platzfuer eine Zeile*/
    /*Platz fuer Matrixelemente und Zeilenpointer */
    if ((b_ptr = realloc(b_ptr,(size_t)(sizeof(matrix)+zsiz+m->z*zmsiz)))==NULL) {
	printf ("OUT OF MEMORY ERROR (%s,%d)\n",__FILE__,__LINE__) ;
	return (NULL) ;
	}
    /*Pointer setzen*/
	m = (matrix *) b_ptr ;
	b_ptr += sizeof (matrix) ;
	m->m = (double **)b_ptr ;
	b_ptr += zsiz ;
	for (i = 0 ; i < m->z ; i++) {
	    m->m[i] = (double*)b_ptr ;
	    b_ptr += zmsiz ;
	    }
    /*Elemente lesen*/
    for (i = 0 ; i < m->z ; i++) {
	for (j = 0 ; j < m->s ; j++) {
	    if (fscanf(fp,"%lf",&m->m[i][j]) != 1) {
		printf ("ERROR READING MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
		delmat (&m) ;
		return (NULL) ;
		}
	    }/*for j*/
	}/*for i*/
    return (m) ;
    }/*Ende ascrmat*/



/*-------------------------------------------------------*/
/*matrix * getmat (long , long)				 */
/*Eingabe einer Matrix von Standardeingabe		 */
/*-------------------------------------------------------*/
matrix * getmat (long z, long s) {
    long i,j ;
    matrix * m ;
    if ( (m = genmat(z,s) ) == (matrix*)NULL) {
	printf ("KEIN SPEICHER VERFUEGBAR(%s,%d)\n",__FILE__,__LINE__) ;
	return (m) ;
	}
    printf ("EINGABE EINER MATRIX MIT %ld ZEILEN UND %ld SPALTEN (MATRIX/GETMAT)\n",z,s) ;
    for (i = 0 ; i < z ; i++ ) {
	for (j = 0 ; j < s ; j++) {
	    printf ("\nELEMENT %ld %ld: ",i,j) ;
	    scanf("%lf",&m->m[i][j]) ;
	    }/*for j*/
	}/*for i*/
    printf ("ENDE DER EINGABE\n") ;
    return (m) ;
    }/*Ende getmat*/



/*--------------------------------------------------------*/
/*int gau (matrix *a , matrix *b)			  */
/*GAUSS-ALGORITHMUS					  */
/*MIT AEQUILIBRIERUNG UND SPALTEN-PIVOTISIERUNG 	  */
/*Ergebnis in b, matrix a wird zerstoert		  */
/*--------------------------------------------------------*/

int gau (matrix *a , matrix *b)
{
/*Variablen*/
long n ; /*Dimension des Gleichungssystems*/
long i,j,l ;
long p ;
long piv ; /*Index des Pivot-Elementes*/
double max ; /*fuer Maximumsuche*/
double tmp ;
double c ; /*Faktor fuer Reduktion*/
double * s ; /*inverse Zeilennormen*/
double * t_ptr ; /*fuer Zeilentausch*/
/*Variablen Ende*/
/*Test auf Dimensionen der Matrizen*/
if ((a->z != a->s)||(b->z != a->z)) {
    printf ("ERROR: INVALID DIMENSION (%s,%d)\n",__FILE__,__LINE__) ;
    return (-1);
    }
n = a->z ;  /*Dimension eintragen*/
/*Speicher fuer Zeilennormen*/
if((s = calloc((size_t)n,sizeof(double)))== NULL) {
    printf ("ERROR : OUT OF MEMORY (%s,%d)\n",__FILE__,__LINE__);
    return (-1);
    }
/*Aequilibrierung: inverse Zeilennormen ermitteln*/
for (i = 0 ; i < n ; i++) {
    tmp = 0.0 ;
    for (j = 0 ; j < n ; j++){
	tmp += fabs(a->m[i][j]);
	}/*for j*/
    if (tmp == 0.0) {
	printf ("ERROR: ZERO-ROW IN MATRIX 'A' (%s,%d)\n",__FILE__,__LINE__) ;
	free (s) ;
	return (-1) ;
	}
    s[i] = 1.0 / tmp ;
    }/*for i*/
/*Aequilibrierung Ende*/
/*Dreieckszerlegung: p ist Spaltennummer*/
for (p = 0 ; p < n-1 ; p++) {

    /*Spaltenpivot-Suche*/
    max = fabs(a->m[p][p])*s[p];
    piv = p ;
    for (i = p+1 ; i < n ;i++) {
	if (max < (tmp=fabs(a->m[i][p])*s[i])) {
	    max = tmp ;
	    piv = i ;
	    }
	}/*for i*/
    if (piv != p){
	/*Zeilen tauschen*/
	t_ptr = a->m[p] ;
	a->m[p] = a->m[piv] ;
	a->m[piv] = t_ptr ;
	t_ptr = b->m[p] ;
	b->m[p] = b->m[piv] ;
	b->m[piv] = t_ptr ;
	tmp = s[p] ;
	s[p] = s[piv] ;
	s[piv] = tmp ;
	}
    if ((fabs(a->m[p][p])*s[p])<= MINPIV) {
	printf ("ERROR: PIVOT - UNDERFLOW (%s,%d)\n",__FILE__,__LINE__);
	free(s) ;
	return (-1) ;
	}
    /*Reduktion*/
    for (i = p+1 ; i < n ; i++) {
	c = a->m[i][p]/a->m[p][p] ;
	for (j = p+1 ; j < n ; j++) {
	    a->m[i][j] -= c*a->m[p][j] ;
	    }/*for j*/
	for (j = 0 ; j < b->s ; j++){
	    b->m[i][j] -= c*b->m[p][j] ;
	    }/*for j*/
	/*TEST*/
	/*a->m[i][p] = c ;*/
	}/*for i*/

    }/*for p*/
/*Dreieckszerlegung Ende*/

/*Loesungsvektoren berechnen*/
if (fabs(a->m[n-1][n-1])*s[n-1] <= MINPIV) {
    printf ("ERROR: UNDERFLOW (%s,%d)\n",__FILE__,__LINE__);
    free(s) ;
    return (-1) ;
    }
for (l = 0 ; l < b->s ; l++){
    b->m[n-1][l] /= a->m[n-1][n-1] ;
    for (i = n-2 ; i >= 0 ; i--) {
	tmp = b->m[i][l] ;
	for (j = i+1 ; j < n ; j++) {
	    tmp -= b->m[j][l]*a->m[i][j] ;
	    }/*for j*/
	b->m[i][l] = tmp/a->m[i][i] ;
	}/*for i*/
    }/*for l*/
/*Loesungsvektoren berechnen Ende*/

/*Speicher freimachen*/
free(s) ;
return (0) ;
}/*Ende gau*/



/*-------------------------------------------------------*/
/*int invdet(matrix *a,matrix*b,double *d)		 */
/*Berechnung von inv(a) und abs(det(a))	aus		 */
/*einer Matrix a: Inverse in b, Determinantenbetrag in d */
/*matrix a bleibt unveraendert				 */
/*-------------------------------------------------------*/
int invdet (matrix *a,matrix *b,double *d)
{
/*Variable*/
long i,j,n ;
double c ;
matrix * aa ;
/*Variable Ende*/

/*Test auf Dimension der Matrizen*/
if ((a->z != a->s)||(b->z != b->s)||(a->z != b->z)) {
    printf ("ERROR: INVALID DIMENSION (%s,%d)\n",__FILE__,__LINE__) ;
    return (-1) ;
    }
/*Dimension merken*/
n = a->z ;
/*Platz schaffen fuer Matrix AA*/
if ((aa = genmat(n,n))== NULL){
    printf("OUT OF MEMORY ERROR (%s,%d)\n",__FILE__,__LINE__);
    return (-1) ;
    }
/*Matrix a nach AA kopieren fuer Gauss-Alg.*/
for (i = 0 ; i < n ; i++) {
    for (j = 0 ; j < n ; j++) {
	aa->m[i][j] = a->m[i][j] ;
	}/*for j*/
    }/*for i*/
/*Ergebnismatrix = Einheitsmatrix setzen*/
for (i = 0 ; i < n ; i++) {
    for (j = 0 ; j < n ; j++) {
	b->m[i][j] = ((i==j) ? 1.0 : 0.0) ;
	}/*for j*/
    }/*for i*/
/*Gauss - Algorithmus*/
if (gau(aa,b)) {
    printf("ERROR: SINGULAR MATRIX (%s,%d)\n",__FILE__,__LINE__) ;
    delmat (&aa) ;
    return (-1) ;
    }
/*Determinante berechnen*/
c = aa->m[0][0] ;
for (i = 1 ; i < n ; i++) {
    c *= aa->m[i][i] ;
    }/*for i*/
*d = fabs(c);
/*Speicher freimachen*/
delmat (&aa);
return (0) ;
}/*Ende invdet*/

