# This script extracts a type 4 tier from a BPF and creates the core part
# of a TextGrid file (not the header!). The input must not contain any other 
# BPF tier!
#
# The beginning of the next segment must have the same value as the end of the 
# previous segment in TextGrid. This is necessary since praat has a bug that
# causes boundaries with a small deviation of end and begin time
# not to be movable. Therefore the script insert empty labelled silence intervals,
# if the input BPF segment are not consecutive.

# Needs commandline variables SAMPLERATE and MAXSAMPLE (= no of samples in signal)

BEGIN   {
          segcnt = 1
	  xmaxold = 0
          oldend = -1  # because the very first BPF segment could start with 0
	}  
	{
          # get label string (can contain white spaces!)
          label = $5
          for(i=6;i<=NF;i++) label = label " " $i

          # check if the end of segment exceeds MAXSAMPLE; this can happen, if the processed
          # class 4 BPF is not consistent to the first class 4 reference tier, from which the
          # MAXSAMPLE was taken (e.g. a SAP tier last segment exceeds the last MAU segment);
          # if the last segment is *shorter* than MAXSAMPLE, we insert an empty labelled segment.
          beginn = $2
          dauer = $3
          if(beginn+dauer+1 > MAXSAMPLE) dauer = MAXSAMPLE - beginn - 1

          # check for deleted segments (only in PHO and SAP as far as we know):
          # deleted segments are labelled witha preceeding '-' (e.g. '-d') and both,
          # begin and duration sample are set to 0. Since we cannot possibly encode 
          # such a thing in TextGrid, we simply ignore those.
          if(beginn == 0 && dauer == 0) next 
          # check for implicit silence intervals (which can happen in BPF 
          # e.g. in TRN tier, but is not allowed in TextGrid)
          # test for silence interval between current and previous BPF segment
          if ( beginn != (oldend+1) ) {
            # print implicite silence interval between non-consecutive BPF segments
            xmin = xmaxold
            xmax = beginn * 1.0 / SAMPLERATE
	    printf("        intervals [%d]:\n",segcnt)
	    printf("            xmin = %f\n",xmin)
	    printf("            xmax = %f\n",xmax)
	    printf("            text = \"\"\n")
            xmaxold = xmax
            segcnt ++
          } 
          # print normal BPF segment
          xmin = xmaxold
          xmax = (beginn + dauer + 1) * 1.0 / SAMPLERATE  
          oldend = beginn + dauer
	  printf("        intervals [%d]:\n",segcnt)
	  printf("            xmin = %f\n",xmin)
	  printf("            xmax = %f\n",xmax)
	  printf("            text = \"%s\"\n",label)
          xmaxold = xmax
          segcnt ++
	}
END     {
          # print a final silence interval, if the last printed BPF segment end before MAXSAMPLE
          if ( (oldend+1) < MAXSAMPLE ) {
            xmin = xmaxold
            xmax = MAXSAMPLE * 1.0 / SAMPLERATE
	    printf("        intervals [%d]:\n",segcnt)
	    printf("            xmin = %f\n",xmin)
	    printf("            xmax = %f\n",xmax)
	    printf("            text = \"\"\n")
          }
        }
