import argparse, sys, shlex

def custom_error(self, message):
    # since we'd like our error messages to be in a consistent format, we have
    # to override the ArgumentParser.error function with our own. the only thing
    # this changes is the prefix of the error message. confirmed to work with
    # python version >= 3.6. If it's causing any problems in the future, you
    # can just delete it.
    # (this really isn't the best style but in this case there's no way around it)
    self.print_usage(sys.stderr)
    self.exit(2, argparse._('ERROR: %s: %s\n') % (self.prog, message))
argparse.ArgumentParser.error = custom_error

##### Validation functions
def bas_compatible_filename(fname):
    '''Verify that `fname` contains no whitespace, non-ASCII characters or regex metacharacters.'''
    metachars = '*+?()[]{}\\|^$'  # excluding the dot since almost all filenames have it
    if all(not char.isspace() for char in fname) and \
       all(ord(char) < 128 for char in fname) and \
       all(metachar not in fname for metachar in metachars):
        return fname
    else:
        raise argparse.ArgumentTypeError(
            'the input file name must not contain any whitespace, non-ASCII '
            'characters, or POSIX regular expression metacharacters: '
            + metachars + ' (dots are allowed)'
        )

def even_length_str(string):
    '''Verify that `string` has an even number of characters.'''
    if len(string) % 2 == 0:
        return string
    else:
        raise argparse.ArgumentTypeError(
            'the brackets must come in pairs (i.e. have an even number of characters)'
        )

def non_whitespace_str(string):
    '''Verify that `string` contains no whitespace.'''
    if all(not char.isspace() for char in string):
        return string
    else:
        raise argparse.ArgumentTypeError(
            'the whitespace replacement character cannot contain any whitespace itself'
        )

#####
if __name__ == '__main__':
    parser = argparse.ArgumentParser('textEnhance', add_help = False, description = 'Prepare text files for processing by the BAS WebServices.')
    parser.add_argument('input', type = bas_compatible_filename, help = "The input file. Must be a regular file")
    parser.add_argument('-o', '--outfile', required = True, help = 'The output file. Does not have to be a regular file')
    parser.add_argument('-v', '--verbose', action = 'store_true', help = 'Print debugging information to STDOUT as the program is running')
    parser.add_argument('--brackets', default = '<>', type = even_length_str, help = 'One or more pairs of characters which mark enclosed comments in the input (default: \'<>\')')
    parser.add_argument('--left-bracket', default = '', help = 'One or more characters which mark comments reaching until the end of the line (default: none)')
    parser.add_argument('--replace-whitespace-char', dest = 'whitespace_replacement', default = '_', type = non_whitespace_str, help = 'The character that whitespace in comments should be substituted by (default: \'_\')')
    parser.add_argument('--version', action = 'version', version = '1.0', help = 'Show version number and exit')
    parser.add_argument('--help', action = 'help', help = 'Show this help message and exit')
    args = parser.parse_args()
    
    print('#sourcethis')  # prepended to output so we don't accidentally
                          # try to parse the help page or the version info
    for var, val in args._get_kwargs():
        if isinstance(val, str):
            val = shlex.quote(val)
        print('%s=%s' % (var, val))
