# Usage: python3 convert-comments.py {outfile_path} {brackets} {left_bracket} {whitespace_replacement}

import sys, re

def convert_comment(match):
    """
    Take an `re.match` object and return it as a string enclosed in angle
    brackets and with all whitespace replaced by `whitespace_replacement`.
    """
    global whitespace_warning_given
    comment = match.group(0)
    if re.search(r'\s', comment) and not whitespace_warning_given:
        print('WARNING: textEnhance: Encountered whitespace in comments; '
              'will replace with \'_\' to comply with the BAS WebServices '
              '(you can specify a different replacement character using '
              'the --replace-whitespace-char option)', file = sys.stderr)
        # don't issue this warning again
        whitespace_warning_given = True
    comment = re.sub(r'\s+', whitespace_replacement, comment)
    if not (comment[0] == '<' and comment[-1] == '>'):
        # if the first and last characters aren't angle brackets, add them here
        comment = '<' + comment + '>'
    return comment

if __name__ == '__main__':
    outfile_path, brackets, left_bracket, whitespace_replacement = sys.argv[1:]
    if whitespace_replacement == '_':
        # this is a warning that we want to issue only once, and only if the
        # user didn't specify their own whitespace replacement character
        whitespace_warning_given = False
    else:
        whitespace_warning_given = True
    
    with open(outfile_path, mode = 'r', encoding = 'utf-8', newline = '\n') as outfile:
        content = outfile.read()
    
    ### Convert bilateral brackets
    for i in range(0, len(brackets), 2):
        left, right = brackets[i:i+2]
        left = re.escape(left)
        right = re.escape(right)
        comment_match = re.compile(left + '.*?' + right, flags = re.DOTALL)
        content = comment_match.sub(convert_comment, content)
    ###
    
    ### Convert left brackets
    for bracket in left_bracket:
        bracket = re.escape(bracket)
        comment_match = re.compile(bracket + '.*?$', flags = re.MULTILINE)
        content = comment_match.sub(convert_comment, content)
    ###
    
    with open(outfile_path, mode = 'w', encoding = 'utf-8', newline = '\n') as outfile:
        outfile.write(content)
