
/**************************************************************************
***************************************************************************

Modul           : label.c
Titel           : identifikation von labelungen in sotschek sprachdaten
Autor           : F. Schiel
Datum/Aenderung : 17.12.90 / 30.07.91

Beschreibung:
dieses modul enthaelt subroutinen zur bestimmung von labeln zu den vorhandenen 
sprachdatenbasen.

Anzulinkende Module	:

Subroutinen:
labgetipsk		: berechnung von IPSK-label aus sample und satz
labgetsilbos           	: berechnung von SILBOS-label aus sample und satz
labgetsamples		: berechnung von label-grenzen in samples zu ipsk-label
labmapipsktosilbos	: mappen von ipsk-gruppen auf silbos-label
labipskdummy		: 1, wenn label gueltig, aber nicht verwendbar
labipskerror		: 1, wenn label verboten
labipskdiacrit		: 1, wenn label ipsk-diacriticum
labipsktosilbos		: mappen 1 ipsk-phonem-label auf silbos-label
labipskgrouptosilbos	: mappen mehrerer ipsk-phonem-label auf silbos-label
labtransipsktosilbos	: uebersetzt ipsk-label-datei in silbos-label-struktur
labdelstlabel		: gibt speicher-bereich von silbos-label-struktur frei
labgetword		: liefert labelgrenzen eines wortes aus best. satz
labgetsatz		: liefert labelgrenzen eines best. satzes
labgetphonem		: liefert labelgrenzen eines phonems aus best. satz

*/
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include "ipkclib.h"

# define MAXLAB		10	/* max. anzahl labeln pro zeile 	*/
# define MAXPHON	100	/* max. anzahl von zurueckgelieferten 
				   segmenten (worte oder phoneme) 	*/


/*------------------------------------------------------------------------

Name            : labgetipsk 
Modul           : label
Titel           : berechnung von IPSK-label aus sample und satz

Beschreibung:
liefert IPSK-Labels zu bestimmten zeitpunkt (sample) in einem bestimmten 
sotschek-satz (auf platte)

Parameter:
satz		: name der labeldatei
sample		: sample-nummer

Return-Value	: anzahl gefundener ipsk-labeln
*/
int labgetipsk(char *satz,long sample,int ipsk[MAXLAB])
{
	int	i,
		count;

	long	end_label,
		begin_label;

	char	buff[80],	/* buffer fuer labelzeile		*/
		*newptr;

	FILE	*fplab;

/* initialisierung							*/
    if(sample <= 0)
    {
	fprintf(stderr,"labgetipsk: sample ist negativ oder 0 : %d\n",sample);
	return('.');
    }

/* label-datei oeffnen							*/
    if(!(fplab = fopen(satz,"r")))
    { 
    	fprintf(stderr,"labgetipsk: %s nicht vorhanden\n",satz);
	perror("labgetipsk");
	exit(1);
    }

/* zeilenweises einlesen der labeldatei					*/
    while((fgets(buff,80,fplab) != NULL))	
    {
/*      kommentarzeilen ueberspringen					*/
	if(*buff == '*') continue;

/*	printf("\nlabgetipsk: %s",buff);					*/

/* labelgrenzen einlesen						*/
	if((begin_label = strtol(buff,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetipsk: keine labelgrenzen lesbar\n");
	    exit(1);
	}
/*	printf("labgetipsk: begin_label : %d\n",begin_label);		*/
	if((end_label = strtol(newptr,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetipsk: keine labelgrenzen lesbar\n");
	    exit(1);
	}
/*	printf("labgetipsk: end_label : %d\n",end_label);			*/

/* labelgrenzen ueberpruefen						*/
	if(sample <= end_label && sample >= begin_label)
	{
/* 	labels einlesen							*/
	    count = 0;
	    while((ipsk[count] = (int)strtol(newptr,&newptr,10)) != 0)
	    {
/*	        printf("labgetipsk: label %d : %d\n",count+1,ipsk[count]);*/
	        count++;
	        if(count == MAXLAB)
	        {
		    fprintf(stderr,"labgetipsk: %d labels in einer zeile\n",
			MAXLAB);
		    break;
	        }
 	    }
	    fclose(fplab);
	    return(count);
	} /* ende if-schleife						*/

    } /* ende while schleife ueber alle zeilen				*/

/* sample lag ausserhalb aller labelungen				*/
    fprintf(stderr,"labgetipsk: sample %d hat kein label\n",sample);
    fclose(fplab);
    return('.');

} /* ende subroutine labgetipsk						*/


/*------------------------------------------------------------------------

Name            : labgetsilbos 
Modul           : label
Titel           : berechnung von SILBOS-label aus sample und satz

Beschreibung:
liefert SILBOS-Label zu bestimmten zeitpunkt (sample) in einem bestimmten 
sotschek-satz (auf platte)

Parameter:
satz		: name der labeldatei
sample		: sample-nummer

Return-Value	: SILBOS-Label
*/
char *labgetsilbos(char *satz,long sample)
{
	int	i,
		count,
		labels[MAXLAB];	/* labels einer zeile			*/

	long	end_label,
		begin_label;

	char	buff[80],	/* buffer fuer labelzeile		*/
		*newptr,
		*silbos;

	FILE	*fplab;

/* initialisierung							*/
    if(sample <= 0)
    {
	fprintf(stderr,"labgetsilbos: sample ist negativ oder 0 : %d\n",sample);
	exit(1);
    }

/* label-datei oeffnen							*/
    if(!(fplab = fopen(satz,"r")))
    { 
    	fprintf(stderr,"labgetsilbos: %s nicht vorhanden\n",satz);
	perror("labgetsilbos");
	exit(1);
    }

/* zeilenweises einlesen der labeldatei					*/
    while((fgets(buff,80,fplab) != NULL))	
    {
/*      kommentarzeilen ueberspringen					*/
	if(*buff == '*') continue;

/*	printf("\nlabgetsilbos: %s",buff);				*/

/* labelgrenzen einlesen						*/
	if((begin_label = strtol(buff,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetsilbos: keine labelgrenzen lesbar\n");
	    continue;
	}
/*	printf("labgetsilbos: begin_label : %d\n",begin_label);		*/
	if((end_label = strtol(newptr,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetsilbos: keine labelgrenzen lesbar\n");
	    continue;
	}
/*	printf("labgetsilbos: end_label : %d\n",end_label);		*/

/* labelgrenzen ueberpruefen						*/
	if(sample <= end_label && sample >= begin_label)
	{
/* 	labels einlesen							*/
	    count = 0;
	    while((labels[count] = (int)strtol(newptr,&newptr,10)) != 0)
	    {
/*	        printf("labgetsilbos: label %d : %d\n",count+1,labels[count]);*/
	        count++;
	        if(count == MAXLAB)
	        {
		    fprintf(stderr,"labgetsilbos: %d labels in einer zeile\n",
			MAXLAB);
		    break;
	        }
 	    }

/*	ipsk-labels auf silbos-labels umsetzen				*/
	    silbos = labmapipsktosilbos(count,labels);
/*	    printf("labgetsilbos: gefundenes silbos-label : %s\n",silbos); */
	    fclose(fplab);
	    return(silbos);
	} /* ende if-schleife						*/

    } /* ende while schleife ueber alle zeilen				*/

/* sample lag ausserhalb aller labelungen				*/
    fprintf(stderr,"labgetsilbos: sample %d hat kein label\n",sample);
    fclose(fplab);
    silbos = (char *)calloc(3,sizeof(char));
    strcpy(silbos,".");
    return(silbos);

} /* ende subroutine labgetsilbos						*/



/*------------------------------------------------------------------------

Name            : labgetsamples 
Modul           : label
Titel           : berechnung aller grenzen (samples) zu bestimmten IPSK-label

Beschreibung:
liefert array von sample-grenzen, innerhalb derer in einem bestimmten sotschek-
satz ein bestimmtes, einzelnes IPSK-label verzeichnet ist.
der speicherbereich des ergebnis-arrays samples kann mit cfree wieder 
freigegeben werden.

Parameter:
satz		: name der labeldatei
ipsk		: IPSK-label
samples		: pointer auf pointer auf 2-dim. array mit 
		  labelgrenzen in samples

Return-Value	: anzahl der gefundenen samples

*/
int labgetsamples(char *satz,int ipsk,long **samples)
{
	int	i,anz,
		count,
		labels[MAXLAB];	/* labels einer zeile			*/

	long	end_label,
		begin_label,
		*hsamples;

	char	buff[80],	/* buffer fuer labelzeile		*/
		*newptr;

	FILE	*fplab;

/* initialisierung							*/
    anz = 0;
    *samples = (long *)calloc(LABANZ,2*sizeof(long));
    hsamples = *samples;

/* label-datei oeffnen							*/
    if(!(fplab = fopen(satz,"r")))
    { 
    	fprintf(stderr,"labgetsamples: %s nicht vorhanden\n",satz);
	perror("labgetsamples");
	exit(1);
    }

/* zeilenweises einlesen der labeldatei					*/
    while((fgets(buff,80,fplab) != NULL))	
    {
/*      kommentarzeilen ueberspringen					*/
	if(*buff == '*') continue;

if(iolog(0)) printf("\nlabgetsamples: %s",buff);

/* labelgrenzen in einlesen						*/
	if((begin_label = strtol(buff,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetsamples: keine labelgrenzen lesbar\n");
	    continue;
	}

if(iolog(0)) printf("labgetsamples: begin_label : %d\n",begin_label);

	if((end_label = strtol(newptr,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetsamples: keine labelgrenzen lesbar\n");
	    continue;
	}

if(iolog(0)) printf("labgetsamples: end_label : %d\n",end_label);


/* labels einlesen							*/
	count = 0;
	while((labels[count] = (int)strtol(newptr,&newptr,10)) != 0)
	{

if(iolog(0)) printf("labgetsamples: label %d : %d\n",count+1,labels[count]);

	    count++;
	    if(count == MAXLAB)
	    {
		fprintf(stderr,"labgetsamples: %d labels in einer zeile\n",
			MAXLAB);
		break;
	    }
	}

/* labels ueberpruefen							*/
	for(i=0; i<count; i++)	
	    if(labels[i] == ipsk)
	    {
		anz++;
		if(anz > LABANZ)
		{
		    fprintf(stderr,"labgetsamples: mehr als %d ident. labels in %s\n",LABANZ,satz);
		    fclose(fplab);
		    return(anz-1);
		}
		*(hsamples++) = begin_label;
		*(hsamples++) = end_label;
	    }

	
    } /* ende while schleife ueber alle zeilen				*/

    fclose(fplab);
    return(anz);
} /* ende subroutine labgetsamples					*/


/*------------------------------------------------------------------------

Name            : labmapipsktosilbos 
Modul           : label
Titel           : mappen von beliebigen ipsk-gruppen auf silbos-label


Beschreibung:
versucht eine eindeutige zuordnung einer beliebig langen ipsk-label-gruppe 
auf ein eindeutiges silbos-label. diese subroutine verwendet im source-
code implementierte mapping-listen in den routinen labipsktosilbos und 
labipskgrouptosilbos. ist n gleich 0 wird ein dummy-label "." 
zurueckgegeben. ist die ipsk-label-gruppe nicht auf silbos-inventar 
abbildbar, wird ebenfalls ein dummy-label zurueckgegeben. der
speicherbereich auf den der return-wert zeigt, kann mit cfree() auf diesen
pointer wieder freigegeben werden. 

Parameter:
n		: anzahl der ipsklabels
ipsk		: array von ipsk-labeln

Return-Value	: silbos-label

*/
char *labmapipsktosilbos(int n,int ipsk[MAXLAB])
{
	int	i,
		phonem_count,	/* zaehler fuer phonem-labels		*/
		phonem[MAXLAB];	/* feld mit phonem-labeln		*/

	char	*silbos,	/* gefundenes silbos-label		*/
		dummy[3] = "."; /* dummy-silbos-label im fehlerfall	*/

/* initialisierung							*/
    phonem_count = 0;

/* schleife ueber ipsk-label-gruppe					*/
    for(i=0; i<n; i++)
    {

/* FALLUNTERSCHEIDUNGEN							*/

/* ipsk-label ist diacriticum						*/
	if(labipskdiacrit(ipsk[i]) == 1)
	{
/*	einzelnes diacriticum erzeugt fehlermeldung und silbos-dummy	*/
	    if(n == 1)
	    {
if(iolog(0)) printf("labmapipsktosilbos: einzelnes Diacriticum als Label: %d\n",ipsk[0]);
		silbos = (char *)calloc(3,sizeof(char));
		if(silbos == NULL)
		{
		    fprintf(stderr,"labmapipsktosilbos: nicht genug speicher fuer silbos\n");
		    perror("labmapipsktosilbos");
		    exit(1);
		}
		strcpy(silbos,dummy);
		return(silbos);
	    }
/*	nicht einzelnes diacriticum wird unterdrueckt			*/
	    continue;
	}

/* ipsk-label ist fuer technische spracherkennung unbrauchbar		*/
	if(labipskdummy(ipsk[i]) == 1)
	{
/*	einzelnes zeichen erzeugt silbos-dummy '.'			*/
	    if(n == 1)
	    {
if(iolog(0)) printf("labmapipsktosilbos: nicht verwendbares IPA-Zeichen als einziges Label: %d\n",ipsk[0]);
		silbos = (char *)calloc(3,sizeof(char));
		if(silbos == NULL)
		{
		    fprintf(stderr,"labmapipsktosilbos: nicht genug speicher fuer silbos\n");
		    perror("labmapipsktosilbos");
		    exit(1);
		}
		strcpy(silbos,dummy);
		return(silbos);
	    }
/*	nicht einzelnes zeichen wird unterdrueckt			*/
	    continue;
	}

/* ipsk-label ist verbotene oktalzahl					*/
	if(labipskerror(ipsk[i]) == 1)
	{
/*	einzelnes zeichen erzeugt fehlermeldung und silbos-dummy '.'	*/
	    if(n == 1)
	    {
if(iolog(0)) printf("labmapipsktosilbos: verbotene Oktalzahl als einzelnes Label: %d\n",ipsk[0]);
		silbos = (char *)calloc(3,sizeof(char));
		if(silbos == NULL)
		{
		    fprintf(stderr,"labmapipsktosilbos: nicht genug speicher fuer silbos\n");
		    perror("labmapipsktosilbos");
		    exit(1);
		}
		strcpy(silbos,dummy);
		return(silbos);
	    }
/*	nicht einzelnes zeichen wird gemeldet und unterdrueckt		*/
if(iolog(0)) printf("labmapipsktosilbos: verbotene Oktalzahl als Label: %d\n",ipsk[i]);
	    continue;
	}

/* ipsk-label ist echtes label (oder unbekannte verbotene oktalzahl)	*/
	phonem[phonem_count] = ipsk[i];
	phonem_count++;

    } /* ende schleife ueber alle ipsk-label				*/

    if(phonem_count == 0)
/* fehler: kein echtes phonem-label in der ipsk-gruppe			*/
    {
if(iolog(0))
{
printf("labmapipsktosilbos: kein phonem-label in ipsk-gruppe:\n");
for(i=0; i<n; i++) printf("   %d",ipsk[i]);
printf("\n");
}
	silbos = (char *)calloc(3,sizeof(char));
	if(silbos == NULL)
	{
	    fprintf(stderr,"labmapipsktosilbos: nicht genug speicher fuer silbos\n");
	    perror("labmapipsktosilbos");
	    exit(1);
	}
	strcpy(silbos,dummy);
    }
 
    else if(phonem_count == 1)
/* genau ein phonem-label aufgetreten -> eindeutig abbildbar		*/
	silbos = labipsktosilbos(phonem[0]);

    else
/* mehr als ein phonem-label aufgetreten 				*/
	silbos = labipskgrouptosilbos(phonem_count,phonem);


    return(silbos);
}


/*------------------------------------------------------------------------

Name            : labipskdiacrit		
Modul           : label
Titel           : 1, wenn label ipsk-diacriticum

Beschreibung:
liefert 1, wenn uebergebenes ipsk-label ein diacriticum ist, sonst 0.

Parameter:
label		: zu untersuchendes ipsk-label

Return-Value	: 0 : kein diacriticum, 1 : diacriticum.

*/
int labipskdiacrit(int ipsk)
{
	int	i,
		n = 16,		/* anzahl bekannter diacritica		*/
		diacrit[16] = 	/* array mit bekannten diacritica	*/
	{	7,
		10,
		27,
		34,
		46,
		52,
		53,
		67,
		70,
		71,
		74,
		75,
		76,
		134,
		173,
		175,
	};

/* normale diacritica							*/
	for(i=0; i<n; i++)
	    if(diacrit[i] == ipsk) return(1);
	
/* sonst								*/
	return(0);
} /* ende subroutine labipskdiacrit					*/

/*------------------------------------------------------------------------

Name            : labipskerror		
Modul           : label
Titel           : 1, wenn label verbotenes ipsk-label

Beschreibung:
liefert 1, wenn uebergebenes ipsk-label ein verbotenes ipsk-label ist, sonst 0.

Parameter:
label		: zu untersuchendes ipsk-label

Return-Value	: 0 : nicht verboten, 1 : verboten

*/
int labipskerror(int ipsk)
{
	int	i,
		n = 57,		/* anzahl verbotener label		*/
		error[57] = 	/* array mit verbotenen labeln		*/
	{	3,
		8,	
		9,
		13,
		15,
		18,
		19,
		26,
		28,
		29,
		31,
		33,
		38,
		39,
		40,
		42,
		45,
		47,
		48,
		49,
		50,
		54,
		55,
		56,
		57,
		58,
		59,
		65,
		66,
		68,
		69,
		72,
		73,
		78,
		79,
		108,
		109,
		118,
		119,
		128,
		129,
		133,
		135,
		136,
		137,
		138,
		139,
		140,
		148,
		149,
		158,
		159,
		168,
		169,
		174,
		178,
		179
	};

	if(ipsk > 79 && ipsk < 100) return(1);
	if(ipsk > 175) return(1);
	for(i=0; i<n; i++)
	    if(error[i] == ipsk) return(1);
	
/* sonst								*/
	return(0);
} /* labipskerror							*/

/*------------------------------------------------------------------------

Name            : labipskdummy
Modul           : label
Titel           : 1, wenn label nutzloses ipsk-label

Beschreibung:
liefert 1, wenn uebergebenes ipsk-label ein ipsk-label ist, das nicht fuer 
die technische spracherkennung verwendbar ist, sonst 0.

Parameter:
label		: zu untersuchendes ipsk-label

Return-Value	: 0 : nicht nutzlos, 1 : nutzlos

*/
int labipskdummy(int ipsk)
{
	int	i,
		n = 6,		/* anzahl verbotener label		*/
		dummy[6] = 	/* array mit verbotenen labeln		*/
	{	2,6,22,36,43,113
	};

	for(i=0; i<n; i++)
	    if(dummy[i] == ipsk) return(1);
	
/* sonst								*/
	return(0);
} /* labipskdummy							*/

/*------------------------------------------------------------------------

Name            : labipsktosilbos
Modul           : label
Titel           : mappen 1 ipsk-phonem-label auf silbos-label

Beschreibung:
liefert zu gegebenem ipsk-phonem-label das entsprechende silbos-label
ACHTUNG: neue SILBOS labelung auf der Basis von SAMPA. der speicherbereich, 
auf den der return-wert zeigt, kann mit cfree() weider freigegeben werden.

Parameter:
ipsk		: zu mappendes ipsk-phonem-label

Return-Value	: silbos-label

*/
char *labipsktosilbos(int ipsk)
{
	int	i,
		n = 78;			/* anzahl bekannter phonem-label*/
	int	phon_map[78][2] =	/* mapping-tabelle ipsk -> silbos*/
    {   {1,'.'},	/* streut zwischen 'O' und 'a:'			*/
	{4,'d'},
	{5,'E'},
	{11,'I'},
	{12,'R'},
	{14,'l'},
	{16,'N'},
	{17,'@'},
	{20,'@'},
	{21,'g'},
	{23,'s'},
	{24,'t'},
	{25,'.'},
	{30,'R'},
	{32,'z'},
	{35,'n'},
	{37,'.'},
	{41,'U'},
	{44,'.'},
	{51,'9'},
	{60,'9'},
	{61,'O'},
	{62,'E'},
	{63,'.'},
	{64,'c'},
	{77,'?'},
	{100,'6'},	/* 100 und 101 sind kaum zu unterscheiden,	*/
	{101,'6'},      /* streuen stark				*/
	{102,'v'},
	{103,'C'},
	{104,'.'},
	{105,'E'},
	{106,'.'},
	{107,'R'},
	{110,'h'},
	{111,'I'},
	{112,'j'},
	{114,'.'},
	{115,'.'},
	{116,'.'},
	{117,'O'},
	{120,'.'},
	{121,'.'},
	{122,'R'},
	{123,'S'},
	{124,'d'},
	{125,'U'},
	{126,'v'},
	{127,'.'},
	{130,'x'},
	{131,'Y'},
	{132,'Z'},
	{141,'a:'},
	{142,'b'},
	{143,'.'},
	{144,'d'},
	{145,'e:'},
	{146,'f'},
	{147,'g'},
	{150,'h'},
	{151,'i:'},
	{152,'j'},
	{153,'k'},
	{154,'l'},
	{155,'m'},
	{156,'n'},
	{157,'o:'},
	{160,'p'},
	{162,'r'},
	{161,'k'},
	{163,'s'},
	{164,'t'},
	{165,'u:'},
	{166,'v'},
	{167,'v'},
	{170,'x'},
	{171,'y:'},
	{172,'z'}	};

	char	*silbos;

	silbos = (char *)calloc(3,sizeof(char));
	if(silbos == NULL)
	{
	    fprintf(stderr,"labipsktosilbos: nicht genug speicher fuer silbos\n");
	    perror("labipsktosilbos");
	    exit(1);
	}
	for(i=0; i<n; i++)
	    if(ipsk == phon_map[i][0])
	    {
		sprintf(silbos,"%.2s",&(phon_map[i][1]));

/*
printf("labipsktosilbos: ipsk-label   : %d\n",phon_map[i][0]);
printf("labipsktosilbos: silbos-label : %s\n",silbos);	
*/

		return(silbos);
	    }

/* unbekanntes label (nicht in mapping-tabelle)				*/
if(iolog(0)) printf("labipsktosilbos: kein phonem-label : %d\n",ipsk);
	strcpy(silbos,".");	
	return(silbos);

} /* ende subroutine labipsktosilbos					*/


/*------------------------------------------------------------------------

Name            : labipskgrouptosilbos
Modul           : label
Titel           : mappen mehrerer ipsk-phonem-label auf silbos-label

Beschreibung:
liefert zu einer gruppe von ipsk-phonem-labels (mehr als 1) ein silbos-
label zurueck. der speicherbereich, auf den der return-wert zeigt, kann 
mit cfree() weider freigegeben werden.


Parameter:
n		: anzahl von labeln in gruppe
phonem		: gruppe von ipsk-phonem-label

Return-Value	: silbos-label

*/
char *labipskgrouptosilbos(int n,int phonem[MAXLAB])
{
	int 	i,
		duon = 17,
		trion = 1;

	int	duo[17][3] =
{	
	{101,111,'aI'},
	{101,145,'aI'},
	{101,157,'aU'},
	{117,111,'OY'},
	{117,145,'OY'},
	{141,20,'aI'},
	{141,105,'aI'},
	{141,111,'aI'},
	{141,117,'aU'},
	{141,125,'aU'},
	{141,145,'aI'},
	{141,157,'aU'},
	{144,163,'ts'},
	{157,111,'OY'},
	{160,146,'pf'},
	{164,123,'tS'},
	{164,163,'ts'}
			};

	int	trio[1][4] =
{	{117,117,145,'OY'}
			};

	char	*silbos;

	silbos = (char *)calloc(3,sizeof(char));
	if(silbos == NULL)
	{
	    fprintf(stderr,"labipskgrouptosilbos: nicht genug speicher fuer silbos\n");
	    perror("labipskgrouptosilbos");
	    exit(1);
	}
    if(n == 2)
    {
	for(i=0; i<duon; i++)
	{
	    if(duo[i][0] == phonem[0] && duo[i][1] == phonem[1])
	    {
		sprintf(silbos,"%.2s",&(duo[i][2]));

/*
printf("labipskgrouptosilbos: silbos-label : %s\n",silbos);	
*/

		return(silbos);
	    }
	}
    }

    if(n == 3)
    {
	for(i=0; i<trion; i++)
	{
	    if(trio[i][0] == phonem[0] && trio[i][1] == phonem[1] && 
			trio[i][2] == phonem[2])
	    {
		sprintf(silbos,"%.2s",&(trio[i][3]));

/*
printf("labipskgrouptosilbos: silbos-label : %s\n",silbos);	
*/

		return(silbos);
	    }
	}
    }

/* sonst								*/
if(iolog(0)) 
{
printf("labipskgrouptosilbos: unbekannte gruppe von phonemen gefunden:\n");
for(i=0; i<n; i++)
	    printf("   %d",phonem[i]);
printf("\n");
}
    strcpy(silbos,".");

    return(silbos);
}


/*------------------------------------------------------------------------

Name            : labtransipsktosilbos
Modul           : label
Titel           : uebersetzt ipsk-label-datei in silbos-label-struktur

Beschreibung:
liefert zu einer ipsk-label-datei den pointer auf eine struktur Label 
zurueck, welche die entsprechende silbos-labelung enthaelt. der 
speicherbereich wird in dieser und nachgeordneten routinen angefordert; um 
den speicherbereich freizugeben muss die routine labdelstlabel aufgerufen 
werden.

Parameter:
satz		: name der ipsk-labeldatei

Return-Value	: pointer auf struktur Label mit silbos-labelung

*/
Label *labtransipsktosilbos(char *satz)
{
	int 	i,anz,		/* anzahl labels			*/
		count,		/* anzahl ipsk-labels in einer zeile	*/
		ipsk[MAXLAB];	/* ipsk-label in einer zeile		*/

	char	*buff,		/* buffer fuer labelzeile		*/
		*newptr,
		*silblab;	/* aktuelles silbos-label		*/

	FILE	*fplab;


	Label	*silbos;

/* initialisierung							*/
    silbos = (Label *)calloc(1,sizeof(Label));
    if(silbos == NULL)
    {
	fprintf(stderr,"labtransipsktosilbos: nicht genug speicher fuer struktur silbos\n");
	perror("labtransipsktosilbos");
	exit(1);
    }
    buff = (char *)calloc(80,sizeof(char));
    if(buff == NULL)
    {
	fprintf(stderr,"labtransipsktosilbos: nicht genug speicher fuer buff\n");
	perror("labtransipsktosilbos");
	exit(1);
    }
    anz = 0;

/* label-datei oeffnen							*/
    if(!(fplab = fopen(satz,"r")))
    { 
    	fprintf(stderr,"labtransipsktosilbos: %s nicht vorhanden\n",satz);
	perror("labtransipsktosilbos");
	exit(1);
    }

/* zeilenweises einlesen der labeldatei					*/
    while((fgets(buff,80,fplab) != NULL))	
    {
/*      kommentarzeilen ueberspringen					*/
	if(*buff == '*') continue;

	if(anz > LABANZ-1)
	{
	    fprintf(stderr,"labtransipsktosilbos: mehr als %d (LABANZ) labels\n",
		LABANZ);
	    exit(1);
	}

/*	printf("\nlabtransipsktosilbos: %s",buff);				*/

/* newline unterdruecken						*/
	buff[strlen(buff)-1] = '\0';

/* labelgrenzen einlesen						*/
	if((silbos->beginn[anz] = strtol(buff,&newptr,10)) == 0)
	{
	    fprintf(stderr,"labtransipsktosilbos: keine labelgrenzen lesbar\n");
	    silbos->end[anz] = 0;
	}
	if((silbos->end[anz] = strtol(newptr,&newptr,10)) == 0)
	{
	    fprintf(stderr,"labtransipsktosilbos: keine labelgrenzen lesbar\n");
	    silbos->beginn[anz] = 0;
	}
/*
printf("labtransipsktosilbos: begin_label : %d\n",silbos->beginn[anz]);
printf("labtransipsktosilbos: end_label : %d\n",silbos->end[anz]);
*/

/* 	ipsk-labels einlesen						*/
	count = 0;
	while((ipsk[count] = (int)strtol(newptr,&newptr,10)) != 0)
	{
/*	        printf("labtransipsktosilbos: label %d : %d\n",count+1,
			ipsk[count]);					*/
	        count++;
	        if(count == MAXLAB)
	        {
		    fprintf(stderr,"labtransipsktosilbos: %d labels in einer zeile\n",
			MAXLAB);
		    exit(1);
	        }
 	}

/* ipsk-labels auf silbos-label mappen					*/
	silblab = labmapipsktosilbos(count,ipsk);

/* silbos-label und ipsk-label in struktur einordnen			*/
	silbos->silbos[anz] = silblab;
	silbos->ipsk[anz] = buff;

	anz++;
	buff = (char *)calloc(80,sizeof(char));
	    if(buff == NULL)
	    {
		fprintf(stderr,"labtransipsktosilbos: nicht genug speicher fuer buff\n");
		perror("labtransipsktosilbos");
		exit(1);
	    }

    } /* ende while schleife ueber alle zeilen				*/

    fclose(fplab);

/* anzahl der gefundenen label in struktur eintragen			*/
    silbos->anz = anz;

    return(silbos);

} /* ende subroutine labtransipsktosilbos					*/


/*------------------------------------------------------------------------

Name            : labdelstlabel
Modul           : label
Titel           : gibt speicher-bereich von silbos-label-struktur frei

Beschreibung:
gibt den gesamten speicherbereich einer silbos-label-struktur (Label), die 
von labtransipsktosilbos erstellt wurde frei und liefert NULL zurueck.

Parameter:
stlab		: adresse eines pointers auf struktur Label

Return-Value	: void

*/
void labdelstlabel(Label **stlab)
{
	int	i;

/* inhalte der struktur freigeben					*/
	for(i=0; i<(*stlab)->anz; i++)
	{
		cfree((*stlab)->silbos[i]);
		cfree((*stlab)->ipsk[i]);
	}
/* struktur selber freigeben und pointer mit NULL belegen		*/
	cfree(*stlab);
	*stlab = NULL;

} /* ende subroutine labdelstlabel					*/

/*------------------------------------------------------------------------

Name            : labgetword
Modul           : label
Titel           : liefert labelgrenzen eines wortes aus best. satz

Beschreibung:
liefert zu einer ipsk-label-datei und einem bestimmten wort die 
labelgrenzen (samples). es koennen max. MAXPHON gleiche woerter pro satz 
gefunden werden.
der speicherbereich des ergebnis-arrays samples kann mit cfree wieder 
freigegeben werden.

Parameter:
satz		: name der ipsk-labeldatei
word		: gesuchtes wort
samples		: pointer auf pointer auf 2-dim. array mit wortgrenzen

Return-Value	: anzahl der gefundenen woerter

*/
int labgetword(char *satz,char *word,long **samples)
{
	int	i,anz;

	long	end_label,
		begin_label,
		*hsamples;
	char	buff[80],	/* buffer fuer labelzeile		*/
		readword[80],	/* wort aus gelesener zeile		*/
		wort[80],	/* auf grossbuchstaben konvertiert	*/
		*newptr;

	FILE	*fplab;

/* initialisierung							*/
    anz = 0;
    *samples = (long *)calloc(MAXPHON,2*sizeof(long));
    if(*samples == NULL)
    {
	fprintf(stderr,"labgetword: nicht genug speicher fuer samples\n");
	perror("labgetword");
	exit(1);
    }
    hsamples = *samples;

/* gesuchtes wort auf grossbuchstaben konvertieren			*/
    strcpy(wort,word);
    for(i=0; wort[i] != '\0'; i++)
	wort[i] = toupper(wort[i]);

if(iolog(0)) printf("labgetword: gesuchtes wort : %s\n",wort);

/* label-datei oeffnen							*/
    if(!(fplab = fopen(satz,"r")))
    { 
    	fprintf(stderr,"labgetword: %s nicht vorhanden\n",satz);
	perror("labgetword");
	cfree(*samples);
	*samples = NULL;
	return(0);
    }

/* zeilenweises einlesen der labeldatei					*/
    while((fgets(buff,80,fplab) != NULL))	
    {
/*      andere zeilen ueberspringen					*/
	if(strncmp(buff,"* *1",4) != 0) continue;
	newptr = buff + 4;
/*	printf("\nlabgetword: %s",newptr);				*/

/* 	labelgrenzen einlesen						*/
	if((begin_label = strtol(newptr,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetword: keine labelgrenzen in %s lesbar\n",satz);
	    continue;
	}

if(iolog(0)) printf("labgetword: begin_label : %d\n",begin_label);

	if((end_label = strtol(newptr,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetword: keine labelgrenzen in %s lesbar\n",satz);
	    continue;
	}

if(iolog(0)) printf("labgetword: end_label : %d\n",end_label);
	
/* wort einlesen und vergleichen					*/
	sscanf(newptr,"%s",readword);
if(iolog(0)) printf("labgetword: gelesenenes wort : %s\n",readword);
	if(strcmp(wort,readword) == 0)
	{
		hsamples[anz] = begin_label;
		hsamples[anz+1] = end_label;
		anz += 2;
	}
	
    } /* ende while schleife						*/

    fclose(fplab);

    return(anz/2);

} /* ende subroutine labgetword						*/


/*------------------------------------------------------------------------

Name            : labgetsatz
Modul           : label
Titel           : liefert labelgrenzen eines best. satzes

Beschreibung:
liefert zu einer ipsk-label-datei die labelgrenzen (samples) des ganzen 
satzes. 

Parameter:
satz		: name der ipsk-labeldatei
begin		: pointer auf satzbegin (samples)
end		: pointer auf satzende (samples)

Return-Value	: void
*/
void labgetsatz(char *satz,int *begin,int *end)
{
	int	labflag;

	long	begin_label,end_label;

	char	buff[80],	/* buffer fuer labelzeile		*/
		*newptr;

	FILE	*fplab;

/* initialisierung							*/
    labflag = 0;
    *begin = 99999;
    *end = 0;

/* label-datei oeffnen							*/
    if(!(fplab = fopen(satz,"r")))
    { 
    	fprintf(stderr,"labgetsatz: %s nicht vorhanden\n",satz);
	perror("labgetsatz");
	return;
    }

/* zeilenweises einlesen der labeldatei					*/
    while((fgets(buff,80,fplab) != NULL))	
    {
/*      andere zeilen ueberspringen					*/
	if(strncmp(buff,"* *1",4) != 0) continue;
	newptr = buff + 4;
/*	printf("\nlabgetsatz: %s",newptr);				*/

/* 	labelgrenzen einlesen						*/
	if((begin_label = strtol(newptr,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetsatz: keine labelgrenzen lesbar (begin)\n");
	    exit(1);
	}

/* 	minimum fuer satzbeginn suchen 					*/
	if(begin_label < *begin) *begin = begin_label;


	if((end_label = strtol(newptr,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetsatz: keine labelgrenzen lesbar (end)\n");
	    return;
	}

/* 	maximum fuer satzende suchen 					*/
	if(end_label > *end) *end = end_label;

    } /* ende while schleife						*/

/*    printf("labgetsatz: begin_label : %d\n",*begin);			*/
/*    printf("labgetsatz: end_label : %d\n",*end);			*/

    fclose(fplab);

} /* ende subroutine labgetsatz						*/


/*------------------------------------------------------------------------

Name            : labgetphonem
Modul           : label
Titel           : liefert labelgrenzen eines phonems aus best. satz

Beschreibung:
liefert zu einer ipsk-label-datei und einem bestimmten phonem (SAMPA) die 
labelgrenzen (samples). es koennen max. MAXPHON gleiche phoneme pro satz 
gefunden werden. Die Liste gueltiger Phoneme entspricht denen der Mapping-
Tabellen in den Funktionen labipsktosilbos und labipskgrouptosilbos in 
diesem Modul.
der speicherbereich des ergebnis-arrays samples kann mit cfree wieder 
freigegeben werden.

Parameter:
satz		: name der ipsk-labeldatei
phonem		: gesuchtes phonem (string mit SAMPA-phonem)
samples		: pointer auf pointer auf 2-dim. array mit phonemgrenzen

Return-Value	: anzahl der gefundenen woerter

*/
int labgetphonem(char *satz,char *phonem,long **samples)
{
	int	i,anz,count,
		labels[MAXLAB];	/* labels einer zeile			*/

	long	end_label,
		begin_label,
		*hsamples;
	char	buff[80],	/* buffer fuer labelzeile		*/
		*newptr,
		*silbos; 	/* pointer auf gelesenes silbos-label	*/

	FILE	*fplab;

/* initialisierung							*/
    anz = 0;

if(iolog(0)) printf("labgetphonem: gesuchtes phonem : %s\n",phonem);

/* label-datei oeffnen							*/
    if(!(fplab = fopen(satz,"r")))
    { 
    	fprintf(stderr,"labgetphonem: %s nicht vorhanden\n",satz);
	perror("labgetphonem");
	*samples = NULL;
	return(0);
    }
/* speicherplatz fuer ergebnis-array					*/
    *samples = (long *)calloc(MAXPHON,2*sizeof(long));
    if(*samples == NULL)
    {
	fprintf(stderr,"labgetphonem: nicht genug speicher fuer samples\n");
	perror("labgetphonem");
	exit(1);
    }
    hsamples = *samples;

/* zeilenweises einlesen der labeldatei					*/
    while((fgets(buff,80,fplab) != NULL))	
    {
/*      andere zeilen ueberspringen					*/
	if(strncmp(buff,"*",1) == 0) continue;
	if(strncmp(buff,"* *1",4) == 0) continue;

if(iolog(0)) printf("\nlabgetphonem: %s",buff);

/* 	labelgrenzen einlesen						*/
	if((begin_label = strtol(buff,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetphonem: keine labelgrenzen in %s lesbar\n",satz);
	    fprintf(stderr,"labgetphonem: %s \n",buff);
	    continue;
	}

if(iolog(0)) printf("labgetphonem: begin_label : %d\n",begin_label);

	if((end_label = strtol(newptr,&newptr,10)) == 0) 
	{	
	    fprintf(stderr,"labgetphonem: keine labelgrenzen in %s lesbar\n",satz);
	    continue;
	}

if(iolog(0)) printf("labgetphonem: end_label : %d\n",end_label);
	
/* phoneme einlesen und nach gefordertem phonem suchen			*/

/* labels einlesen							*/
	count = 0;
	while((labels[count] = (int)strtol(newptr,&newptr,10)) != 0)
	{

if(iolog(0)) printf("labgetphonem: label %d : %d\n",count+1,labels[count]);

	    count++;
	    if(count == MAXLAB)
	    {
		fprintf(stderr,"labgetphonem: %d labels in einer zeile\n",
			MAXLAB);
		break;
	    }
	}

/* eingelesenes array von ipsk-labeln auf SAMPA-Label umwandeln		*/
        silbos = labmapipsktosilbos(count,labels);

if(iolog(0)) printf("labgetphonem: silbos-label : %s\n",silbos);

/* wenn gelesenes label mit gesuchtem label uebereinstimmt -> segment 
   aufnehmen in sample-array						*/
	if(strcmp(phonem,silbos) == 0)
	{
		hsamples[anz] = begin_label;
		hsamples[anz+1] = end_label;
		anz += 2;
	}

/* speicherbereich fuer silbos-label freigeben				*/
	cfree(silbos);
	
    } /* ende while schleife ueber zeilen der labeldatei		*/

    fclose(fplab);

    return(anz/2);

} /* ende subroutine labgetphonem					*/

