
/**************************************************************************
***************************************************************************

Modul           : cluster_uti.c
Directory       : vb$dua1:[sch.lib]
Titel           : Sammlung von Subroutinen fuer die Clusteranalyse nach
		  split-k-means

Autor           : F. Schiel
Datum/Aenderung : 09.11.90 / 17.06.92

Beschreibung:
enthaelt spezifische routinen zur clusteranalyse cluster. 

Anzulinkende Module (ausser SCHCLIB/LIB):

Subroutinen:
readparamclusmain	: einlesen parameter fuer split-k-means
clusterini		: erzeugung eines codebuchs mit 2 prototypen
clusterencode		: vektorquantisierung
clusterupdate		: neuberechnung centroiden und lochbehandlung
interset		: berechnung des interset-abstandes eines codebuchs
clustersplit		: splitten der prototypen eines codebuchs
readcodebook		: einlesen eines codebuchs
writecodebook		: abspeichern eines codebuchs
writeasciicb		: Codebuch in ASCII schreiben (Format ASL, Plannerer)
distorsion		: messung der distorsion von daten zu codebuch
partition		: berechnung der verteilung von daten auf symbole
calkovar		: berechnung der kovarianz-matrizen eines cbs
kovarianzmat		: berechnung einer kovarianz-matrix aus cluster
delstcb			: codebuch loeschen

*/
# include <ctype.h>
# include <string.h>
# include <math.h>
# include <stdio.h>
# include "ipkclib.h"

/* DEFINES, die nur innerhalb dieses Moduls Verwendung finden ***********/
# define PARTS 1024		/* alloc-stufen fuer pointer-array in 
				struktur Stclus fuer daten-partition	*/
# define DETNORM 100		/* normierungsfaktor fuer datenvektoren
					zur berechnung der determinate	*/


/*----------------------------------------------------------------------

Name            : readparamclusmain
Modul           : cluster_uti.c
Titel           : einlesen parameter fuer split-k-means

Beschreibung:
liest parameterwerte aus datei fdef ein, die jeweils nach einem 
'='-zeichen in einer zeile stehen.
 
Parameter:
fdef		: filedeskriptor der def.datei  		

Return-Value	: void
*/
void readparamclusmain(FILE *fdef,float *splitpar,
			int *itermax,int *itermin,float *epsilon)
{ 
	char  buf[100];
          
          sscanf(getwert(fdef,buf),"%f",splitpar);
		printf("readparamclusmain: splitparameter: \t%f\n",
		 *splitpar);	
          sscanf(getwert(fdef,buf),"%d",itermax);
		printf("readparamclusmain: max. iteration: \t%d\n",
		 *itermax);	
          sscanf(getwert(fdef,buf),"%d",itermin);
		printf("readparamclusmain: min. iteration: \t%d\n",
		 *itermin);	
          sscanf(getwert(fdef,buf),"%f",epsilon);
		printf("readparamclusmain: epsilon: \t%f\n",
		 *epsilon);	
        }
 

/*----------------------------------------------------------------------

Name            : clusterini
Modul           : cluster_uti.c
Titel           : initialisierung codebuch

Beschreibung:
erzeugt aus der gesamtdatenmenge ein codebuch mit zwei prototypen.
aus allen datenvektoren wird der arithmetische mittelwertsvektor berechnet. 
dieser wird laengs der verbindungsgeraden zum ursprung um splitpar in beide 
richtungen verschoben -> 2 prototypen. ergebnis liegt in den ersten beiden 
prototypen des codebuchs cb.
 
Parameter:
material	: p. auf array von p. auf datenvektoren material[anz]
cb		: pointer auf codebuch cb[n][dim]
dim		: dimension der vektoren
anz		: anzahl der datenvektoren
splitpar	: splitting-faktor

Return-Value	: void
*/
void clusterini(float **material,float *cb,int dim,int anz,float splitpar)
{ 

	int	i,j,
		n;
	float   splitpar1,
		splitpar2,
		*cbh,
		*matp;

	splitpar1=1-splitpar;
	splitpar2=(1+splitpar)/splitpar1;

/* mittelwertsbildung ueber alle datenvektoren				*/
	for(i=0; i<anz; i++)
	{		  
		cbh = cb;
		matp = *(material++);
		for(j=0; j<dim; j++) *(cbh++) += *(matp++);
	}

/* splitten des gesamtmittelpunkts laengs der ursprungsgeraden		*/
	for(j=0; j<dim; j++)		    /* splitting    */
	{
		*cb /= anz;
		*cb *= splitpar1;
		*(cb+dim) = splitpar2 * *cb;
		cb++;
	}
} /* ende subroutine clusterini						*/


/*----------------------------------------------------------------------

Name            : clusterencode
Modul           : cluster_uti.c
Titel           : vektorquantisierung

Beschreibung:
quantisiert das datenmaterial auf das uebergebene codebuch, bildet ueber alle 
datenvektoren, die auf ein cb-symbol fallen die summe (cbh) und liefert die 
anzahl pro cb-symbol zurueck.

Parameter:
material	: p. auf array von p. auf datenvektoren material[anz]
cb		: pointer auf codebuch cb[n][dim]
mintra		: p. auf mittleren intraset-abstand ueber alle datenvektoren
varintra	: p. auf varianz des intrasetabstands aller datenvektoren
zaehler		: pointer auf array mit anzahl der quantisierten datenvektoren
		  pro cb-symbol
cbh		: summen der zugeordneten datenvektoren
anz		: anz von datenvektoren
n		: anzahl cb-prototypen
dim		: dimension der vektoren
dist		: pointer auf funktion, die abstand zweier vektoren berechnet

Return-Value	: void
*/
void clusterencode(float **material,float *cb,float *mintra,float *varintra,
		int *zaehler,float *cbh,int anz,int n,int dim,
		double (*dist)(int,float *,float *,matrix *))
{ 
	int	i,j,l,
		index; 		/* nr. des aktuell naechsten cb-symbols */

	double 	dmin,		/* aktueller mindestabstand		*/
		d;		/* aktueller abstand			*/

	float	*cbm;		/* hilfpointer fuer cb			*/

	matrix	*dummym;	/* dummypointer, ev. fuer kovarianz	*/

/* initialisierung							*/
	*mintra = 0.;
	*varintra = 0.;
	reset_float2(cbh,n,dim);
	reset_int2(zaehler,1,n);

/* vektorzuordnung auf naechsten cb-prototypen				*/
	for(l=0; l<anz; l++)
	{
		dmin = 1.5E23;
		cbm = cb;
		for(i=0; i<n; i++)
		{
			if((d = (*dist)(dim,cbm,*material,dummym)) < dmin)
			{
				dmin = d;
				index = i;
			}
			cbm += dim;
		}

/* zaehler des zugeordneten cb-symbols erhoehen und datenvektor aufsummieren */
		(*(zaehler + index))++;
		float2_add_float2(cbh+index*dim,*material,cbh+index*dim,1,dim);

/* auf naechsten datenvektor weiterschalten				*/
		material++;

/* minimalen abstand und quadrat aufsummieren				*/
		*mintra += dmin;
		*varintra += dmin*dmin;

	} /* ende for-schleife ueber l					*/

/* aufsummierte minimale distanzen durch anzahl der datenvektoren
   teilen, ergibt mittleren intrasetabstand				*/
	*mintra /= anz;

/* varianz  =  E(x^2)  -  (E(x))^2					*/
	*varintra = (*varintra / anz)  -  (*mintra * *mintra);

} /* ende subroutine clusterencode					*/


/*----------------------------------------------------------------------

Name            : clusterupdate
Modul           : cluster_uti.c
Titel           : neuberechnung centroiden und lochbehandlung

Beschreibung:
aus den summierten zuordnungen (cbh) werden neue centroiden berechnet und 
in cb abgelegt. hat fuer einen prototypen keine zuordnung stattgefunden, so 
wird er ersetzt durch eine splittung desjenigen prototypen mit den meisten 
zuordnungen (lochbehandlung).

Parameter:
zaehler		: pointer auf array mit anzahl der zuordnungen zaehler[n]
cb		: p. auf p. auf codebuch mit neuen prototypen cb[n][dim]
cbh		: p. auf p. auf array mit summierten zuordnungen cbh[n][dim]
dim		: dimension der vektoren
n		: anzahl der prototypen im aktuellen codebuch
splitpar	: splitting-faktor

Return-Value	: int  1 = loch aufgetreten  0 = kein loch aufgetreten
*/
int clusterupdate(int *zaehler,float **cb,float **cbh,
		int dim,int n,float splitpar)
{ 
	int	i,j,index,maxi,merker,
		*zaehlerp,*zaehlerpp;	/* hilfspointer auf zaehler array */
	float	split1,split2,		/* splitfaktoren lochbehandlung */
		*cbhp;			/* hilfspointer auf codebuecher	*/

/* initialisierung							*/
    split1 = 1 - splitpar;
    split2 = 1 + splitpar;
    merker = 0;

/* centroiden berechnen und auf loecher untersuchen			*/
    zaehlerp = zaehler;
    for(i=0; i<n; i++)
    {
	if(*zaehlerp == 0) merker = 1;
	else
	    float2_div_float(*cbh+i*dim,*zaehlerp,*cbh+i*dim,1,dim);
	zaehlerp++;
    }

/* wenn loecher auftreten, groesste gruppe von vektoren suchen, splitten und 
ergebnisse auf orginal und loch verteilen				*/
    if(merker == 1)
    {
	zaehlerp = zaehler;
	for(i=0; i<n; i++)
/* loecher aufsuchen							*/
	{
	    if(*zaehlerp == 0)
	    {
		printf("clusterupdate: loch wird ersetzt\n");
		maxi = 0;
		cbhp = *cbh;
		zaehlerpp = zaehler;
		for(j=0; j<n; j++)
/* groesste gruppe suchen (index)					*/
		{
		    if(*zaehlerpp > maxi)
		    {
			maxi = *zaehlerpp;
			index = j;
		    }
		}
/* symbol mit groesster gruppe (index) splitten				*/
		for(j=0; j<dim; j++)
		{
		    *(cbhp+i*dim+j) = split1 * *(cbhp+index*dim+j);
		    *(cbhp+index*dim+j) *= split2;
		}
/* zaehler fuer die neuen gesplitteten symbole setzten			*/
		*zaehlerp = *(zaehler+index) / 2;
		*(zaehler+index) /= 2;
	    }
	    zaehlerp++;
	}
    }

/* codebuch und hilfscodebuch vertauschen, neues codebuch dann in cb	*/
    cbhp = *cbh;
    *cbh = *cb;
    *cb = cbhp;

    return(merker);
} /* ende subroutine clusterupdate					*/


/*----------------------------------------------------------------------

Name            : interset
Modul           : cluster_uti.c
Titel           : berechnung des interset-abstands eines codebuchs

Beschreibung:
der mittlere abstand (dist) jedes prototypen zu jedem anderen prototypen wird 
berechnet. ausserdem die varianz dieser groesse.

Parameter:
cb		: pointer auf codebuch cb[anz][dim]
anz		: anzahl der codebuch-symbole
dim		: dimension der cb-vektoren
dist		: pointer auf abstandsberechnung zweier vektoren
varinter	: p. auf varianz des intersetabstands

Return-Value	: float interset-abstand
*/
float interset(float *cb,int anz,int dim,
		double (*dist)(int,float *,float *,matrix *),float *varinter)
{
	int       i,j,count;
	double    d,abstand;

	float	  *cbi,*cbj;

	matrix	  *dummym;

 	count = 0;
	*varinter = 0.;
	abstand = 0.;
	cbi = cb;
	for(i=0; i<anz; i++)
	{
		cbj = cb + (i+1)*dim;
		for(j=i+1; j<anz; j++)
	        {
		    d = dist(dim,cbi,cbj,dummym);
		    abstand += d;
		    *varinter += d*d;
		    count++;
		    cbj += dim;
		}
		cbi += dim;
	}
abstand = abstand/count;
*varinter = (*varinter/count) - (abstand * abstand);

return((float)abstand);
} /* ende subroutine interset						*/     



/*-----------------------------------------------------------------------

Name            : clustersplit
Modul           : cluster_uti.c
Titel           : splitten der prototypen eines codebuchs

Beschreibung:
splittet jeden prototypen im codebuch cb in zwei prototypen laengs der 
verbindungsgeraden zum ursprung. die 'neuen' prototypen werden in cb 
angehaengt, die alten ueberschrieben.

Parameter:
cb		: pointer auf codebuch cb[n][dim]
n		: pointer auf anzahl der codebuch-symbole vor/nach splitting
dim		: dimension der cb-vektoren
splitpar	: splitting-faktor

Return-Value	: void
*/
void clustersplit(float *cb,int dim,int *n,float splitpar)
{
	int	i,j;

	float	split1,split2,	/* splitfaktoren			*/
		*cbh;		/* pointer auf noch leeren bereich des cbs */

/* initialisierung							*/
    split1 = 1 - splitpar;
    split2 = (1 + splitpar) / split1;
    cbh = cb + *n * dim;

/* splitting, verdoppeln der anzahl der prototypen			*/
    for(i=0; i<*n; i++)
	for(j=0; j<dim; j++)
	{
	    *cb *= split1;
	    *(cbh++) = split2 * *(cb++);
 	}
    *n += *n;

} /* ende subroutine clustersplit					*/


/*----------------------------------------------------------------------

Name            : readcodebook
Modul           : cluster_uti.c
Titel           : einlesen eines codebuchs

Beschreibung:
liest codebuch im standardformat ein. Wenn erweitertes codebuch mit 
kovarianzmatrizen, werden diese auch eingelesen; sonst werden die pointer 
auf das determinanten-array det und das matrizenpointer-array invkov zu 
NULL gesetzt. die routine legt speicherplatz fuer die gesamte struktur an, 
dieser speicherplatz kann mit hilfe der routine delstcb (modul cluster_uti) 
freigegeben werden.

Parameter:
cbfnam		: filename des codebuchs

Return-Value	: pointer auf eingelesenes codebuch

*/
Stcb *readcodebook(char *cbfnam)
{
	int	i;
	FILE  *fcb;
	Stcb	*cb;

    cb = (Stcb *)calloc(1,sizeof(Stcb));
    if(cb == NULL)
    {
	fprintf(stderr,"readcodebook: nicht genug speicher fuer cb\n");
	perror("readcodebook");
	exit(1);
    }	

    if (!(fcb = fopen(cbfnam,"r")))
    {
        fprintf(stderr,"readcodebook: codebook %s nicht vorhanden\n",cbfnam);
        exit(1);
    }
    fread(&cb->n,sizeof(int),1,fcb);
    fread(&cb->dim,sizeof(int),1,fcb);
    cb->cb = (float *)calloc(cb->n * cb->dim,sizeof(float));
    if(cb->cb == NULL)
    {
	fprintf(stderr,"readcodebook: nicht genug speicher fuer cb.cb\n");
	perror("readcodebook");
	exit(1);
    }	
    fread(cb->abstand,sizeof(char),20,fcb);
    fread(cb->codetyp,sizeof(char),20,fcb);
    fread(&cb->intraset,sizeof(float),1,fcb);
    fread(&cb->interset,sizeof(float),1,fcb);
    fread(&cb->varintra,sizeof(float),1,fcb);
    fread(&cb->varinter,sizeof(float),1,fcb);
    fread(cb->cb,cb->dim * sizeof(float),cb->n,fcb);
/* speicher fuer determinantenvektor anlegen				*/
    cb->det = (float *)calloc(cb->n,sizeof(float));
    if(cb->det == NULL)
    {
	fprintf(stderr,"readcodebook: nicht genug speicher fuer cb.det\n");
	perror("readcodebook");
	exit(1);
    }	
/* deteminatenvektor versuchsweise einlesen				*/
    if(fread(cb->det,sizeof(float),cb->n,fcb) == 0)
    {
/*	misserfolg -> kein erweitertes codebuch, speicher wieder freigeben*/
	if(iolog(0)) printf("readcodebook: normales codebuch %s (keine kovarianzen)\n",
		cbfnam);
	cfree(cb->det);
	cb->det = NULL;
	cb->invkov = NULL;
	cb->invdia = NULL;
    }
    else
    {
/*	erfolg -> auch inverse kovarianzen einlesen			*/
	cb->invkov = (matrix **)calloc(cb->n,sizeof(matrix *));
	if(cb->invkov == NULL)
	{
	    fprintf(stderr,"readcodebook: nicht genug speicher fuer cb.invkov\n");
	    perror("readcodebook");
	    exit(1);
	}	
	for(i=0; i<cb->n; i++)
	    cb->invkov[i] = readmat(fcb);

	fread(&cb->detnorm,sizeof(long),1,fcb);
        if(iolog(0)) printf("readcodebook: erweitertes codebuch %s (kovarianzen)\n",
		cbfnam);

    
/* speicherbereich fuer pointerarray auf inv. hauptdiagonal-elemente 
   reservieren								*/
        cb->invdia = (float **)calloc(cb->n,sizeof(float *));    
        if(cb->invdia == NULL)
        {
	    fprintf(stderr,"kluster: nicht genug speicher fuer cb.invdia\n");
	    perror("kluster");
	    exit(1);
        }	

/* speicher fuer inv. haupt-diagonal-elemente anlegen			*/
        for(i=0; i<cb->n; i++)
        {
	    cb->invdia[i] = (float *)calloc(cb->dim,sizeof(float));
	    if(cb->invdia[i] == NULL)
	    {
	        fprintf(stderr,"kluster: kein speicher fuer inv. hauptdiagonal-elemente\n");
	        perror("kluster");
	        exit(1);
	    }
        }

/* erste diagonale versuchsweise einlesen				*/
        if(fread(cb->invdia[0],sizeof(float),cb->dim,fcb) == 0)
        {
/*	    misserfolg -> keine diagonalern vorhanden, speicher freigeben*/
 	    for(i=0; i<cb->n; i++) cfree(cb->invdia[i]);
	    cfree(cb->invdia);
	    cb->invdia = NULL;
        }
        else
        {
/*	    erfolg -> restlichen diagonalen auch einlesen		*/
	    for(i=1; i<cb->n; i++)
	        fread(cb->invdia[i],sizeof(float),cb->dim,fcb);

	    if(iolog(0)) printf("readcodebook: + diagonal-matrix\n",
		cbfnam);
        }	
    }

    fclose(fcb);

    return(cb);
}

/*----------------------------------------------------------------------

Name            : writecodebook
Modul           : cluster_uti.c
Titel           : abspeichern eines codebuchs

Beschreibung:
speichert codebuch im standardformat ab. wenn erweitertes codebuch 
(kovarianzen) werden die determinanten und kovarianzen mit abgespeichert. 
erweitertes codebuch dann, wenn pointer auf determinatenvektor und pointer-
array auf kovarianzmatrizen nicht NULL ist. ist datei mit angegebenem nanen
cbfnam nicht anlegbar, wird das codebuch unter writecodebook.cod im default 
directory angelegt.

Parameter:
cbfnam		: filename des codebuchs
cb		: pointer auf codebuch

Return-Value	: void

*/
void writecodebook(char *cbfnam,Stcb *cb)
{
	int	i;
	FILE  *fcb;

    if (!(fcb = fopen(cbfnam,"w")))
    {
        fprintf(stderr,"writecodebook: datei %s nicht anlegbar\n",cbfnam);
        perror("writecodebook");
        fprintf(stderr,"writecodebook: speichere unter WRITECODEBOOK.COD\n");
	if((fcb = fopen("writecodebook.cod","w")) == NULL)
	{
	    fprintf(stderr,"writecodebook: keine datei im aktuellen directory anlegbar\n");
	    perror("writecodebook");
	    exit(1);
	}
    }
    fwrite(&cb->n,sizeof(int),1,fcb);
    fwrite(&cb->dim,sizeof(int),1,fcb);
    fwrite(cb->abstand,sizeof(char),20,fcb);
    fwrite(cb->codetyp,sizeof(char),20,fcb);
    fwrite(&cb->intraset,sizeof(float),1,fcb);
    fwrite(&cb->interset,sizeof(float),1,fcb);
    fwrite(&cb->varintra,sizeof(float),1,fcb);
    fwrite(&cb->varinter,sizeof(float),1,fcb);
    fwrite(cb->cb,cb->dim * sizeof(float),cb->n,fcb);
    if((cb->det != NULL) && (cb->invkov != NULL))
    {
	fwrite(cb->det,sizeof(float),cb->n,fcb);
	for(i=0; i<cb->n; i++)
	    writemat(fcb,cb->invkov[i]);
	fwrite(&cb->detnorm,sizeof(long),1,fcb);
    }
    if(cb->invdia != NULL)
    {
	for(i=0; i<cb->n; i++)
	    fwrite(cb->invdia[i],sizeof(float),cb->dim,fcb);
    }
    fclose(fcb);
}

/*----------------------------------------------------------------------

Name		: writeasciicb
Modul		: cluster_uti
Titel		: Codebuch in ASCII schreiben (Format ASL, Plannerer)

Beschreibung:
Schreibt ein Codebuch (Struktur Stcb) in ASCII-Format in Datei. Das Format
entspricht dem offiziellen ASL-Format nach B. Plannerer.
Achtung: schreibt nur Codebuecher vom Typ 30 (keine Kovarianzmatrizen)

Parameter:
asccicbname	: Name der Datei mit ASCII-Darstellung (Output)
bincb		: pointer auf zu schreibendes Codebuch

Return-Value	: 0, Fehler: -1
----------------------------------------------------------------------*/
int writeasciicb(char *asciicbname,Stcb *bincb)
{
	char inp[3];
	int i,k;
	double log_det;
	FILE *fpasciicb;		/* ascii codebuch		*/

/* Dateiname des ascii-codebuchs und datei oeffnen			*/

if(iolog(0)) printf("writeasciicb: ASCII Codebuch = %s\n",asciicbname);

    if((fpasciicb = fopen(asciicbname,"w")) == NULL)
    {
	fprintf(stderr,"writeasciicb: cannot open ascii-file %s\n",asciicbname);
	perror("writeasciicb");
	return(-1);
    }

/* titel schreiben							*/
    fprintf(fpasciicb,"TYPE: 30 CBE: %d  DIMENSION: %d\n",
	bincb->n,bincb->dim);

/* prototypen schreiben							*/
    fprintf(fpasciicb,"CENTROIDS:\n");
    for(i=0;i<bincb->n;i++)
    {
 	fprintf(fpasciicb,"CENT %d:\n",i);
	for(k=0;k<bincb->dim;k++)
	    fprintf(fpasciicb,"%e\n",bincb->cb[i*bincb->dim+k]);
    }

/* logarithmierte, inverse determinanten schreiben			
   da nur codebuecher vom typ 30 geschrieben werden, wird die 
   logarithmierte, inverse determinante 
   nicht aus dem codebuch berechnet (echte det.) sondern aus den inversen
   elementen der diagonalen kovarianzmatrix neu berechnet		*/
    fprintf(fpasciicb,"LOG_IDET:\n");
    do {
		printf("Log. Determinanten alle Null (Y/N) := ");
		if(scanf("%s",inp) == EOF)
			exit(0);
	} while((strcmp(inp,"Y") != 0) && (strcmp(inp,"N") != 0));

    if(strcmp(inp,"Y") == 0)
	for(i=0;i<bincb->n;i++)
	    fprintf(fpasciicb,"LOG_IDET %d: %e\n",i,0.0);
    else
	for(i=0;i<bincb->n;i++)
	{
	    log_det = 0.;
	    for(k=0;k<bincb->dim;k++)
		log_det += log(bincb->invdia[i][k]);

	    fprintf(fpasciicb,"LOG_IDET %d: %e\n",i,log_det);
	}

/* inverse der hauptdiagonalen schreiben (typ 30: keine kovarianzen)	*/
    fprintf(fpasciicb,"IVAR:\n");
    for(i=0;i<bincb->n;i++)
    {
        fprintf(fpasciicb,"IVAR %d:\n",i);
	for(k=0;k<bincb->dim;k++)
	    fprintf(fpasciicb,"%e\n",bincb->invdia[i][k]);
    }

    fclose(fpasciicb);

    return(0);
}

/*-----------------------------------------------------------------------

Name            : distorsion
Modul           : cluster_uti.c
Titel           : messung der distorsion von daten zu codebuch

Beschreibung:
quantisiert alle datenvektoren in material auf codebuch cb mit abstandsfunktion 
distanz. misst den mittleren abstand aller datenvektoren zu ihren zugeordneten 
cb-symbolen und gibt mittelwert zurueck

Parameter:
material	: p. auf p. auf datenvektoren
anz		: anz der datenvektoren
cb		: pointer auf codebuch
distanz		: p. auf funktion zur abstandsberechnung

Return-Value	: mittlerer abstand (distorsion)

*/
float distorsion(float **material,int anz,Stcb *cb,
	double (*distanz)(int,float *,float *,matrix *))
{
	int	i,
		index;		/* codebuch-index			*/

	double  disto,		/* distorsion				*/
		abstand;	/* abstand datenvektor zu prototyp	*/
	float	*matp;		/* hilfspointer auf material		*/

/* initialisierung							*/
    disto = 0.;
    
/* schleife ueber alle daten vektoren					*/
    for(i=0; i<anz; i++)
    {
	matp = *(material++);

/* quantisierung							*/
	index = quant1(matp,cb,&abstand,distanz);	
	disto += abstand;
    }
    printf("\n");
    disto /= anz;
    return(disto);
} /* ende subroutine distorsion						*/

/*----------------------------------------------------------------------

Name            : partition
Modul           : cluster_uti.c
Titel           : berechnung der verteilung von daten auf symbole

Beschreibung:
quantisiert alle datenvektoren in daten auf das codebuch cb und traegt die 
pointer auf die datenvektoren in die listen der struktur clus ein. bei 
jedem eintrag wird ausserden der zaehler fuer jedes cb-symbol in clus.anz 
erhoeht. als ergebnis stehen in der struktur clus fuer jedes cb-symbol i
unter clus.anz[i] die anzahl der auf i geclusterten datenvektoren und in 
clus.part[i] ein pointer auf ein array von pointer der laenge PARTS, 
die wiederum auf die datenvektoren innerhalb von daten zeigen, welche auf i 
geclustert wurden. uebersteigt die anzahl der auf ein symbol quantisierten 
datenvektoren den Wert PARTS, so wird der zaehler fuer das betreffende 
symbol auf einen bruchteil von PARTS zurueckgesetzt, der dem bruchteil der 
bereits verarbeiteten datenvektoren entspricht. dadurch soll einen 
moeglichst gleichmaessige verteilung der datenvektoren eines clusters 
innerhalb der auf PARTS begrenzten partition erreicht werden.

Parameter:
cb		: pointer auf cb-struktur
daten		: pointer auf array von anz pointern auf datenvektoren
anz		: anzahl der pointer in daten
distanz		: p. auf funktion zur abstandsberechnung
clus		: pointer auf struktur mit cluster-verteilung

Return-Value	: void

*/
void partition(Stcb *cb,float **daten,int anz,	
	double (*distanz)(int,float *,float *,matrix *),Stclus *clus)
{
	int	i,j,l,
		index; 		/* nr. des aktuell naechsten cb-symbols */

	float	dmin,		/* aktueller mindestabstand		*/
		d,		/* aktueller abstand			*/
		*cbm;		/* hilfpointer fuer cb			*/
	matrix	*dummym;	/* dummypointer, ev. fuer kovarianz	*/

/* initialisierung							*/
    clus->n = cb->n;
/*  zaehlervektor erzeugen: pro cb-symbol ein element			*/
    clus->anz = (int *)calloc(clus->n,sizeof(int));
    if(clus->anz == NULL)
    {
	fprintf(stderr,"kluster: kein speicherplatz fuer clus.anz\n");
	perror("kluster");
	exit(1);
    }
/* pointervektor erzeugen: enthaelt pro cb-symbol einen pointer. dieser 
   pointer zeigt wiederum auf einen vektor von pointern (laenge PARTS) 
   diese pointer zeigen auf die datenvektoren im speicherbereich von 
   daten								*/
    clus->part = (float ***)calloc(clus->n,sizeof(float **));
    if(clus->part == NULL)
    {
	fprintf(stderr,"kluster: kein speicherplatz fuer clus.part\n");
	perror("kluster");
	exit(1);
    }
/* fuer pointer-array pro symbol platz fuer PARTS pointer schaffen	*/
    for(i=0; i < clus->n; i++)
    {
	clus->part[i] = (float **)calloc(PARTS,sizeof(float *));
	if(clus->part[i] == NULL)
	{
	    fprintf(stderr,"kluster: kein speicherplatz fuer clus.part[%d]\n",i);
	    perror("kluster");
	    exit(1);
	}
    }
    
/* schleife ueber alle datenvektoren					*/
	for(l=0; l<anz; l++)
	{
/* 		vektorzuordnung auf naechsten cb-prototypen		*/
		dmin = 1.5E23;
		cbm = cb->cb;
		for(i=0; i < cb->n; i++)
		{
			if((d = (*distanz)(cb->dim,cbm,*daten,dummym)) < dmin)
			{
				dmin = d;
				index = i;
			}
			cbm += cb->dim;
		}

/* pointer auf datenvektor in i-ten array unter clus.part eintragen	*/
		if(clus->anz[index] > PARTS-1)
		{
/* wenn zahl der datenvektoren in einem cluster groesser PARTS, zaehler 
   zuruecksetzen auf bruchteil der verarbeiteten datenvektoren		*/
		    clus->anz[index] = (int)((PARTS-1) * (float)l/anz);
/*		    fprintf(stderr,"partition: zaehler fuer symbol %d zurueckgesetzt auf %d\n",
			index,clus->anz[index]);			*/
		    clus->part[index][clus->anz[index]] = *daten;
		    clus->anz[index]++;
		}
		else
		{
		    clus->part[index][clus->anz[index]] = *daten;

/* zaehler des zugeordneten cb-symbols erhoehen 			*/
/*		    printf("partition: clus->anz[%d] : %d\n",
			index,clus->anz[index]);			*/
		    clus->anz[index]++;
		}

/* auf naechsten datenvektor weiterschalten				*/
		daten++;

	} /* ende for-schleife ueber l					*/

} /* ende subroutine partition						*/

/*----------------------------------------------------------------------

Name            : calkovar
Modul           : cluster_uti.c
Titel           : berechnung der kovarianz-matrix eines clusters

Beschreibung:
ruft fuer jedes cb-symbol die berechnung der kovarianz-matrix aus den 
entsprechenden partitionen in der struktur clus auf. die berechneten 
kovarianz-matrizen stehen als ergebnis in der struktur cb (cb->invkov[] 
ist array von pointern auf matrizen).

Parameter:
clus		: pointer auf struktur mit cluster-verteilung
cb		: pointer auf cb-struktur

Return-Value	: void

*/
void calkovar(Stclus *clus,Stcb *cb)
{
	int	i,j;
	
	float	*mitt;		/* pointer auf aktuellen prototypen	*/

/* schleife ueber alle symbole						*/
    for(i=0; i<cb->n; i++)
    {
/* alten prototyp auswaehlen						*/
	mitt = cb->cb + i * cb->dim;

/* kovarianzmatrix und neuen prototypen berechnen			*/
	kovarianzmat(clus->anz[i],cb->dim,clus->part[i],mitt,cb->invkov[i]);

/* inverse der hauptdiagonal-elemente berechnen	und in cb eintragen	*/
	for(j=0; j<cb->dim; j++)
	    cb->invdia[i][j] = 1 / (float)cb->invkov[i]->m[j][j];


    } /* ende schleife i ueber alle cb-symbole				*/

/*
printf("calkovar: neuer prototyp im symbol 0:\n");
disp_float2(cb->cb,1,cb->dim);
printf("calkovar: inverse diagonal-elemente im symbol 0:\n");
disp_float2(cb->invdia[0],1,cb->dim);
printf("calkovar: kovarianz-matrix von symbol 0:");
prmat(cb->invkov[0]);
*/

} /* ende subroutine calkovar						*/

/*------------------------------------------------------------------------

Name            : kovarianzmat
Modul           : cluster_uti.c
Titel           : berechnung einer kovarianz-matrix aus cluster

Beschreibung:
berechnung der kovarianzmatrix zu einer sammlung von vektoren, die einem 
cluster angehoeren. der prototyp (mittelwertsvektor) wird aus der uebergebenen 
partition (part) neu berechnet und in mitt eingetragen.

Parameter:
anz		: anzahl vektoren im cluster
dim		: dimension der datenvektoren
part		: pointer-array mit pointern auf datenvektoren (float)
mitt		: pointer auf prototypen
kov		: pointer auf ergebnis-matrix

Return-Value	: void

*/
void kovarianzmat(int anz,int dim,float **part,float *mitt,matrix *kov)
{
	int	i,j,k;

	matrix	*diadm;		/* diadisches produkt des prototypen	*/

/* ueberpruefung der vektorzahl im cluster				*/
    if(anz < 2)
	fprintf(stderr,"kovarianzmat: zuwenig daten im cluster %d\n",anz);

/* initialisierung							*/
    diadm = genmat((long)dim,(long)dim);

/*
printf("alter prototyp:\n");
disp_float2(mitt,1,dim);
*/

/* prototypen (mittelwert) aus partition neu berechnen			*/
    for(j=0; j<dim; j++) mitt[j] = 0.;
    for(i=0; i<anz; i++)
	for(j=0; j<dim; j++)
	    mitt[j] += part[i][j];
    for(j=0; j<dim; j++) mitt[j] /= (float)anz;

/*
printf("neuer prototyp:\n");
disp_float2(mitt,1,dim);
*/

/* diadisches produkt des prototypen berechnen und ergebnis-matrix 	
   kov	null setzen							*/
	for(i=0; i<dim; i++)
	    for(j=0; j<dim; j++)
	    {
		diadm->m[i][j] = mitt[i] * mitt[j];
		kov->m[i][j] = 0.;
	    }

/*
printf("diadischer prototyp:\n");
prmat(diadm);
*/

/* schleife ueber alle vektoren des clusters				*/
	for(k=0; k<anz; k++)
/*          kovarianzen ohne mittelwert bestimmen			*/
	    for(i=0; i<dim; i++)
	    	for(j=0; j<dim; j++)
		    kov->m[i][j] += part[k][i] * part[k][j];

/* normieren auf vektorzahl und diadisches produkt aus prototypen abziehen */
	for(i=0; i<dim; i++)
	    for(j=0; j<dim; j++)
		kov->m[i][j] /= anz;

/*
printf("kovarianz ohne mittelwert:\n");
prmat(kov);
*/

	for(i=0; i<dim; i++)
	    for(j=0; j<dim; j++)
		kov->m[i][j] -= diadm->m[i][j];

/*
printf("kovarianz mit mittelwert:");
prmat(kov);
*/

 	delmat(&diadm);

} /* ende subroutine kovarianzmat					*/


/*------------------------------------------------------------------------

Name            : delstcb
Modul           : cluster_uti.c
Titel           : codebuch loeschen

Beschreibung:
gibt speicherbereich eines codebuchs (struktur Stcb) vollstaendig frei und 
setz den cb-pointer auf NULL

Parameter:
cbptr		: adresse des codebuch-pointers

Return-Value	: void

*/
void	delstcb(Stcb **cbptr)
{
	int	i;

	cfree((*cbptr)->cb);
	cfree((*cbptr)->det);
	for(i=0; i<(*cbptr)->n; i++)
	{
		delmat(&((*cbptr)->invkov[i]));
		cfree((*cbptr)->invdia[i]);
	}
	cfree((*cbptr)->invkov);
	cfree((*cbptr)->invdia);
	cfree(*cbptr);
	*cbptr = NULL;
} /* ende subroutine delstcb						*/
