/***********************************************************************
************************************************************************

Module          : linuxaudio.c 
Title           : Audioausgabe unter Linux

Author          : M. Lu"cke
Date/revision   : 16.06.95/23.06.95

Description     : 
Mit diesem Modul koennen unter Linux Audiodaten in folgenden Formaten
abgespielt werden:
U-Law 8 Bit, A-Law 8 Bit und Linear 16 Bit.

Modifying       : Klaus Jaensch, 07 Dec 99
                  -DSP buffer size greater equal zero allowed
                   (before: >= 4k) 
                  -Opening of DSP device with command 'fopen' 
                   for correct functionality on 'libc6' based systems
                  -ioctl sequence is now same as 'sox'
                  


Link modules and libraries:

Contained functions:
linux_audio: gibt Audiodaten ueber die Soundkarte aus 
ulaw2linear : konvertiert ein Byte von ulaw nach linear (16 Bit)
alaw2linear : konvertiert ein Byte von alaw nach linear (16 Bit)

*************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <fcntl.h>
#include <limits.h>
#include <ctype.h>
#include <sys/ioctl.h>

#include "ipkclib.h"
#include <sys/soundcard.h>

/* DEFINES, only used within this module ********************************/


/* TYPE DEFINITIONS, only used within this module  **********************/


/* GLOBAL VARIABLES (avoid, if possible !) ******************************/


/* FUNCTION PROTOTYPES, capsuled in this module *************************/
/*----------------------------------------------------------------------
Name            : linux_audio
Module          : linuxaudio.c
Title           : Konvertiert und schickt Audiodaten an die Soundkarte

Description:
Audiodaten, die sich im Speicher befinden werden falls noetig auf das
Format "Linear 16 Bit" konvertiert und zur Audiokarte geschickt. Die
Lautstaerkeregelung erfolgt ueber die Multiplikation der (konvertierten)
Samples mit dem Parameter "volume".
Evtl. vorhandene "Header" in den Audiodaten werden nicht beachtet und
unbearbeitet an die Soundkarte geschickt.

Parameters:
buf             : Zeiger auf die Audiodaten
len             : Laenge der Audiodaten in Bytes
type		: Format der Audiodaten (alaw, ulaw, linear) s.h. linuxaudio.h
		  DSP_TYPE_LINEAR_16 1    // Uebergebene Daten sind Linear 16 Bit
		  DSP_TYPE_ALAW_8 2       // Uebergebene Daten sind ALaw 8 Bit
		  DSP_TYPE_ULAW_8 3       // Uebergebene Daten sind ULaw 8 Bit
rate		: Abtastrate in Hz
volume		: Lautstaerke(faktor) (1.0 = keine Aenderung)
 
Return-Value    : Status Rueckgabe
		  DSP_OPEN_FAILED 100     // Audio Device konnte nicht geoeffnet werden
		  DSP_UNKNOWN_TYPE 101    // Uebergebener Datentyp unbekannt
		  DSP_OUT_OF_MEM 102      // Kein Speicherplatz fuer Zwischenpuffer
		  DSP_ILLEGAL_BUFSIZE 103 // Puffer der Karte nicht zw. 4096 und 65536
		  DSP_NO_ERROR 0          // Kein Fehler
-------------------------------------------------------------------------*/

int linux_audio(short int *buf,int len, int type, int rate, float volume)
{
  int i,j,tmp;
  unsigned bits;
  unsigned channels = 1;
  unsigned char *t;
  short int *bufx = 0;
  unsigned short *audio_buf;
  int buf_size;
  FILE *f;

  if(!(f = fopen( "/dev/dsp","w" )))
    {
      fprintf(stderr, "fopen: failed\n" );
      return DSP_OPEN_FAILED;
    }
  switch(type)
    {
    case DSP_TYPE_LINEAR_16:
      bits = 16;
      break;
    case DSP_TYPE_ALAW_8:
      bufx = (short int *)malloc(len*2);
      bits = 16;
      t = (unsigned char*)buf;
      for(i=0;i<len;i++)
        {
          bufx[i] = ((short int)alaw2linear(t[i]));
        }
      len *= 2;
      buf = (short int *)bufx;
      break;
    case DSP_TYPE_ULAW_8:
      bufx = (short int *)malloc(len*2);
      bits = 16;
      t = (unsigned char*)buf;
      for(i=0;i<len;i++)
        {
          bufx[i] = ((short int)ulaw2linear(t[i]));
        }
      len *= 2;
       buf = bufx;
      break;
    default:
      fprintf(stderr,"linuxaudio: Unbekannter DSP Typ !\n");
      return DSP_UNKNOWN_TYPE;
    }

  ioctl(fileno(f), SNDCTL_DSP_RESET, 0);
  if(ioctl (fileno(f), SNDCTL_DSP_GETBLKSIZE, &buf_size)==-1) {
   fprintf(stderr,"linuxaudio: DSP gettin blockSize failed !\n");
     return DSP_OPEN_FAILED;
  }
 if (ioctl(fileno(f), SNDCTL_DSP_SYNC, NULL) < 0) {
    fprintf(stderr,"Unable to sync DSP !\n");
    return DSP_OPEN_FAILED;
    }

    tmp = bits;
    ioctl(fileno(f), SNDCTL_DSP_SAMPLESIZE, &tmp);
    if (tmp != bits) {
	fprintf(stderr,"Unable to set the sample size to %d",bits);
         return DSP_OPEN_FAILED;
       
    }
   
    tmp=0;
    if (channels==2) tmp=1;
   if(ioctl( fileno(f),SNDCTL_DSP_STEREO, &tmp )==-1) {
      fprintf(stderr,"linuxaudio: DSP channel setting failed !\n");
       return DSP_OPEN_FAILED;
    }
  tmp=rate;
  if (ioctl( fileno(f), SNDCTL_DSP_SPEED, &tmp )==-1) {
     fprintf(stderr,"linuxaudio: DSP rate setting failed !\n");
     return DSP_OPEN_FAILED;
     }

  
  if (buf_size <= 0)
    {
      if(bufx != 0)
	free(bufx);
       fprintf(stderr,"linuxaudio: Buffersize: %d\n",buf_size); 
       fprintf(stderr,"linuxaudio: Illegal buffersize !\n");
      return DSP_ILLEGAL_BUFSIZE;
    }

  if((audio_buf = (unsigned short*)malloc(buf_size)) == NULL)
    {
      if(bufx != 0)
	free(bufx);
      return DSP_OUT_OF_MEM;
    }

  /* len = laenge in bytes; buf&audio_buf vom typ short int -> len/2 ! */

  for(i=0;i<len/2;)
    {
      for(j=0;(j<buf_size/2) && (i<len/2);j++,i++)
	audio_buf[j] = (unsigned short)(buf[i] * volume); 
      /* audio_buf[j] = (unsigned short)(buf[i]); */

      if(j<buf_size)
	for(;j<buf_size/2;j++)
	  audio_buf[j] = 0;

      write(fileno(f),audio_buf,buf_size);    /* spielt einen puffer */
    }

  if(bufx != 0)
    free(bufx);

  fclose(f);
  return DSP_NO_ERROR;
} /* end subroutine :  */

/*----------------------------------------------------------------------
Name            : ulaw2linear
Module          : linuxaudio.c
Title           : Konvertiert ulaw nach linear 

Description:
Diese Funktion konvertiert das uebergebene Byte von ulaw nach linear
und liefert das Ergebnis (16 Bit) als Rueckgabewert.

Parameters:
u_val           : ulaw Byte

Return-Value    : 16 Bit Sample
-------------------------------------------------------------------------*/
/*
 * This source code is a product of Sun Microsystems, Inc. and is provided
 * for unrestricted use.  Users may copy or modify this source code without
 * charge.
 *
 * SUN SOURCE CODE IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING
 * THE WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun source code is provided with no support and without any obligation on
 * the part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY THIS SOFTWARE
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
/*
 * ulaw2linear() - Convert a u-law value to 16-bit linear PCM
 *
 * First, a biased linear code is derived from the code word. An unbiased
 * output can then be obtained by subtracting 33 from the biased code.
 *
 * Note that this function expects to be passed the complement of the
 * original code word. This is in keeping with ISDN conventions.
 */

#define SIGN_BIT        (0x80)          /* Sign bit for a A-law byte. */
#define QUANT_MASK      (0xf)           /* Quantization field mask. */
#define NSEGS           (8)             /* Number of A-law segments. */
#define SEG_SHIFT       (4)             /* Left shift for segment number. */
#define SEG_MASK        (0x70)          /* Segment field mask. */
#define BIAS            (0x84)          /* Bias for linear code. */

short int ulaw2linear(unsigned char u_val)
{
  short int t;

  /* Complement to obtain normal u-law value. */
  u_val = ~u_val;

  /*
   * Extract and bias the quantization bits. Then
   * shift up by the segment number and subtract out the bias.
   */
  t = ((u_val & QUANT_MASK) << 3) + BIAS;
  t <<= ((unsigned)u_val & SEG_MASK) >> SEG_SHIFT;

  return ((u_val & SIGN_BIT) ? (BIAS - t) : (t - BIAS));
} /* end subroutine :  */

/*----------------------------------------------------------------------
Name            : alaw2linear
Module          : linuxaudio.c
Title           : Konvertiert alaw nach linear

Description:
Diese Funktion konvertiert das uebergebene Byte von alaw nach linear
und liefert das Ergebnis (16 Bit) als Rueckgabewert.

Parameters:
a_val           : alaw Byte

Return-Value    : 16 Bit Sample
-------------------------------------------------------------------------*/
/*
 * This source code is a product of Sun Microsystems, Inc. and is provided
 * for unrestricted use.  Users may copy or modify this source code without
 * charge.
 *
 * SUN SOURCE CODE IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING
 * THE WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun source code is provided with no support and without any obligation on
 * the part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY THIS SOFTWARE
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
/*
 * alaw2linear() - Convert an A-law value to 16-bit linear PCM
 *
 */

#define SIGN_BIT        (0x80)          /* Sign bit for a A-law byte. */
#define QUANT_MASK      (0xf)           /* Quantization field mask. */
#define NSEGS           (8)             /* Number of A-law segments. */
#define SEG_SHIFT       (4)             /* Left shift for segment number. */
#define SEG_MASK        (0x70)          /* Segment field mask. */
#define BIAS            (0x84)          /* Bias for linear code. */

short int alaw2linear(unsigned char a_val)
{
  short int             t;
  short int             seg;

  a_val ^= 0x55;

  t = (a_val & QUANT_MASK) << 4;
  seg = ((unsigned)a_val & SEG_MASK) >> SEG_SHIFT;
  switch (seg)
    {
    case 0:
      t += 8;
      break;
    case 1:
      t += 0x108;
      break;
    default:
      t += 0x108;
      t <<= seg - 1;
    }
  return ((a_val & SIGN_BIT) ? t : -t);
} /* end subroutine :  */










