/***********************************************************************
************************************************************************

Module		: hear_raw-2.1
Title		: playing raw sound files to audio

Author		: F. Schiel
Datum/Aenderung : 21.02.95 / 28.08.95

Description	: 
Takes all filenames from command line that are not icsiargs
and plays them to a audio device.
Version for SUNOS and LINUX platforms (use option -DSUNOS
and -DLINUX for gcc respectively)

Usage:
hear_raw [v=yes][port=0][swap=no][rate=<rate>]
         [length=<secs>] file1 [file2 ...]

v=yes : prints infos about each file to stdout (default=no)
port=0|1|2 : 0 = speaker (default), 1 = headphone, 2 = line out
vol=0.0-1.0 : SUNOS: audio volume (default 0.8)
              LINUX: audio volume factor 
rate=16000 : sampling rate (default 16000)
swap=yes : swap the short samples (default no)
length=<seconds> : outputs only seconds from beginning
code=linear|ulaw|alaw : coding of data (default: linear)

Linking modules or libraries:
ipkclib

Contained functions:
			: 

*************************************************************************/
# include <stdio.h>
# include <ipkclib.h>

/* DEFINES, only used within this module ********************************/
# define VOLUME 0.80      /* Default Laustaerke 0.0 ... 1.0 */
# define PORT 0           /* Default Ausgabeport: 0 = speaker */
# define RATE 16000       /* Default sampling rate */
# define CODE linear      /* Default coding */


/* TYPE DEFINITIONS, only used within this module  **********************/


/* GLOBAL VARIABLES (avoid, if possible !) ******************************/


/* FUNCTION PROTOTYPES, capsuled in this module *************************/

main(int argc,const char **argv)
{
	char 	filenam[80],buff,
		*appname;
	char 	*charpoint,*code;
        char    defaultcode[7] = "linear";
	short   *data, *dataptr;
	int	i,k,l,ret;
        int 	verb = 0;
        int     port = PORT;
        int     swap = 0;
        int     bit;
	long	rate = RATE,skip = 0;
        long	samples,bytes;
	float	length = 0.;
	double	volume = VOLUME;
	FILE	*file;
	icsiargtab_t args[] =
        {
          { 0, "2.0 : playing raw sound files to audio", ARG_DESC},
          { "v","verbose mode", ARG_BOOL, &verb },
          { "swap","swap short samples", ARG_BOOL, &swap },
	  { "port","audio port (0:speaker, 1:headphone, 2:lineout)", ARG_INT, &port },
	  { "rate","sampling rate", ARG_LONG, &rate },
	  { "vol","audio volume (0.0 - 1.0)", ARG_DOUBLE, &volume },
	  { "length","seconds of played signal", ARG_FLOAT, &length },
	  { "code","coding of data", ARG_STR, &code },
          { "skip","# of chars skipped in front", ARG_LONG, &skip },
	  {0,0,0}
	};

  code = defaultcode;
  /* Test for icsiargs if any */
  icsiargs(args,&argc,&argv,&appname);
  if(verb) printargs(stdout,appname,args);
  if(argc < 1)
    {
    fprintf(stderr,"usage: %% %s [v=yes][port=0][vol=0.8][swap=no][skip=0]\n\t\t[length=<secs>][code=linear]  file1 [file2 ...]\n",appname);
    exit(-1);
    }
  if(strcmp(code,"linear"))
  {
    if(verb) fprintf(stdout,"%s: ulaw or alaw are always 8 bit - set bit to 8\n",appname); 
    bit = 8;
  }
  else
    bit = 16;
  if((rate != 8000) && strcmp(code,"linear"))
  {
    if(verb) fprintf(stdout,"%s: ulaw or alaw are always 8 kHz - set rate to 8 kHz\n",appname); 
    rate = 8000;
  }
#ifdef LINUX
  if(port != 0)
  {
    fprintf(stderr,"%s: other ports than 0 not implemented on LINUX\n",appname); 
    port = 0;
  }
#endif
  /* processing filenames from line */
  for(k=0;k<argc;k++)
  {
    strcpy(filenam,argv[k]);
    if(verb) printf("%s: File = %s\n",appname,filenam);
    if((file = fopen(filenam,"r")) == NULL)
    {
	fprintf(stderr,"%s: cannot open file %s\n",appname,filenam);
	perror(appname);
        continue;
    }
    /* check for size */
    fseek(file,0,2);
    bytes = ftell(file) - skip;
    if(bit == 16)
      samples = bytes/2;
    else 
      samples = bytes;
    rewind(file);
    if(skip) fseek(file,skip,0);
    if(length)
    {
      if(length > (float)samples/rate)
        fprintf(stderr,"%s: not enough samples in file %s, playing all\n",
                appname,filenam);
      else
      {
        samples = (int)(length*rate);
        if(bit == 16)
          bytes = 2*samples;
        else 
          bytes = samples;
      }
    }
    /* get memory for block-wise read */
    if((data = (short *)malloc(bytes)) == NULL)
    {
      fprintf(stderr,"%s: not enough memory to read data in file %s\n",
            appname,filenam);
      perror(appname);
      fclose(file);
      continue;
    }
    if(verb) printf("%s: number of samples = %ld (%.2f sec)\n",
                    appname,samples,(float)samples/(float)rate);
    /* read speech data as a block (faster) */
    if(fread(data,sizeof(char),bytes,file) != bytes)
    {                                                                    
      fprintf(stderr,"%s: could not read data from file %s\n",  
              appname,filenam);                                          
      perror(appname);                                                   
      free(data);
      fclose(file);
      continue;
    }                                                                    
    fclose(file);
    if(verb) printf("%s: read %ld bytes successfully\n",
                    appname,bytes);
    /* swap samples if necessary */
    if(swap && (bit == 16))
    {
    if(verb) printf("%s: swapping\n", appname);
      charpoint = (char *)data;        
      for(i=0;i<samples;i++)           
       {                              
         buff = charpoint[0];         
         charpoint[0] = charpoint[1]; 
         charpoint[1] = buff;         
         charpoint += 2;              
       }                              
    }
    /* play data linear to audio device */
    if(verb) printf("%s: code = %s\n",appname,code);
    if(!strcmp(code,"linear"))
    {
#ifdef SUNOS
      if(verb) printf("%s: invoking sparc_audio\n", appname);
      if(!sparc_audio(data,samples,rate,volume,port))
      {
        fprintf(stderr,"%s: cannot use sparc audio device\n",appname);
        exit(-1);
      }
#endif
#ifdef LINUX
      if(verb) printf("%s: invoking linux_audio\n", appname);
      if(linux_audio(data,2*samples,DSP_TYPE_LINEAR_16,rate,volume) 
         != DSP_NO_ERROR)
      {
        fprintf(stderr,"%s: could not send data to linux audio card (error %d)\n",
                appname,ret); 
        exit(-1);
      }
#endif 
    }
    else if(!strcmp(code,"alaw"))
    {
#ifdef SUNOS
      if(!sparc_audio_alaw((unsigned char *)data,samples,volume,port))
      {
        fprintf(stderr,"%s: cannot use sparc audio device\n",appname);
        exit(-1);
      }
#endif
#ifdef LINUX
      if(linux_audio(data,samples,DSP_TYPE_ALAW_8,rate,volume) 
         != DSP_NO_ERROR)
      {
        fprintf(stderr,"%s: could not send data to linux audio card (error %d)\n",
                appname,ret); 
        exit(-1);
      }
#endif
    }
    else if(!strcmp(code,"ulaw"))
    {
#ifdef SUNOS
      if(!sparc_audio_ulaw((unsigned char *)data,samples,volume,port))
      {
        fprintf(stderr,"%s: cannot use audio device\n",appname);
        exit(-1);
      }
#endif
#ifdef LINUX
      if(linux_audio(data,samples,DSP_TYPE_ULAW_8,rate,volume) 
         != DSP_NO_ERROR)
      {
        fprintf(stderr,"%s: could not send data to linux audio card (error %d)\n",
                appname,ret); 
        exit(-1);
      }
#endif
    }
    else
    {
      fprintf(stderr,"%s: unknown coding: %s\n",appname,code); 
      exit(-1);                                                
    }                                                            
    free(data);
  } /* ende schleife ueber argumente */
  return(0);
}
