//    Speechrecorder
//    (c) Copyright 2012-2020
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

/**
* PromptImageViewer fetches the images found in the image directory
* and provides methods for selecting them.
*
* @author Christoph Draxler
* @version 1.0
* @since JDK 1.0
*/

package ipsk.apps.speechrecorder.prompting;

import java.awt.Dimension;
import java.awt.Image;
import java.net.URL;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JLabel;

import ipsk.apps.speechrecorder.MIMETypes;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenterException;
import ipsk.apps.speechrecorder.prompting.presenter.UnsupportedContentException;
import ipsk.db.speech.script.prompt.Mediaitem;
import ipsk.util.LocalizableMessage;
import ipsk.util.services.Description;
import ipsk.util.services.Title;
import ipsk.util.services.Vendor;
import ipsk.util.services.Version;

@Title("Image icon prompter")
@Description("Presents image icons (GIF files).")
@Vendor("Institute of Phonetics and Speech processing, Munich")
@Version(major=1)
public class PromptImageIconViewer extends BasicPromptPresenter implements PromptPresenter {
    
	private static final long serialVersionUID = 1L;
	public static final PromptPresenterServiceDescriptor DESCRIPTOR=new BasicPromptPresenterServiceDescriptor(PromptImageIconViewer.class.getName(),new LocalizableMessage("Image prompter"), "Institute of Phonetics and Speech processing, Munich", new ipsk.text.Version(new int[]{1,0,0}), new LocalizableMessage("Presents image icons."),PromptImageIconViewer.getSupportedMIMETypes());
	
	private Image image;
	private ImageIcon imageIcon=new ImageIcon();
	private JLabel label=null;
	
	/**
	 * Create prompt image icon viewer especially for animated GIF files.
	 */
	public PromptImageIconViewer() {
		super();
		setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
		label=new JLabel(imageIcon);
		add(label);
	}

	public Dimension getPreferredSize() {
		return new Dimension(getSize());
	}

	public void showContents() {
	};
	
	public void hideContents() {
	};
	
	public void loadContents(URL url) {
	};

	/**
	* Selects the next image to display. Not implemented.
	*
	* @param imageName name of image file
	*/
	public void setContents(String imageName) {

	}

	/**
	* Selects the next image to display. Not implemented.
	*
	* @param imageName name of image file
	* @param description description of image
	*/
	public void setContents(String imageName, String description) {

	}
      
	/**
	 * Not implemented
	 * @param cts content
	 * @param description description
	 * @param type content type
	 */
    public void setContents(String cts, String description, String type) {
          
     }
      
	/**
	* Set contents.
	*
	* @param imageIconURL URL of image icon file
	*/
	public synchronized void setContents(URL imageIconURL) {
		loadImageIcon(imageIconURL);
	}

	/**
	* Set contents.
	*
	* @param imageURL URL of image icon file
	* @param d description of image (not used)
	*/
	public void setContents(URL imageURL, String d) {
		setContents(imageURL);
	}
	
	/**
	* Set contents.
	*
	* @param imageURL URL of image icon file
	* @param d description of image (not used)
	* @param t MIME-type of image (not used)
	*/
	public void setContents(URL imageURL, String d, String t) {
		setContents(imageURL);
	}
	
	/**
	 * Set contents.
	 * 
	* @param imageUrl URL of image icon file
	* @param description description of image (not used)
	* @param type MIME-type of image (not used)
	 * @param charset name of character set (not used)
	 */
    public void setContents(URL imageUrl, String description, String type,String charset) {
        setContents(imageUrl);    
    }
    
    public void doLayout() {
        if (image!= null) {
            
            int cw = getSize().width;
            int ch = getSize().height;
           
            int iw = image.getWidth(this);
            int ih = image.getHeight(this);
            
            int w;
            int h;
    
            float wr = (float) cw / (float) iw;
            float hr = (float) ch / (float) ih;
            
            if (wr < hr) {
                w = (int) (iw * wr);
                h = (int) (ih * wr);
            } else {
                w = (int) (iw * hr);
                h = (int) (ih * hr);
            }
            
            // compute origin of image so that it is centered horizontally and vertically
            int x = (cw - w) / 2;
            int y = (ch - h) / 2;
            
            // Animated GIFs are only scaled with Image.SCALE_DEFAULT scaling mode.
            Image scaledImage=image.getScaledInstance(w, h, Image.SCALE_DEFAULT);
            imageIcon.setImage(scaledImage);
           
            label.setBounds( x, y, w, h);
        }
    }
   
    /**
     * Load contents.
     */
    public void loadContents()
    throws PromptPresenterException {
        if(mediaitems==null || mediaitems.length==0){
            throw new UnsupportedContentException("No media item to display!");
        }else if(mediaitems.length > 1){
            throw new UnsupportedContentException("Multiple media items not supported!");
        }  
        
        Mediaitem mi=mediaitems[0];
        URL imageIconURL=applyContextToMediaitemURL(mi);
        loadImageIcon(imageIconURL);
    }
    
    private void loadImageIcon( URL imageIconURL) {
    	if (image==null){
    		ImageIcon imgIcon=new ImageIcon(imageIconURL);
    		image=imgIcon.getImage();
    	}
    }
    
	public static String[][] getSupportedMIMETypes() {
	        return getSupportedMIMETypes(MIMETypes.IMAGEICONMIMETYPES);
	}

    /* (non-Javadoc)
     * @see ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter#getServiceDescriptor()
     */
    public PromptPresenterServiceDescriptor getServiceDescriptor() {
        return DESCRIPTOR;
    }

}