//    Speechrecorder
// 	  (c) Copyright 2017
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.



package ipsk.apps.speechrecorder.script.ui.prompt.styled;

import java.awt.Graphics;
import java.awt.Shape;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JEditorPane;
import javax.swing.text.AbstractDocument.BranchElement;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.AbstractDocument;
import javax.swing.text.BoxView;
import javax.swing.text.Caret;
import javax.swing.text.ComponentView;
import javax.swing.text.Document;
import javax.swing.text.Element;
import javax.swing.text.IconView;
import javax.swing.text.JTextComponent;
import javax.swing.text.LabelView;
import javax.swing.text.MutableAttributeSet;
import javax.swing.text.ParagraphView;
import javax.swing.text.Position.Bias;
import javax.swing.text.Segment;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledEditorKit;
import javax.swing.text.View;
import javax.swing.text.ViewFactory;
import javax.swing.text.html.InlineView;

import ipsk.db.speech.script.prompt.doc.P;
import ipsk.db.speech.script.prompt.doc.PromptDoc;
import ipsk.db.speech.script.prompt.doc.Font.FontSize;
import ipsk.text.ParserException;





/**
 * @author klausj
 *
 */
public class PromptDocEditorKit extends StyledEditorKit {

	public static class EmptyView extends View{

		public EmptyView(Element elem) {
			super(elem);
			
		}

		@Override
		public float getPreferredSpan(int axis) {
			
			return 0;
		}

		@Override
		public void paint(Graphics g, Shape allocation) {
			// Intentionally do nothing
		}

		@Override
		public Shape modelToView(int pos, Shape a, Bias b) throws BadLocationException {
		
			return null;
		}

		@Override
		public int viewToModel(float x, float y, Shape a, Bias[] biasReturn) {
			return 0;
		}
		
	}
	
	public static class LineBreakView extends LabelView {

	    /**
	     * 
	     *
	     * @param elem the element to create a view for
	     */
	    public LineBreakView(Element elem) {
	        super(elem);
	    }

	    /**
	     * Forces a line break.
	     *
	     * @return View.ForcedBreakWeight
	     */
	    public int getBreakWeight(int axis, float pos, float len) {
	        if (axis == X_AXIS) {
	            return ForcedBreakWeight;
	        } else {
	            return super.getBreakWeight(axis, pos, len);
	        }
	    }
	}
	
	public static class PromptDocFactory implements ViewFactory {

		private boolean viewOnly;
		
		public PromptDocFactory() {
			this(false);
		}
		
		public PromptDocFactory(boolean viewOnly) {
			super();
			this.viewOnly=viewOnly;
		}
		
        /**
         * Creates a view from an element.
         *
         * @param elem the element
         * @return the view
         */
		public View create(Element elem) {

			//System.out.println("Class: "+elem.getClass().getName());
			String kind = elem.getName();
			if (kind != null) {
				if (kind.equals(AbstractDocument.ContentElementName)) {
					//System.out.println("LabelView");
					return new LabelView(elem);
				} else if (kind.equals(AbstractDocument.ParagraphElementName)) {
					//System.out.println("ParagraphView");
					AttributeSet pAttrs=elem.getAttributes();
					Object viewAttr=pAttrs.getAttribute(PromptStyledDocument.VIEW_ELEMENT_NAME);
					if(!viewOnly || Boolean.TRUE.equals(viewAttr)) {
						return new ParagraphView(elem);
					}else {
						return new EmptyView(elem);
					}
					
				} else if (kind.equals(AbstractDocument.SectionElementName)) {
					//System.out.println("BoxView");
					return new BoxView(elem, View.Y_AXIS);
				} else if (kind.equals(StyleConstants.ComponentElementName)) {
					//System.out.println("ComponentView");
					return new ComponentView(elem);
				} else if (kind.equals(StyleConstants.IconElementName)) {
					return new IconView(elem);
				}else if(PromptStyledDocument.LINE_BREAK_ELEMENT_NAME.equals(kind)) {
					//System.out.println("LineBreakView");
					return new LineBreakView(elem);
				}
			}

			// default to text display
			return new LabelView(elem);
		}

	}
	
	JEditorPane installedToPane;
	
	private ViewFactory defaultViewFactory;
	
	/**
	 * 
	 */
	public PromptDocEditorKit() {
		this(false);
	}
	public PromptDocEditorKit(boolean viewOnly) {
		super();
		defaultViewFactory=new PromptDocFactory(viewOnly);
	}
	
	  /**
     * Called when the kit is being installed into
     * a JEditorPane.
     *
     * @param c the JEditorPane
     */
    public void install(JEditorPane c) {
        this.installedToPane=c;
        super.install(c);
    }

    /**
     * Called when the kit is being removed from the
     * JEditorPane.  This is used to unregister any
     * listeners that were attached.
     *
     * @param c the JEditorPane
     */
    public void deinstall(JEditorPane c) {
    	this.installedToPane=null;
    	super.deinstall(c);
    }
	
	
	 /**
     * Returns view factory for this prompt document editor kit.
     *
     * @return the factory
     */
    public ViewFactory getViewFactory() {
        return defaultViewFactory;
    }

	/* (non-Javadoc)
	 * @see javax.swing.text.EditorKit#getContentType()
	 */
	@Override
	public String getContentType() {
		return "text/x-prompt";
	}
	
	/**
     * An action to toggle underline attribute.
     
     */
    public static class LineBreakAction extends StyledTextAction {

    	public final static String ACTION_COMMAND = new String("br");

        public LineBreakAction() {
            super(ACTION_COMMAND);
            putValue(ACTION_COMMAND_KEY, ACTION_COMMAND);
        }

        /**
         * Toggles the underline attribute.
         *
         * @param e the action event
         */
        public void actionPerformed(ActionEvent e) {
        	//System.out.println("Custom line break action!");
            JEditorPane editor = getEditor(e);
            if (editor != null) {
                StyledEditorKit kit = getStyledEditorKit(editor);
                MutableAttributeSet attr = kit.getInputAttributes();
                MutableAttributeSet attrSave=new SimpleAttributeSet(attr);
                Document d=editor.getDocument();
                if(d instanceof PromptStyledDocument) {
                	PromptStyledDocument psd=(PromptStyledDocument)d;
                	 
                  	
                	 Caret caret = editor.getCaret();
                     int dot = caret.getDot();
                    
						try {
							psd.insertLineBreak(dot);
						} catch (BadLocationException e1) {
							// TODO Auto-generated catch block
							e1.printStackTrace();
						}
						setCharacterAttributes(editor, attrSave, false);
						caret.setDot(dot+1);
						
						
					
                }
            }
        }
    }
	
    /**
     * An action to toggle underline attribute.
     
     */
    public static class UnderlineAction extends StyledTextAction {



        public UnderlineAction() {
            super("underline");
        }

        /**
         * Toggles the underline attribute.
         *
         * @param e the action event
         */
        public void actionPerformed(ActionEvent e) {
            JEditorPane editor = getEditor(e);
            if (editor != null) {
                StyledEditorKit kit = getStyledEditorKit(editor);
                MutableAttributeSet attr = kit.getInputAttributes();
                boolean underline = (StyleConstants.isUnderline(attr)) ? false : true;
                SimpleAttributeSet sas = new SimpleAttributeSet();
                StyleConstants.setUnderline(sas, underline);
                if(underline) {
                	StyleConstants.setStrikeThrough(sas, false);
                }
                setCharacterAttributes(editor, sas, false);
            }
        }
    }
	
	 /**
     * An action to toggle the strike through attribute.
     
     */
    public static class StrikethroughAction extends StyledTextAction {

        /**
         * Constructs a new StrikeThroughAction.
         */
        public StrikethroughAction() {
            super("strike-through");
        }

        /**
         * Toggles the Strikethrough attribute.
         *
         * @param e the action event
         */
        public void actionPerformed(ActionEvent e) {
            JEditorPane editor = getEditor(e);
            if (editor != null) {
                StyledEditorKit kit = getStyledEditorKit(editor);
                MutableAttributeSet attr = kit.getInputAttributes();
                boolean strikethrough = (StyleConstants.isStrikeThrough(attr)) ? false : true;
                SimpleAttributeSet sas = new SimpleAttributeSet();
                StyleConstants.setStrikeThrough(sas, strikethrough);
                if(strikethrough) {
                	StyleConstants.setUnderline(sas, false);
                }
                setCharacterAttributes(editor, sas, false);
            }
        }
    }
    
    public static class SetFontSizeActionEvent extends ActionEvent{
    	/**
		 * 
		 */
		private static final long serialVersionUID = 1L;
		public SetFontSizeActionEvent(Object source, String command, FontSize fontSize,int size) {
			super(source, ActionEvent.ACTION_PERFORMED, command);
			this.fontSize=fontSize;
			this.size = size;
		}
		private int size;
		private FontSize fontSize;
		public FontSize getFontSize() {
			return fontSize;
		}
		public int getSize() {
			return size;
		}
    
    }
    
    /**
     * An action to set font size attribute.
     */
    public static class SetFontSizeAction extends StyledTextAction {

    	/**
    	 * Constructs a new StrikeThroughAction.
    	 */
    	public SetFontSizeAction() {
    		super("set-font-size");
    	}


    	public void actionPerformed(ActionEvent e) {
    		JEditorPane editor = getEditor(e);
    		if (editor != null && e instanceof SetFontSizeActionEvent) {
    			SetFontSizeActionEvent sfae=(SetFontSizeActionEvent)e;
    			StyledEditorKit kit = getStyledEditorKit(editor);
    			MutableAttributeSet attr = kit.getInputAttributes();
    			FontSize fs=sfae.getFontSize();
    			SimpleAttributeSet sas;
    			boolean replace=false;
    			
    			if(fs==null) {
    				// To really remove the font size attribute, copy the existing attributes, remove our custom STYLE_ATTR_FONTSIZE_RELATIVE_OR_ABSOLUTE attribute and replace the editors attributes.
    				sas = new SimpleAttributeSet(attr);
    				replace =true;
    				sas.removeAttribute(PromptStyledDocument.STYLE_ATTR_FONTSIZE_RELATIVE_OR_ABSOLUTE);
    				Document d=editor.getDocument();
    				if( d instanceof PromptStyledDocument) {
    					PromptStyledDocument psd=(PromptStyledDocument)d;

    					Style defStyle=psd.getStyle(StyleContext.DEFAULT_STYLE);
    					Object defFontSize=defStyle.getAttribute(StyleConstants.FontSize);

    					if(defFontSize instanceof Number){
    						Number defFontSizeNumber=(Number)defFontSize;
    						StyleConstants.setFontSize(sas,defFontSizeNumber.intValue());
    					}
    				}
    			}else {
    				sas = new SimpleAttributeSet();
    				sas.addAttribute(PromptStyledDocument.STYLE_ATTR_FONTSIZE_RELATIVE_OR_ABSOLUTE,fs);
    				StyleConstants.setFontSize(sas,sfae.getSize());
    			}
    			setCharacterAttributes(editor, sas, replace);
    		}
    	}
    }

	/* (non-Javadoc)
	 * @see javax.swing.text.EditorKit#createDefaultDocument()
	 */
	@Override
	public Document createDefaultDocument() {
		
		PromptStyleContext c=new PromptStyleContext();
		PromptStyledDocument d=null;
		try {
			d = new PromptStyledDocument(c,new PromptDoc());
		} catch (ParserException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		return d;
//		return new DefaultStyledDocument();
	}


}
