//    Speechrecorder
//    (c) Copyright 2009-2011
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

//
//  ProjectManager.java
//  JSpeechRecorder
//
//  Created by Christoph Draxler on Thu Dec 05 2002.
//

package ipsk.apps.speechrecorder;

import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.Console;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.net.Authenticator;
import java.net.CookieHandler;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.xml.bind.JAXB;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import ips.annot.BundleAnnotationPersistorServiceDescriptor;
import ips.annot.autoannotator.AutoAnnotationServiceDescriptor;
import ips.annot.autoannotator.AutoAnnotator;
import ips.annot.model.emu.EmuBundleAnnotationPersistor;
import ips.annot.textgrid.TextGridFilePersistor;
import ips.incubator.util.apps.consent.Consent;
import ipsk.apps.speechrecorder.annotation.auto.AutoAnnotationPluginManager;
import ipsk.apps.speechrecorder.annotation.auto.AutoAnnotationWorker;
import ipsk.apps.speechrecorder.annotation.auto.MAUSTemplateTextFilePersistor;
import ipsk.apps.speechrecorder.annotation.auto.TemplateTextFilePersistor;
import ipsk.apps.speechrecorder.annotation.auto.impl.PromptAutoAnnotatorServiceDescriptor;
import ipsk.apps.speechrecorder.annotation.auto.impl.TemplateAutoAnnotatorServiceDescriptor;
import ipsk.apps.speechrecorder.config.ConfigHelper;
import ipsk.apps.speechrecorder.config.Formatter;
import ipsk.apps.speechrecorder.config.ProjectConfiguration;
import ipsk.apps.speechrecorder.config.WorkspaceProject;
import ipsk.apps.speechrecorder.project.ActiveProjectManager;
import ipsk.apps.speechrecorder.project.ProjectManager;
import ipsk.apps.speechrecorder.project.ProjectManagerEvent;
import ipsk.apps.speechrecorder.project.ProjectManagerException;
import ipsk.apps.speechrecorder.project.ProjectManagerListener;
import ipsk.apps.speechrecorder.project.ProjectManagerProjectClosedEvent;
import ipsk.apps.speechrecorder.project.ProjectManagerProjectOpenedEvent;
import ipsk.apps.speechrecorder.project.ProjectManagerProjectReadyForShutdownEvent;
import ipsk.apps.speechrecorder.prompting.FormattedPromptTextViewer;
import ipsk.apps.speechrecorder.prompting.PromptBufferedImageViewer;
import ipsk.apps.speechrecorder.prompting.PromptFormattedTextViewer;
import ipsk.apps.speechrecorder.prompting.PromptImageIconViewer;
import ipsk.apps.speechrecorder.prompting.PromptPlainTextViewer;
import ipsk.apps.speechrecorder.prompting.PromptPresenterPluginManager;
import ipsk.apps.speechrecorder.prompting.PromptPresenterServiceDescriptor;
import ipsk.apps.speechrecorder.prompting.ScrollableFormattedPromptTextViewer;
import ipsk.apps.speechrecorder.prompting.combined.FormattedTextAndAudioJavaSoundViewer;
import ipsk.apps.speechrecorder.prompting.combined.ImageAndAudioJavaSoundViewer;
import ipsk.apps.speechrecorder.prompting.combined.PlainTextAndAudioJavaSoundViewer;
import ipsk.apps.speechrecorder.prompting.combined.TextButtonAudioJavaSoundViewer;
import ipsk.apps.speechrecorder.prompting.sound.javasound.PromptAudioJavaSoundViewer;
import ipsk.apps.speechrecorder.script.RecscriptManagerException;
import ipsk.apps.speechrecorder.session.action.CloseSpeakerDisplayAction;
import ipsk.apps.speechrecorder.storage.StorageManagerException;
import ipsk.apps.speechrecorder.workspace.WorkspaceException;
import ipsk.apps.speechrecorder.workspace.WorkspaceManager;
import ipsk.audio.AudioController4;
import ipsk.audio.AudioControllerException;
import ipsk.audio.mixer.MixerManager;
import ipsk.awt.Worker.State;
import ipsk.awt.WorkerException;
import ipsk.beans.DOMCodec;
import ipsk.beans.DOMCodecException;
import ipsk.net.SimplePasswordAuthentication;
import ipsk.net.URLContext;
import ipsk.net.cookie.SessionCookieHandler;
import ipsk.swing.JProgressDialogPanel;
import ipsk.swing.legal.JConsentDialog;
import ipsk.text.ParserException;
import ipsk.text.Version;
import ipsk.util.LocalizableMessage;
import ipsk.util.SystemHelper;
import ipsk.util.apps.UpdateManager;
import ipsk.util.apps.UpdateManagerConfig;
import ipsk.util.apps.UpdateManagerEvent;
import ipsk.util.apps.UpdateManagerListener;
import ipsk.util.apps.descriptor.Change;
import ipsk.util.apps.event.UpdateAvailableEvent;
import ipsk.util.apps.ui.JUpdateManagerConfigDialog;
import ipsk.util.apps.ui.JUpdateManagerConsentDialog;
import ipsk.util.optionparser.Option;
import ipsk.util.optionparser.OptionParser;
import ipsk.util.optionparser.OptionParserException;
import ipsk.util.zip.UnzipWorker;
import ipsk.xml.DOMConverter;
import ipsk.xml.DOMConverterException;


public class SpeechRecorder implements ActionListener, ProjectManagerListener, UpdateManagerListener{

   
    public final static boolean DEBUG=false;
    
    // version of Speechrecorder application
	public final static String VERSION = SpeechRecorder.class.getPackage().getImplementationVersion();
	public final static String FULL_QUALIFIED_APPLICATION_NAME="de.uni-muenchen.phonetik.SpeechRecorder";
	public final static String APPLICATION_NAME="SpeechRecorder";
	public final static String FREEDESKTOP_APPLICATION_ICON_NAME="speechrecorder";
	
	// set to true for stable releases
	public final static boolean IS_STABLE_RELEASE=true;
	// user warning message if (non stable) development version
	private final static String NON_STABLE_VERSION_WARNING="You are using a non stable development version.\nPlease use this version only for evaluation!\nProjects configured by this version might be incompatible to stable releases!";

	public final static String AUTHORS = " Chr. Draxler, K. J\u00e4nsch";

	public final static String COPYRIGHT = "Copyright \u00A9 2004-2024";
	
	public static final String CONTACT_URI="mailto:speechrecorder@phonetik.uni-muenchen.de";
	
	private final String DEF_REL_WORKSPACE_DIR = "speechrecorder";
	
	private final String USER_CONFIG_DIR = ".speechrecorder";
	
	private final String USER_CONFIG_LEGAL_DIR = "legal";
	
	public final String APPLICATION_DESCRIPTOR_KEY="ipsk.util.apps.descriptor.url";
	public final String PREFERRED_START_STOP_SIGNAL_PLUGIN="ips.apps.speechrecorder.startstopsignal.Ampelmaennchen";
	
	public final boolean USE_MAX_REC_TIMER=true;
	
    public static final String PROJECT_FILE_EXTENSION = "_project.prj";

	private long SHUTDOWN_RETRY_DELAY = 2000;

	public String LOG_HANDLER_NAME = "default";

	public String TIMELOG_HANDLER_NAME = "timelog";

	public static ipsk.apps.speechrecorder.config.Handler[] DEF_LOG_HANDLERS = new ipsk.apps.speechrecorder.config.Handler[0];

	protected final static Formatter TIME_LOG_FORMATTER_CFG = new Formatter(
			TimeLogFormatter.class.getName(), "Time logger");

	public final static Formatter[] LOG_FORMATTERS = {
			new Formatter(null, "(Default)"),
			new Formatter("java.util.logging.SimpleFormatter", "Plain Text"),
			new Formatter("java.util.logging.XMLFormatter", "XML"),
			TIME_LOG_FORMATTER_CFG };

	public static ipsk.apps.speechrecorder.config.Logger[] AVAIL_LOGGERS = new ipsk.apps.speechrecorder.config.Logger[0];

	private Logger logger;

	private Logger timeLogger;

	private DOMCodec domCodec;

	private DOMConverter domConverter;

	private File defWorkspaceDir;

	private WorkspaceProject[] workspaceProjects;

	private GraphicsConfiguration speakerScreenConfig=null;

	private GraphicsConfiguration experimenterScreenConfig=null;

	
	private ActiveProjectManager projectManager;

	private AudioController4 audioController;


	private UIResources uiString;

	private SpeechRecorderUI speechRecorderUI;

	
	private CloseSpeakerDisplayAction closeSpeakerDisplayAction;
	

    private WorkspaceManager workspaceManager;
  
    
    private PromptPresenterPluginManager promptPresenterPluginManager;
    private AutoAnnotationPluginManager autoAnnotatorPluginManager;
    public AutoAnnotationPluginManager getAutoAnnotatorPluginManager() {
        return autoAnnotatorPluginManager;
    }

    private AutoAnnotationWorker autoAnnotationWorker;
    private UpdateManager updateManager;
    private Consent appCheckUpdatesConsent=null;
    private UpdateManagerConfig updateManagerConfig;

    private List<BundleAnnotationPersistorServiceDescriptor> availableBundleAnnotationServiceDescriptors;

    private Path userConfigDir;
    
    private Path userDataDir;

    private Path userConfigUpdateManagerDir;
    
    private Path userDataUpdateManagerDir;

    private SystemHelper systemHelper;
    
	public UpdateManager getUpdateManager() {
		return updateManager;
	}


	/** Create Speechrecorder application
     * Must be called from AWT event thread ! 
     * @param projectFileURL URL of the project configuration file
	 * @param projectName name of project
     * @param user user for web authentication
     * @param password password for web authentication
     * @throws ClassNotFoundException class not found
     * @throws DOMCodecException DOM codec exception
     * @throws DOMConverterException DOM converter exception
     * @throws IOException I/O exception
     * @throws PluginLoadingException plugin loading exception
     * @throws AudioControllerException audio controller exception
     * @throws ParserConfigurationException parser configuration exception
     * @throws SAXException SAX XML parser exception
     * @throws StorageManagerException storage manager exception
     * @throws InstantiationException instantiation exception
     * @throws IllegalAccessException illegal access
     * @throws WorkspaceException workspace exception
	 * @throws ProjectManagerException project manager exception
     */
    public SpeechRecorder(String projectFileURL, String projectName,String user, String password)
    throws ClassNotFoundException, DOMCodecException,
    DOMConverterException, IOException, PluginLoadingException,
    AudioControllerException, ParserConfigurationException,
    SAXException, StorageManagerException, InstantiationException,
    IllegalAccessException, WorkspaceException, ProjectManagerException {

        systemHelper=SystemHelper.getInstance();
        
        // initialize the logging mechanism
        ipsk.apps.speechrecorder.config.Handler logHandler = new ipsk.apps.speechrecorder.config.Handler(
                LOG_HANDLER_NAME);
        ipsk.apps.speechrecorder.config.Handler timeLogHandler = new ipsk.apps.speechrecorder.config.Handler(
                TIMELOG_HANDLER_NAME);
        timeLogHandler.setFormatter(TIME_LOG_FORMATTER_CFG);
        DEF_LOG_HANDLERS = new ipsk.apps.speechrecorder.config.Handler[] {
                logHandler, timeLogHandler };
        ipsk.apps.speechrecorder.config.Logger defLogger = new ipsk.apps.speechrecorder.config.Logger();
        defLogger.setName("ipsk.apps.speechrecorder");
        defLogger.setHandlerName(logHandler.getName());
        ipsk.apps.speechrecorder.config.Logger defTimeLogger = new ipsk.apps.speechrecorder.config.Logger();
        defTimeLogger.setName("time");
        defTimeLogger
        .setHandlerName(timeLogHandler.getName());
        logger = Logger.getLogger("ipsk.apps.speechrecorder");
        AVAIL_LOGGERS = new ipsk.apps.speechrecorder.config.Logger[] {
                defLogger, defTimeLogger };

        timeLogger = Logger.getLogger("time");
        // logger.setLevel(logLevel);
        // logger.info("Starting up ProjectManager");

        uiString = UIResources.getInstance();
        String userHomePropStr=null;
        try{
            userHomePropStr=System.getProperty("user.home");
        }catch(SecurityException se){
            JOptionPane.showMessageDialog(speechRecorderUI,"Could not get user home directory path: "+se.getLocalizedMessage(),"Fatal security excption",JOptionPane.ERROR);
            shutdown();
        }
        File homeDir=new File(userHomePropStr);

//        userConfigDir = homeDir.toPath().resolve(USER_CONFIG_DIR);
        
        // Note: userConfigDir and userDataDir are only different on Linux! 
        
        userConfigDir=systemHelper.applicationUserConfigFolder().resolve(FULL_QUALIFIED_APPLICATION_NAME);
        if(Files.exists(userConfigDir)){
            if(!Files.isDirectory(userConfigDir)){
                JOptionPane.showMessageDialog(speechRecorderUI,"Could not use \""+userConfigDir+". It is not a directory!","Fatal I/O excption",JOptionPane.ERROR);
                shutdown();
            }
        }else{
            try{
            	
                Files.createDirectories(userConfigDir);
//                FileStore fileStore=Files.getFileStore(userConfigDir);
//                if(fileStore.supportsFileAttributeView(DosFileAttributeView.class)) {
//                	DosFileAttributeView dfav=Files.getFileAttributeView(userConfigDir,DosFileAttributeView.class);
//                	if(dfav!=null) {
//                		dfav.setHidden(true);
//                	}
//                	//Files.setAttribute(userConfigDir, "dos:hidden", true);
//                }
            }catch(IOException ioe){
                JOptionPane.showMessageDialog(speechRecorderUI,"Could not create user config directory: "+ioe.getLocalizedMessage(),"Fatal I/O excption",JOptionPane.ERROR);
                shutdown();
            }
        }
        
        
        userDataDir=systemHelper.applicationUserDataFolder().resolve(FULL_QUALIFIED_APPLICATION_NAME);
        if(Files.exists(userDataDir)){
            if(!Files.isDirectory(userDataDir)){
                JOptionPane.showMessageDialog(speechRecorderUI,"Could not use \""+userDataDir+". It is not a directory!","Fatal I/O excption",JOptionPane.ERROR);
                shutdown();
            }
        }else{
            try{
                
                Files.createDirectories(userDataDir);
            }catch(IOException ioe){
                JOptionPane.showMessageDialog(speechRecorderUI,"Could not create user data directory: "+ioe.getLocalizedMessage(),"Fatal I/O excption",JOptionPane.ERROR);
                shutdown();
            }
        }


//        // Check preferences permissions
//        boolean systemPreferencesReadable=false;
//        boolean systemPrefrencesWritable=false;
//        
//        boolean checkSystemPrefsAccess=false;
//        
//        if(checkSystemPrefsAccess){
////        SecurityManager sm=System.getSecurityManager();
////        try{
////        sm.checkPermission(new RuntimePermission("preferences", "write"));
////        System.out.println("write access");
////        }catch(SecurityException se){
////            System.out.println("no write access");
////        }
////        System.setProperty("java.util.prefs.syncInterval","500");
//        try{
//        Preferences sysPrefs=Preferences.systemNodeForPackage(this.getClass());
//        systemPreferencesReadable=true;
//        sysPrefs.put("primaryRecordingTarget","TEMP_RAW_FILE");
//        try {
//            sysPrefs.flush();
//            systemPrefrencesWritable=true;
//        } catch (BackingStoreException e1) {
//            systemPrefrencesWritable=false;
//            
//        }
//        }catch(SecurityException se){
//            systemPreferencesReadable=false;
//        }
//       
//        System.out.println("System prefs: readable: "+systemPreferencesReadable+" writable: "+systemPrefrencesWritable);
//    }
//        boolean userPreferencesReadable=false;
//        boolean userPrefrencesWritable=false;
//        
////        SecurityManager sm=System.getSecurityManager();
////        try{
////        sm.checkPermission(new RuntimePermission("preferences", "write"));
////        System.out.println("write access");
////        }catch(SecurityException se){
////            System.out.println("no write access");
////        }
////        System.setProperty("java.util.prefs.syncInterval","500");
//        try{
//        Preferences userPrefs=Preferences.userNodeForPackage(this.getClass());
//        userPreferencesReadable=true;
//        userPrefs.put("primaryRecordingTarget","TEMP_RAW_FILE");
//        try {
//            userPrefs.flush();
//            userPrefrencesWritable=true;
//        } catch (BackingStoreException e1) {
//            userPrefrencesWritable=false;
//            
//        }
//        }catch(SecurityException se){
//            userPreferencesReadable=false;
//        }
//       System.out.println("User prefs: readable: "+userPreferencesReadable+" writable: "+userPrefrencesWritable);
//        
        
//        sequenceGenerator=new AtomicIntegerSequenceGenerator();
        
        
        
        
        
        defWorkspaceDir = new File(homeDir,DEF_REL_WORKSPACE_DIR);
        workspaceManager=new WorkspaceManager(defWorkspaceDir,this);
        Package configBasePack = Class.forName(
        "ipsk.apps.speechrecorder.config.ProjectConfiguration")
        .getPackage();
//        audioEnabled = false;
    	new MixerManager();
        domCodec = new DOMCodec(configBasePack);
        domConverter = new DOMConverter();
//        speakerManager = new SpeakerManager();
      
//        sessionManager=ProgressManager.getInstance();
     //   projectConfigurationSaved = true;
        
        if (user != null) {
            Authenticator.setDefault(new SimplePasswordAuthentication(user,
                    password));
            logger.info("Set authenticator for user " + user);
        }
        URL projectURL=null;
        if (projectFileURL != null) {
            projectURL = new URL(projectFileURL);
            // For testing ??
            InputStream projectFileStream = projectURL.openStream();
            projectFileStream.close();
        }
     
        //              now determine the graphics environment: one or two displays, what
        // resolution, etc.
        int experimenterScreenIdx=0;
        GraphicsEnvironment ge = GraphicsEnvironment
        .getLocalGraphicsEnvironment();
        GraphicsDevice[] gds = ge.getScreenDevices();
        GraphicsDevice defGd=ge.getDefaultScreenDevice();
        if(defGd==null && gds.length>0){
        	// fallback should not happen
        	defGd=gds[0];
        }
        if(gds==null || gds.length==0){
            logger.severe("No display connected!");
        }
        for(int i=0;i<gds.length;i++){
    		GraphicsDevice gd=gds[i];
    		GraphicsConfiguration gc=gd.getDefaultConfiguration();
    		Rectangle gb=gc.getBounds();
            logger.info("Display "+i+": "+ gb.getWidth() + " x "+ gb.getHeight());
        }
        
        if(defGd!=null){
        	for(int i=0;i<gds.length;i++){
        		GraphicsDevice gd=gds[i];
        		if(gd.equals(defGd)){
        			experimenterScreenConfig = defGd.getDefaultConfiguration();
        			experimenterScreenIdx=i;
        			logger.info("Selected default display "+i+" as experimenter screen");
        		}
        	}
        }
        if (gds.length == 1) {
            // only one display connected; both the experimenter and the
            // speaker display share the same display
            speakerScreenConfig = experimenterScreenConfig;
            logger.info("Selected default display 0 as speaker screen");
        } else if (gds.length >= 2) {
            // at least two displays connected
        	// Fix Bug ID 0049
        	// On Mac OS X the first display is not necessarily the default display
        	// Choose the first display which is not the default
            
        	for(int i=0;i<gds.length;i++){
        		GraphicsDevice gd=gds[i];
        		if(speakerScreenConfig==null && ! gd.equals(defGd)){
        			speakerScreenConfig = gd.getDefaultConfiguration();
        			logger.info("Selected display "+i+" as speaker screen");
        		}
        	}
          
        } 
   
 
        // now additional plugins
        promptPresenterPluginManager=new PromptPresenterPluginManager();
        
        List<PromptPresenterServiceDescriptor> promptPresenterClassList=promptPresenterPluginManager.getPromptPresenterServiceDescriptors();
        
        PromptPresenterServiceDescriptor ftpctvsd=FormattedPromptTextViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(ftpctvsd)){
            promptPresenterClassList.add(ftpctvsd);
        }
        PromptPresenterServiceDescriptor scftpctvsd=ScrollableFormattedPromptTextViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(scftpctvsd)){
            promptPresenterClassList.add(scftpctvsd);
        }
//        
//        PromptPresenterServiceDescriptor ftpctvsd=SprCompatTextViewer.DESCRIPTOR;
//        if(!promptPresenterClassList.contains(ftpctvsd)){
//        	promptPresenterClassList.add(ftpctvsd);
//        }
        PromptPresenterServiceDescriptor pptvsd=PromptPlainTextViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(pptvsd)){
            promptPresenterClassList.add(pptvsd);
        }
        PromptPresenterServiceDescriptor pftvsd=PromptFormattedTextViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(pftvsd)){
            promptPresenterClassList.add(pftvsd);
        }
        PromptPresenterServiceDescriptor pbivsd=PromptBufferedImageViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(pbivsd)){
            promptPresenterClassList.add(pbivsd);
        }
        PromptPresenterServiceDescriptor piivsd=PromptImageIconViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(piivsd)){
            promptPresenterClassList.add(piivsd);
        }
        PromptPresenterServiceDescriptor pajsvsd=PromptAudioJavaSoundViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(pajsvsd)){
            promptPresenterClassList.add(pajsvsd);
        }
//        if(!promptPresenterClassList.contains(PromptAudioViewer.class)){
//            promptPresenterClassList.add(PromptAudioViewer.class);
//        }
        
        PromptPresenterServiceDescriptor ptbtaajsvsd=TextButtonAudioJavaSoundViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(ptbtaajsvsd)){
            promptPresenterClassList.add(ptbtaajsvsd);
        }
        PromptPresenterServiceDescriptor ptaajsvsd=PlainTextAndAudioJavaSoundViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(ptaajsvsd)){
            promptPresenterClassList.add(ptaajsvsd);
        }
        PromptPresenterServiceDescriptor pftaajsvsd=FormattedTextAndAudioJavaSoundViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(pftaajsvsd)){
            promptPresenterClassList.add(pftaajsvsd);
        }
        PromptPresenterServiceDescriptor piaajsvsd=ImageAndAudioJavaSoundViewer.DESCRIPTOR;
        if(!promptPresenterClassList.contains(piaajsvsd)){
            promptPresenterClassList.add(piaajsvsd);
        }
//        for(Class<PromptPresenter> promptPresenterClass: promptPresenterClassList){
//            PromptPresenter pp=promptPresenterClass.newInstance();
//            promptPresenterList.add(pp);
//        }
        // Some components are realized before, so we should run the show method
        // after all events are dispatched
        
        availableBundleAnnotationServiceDescriptors=new ArrayList<BundleAnnotationPersistorServiceDescriptor>();
        availableBundleAnnotationServiceDescriptors.add(new EmuBundleAnnotationPersistor());
        availableBundleAnnotationServiceDescriptors.add(new TextGridFilePersistor());
        availableBundleAnnotationServiceDescriptors.add(new TemplateTextFilePersistor());
        availableBundleAnnotationServiceDescriptors.add(new MAUSTemplateTextFilePersistor());
        
        autoAnnotatorPluginManager=new AutoAnnotationPluginManager();
        List<AutoAnnotationServiceDescriptor> autoAnnotatorDescriptors=autoAnnotatorPluginManager.getAutoAnnotatorServiceDescriptors();

        // list should contain plugins such as G2P and MAUS plugin
        
        // insert standard "plugins"
        autoAnnotatorDescriptors.add(0,new PromptAutoAnnotatorServiceDescriptor());
        autoAnnotatorDescriptors.add(1,new TemplateAutoAnnotatorServiceDescriptor());
        
     
        projectManager=new ActiveProjectManager(promptPresenterPluginManager,autoAnnotatorPluginManager,projectFileURL, user, password);
        projectManager.setWorkspaceDir(defWorkspaceDir);
        projectManager.setListener(this);
       
                
        if (speechRecorderUI != null){
        	// can be disposed on all platforms it is not reused
        	speechRecorderUI.dispose();
        }
        
        String appDescrUrlParam=System.getProperty(APPLICATION_DESCRIPTOR_KEY);
        if(appDescrUrlParam !=null){
            // version should be available from package manifest, set to 0.0.1 from debugging purposes
            Version currentVersion=null;
            if(VERSION==null){
                currentVersion=new Version(new int[]{0,0,1});
            }else{
                try {
                    currentVersion=Version.parseString(VERSION);
                } catch (ParserException e) {
                    // ignore
                }
            }
            updateManager=new UpdateManager(currentVersion);
            
            updateManager.addUpdateManagerListener(this);
        }
        
        speechRecorderUI = new SpeechRecorderUI(this,experimenterScreenIdx, experimenterScreenConfig,
                speakerScreenConfig);
       
        speechRecorderUI.setPromptPresenterServiceDescriptors(promptPresenterClassList);
        //speechRecorderUI.setAutoAnnotatorServiceDescriptors(autoAnnotatorDescriptors);
        
        // TODO very dirty dependencies
        speechRecorderUI.createUI(projectManager);
        projectManager.setSpeechRecorderUI(speechRecorderUI);
        
        speechRecorderUI.pack();
        speechRecorderUI.setExtendedState(JFrame.MAXIMIZED_BOTH);
        speechRecorderUI.setVisible(true);
//       
//        closeSpeakerDisplayAction=new CloseSpeakerDisplayAction(this,"Close speaker display");
//        ACTIONS = new Action[] { closeSpeakerDisplayAction};
       
        //  do not show splash screen anymore. The icon logo already appears on startup
        // two splash screens are too much
        
//        speechRecorderUI.showSplashScreen(true);

        // create splash screen
        // splashPanel = new SplashScreen(experimenterScreenConfig);

        if(!IS_STABLE_RELEASE){
            int ans=JOptionPane.showConfirmDialog(speechRecorderUI, NON_STABLE_VERSION_WARNING, "Warning!", JOptionPane.WARNING_MESSAGE);
            if(ans==JOptionPane.CANCEL_OPTION){
                shutdown();
            }
        }
        
        if(updateManager !=null){
            URL appDescrUrl=new URL(appDescrUrlParam);
            updateManager.setApplicationDescriptorURL(appDescrUrl);
            userConfigUpdateManagerDir = userConfigDir.resolve("update_manager");
            userDataUpdateManagerDir = userDataDir.resolve("update_manager");
            loadAppUpdatesConfig();
            if(updateManagerConfig==null){
                boolean consent=checkAppUpdatesConsent(true);
                configureUpdateManager(consent);
            }
            if(updateManagerConfig!=null && updateManagerConfig.isCheckOnStartup()){
                checkAppUpdatesConsent(true);
               updateManager.startLoadApplicationDescriptor();
            }
        }
        speechRecorderUI.setWaiting(true);
        if (projectURL != null) {
        	boolean canceled=false;
            try {
                canceled=configureProject(projectURL);
            } catch (Exception e) {
                e.printStackTrace();
                JOptionPane.showMessageDialog(null, e.getLocalizedMessage(),
                        "ERROR", JOptionPane.ERROR_MESSAGE);
                System.exit(-1);
            } 
            if(canceled){
            	// should never happen here
            	// only for interactive selection of prompt script or speaker DB files
            	return;
            }
            projectManager.start();
        } else {
            try {
                workspaceProjects=workspaceManager.scanWorkspace();
            } catch (WorkspaceException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
            speechRecorderUI.setWorkspaceProjects(workspaceProjects);
            speechRecorderUI.setEnableOpenOrNewProject(true);
        }
        speechRecorderUI.setWaiting(false);
        if(projectName!=null) {
        	try {
				openProject(projectName);
			} catch (SpeechRecorderException e) {
				speechRecorderUI.displayError("Open project error", e.getLocalizedMessage());
				System.err.println(e.getLocalizedMessage());
				e.printStackTrace();
				System.exit(-2);
			}
        }

//        System.out.println("Try to observe system sleep...");
        
//        if(Desktop.isDesktopSupported()) {
//        	Desktop deskt=Desktop.getDesktop();
        	// Does not work on Windows!! Reports supported, but no event detected
//        	if(deskt.isSupported(Desktop.Action.APP_EVENT_SYSTEM_SLEEP)){
//        		deskt.addAppEventListener(new SystemSleepListener() {
//
//        			@Override
//        			public void systemAwoke(SystemSleepEvent e) {
//        				System.out.println("System awoke!");
//        			}
//
//        			@Override
//        			public void systemAboutToSleep(SystemSleepEvent e) {
//        				System.out.println("System sleep!");
//        			}
//        		});
//        	}else {
//        		System.out.println("Desktop system sleep observe not supported!");
//        	}
//        }
    }
    
    
    public boolean checkAppUpdatesConsent(boolean forceUserInteractice){
        
        if(updateManager !=null){
            
            
            Path userUpdateManagerLegalDir=userDataUpdateManagerDir.resolve(USER_CONFIG_LEGAL_DIR);
            Path userUpdateManagerLegalFile=userUpdateManagerLegalDir.resolve("consent_update_check.xml");
            
            if(Files.exists(userUpdateManagerLegalFile)){
                appCheckUpdatesConsent=JAXB.unmarshal(userUpdateManagerLegalFile.toFile(),Consent.class);
            }
            
            if(appCheckUpdatesConsent==null || (forceUserInteractice && !appCheckUpdatesConsent.isAccepted())){
                // Ask user
               String consentText=uiString.getString("updatemanager.consent");
                JUpdateManagerConsentDialog pcd=new JUpdateManagerConsentDialog(consentText);
                Object res=pcd.showDialog(speechRecorderUI);
                try {
                    Files.createDirectories(userUpdateManagerLegalDir);
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
                appCheckUpdatesConsent=new Consent();
                appCheckUpdatesConsent.setAccepted(false);
                
                if(res.equals(JConsentDialog.OK_OPTION)){
                    appCheckUpdatesConsent.setText(pcd.getText());
                    appCheckUpdatesConsent.setAccepted(true);
                }
                JAXB.marshal(appCheckUpdatesConsent, userUpdateManagerLegalFile.toFile());
             }    
            
      
        }
        return (appCheckUpdatesConsent!=null && appCheckUpdatesConsent.isAccepted());
    }
    
    public void loadAppUpdatesConfig(){ 
        Path userUpdateManagerConfigDir=userConfigUpdateManagerDir.resolve("config");
        Path userUpdateManagerConfigFile=userUpdateManagerConfigDir.resolve("update_check_config.xml");

        if(Files.exists(userUpdateManagerConfigFile)){
            updateManagerConfig=JAXB.unmarshal(userUpdateManagerConfigFile.toFile(),UpdateManagerConfig.class);
        }
    }
    
    public void configureUpdateManager(boolean checkOnStartup) throws IOException{ 
        Path userUpdateManagerConfigDir=userConfigUpdateManagerDir.resolve("config");
        if(!Files.exists(userUpdateManagerConfigDir)){
            Files.createDirectories(userUpdateManagerConfigDir);
        }
        Path userUpdateManagerConfigFile=userUpdateManagerConfigDir.resolve("update_check_config.xml");
        if(updateManagerConfig==null){
            updateManagerConfig=new UpdateManagerConfig();
        }
        updateManagerConfig.setCheckOnStartup(checkOnStartup);
        JAXB.marshal(updateManagerConfig, userUpdateManagerConfigFile.toFile());
    }
    
    public void configureUpdateManagerInteractive(){ 
        Path userUpdateManagerConfigDir=userConfigUpdateManagerDir.resolve("config");
        Path userUpdateManagerConfigFile=userUpdateManagerConfigDir.resolve("update_check_config.xml");

        if(updateManagerConfig==null){
            updateManagerConfig=new UpdateManagerConfig();

        }

        // Ask user
        JUpdateManagerConfigDialog umcd=new JUpdateManagerConfigDialog(APPLICATION_NAME);
        umcd.setConfig(updateManagerConfig);
        Object res=umcd.showDialog(speechRecorderUI);
        if(res.equals(JConsentDialog.OK_OPTION)){
            umcd.apply();
            UpdateManagerConfig newCfg=umcd.getConfig();
            boolean reqCheckOnstartup=newCfg.isCheckOnStartup();
            if(reqCheckOnstartup){
                boolean cons=checkAppUpdatesConsent(true);
                newCfg.setCheckOnStartup(cons && reqCheckOnstartup);
            }
            updateManagerConfig=newCfg;
            try {
                Files.createDirectories(userUpdateManagerConfigDir);
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
            JAXB.marshal(updateManagerConfig, userUpdateManagerConfigFile.toFile());
        }


    }

	
	public List<AutoAnnotator> getAutoAnnotators(){
	    if(autoAnnotationWorker!=null){
	        return autoAnnotationWorker.getAutoAnnotators();
	    }else{
	        return null;
	    }
	}


	/**
	 * Configure a project.
	 * 
	 * @param projectUrl must contain the project configuration
	 * @return true if cancelled interactively during configuration
	 * @throws URISyntaxException URI syntax exception
	 * @throws RecscriptManagerException recording script manager exception
	 * @throws SpeechRecorderException SpeechRecorder exception
	 * @throws ClassNotFoundException class not found
     * @throws DOMCodecException DOM codec exception
     * @throws DOMConverterException DOM converter exception
     * @throws IOException I/O exception
     * @throws PluginLoadingException plugin loading exception
     * @throws AudioControllerException audio controller exception
     * @throws ParserConfigurationException parser configuration exception
     * @throws SAXException SAX XML parser exception
     * @throws StorageManagerException storage manager exception
     * @throws InstantiationException instantiation exception
     * @throws IllegalAccessException illegal access
     * @throws WorkspaceException workspace exception
	 * @throws ProjectManagerException project manager exception
	 */
	public boolean configureProject(URL projectUrl) throws ClassNotFoundException,
			DOMCodecException, DOMConverterException, StorageManagerException,
			PluginLoadingException, AudioControllerException,
			ParserConfigurationException, SAXException, IOException,
			InstantiationException, IllegalAccessException, WorkspaceException, URISyntaxException, RecscriptManagerException, SpeechRecorderException, ProjectManagerException{
		
		URLConnection prUrlConn=projectUrl.openConnection();
		prUrlConn.setUseCaches(false);
		InputStream prInStream=prUrlConn.getInputStream();
		Document d = domConverter.readXML(prInStream);
		ProjectConfiguration p = (ProjectConfiguration) domCodec
				.readDocument(d);
		projectManager.setProjectURL(projectUrl);
		URL projectContext=URLContext.baseContextFromResourceURL(projectUrl);
		projectManager.setProjectContext(projectContext);
		ConfigHelper.applyLegacyToStrictConversions(p);
		boolean canceled=projectManager.configure(p);
		
		projectManager.setProjectConfigurationSaved(true);
		return canceled;
	}

	public synchronized void actionPerformed(ActionEvent e) {
	  
	}
	
	
	public void openProject(String projectName) throws SpeechRecorderException{
	    boolean canceled;
		try {
			canceled = configureProject(projectName);
		} catch (RecscriptManagerException | ProjectManagerException | AudioControllerException |  WorkspaceException e1) {
			throw new SpeechRecorderException(e1.getLocalizedMessage(),e1);
		} catch (PluginLoadingException e) {
			throw new SpeechRecorderPluginException(e.getLocalizedMessage(),e);
		} 
	    if(!canceled){
	    	speechRecorderUI.closeWorkspaceDialog();
	        speechRecorderUI.showSpeakerDatabase();
	        try {
				projectManager.start();
			} catch (ProjectManagerException | AudioControllerException e) {
				throw new SpeechRecorderException(e);
			}
	    }
	}

    public void setEditingEnabled(boolean b) {  
       projectManager.setEditingEnabled(b);
       speechRecorderUI.setEditingEnabled(b);
   }
    
  
	

	


	/**
	 * Shutdown the application. If an uploading cache is used this method waits
	 * for complete upload.
	 */
	public void shutdown() {
	    projectManager.shutdown();
	}

	/**
	 * Returns the UI object.
	 * 
	 * @return UI object
	 */
	public SpeechRecorderUI getSpeechRecorderUI() {
		return speechRecorderUI;
	}

	/**
	 * Stops playback.
	 * @throws AudioControllerException audio controller exception
	 */
	public void stopPlayback() throws AudioControllerException {
		audioController.stopPlayback();
	}

	/**
	 * Pauses playback.
	 * @throws AudioControllerException audio controller exception
	 */
	public void pausePlayback() throws AudioControllerException {
		audioController.pausePlayback();
	}

	/**
	 * Continues playback after pause.
	 */
	public void continuePlayback() {

		try {
			audioController.startPlayback();
		} catch (AudioControllerException e) {
			speechRecorderUI.displayError("AudioController Error", e
					.getLocalizedMessage());
			e.printStackTrace();
		}

	}

	/**
	 * Returns the default workspace directory.
	 * 
	 * @return default workspace directory
	 */
	public File getDefWorkspaceDir() {
		return defWorkspaceDir;
	}

	/**
	 * Configure project.
	 * @param projectName project name
	 * @throws PluginLoadingException plugin loading exception
	 * @throws AudioControllerException audio controller exception
	 * @throws WorkspaceException workspace exception
	 * @throws RecscriptManagerException recording script manager exception
	 * @throws SpeechRecorderException speech recorder exception
	 * @throws ProjectManagerException project manager exception
	 * @return true if configuration was canceled
	 */
	public boolean configureProject(String projectName) throws PluginLoadingException,
			AudioControllerException, WorkspaceException, RecscriptManagerException, SpeechRecorderException, ProjectManagerException{
		List<WorkspaceProject> workspaceProjects=workspaceManager.getWorkspaceProjects();
		boolean canceled=false;
		boolean found=false;
		for (WorkspaceProject wsp:workspaceProjects) {
			
			
			if (wsp.getName().equals(projectName)) {
				found=true;
			    ProjectConfiguration p = wsp.getConfiguration();
			    File projFile=wsp.getProjectFile();
			    if(p==null){
			        Document d =null;
			        try{
			            d= domConverter
			                    .readXML(new FileInputStream(
			                            projFile));
			            // get project version
			            p = (ProjectConfiguration) domCodec
			                    .readDocument(d);
			        }catch(Exception e){
			            e.printStackTrace();
			            // try to read project config version an check if it is from a newer Spr
			            // fixes Bug ID0058
			            if(d!=null){
			                Element projElem=d.getDocumentElement();
			                if(projElem!=null && projElem.getTagName().equals("ProjectConfiguration")){

			                    String attrProjVers=projElem.getAttribute("version");
			                    if(attrProjVers!=null){
			                        try {

			                            Version projVers=Version.parseString(attrProjVers);

			                            Version sprProjVers=Version.parseString(ProjectManager.PROJECT_VERSION);
			                            int versComp=sprProjVers.compareTo(projVers);
			                            if(versComp<0){
			                                // throw exception with user information to update currently running Spr
			                                throw new SpeechRecorderException("Project configuration of project \""+projectName+"\" could not be loaded.\nIt was configured with a newer version of Speechrecorder.\n(Project configuration version "+projVers+">"+sprProjVers+")\nPlease update Speechrecorder to newest version. (Help->Check for updates)");
			                            }
			                        } catch (ParserException e1) {
			                            // XML config must be broken
			                            e.printStackTrace();
			                        }
			                    }
			                }
			            }
			           // In all other cases throw exception
			            throw new SpeechRecorderException("Error loading project configuration from file "+projFile+":\n"+e.getLocalizedMessage());
			        }
			    }
			    try {
			    	projectManager.setProjectURL(projFile.toURI().toURL());

			    	File projectDir=new File(defWorkspaceDir, p.getName());
			    	URI projectURI=projectDir.toURI();
			    	// convert special chars, e.g. Umlaute
			    	String projURLStr=projectURI.toASCIIString();
			    	URL projectContext=new URL(projURLStr);

			    	ConfigHelper.applyLegacyToStrictConversions(p);
			    	projectManager.setProjectContext(projectContext);
			    	canceled=projectManager.configure(p);
			    	projectManager.setProjectConfigurationSaved(true);
			    	break;
			    } catch (IOException | DOMConverterException e) {
			    	throw new SpeechRecorderException("Could not configure project: "+e.getMessage(),e);
			    }
			}
		}
		if(!found) {
			throw new WorkspaceException("Project "+projectName+" not found!");
		}
		
		return canceled;
	}

	
	private static void printUsageAndExit() {
		
		String usageStr="Usage:\n\n\t{SPECHRECORDER_BIN}\n\n\t{SPECHRECORDER_BIN} [PROJECT_FILE_URL]\n\n\t{SPECHRECORDER_BIN} open project PROJECT_NAME\n\n{SPECHRECORDER_BIN}: the SpeechRecorder application executable";
		String osDepExpl="";
		SystemHelper sh=SystemHelper.getInstance();
		if(sh.isMacOSX()) {
			osDepExpl=",\n\"/Applications/SpeechRecorder.app/Contents/MacOS/JavaAppLauncher\" for default installations on macOS.";
		}else if(sh.isWindows()) {
			osDepExpl=",\n\"C:\\Program Files\\BAS\\SpeechRecorder\\speechrecorder.exe\" for default installations on Windows.";
		}else if(sh.isLinux()) {
			osDepExpl=",\n\"/usr/local/bin/speechrecorder\" on Linux";
		}
		usageStr=usageStr.concat(osDepExpl);
		System.out.println(usageStr);
		Console console=System.console();
		if(console==null) {
			JOptionPane.showMessageDialog(null, usageStr,
					"Usage", JOptionPane.ERROR_MESSAGE);
			
		}
		System.exit(-1);
	}
	
	public static void main(String[] args) {

	    // TODO implement crash report uploader
//	    Thread.setDefaultUncaughtExceptionHandler(new UncaughtExceptionHandler() {
//            
//            @Override
//            public void uncaughtException(Thread t, Throwable e) {
//                System.err.println("Uncaught exception: "+e+" in thread "+t);
//                e.printStackTrace(System.err);
//            }
//        });
		String projectName=null;
		String projectFileURL = null;
		String[] params;
		String user = null;
		String password = "";
		OptionParser op = new OptionParser();
		
		op.addOption("h");
		op.addOption("u", "");
		op.addOption("p", "");
		op.addOption("s", "");
		try {
			op.parse(args);
		} catch (OptionParserException e) {
			System.err.println(e.getLocalizedMessage());
			JOptionPane.showMessageDialog(null, e.getLocalizedMessage(),
					"ERROR", JOptionPane.ERROR_MESSAGE);
			System.exit(-1);
		}
		Option[] options = op.getOptions();
		for (int i = 0; i < options.length; i++) {
			Option option=options[i];
			String optionName=option.getOptionName();
			if ("h".equals(optionName) && option.isSet()) {
				printUsageAndExit();
			} else if ("u".equals(optionName)) {
				user = option.getParam();
			} else if ("p".equals(optionName)) {
				password = option.getParam();
			} else if ("s".equals(optionName)) {
				String sessionCookie = option.getParam();
				
				// Cookie handling: method 1:
				// disable cookie handler
				// Java web start version 6 receives cookies from requests which do not require a authentication
				// e.g. to download jar files
				// the cookie handler then holds a JSESSIONID which belongs to a new unauthenticated session.
				// If Tomcat receives the two session id's and seems to use only the first and rejects the request
				// with HTTP 401 Unauthorized
				// Tomcat bug ?
				// so I reset the cookie handler and set the given (authenticated) cookie for each URLConnection.
				
				// method 2: (used now)
				// implemented own cookie handler to avoid applying to each URLConnection.
			
				// Setting the cookie handler requires all permissions (signed jars) !!
				CookieHandler.setDefault(new SessionCookieHandler(sessionCookie));
				
//				// method 3:
//				// jar files download requires authentication as well
//				// add cookie to default cookie handler
//				SimpleCookie sc=new SimpleCookie(sessionCookie);
//				URI uri;
//				try {
//					uri = new URI(sc.getProperty("path"));
//					CookieHandler ch=CookieHandler.getDefault();
//					ch.put(uri, sc.getResponseHeaders());
//					
//				} catch (URISyntaxException e) {
//					// TODO Auto-generated catch block
//					e.printStackTrace();
//				} catch (IOException e) {
//					// TODO Auto-generated catch block
//					e.printStackTrace();
//				}
			}
		}
		params = op.getParams();

		if (params.length > 3) {
			SpeechRecorder.printUsageAndExit();
		}
		if (params.length == 1) {
			projectFileURL = params[0];
		}else if(params.length==2){
			printUsageAndExit();
		}else if(params.length==3){
			String cmd=params[0];
			String object=params[1];
			if("open".equals(cmd)) {
				if("project".equals(object)) {
					projectName=params[2];
				}else {
					printUsageAndExit();
				}
			}else {
				printUsageAndExit();
			}
		}
		
		// "Delegate" to AWT event thread for Swing thread safety
		
		final String fprojectName = projectName;
		final String fprojectFileURL = projectFileURL;
		final String fuser = user;
		final String fpassword = password;

		// for thread safety the constructor is called in the AWT event thread from the main method
        // if we really want to do configuring,plugin loading,etc in the background we need to
        // separate the jobs which call no Swing methods and could be run safety in the background by the main thread.
		Runnable doStart=new Runnable(){
		    public void run() {
		        try {
		        	// Dirty workaround for the problem that the default JavaSound implementation
		        	// does not initialize Windows COM correctly before using DirectSound
		        	// This causes device names truncated to 32 characters
		        	
		        	// 
		        	SystemHelper sh=SystemHelper.getInstance();
		        	if(sh.isWindows()) {
		        		try {
		        			Class<?> dsSndMpCls=Class.forName("ips.audio.ds.DSMixerProvider");

		        			if(dsSndMpCls!=null) {
		        				Constructor<?> dsSndMpCstr=dsSndMpCls.getConstructor();
		        				Object dsSndMpObj=dsSndMpCstr.newInstance();
		        				Method dsSndMpInit=dsSndMpCls.getMethod("init");
		        				dsSndMpInit.invoke(dsSndMpObj);
		        			}
		        		}catch(ClassNotFoundException cnfe) {
		        			// we can't do anything here
		        			// just proceed ...
		        		}
		        	}
		            new SpeechRecorder(fprojectFileURL,fprojectName, fuser, fpassword);
		        } catch (Exception e) {
		            e.printStackTrace();
		            JOptionPane.showMessageDialog(null, e.getLocalizedMessage(),
		                    "ERROR", JOptionPane.ERROR_MESSAGE);
		            System.exit(-1);
		        }
		    }
		};
		SwingUtilities.invokeLater(doStart);
	}

	public void importProject(File file) throws PluginLoadingException,
			AudioControllerException, IOException, InstantiationException,
			IllegalAccessException, ClassNotFoundException,
			DOMConverterException, WorkspaceException, DOMCodecException, URISyntaxException, RecscriptManagerException, StorageManagerException, SpeechRecorderException, ProjectManagerException{

		// first search project main directory
		ZipFile zipFile=new ZipFile(file);
		Enumeration<? extends ZipEntry> entries = zipFile.entries();

		ZipEntry firstEntry = (ZipEntry) entries.nextElement();
		// some versions of speechrecorder (respectively the ZipPacker class ) wrote
		// no base directory entry (and in general no non-empty directory entries) 
		// so we have to check for the parent dir first to get the project name
		String firstFilename=firstEntry.getName();
		File firstFile=new File(firstFilename);
		File parentFile=firstFile;
		
		while (parentFile.getParentFile() != null) {
		    parentFile=parentFile.getParentFile();
        }
		
		String projectDirName = parentFile.getName();
		File projectDirFile = new File(projectDirName);
		String projectName = projectDirFile.getName();
		File projectDir = new File(defWorkspaceDir, projectDirName);
		// found project directory
		if (projectDir.exists()) {
			speechRecorderUI.displayError(uiString.getString("MenuItemImport"),
					"Project " + projectDirName + " already exists !");
			zipFile.close();
			return;
		}
		zipFile.close();
		
		// Open again to check project file
		zipFile=new ZipFile(file);
		File projectCfgPrototypeFile=new File(projectDirFile,projectName+PROJECT_FILE_EXTENSION);
		entries = zipFile.entries();
		ZipEntry entry=null;
		boolean hasProjectCfgFile=false;
		while(entries.hasMoreElements()){
		    entry=entries.nextElement();
		    String entryFn=entry.getName();
		    File entryF=new File(entryFn);
		    if(projectCfgPrototypeFile.equals(entryF)){
		        hasProjectCfgFile=true;
		        break;
		    }
		}
		zipFile.close();
		
		if(!hasProjectCfgFile){
		    throw new IOException("Speechrecorder project file not found in ZIP file.\nThis does not look like a Speechrecorder export.");
		}
		
		UnzipWorker unzipWorker=new UnzipWorker();
		unzipWorker.setSourceZipFile(file);
		unzipWorker.setTrgDir(defWorkspaceDir);
	
		 JProgressDialogPanel progressDialog=new JProgressDialogPanel(unzipWorker,"Import project","Importing...");
        try {
            unzipWorker.open();
        } catch (WorkerException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        unzipWorker.start();
        
        Object val=progressDialog.showDialog(speechRecorderUI);
       
        try {
            unzipWorker.close();
        } catch (WorkerException e) {
           
            e.printStackTrace();
        }
       
        // TODO what is best practice to ask a worker for final status
        State s=unzipWorker.getStatus();
        if(!State.DONE.equals(s)){
        	if(State.CANCELLED.equals(s) || val.equals(JProgressDialogPanel.CANCEL_OPTION)){
        		JOptionPane.showMessageDialog(speechRecorderUI, "Project import canceled.");
        	}else{
        		LocalizableMessage lm=unzipWorker.getProgressStatus().getMessage();
        		JOptionPane.showMessageDialog(speechRecorderUI, lm.localize(),"Project import error.",JOptionPane.ERROR_MESSAGE);
        	}
        }else{
        	workspaceProjects=workspaceManager.scanWorkspace();
        	speechRecorderUI.setWorkspaceProjects(workspaceProjects);
        	openProject(projectName);
        }
	}
	
//	private Bundle buildBundle() throws IOException, UnsupportedAudioFileException{
//	    if(promptAutoAnnotator!=null){
//	        promptAutoAnnotator.setPromptText(null);
//	    }
//	    if(templateAutoAnnotator!=null){
//	        templateAutoAnnotator.setTemplateText(null);
//	    }
//	    // build Bundle
//        Bundle bundle=new Bundle();
//        bundle.setSession(annotationSession);
//        String targetRootFn=storageManager.getNewRootFileName();
//        bundle.setName(targetRootFn);
//        File[] recFiles;
//        try {
//            recFiles=storageManager.getCurrentItemRecordingFiles();
//            if(recFiles!=null && recFiles.length>0){
//                File masterFile=recFiles[0];
//                bundle.setAnnotates(masterFile.getName());
//                ConvenienceFileAudioSource cfas=new ConvenienceFileAudioSource(masterFile);
//                AudioFormat af=cfas.getFormat();
//                long fl=cfas.getFrameLength();
//                bundle.setSampleRate(af.getSampleRate());
//                bundle.setFrameLength(fl);
//            }
//            List<String> sigPathes=new ArrayList<String>();
//            for(File rf:recFiles){
//                sigPathes.add(rf.getAbsolutePath());
//            }
//            bundle.setSignalpaths(sigPathes);
//            
//            String prDescr=promptItem.getDescription();
//            if(promptAutoAnnotator!=null){
//                promptAutoAnnotator.setPromptText(prDescr);
//            }
//            
//            List<Mediaitem> mis=promptItem.getMediaitems();
//            for(Mediaitem mi:mis){
//               if(mi.getAnnotationTemplate()){
//                   if(templateAutoAnnotator!=null){
//                       templateAutoAnnotator.setTemplateText(mi.getText());
//                   }
//                   break;
//               }
//            }
//        } catch (AudioSourceException e) {
//			// TODO Auto-generated catch block
//			e.printStackTrace();
//        }
//       
//        return bundle;
//	}
//	
	


    /* (non-Javadoc)
     * @see ipsk.util.apps.UpdateManagerListener#update(ipsk.util.apps.UpdateManagerEvent)
     */
    @Override
    public void update(UpdateManagerEvent event) {
        if(event instanceof UpdateAvailableEvent){
            UpdateAvailableEvent uae=(UpdateAvailableEvent)event;
            Change.Priority priority=uae.getPriority();
            Change.Priority informPriority=null;
            if(updateManagerConfig!=null){
                informPriority=updateManagerConfig.getInformOnUpdatePriority();
            }
            if(informPriority==null){
                informPriority=Change.Priority.STRONGLY_RECOMMENDED;
            }
            int cp=informPriority.compareTo(priority);
            if(cp<=0){
                Runnable checkUpdatesRunnable=new Runnable() {
                    @Override
                    public void run() {
                        speechRecorderUI.doCheckUpdates();
                    }
                };
                SwingUtilities.invokeLater(checkUpdatesRunnable);
            }
        }
    }

	public List<AutoAnnotationServiceDescriptor> getAutoAnnotatorServiceDescriptors() throws IOException {
		return autoAnnotatorPluginManager.getAutoAnnotatorServiceDescriptors();
	}

	

    public WorkspaceManager getWorkspaceManager() {
        return workspaceManager;
    }
    
    public List<BundleAnnotationPersistorServiceDescriptor> getBundleAnnotationPersistorServiceDescriptors() {
        return availableBundleAnnotationServiceDescriptors;
    }
 


    /* (non-Javadoc)
     * @see ipsk.apps.speechrecorder.project.ProjectManagerListener#update(ipsk.apps.speechrecorder.project.ProjectManagerEvent)
     */
	@Override
	public void update(ProjectManagerEvent e) {
	    if(e instanceof ProjectManagerProjectOpenedEvent){
           
            workspaceManager.lock(e.getProjectName());
        }else if(e instanceof ProjectManagerProjectClosedEvent){
	        
	        if (!projectManager.isUsingUploadCache()){
	            try {
	                workspaceProjects=workspaceManager.scanWorkspace();
	            } catch (WorkspaceException e1) {
	                // TODO Auto-generated catch block
	                e1.printStackTrace();
	            }
	            speechRecorderUI.setWorkspaceProjects(workspaceProjects);
	        }
//	        if(project!=null){
	        workspaceManager.unlock(e.getProjectName());
//	        }
	    }else if (e instanceof ProjectManagerProjectReadyForShutdownEvent ){
	     // shutdown thread is not AWT event thread !!
	        System.exit(1);
	    }

	}


	public void closeActiveProject() throws SpeechRecorderException {
		try {
			projectManager.close();
		} catch (AudioControllerException | StorageManagerException | WorkspaceException | SpeechRecorderException e) {
			throw new SpeechRecorderException(e);
		}
	}
	
	

}
