package ipsk.webapps;

import java.security.SecureRandom;
import java.util.Base64;
import java.util.HashSet;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

/**
 * Helper class to generate and store secure request tokens and validate requests against them.
 * The secure request tokens should prevent Cross Site Request Forgery (CSRF) attacks. 
 * @author klausj
 *
 */
public class SecureRequestTokenProvider {

	public static final int SECURE_REQUEST_TOKEN_BYTE_LEN=64;
	
	// Property name for secure request token in HTML forms. 
	public static final String SECURE_REQUEST_TOKEN_NAME="_secureRequestToken";
	
	// Key to retrieve the set of secure tokens in the users session scope.
	public static final String SECURE_REQUEST_TOKEN_ATTR_KEY=SecureRequestTokenProvider.class.getName()+"."+SECURE_REQUEST_TOKEN_NAME;
	
	/**
	 * Returns session secure token.
	 * If the secure does not exist yet, it is created.
	 * @param req the HTTP request
	 * @return the secure random token
	 */
	public static String generateSecureRequestToken(HttpServletRequest req) {
		return SecureRequestTokenProvider.generateSecureRequestToken(req,true);
	}
	
	
	/**
	 * Returns session secure token.
	 * @param req the HTTP request
	 * @param create if true and the token does not exist, create it
	 * @return the secure random token
	 */
	public static String generateSecureRequestToken(HttpServletRequest req,boolean create) {
		String secureToken=null;
		HttpSession sess=req.getSession();
		if(sess!=null) {
			Object secToksSetObj=sess.getAttribute(SECURE_REQUEST_TOKEN_ATTR_KEY);
			if(secToksSetObj!=null) {
				if(secToksSetObj instanceof String) {
					secureToken=(String)secToksSetObj;
				}
			}	
			if(secureToken==null) {
				SecureRandom sr=new SecureRandom();
				byte[] bytes=new byte[SECURE_REQUEST_TOKEN_BYTE_LEN];
				sr.nextBytes(bytes);
				Base64.Encoder b64Enc=Base64.getUrlEncoder().withoutPadding();
				secureToken=b64Enc.encodeToString(bytes);
				sess.setAttribute(SECURE_REQUEST_TOKEN_ATTR_KEY, secureToken);
			}
		}
		return secureToken;
	}
	
	
	/**
	 * Check secure request token.
	 * Checks if the token is set and matches.
	 * 
	 * @param req the HTTP request
	 * @return true if the token was available
	 */
	private boolean _checkSecureRequestToken(HttpServletRequest req) {
		String secReqTokSent=req.getParameter(SECURE_REQUEST_TOKEN_NAME);
		if(secReqTokSent!=null) {
			String secureToken=generateSecureRequestToken(req, false);
			if(secureToken!=null) {
				return secureToken.equals(secReqTokSent);
			}
		}
		// failed to validate secure token
		return false;
	}
	
	/**
	 * Check if secure request token of the request is set and is valid.
	 * @param req the HTTP request
	 * @throws ControllerException if the token was not set or not found in the set of generated tokens 
	 */
	public void checkSecureRequestToken(HttpServletRequest req) throws ControllerException {
		boolean res=_checkSecureRequestToken(req);
		if(!res) {
			throw new ControllerException("Invalid request!");
		}
	}
	
}
