package ipsk.webapps.db.speech;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import javax.servlet.Servlet;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.annotation.MultipartConfig;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.Part;

import ipsk.db.speech.Project;
import ipsk.db.speech.UserRoleId;
import ipsk.io.StreamCopy;
import ipsk.text.StringTokenizer;
import ipsk.text.quoting.QuoteParser;
import ipsk.text.quoting.TextPart;
import ipsk.webapps.ControllerException;
import ipsk.webapps.PermissionDeniedException;
import ipsk.webapps.ProcessResult;



@MultipartConfig
public abstract class BasicResourceFilesystemController extends BasicWikiSpeechController<Project> {
	public final static boolean DEBUG=true;
	
	private volatile static Charset cs;
	private static  String CS_NAME="UTF-8";
	protected static File baseDir=null;

	private boolean upDirPossible=false;

	
	protected abstract Path resourcesDirPath(HttpServletRequest req);
	
	public boolean isUpDirPossible() {
		return upDirPossible;
	}

	public void setUpDirPossible(boolean upDirPossible) {
		this.upDirPossible = upDirPossible;
	}
	
	private Path currentPath;
	

	
	public Path getCurrentPath() {
		return currentPath;
	}

	public void setCurrentPath(Path currentPath) {
		this.currentPath = currentPath;
	}

	
	public String getCurrentURIDirPath() {
		return buildPathString(getCurrentPath());
	}
	
	private String currentFilename;
	

	public String getCurrentFilename() {
		return currentFilename;
	}

	public void setCurrentFilename(String currentFilename) {
		this.currentFilename = currentFilename;
	}

	public static class FileItem {
		private File file;
		public File getFile() {
			return file;
		}
		public boolean isSelected() {
			return selected;
		}
		private boolean selected;

		public FileItem(File file) {
			this(file, false);
		}
		public FileItem(File file, boolean selected) {
			super();
			this.file = file;
			this.selected = selected;
		}
		
		public boolean isFileOrEmptyDir() {
			return (! (file.isDirectory() && file.listFiles().length>0)); 
				
		}
		
		public static FileItem[] fromFilesArray(File[] fileArr) {
			FileItem[] itsArr=new FileItem[fileArr.length];
			for(int i=0;i<fileArr.length;i++) {
				itsArr[i]=new FileItem(fileArr[i]);
			}
			return itsArr;
		}
	}
	
	private FileItem[] fileItems=null;

	public FileItem[] getFileItems() {
		return fileItems;
	}

	public void setFileItems(FileItem[] fileItems) {
		this.fileItems = fileItems;
	}
	
	private boolean deleteSelectedRequest=false;

//	private File[] files=null;
//	
//	public File[] getFiles() {
//		return files;
//	}
//
//	public void setFiles(File[] files) {
//		this.files = files;
//	}
	
	public boolean isDeleteSelectedRequest() {
		return deleteSelectedRequest;
	}

	public void setDeleteSelectedRequest(boolean deleteSelectedRequest) {
		this.deleteSelectedRequest = deleteSelectedRequest;
	}

	private ProcessResult processResult;
	
	public ProcessResult getProcessResult() {
		return processResult;
	}

	public BasicResourceFilesystemController() {
		super("WebSpeechDBPU",Project.class);
		cs=Charset.forName(CS_NAME);
	}

	@Override
	public ProcessResult process(HttpServletRequest request, HttpServletResponse response) throws ControllerException {
		processRequest(request);
		return getProcessResult();
	}

	@Override
	public ProcessResult process(HttpServletRequest request, HttpServletResponse response, Servlet servlet)
			throws ControllerException {
		processRequest(request);
		return getProcessResult();
	}

	@Override
	public void setServletContext(ServletContext servletContext) {
		// TODO Auto-generated method stub
		
	}
	
	private boolean isAbsolutePath(String path) {
		return path.startsWith("/") || path.startsWith(File.separator);
	}

	private Path securePath(String path) throws PermissionDeniedException{
		if(isAbsolutePath(path)) {
			throw new PermissionDeniedException();
		}
		Path insecurePath=Paths.get(path);
		return securePath(insecurePath);
	}
	private Path securePath(Path insecurePath) throws PermissionDeniedException{
		// allow only relative paths ...
		
		if(insecurePath.isAbsolute()){
			// Does not work on Windows for paths like "\foo" and "/foo"
			throw new PermissionDeniedException();
		}
		
		
		Path insecureNormPath=insecurePath.normalize();
		// ... which are normalized
		
		// However we double check, that there are no path elements to parent directories and that is not absolute
		int pLen=insecureNormPath.getNameCount();
		
		if(pLen>0){
			Path pNm=insecureNormPath.getName(0);
			if(pNm.startsWith("/") || pNm.startsWith(File.separator)){
				throw new PermissionDeniedException();
			}
		}
		
		for(int i=0;i<pLen;i++){
			Path pNm=insecureNormPath.getName(i);
			if(pNm.startsWith("..")){
				throw new PermissionDeniedException();
			}
		}
		return insecureNormPath;
	}
	
	private boolean isRoot(Path path){
		int nmCnt=path.getNameCount();
		boolean emptyPath=false;
		if(nmCnt==1){
			Path p=path.getName(0);
			String pNm=p.toString();
			emptyPath="".equals(pNm);
			
		}
		boolean isRoot=(nmCnt<1) || emptyPath;
		return isRoot;
	}
	
	private File currentDir(Path securePath,Path resourcesDir) throws PermissionDeniedException {	

		File dir=null;
		if(securePath!=null){
			Path reqPath=resourcesDir.resolve(securePath);
			Path reqDirPath=null;
			Path secureDirPath=null;
			
			if(Files.isDirectory(reqPath)) {
				dir=reqPath.toFile();
				secureDirPath=securePath;
			}else {
				
				Path dirPath=securePath.getParent();
				if(dirPath==null) {
					dirPath=Paths.get(".");
				}
				secureDirPath=securePath(dirPath);
				if(secureDirPath!=null) {
					reqDirPath=resourcesDir.resolve(secureDirPath);
					if(Files.isDirectory(reqDirPath)) {
						dir=reqDirPath.toFile();
					}
				}
			}
		}
		return dir;
	}
	
	
	private String buildPathString(Path path) {
		
		String pStr="";
		int nc=path.getNameCount();
		for(int i=0;i<nc;i++) {
			String fNmStr=path.getName(i).toString();
			if(fNmStr!=null && !fNmStr.isEmpty()) {
				pStr=pStr.concat(fNmStr);
				pStr=pStr.concat("/");
			}
		}
		return pStr;
	}
	
	
	@Override
	public void processRequest(HttpServletRequest req) throws ControllerException {
		ServletContext sCtx=req.getServletContext();
		setDeleteSelectedRequest(false);
		boolean secure=true;
		// Bug: If file is uploaded in top level dir, the user is able to go through whole filesystem !!
		Path resourcesDir= resourcesDirPath(req);
		if(resourcesDir==null) {
			throw new ControllerException("Resources directory path not available");
		}
		
		if(!Files.exists(resourcesDir)) {
			try {
				Files.createDirectories(resourcesDir);
			} catch (IOException e1) {
				throw new ControllerException("Could not create resources directory: "+resourcesDir+": Error:"+e1.getMessage());
			}
		}
		
		if(secure && req.isUserInRole(UserRoleId.RoleName.PROJECT_ADMIN.name())){
			
			String cmd=processCommand(req, new String[] {"delete_request","delete"});
			String method=req.getMethod();
			
			if("POST".equalsIgnoreCase(method)){
				secureRequestTokenProvider.checkSecureRequestToken(req);
				String path=null;
				Part pathPart;
				Path securePath=null;
				String filename=null;

				String ctType=req.getContentType();
				if(ctType.toLowerCase(Locale.US).startsWith("multipart/form-data")){

					Part fPart=null;
					try {

						Part cmdPart=req.getPart("_cmd");
						if(cmdPart!=null){
							InputStream cmdIs=cmdPart.getInputStream();
							cmd=StreamCopy.readTextStream(cmdIs,cs);
						}

						if("upload".equals(cmd)) {

							pathPart = req.getPart("_path");
							if(pathPart!=null){
								InputStream pathIs=pathPart.getInputStream();
								path=StreamCopy.readTextStream(pathIs,cs);
								securePath=securePath(path);
							}
							fPart= req.getPart("file");

							//String filename=fPart.getSubmittedFileName(); // TODO Requires Servlet API 3.1.0 (Ubuntu 16.04 ?) 

							Collection<String> contentDispos=fPart.getHeaders("content-disposition");
							HashMap<String,List<String>> pMap=new HashMap<String, List<String>>();
							for(String cd:contentDispos){
								// parse quotes
								List<TextPart> textPartList=QuoteParser.parseText(cd,'"', '\\', false);
								// separate fields
								String[] fields=StringTokenizer.split(textPartList, ';', true);
								// first field is media type 

								// subsequent fields are parameters
								for(int i=1;i<fields.length;i++){
									String field=fields[i];
									int equInd=field.indexOf('=');
									if(equInd>0){
										// rfc2616: parameter names are case insensitive
										String key=field.substring(0, equInd).trim().toLowerCase(Locale.US);
										String val=field.substring(equInd+1).trim();
										List<String> paramVals=pMap.get(key);
										if(paramVals==null){
											paramVals=new ArrayList<String>();
											pMap.put(key,paramVals);
										}
										paramVals.add(val);
									}
								}
							}

							List<String> filenames=pMap.get("filename");
							if(filenames!=null && filenames.size()==1){
								String qtFilename=filenames.get(0);
								// Remove surrounding double quotes
								qtFilename=qtFilename.replaceAll("^\"","");
								filename=qtFilename.replaceAll("\"$","");

							}
						}
					} catch (IOException | ServletException e) {

						e.printStackTrace();
						throw new ControllerException("could not get path parameter from multipart upload: "+e.getMessage(),e);
					}


					if(securePath!=null){
						Path reqDirPath=resourcesDir.resolve(securePath);
						Path secureFilepath=securePath(filename);
						Path reqUploadPath=reqDirPath.resolve(secureFilepath);
						if(fPart!=null){
							InputStream upStr;
							try {
								upStr = fPart.getInputStream();
								StreamCopy.copy(upStr, reqUploadPath.toFile(),false);
								setFileItems(FileItem.fromFilesArray(reqDirPath.toFile().listFiles()));
								setUpDirPossible(!isRoot(securePath));
								setCurrentPath(securePath);
								processResult=new ProcessResult(ProcessResult.Type.SUCCESS);
							} catch (IOException e) {

								e.printStackTrace();
								processResult=new ProcessResult(ProcessResult.Type.ERROR);

							}

						}
					}

				}else {

					if("mkdir".equals(cmd)) {
						// TODO no error handling for now
						Path currPath=null;
						String[] pathParams=req.getParameterValues("_path");
						if(pathParams!=null && pathParams.length==1){
							securePath=securePath(pathParams[0]);
							if(securePath!=null){
								currPath=resourcesDir.resolve(securePath);
								String dirname=req.getParameter("dirname");
								
								if(dirname!=null && !"".equals(dirname)) {
									String trimmedDirName=dirname.trim();
									if(trimmedDirName.startsWith("/") || trimmedDirName.startsWith(File.separator)) {
										throw new PermissionDeniedException();
									}
									Path newDirSecurePath=securePath(dirname);
									if(newDirSecurePath.getNameCount()==1) {
										Path newDir=currPath.resolve(newDirSecurePath);
										try {
											Files.createDirectory(newDir);
										} catch (IOException e) {
											// TODO
											e.printStackTrace();
										}
									}
								}
							}
						}else {
							securePath=securePath("");
						}
						FileItem[] fileItems=null;
						if(currPath!=null) {
							File[] files=currPath.toFile().listFiles();
							fileItems=FileItem.fromFilesArray(files);
						}
						setFileItems(fileItems);
						setUpDirPossible(!isRoot(securePath));
						setCurrentPath(securePath);
					}else if("delete_request".equals(cmd)) {
						Path currPath=null;
						String[] pathParams=req.getParameterValues("_path");
						if(pathParams!=null && pathParams.length==1){
							securePath=securePath(pathParams[0]);
							currPath=resourcesDir.resolve(securePath);
						}else {
							securePath=securePath(getCurrentPath());
						}
						
						File currDir=currentDir(securePath, resourcesDir);
						
						Set<String> selectedFNms=new HashSet<>();
						String[] selectedFileDirNms=req.getParameterValues("_select");
						if(selectedFileDirNms!=null) {
							for(String selectedFileDirNm:selectedFileDirNms) {
								selectedFNms.add(selectedFileDirNm);
							}
						}
						setDeleteSelectedRequest(selectedFNms.size()>0);
						FileItem[] fileItems=null;
						if(currDir!=null) {
							File[] files=currDir.listFiles();
							FileItem[] fits=new FileItem[files.length];
							for(int i=0;i<files.length;i++) {
								File f=files[i];
								boolean sel=selectedFNms.contains(f.getName());
								fits[i]=new FileItem(files[i],sel);
							}
							fileItems=fits;
						}
						setFileItems(fileItems);
						setUpDirPossible(!isRoot(securePath));
						setCurrentPath(securePath);
					}else if("delete".equals(cmd)) {

						String[] pathParams=req.getParameterValues("_path");
						if(pathParams!=null && pathParams.length==1){
							securePath=securePath(pathParams[0]);
						}else {
							securePath=securePath(getCurrentPath());
						}

						File currDir=currentDir(securePath, resourcesDir);

						
						String[] selectedFileDirNms=req.getParameterValues("_select");
						if(selectedFileDirNms!=null) {
							Path currPath=resourcesDir.resolve(securePath);
							for(String selectedFileDirNm:selectedFileDirNms) {
								Path fileDirNmToDeletePath=securePath(selectedFileDirNm);
								Path fileDirToDeletePath=currPath.resolve(fileDirNmToDeletePath);
								try {
									Files.delete(fileDirToDeletePath);
								} catch (IOException e) {
									// TODO Auto-generated catch block
									e.printStackTrace();
								}
							}
						}
						FileItem[] fileItems=null;
						if(currDir!=null) {
							File[] files=currDir.listFiles();
							fileItems=FileItem.fromFilesArray(files);
						}
						setFileItems(fileItems);
						setDeleteSelectedRequest(false);
						setUpDirPossible(!isRoot(securePath));
						setCurrentPath(securePath);
					}
				}

				}else if("GET".equalsIgnoreCase(method)){
					cmd=req.getParameter("_cmd");
					String[] path=req.getParameterValues("_path");
					Path securePath=null;
					String filename=null;
					if(path!=null && path.length==1){
						securePath=securePath(path[0]);
					}else {
						securePath=securePath("");
					}
					if(securePath!=null){
						Path reqPath=resourcesDir.resolve(securePath);
						Path reqDirPath=null;
						Path secureDirPath=null;
						File dir=null;
					FileItem[] dirFileList=null;
					if(Files.isDirectory(reqPath)) {
						dir=reqPath.toFile();
						secureDirPath=securePath;
					}else {
						Path fileNamePath=securePath.getFileName();
						filename=fileNamePath.toString();
						Path dirPath=securePath.getParent();
						if(dirPath==null) {
							dirPath=Paths.get(".");
						}
						secureDirPath=securePath(dirPath);
						if(secureDirPath!=null) {
							reqDirPath=resourcesDir.resolve(secureDirPath);
							if(Files.isDirectory(reqDirPath)) {
								dir=reqDirPath.toFile();
							}
						}
					}
					if(dir!=null) {
						File[] dirList=dir.listFiles();
						dirFileList=FileItem.fromFilesArray(dirList);
					}
					setFileItems(dirFileList);
					
					setUpDirPossible(!isRoot(secureDirPath));
					setCurrentPath(securePath);
					setCurrentFilename(filename);
					processResult=new ProcessResult(ProcessResult.Type.SUCCESS);
				}
			}
		}else{
			throw new PermissionDeniedException();
		}

	}

	

}
