package ipsk.webapps.db.speech.ws;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.Files;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Stream;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.PersistenceException;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.HEAD;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;

import org.glassfish.jersey.media.multipart.FormDataParam;

import ipsk.audio.AudioFormatNotSupportedException;
import ipsk.audio.AudioSource;
import ipsk.audio.AudioSourceException;
import ipsk.audio.FileAudioSource;
import ipsk.audio.PluginChain;
import ipsk.audio.ThreadSafeAudioSystem;
import ipsk.audio.plugins.AppendPlugin;
import ipsk.db.speech.RecordingFile;
import ipsk.db.speech.Session;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.script.Recording;
import ipsk.db.speech.script.Script;
import ipsk.io.StreamCopy;
import ipsk.net.Utils;
import ipsk.persistence.EntityManagerProvider;
import ipsk.text.InvalidInputException;
import ipsk.webapps.EntityManagerFactoryInitializer;
import ipsk.webapps.PermissionDeniedException;
import ipsk.webapps.audio.DSPProcessor;
import ipsk.webapps.db.speech.ChunkedRecordingStates;
import ipsk.webapps.db.speech.ChunkedRecordingStates.ChunkedRecordingState;
import ipsk.webapps.db.speech.SessionController;
import ipsk.webapps.db.speech.WikiSpeechSecurityManager;

public class SessionRecfileResource extends BasicRecordingfileResource{

	public static String CHUNKS_TEMP_DIRECTORY_NAME=SessionRecfileResource.class.getName()+"_chunks";
	public static int CHUNK_LOCK_TIMEOUT_MINUTES=10;
	
	/*
	 * Use system temp dir for chunk uploads if problems appear using the application data dir (which might be on a network filesystem NFS/SMB).
	 * Drawback of system temp dir is that it is deleted on server stop of Tomcat. Uploaded chunks meght be lost.
	 */
	public static boolean USE_SYSTEM_TMP_DIR_FOR_CHUNKS=false;
	
	private DateFormat jsonDtFmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
	
	public SessionRecfileResource(String sessionIdStr) {
		super(sessionIdStr);
	}

	
	private List<RecordingFile> recordingFilesOfSessionOfRecording(EntityManager em,Session session,Recording recording){
		
		String recFilesOfSessionOfRecordingQueryStr="SELECT rf FROM RecordingFile AS rf,Recording AS r WHERE r = :recording AND rf MEMBER OF r.recordingFiles AND rf.session = :session";
		TypedQuery<RecordingFile> q = em.createQuery(recFilesOfSessionOfRecordingQueryStr,RecordingFile.class);
		q.setParameter("recording", recording);
		q.setParameter("session", session);
		return q.getResultList();
		
		// The following criteria query did not work
		
//		CriteriaBuilder cb = em.getCriteriaBuilder();
//		CriteriaQuery<RecordingFile> cq = cb.createQuery(RecordingFile.class);
//		
//		Root<RecordingFile> rrf = cq.from(RecordingFile.class);
//		Root<Recording> rr=cq.from(Recording.class);
//		
//		cq.select(rrf);
//		
//		Predicate rp=cb.equal(rr,recording);
//		Predicate rf=rrf.in(rr.get("recordingFiles"));
//		Predicate sp=cb.equal(rr.get("session"), session);
//		cq.where(cb.and(rp,rf,sp));
//		
//		TypedQuery<RecordingFile> rfq = em.createQuery(cq);
//		List<RecordingFile> rfList=rfq.getResultList();
//		return rfList;
	}
	
	
	@GET
	@Produces({ MediaType.APPLICATION_JSON })
	public Response getRecordingFileList(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req,@QueryParam("latestVersionsOnly") Boolean latestVersionsOnly) {
		return recordingFileList(sc, sec, req, latestVersionsOnly);
	}

	@HEAD
	@Path("/{itemcode}")
	@Consumes("*/*")
	@Produces("audio/wav")
	public Response headRecordingFiles(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, InputStream inputStream, @PathParam("itemcode") String itemCode) {
		int sessionId;
		try {
			sessionId = Integer.parseInt(sessionIdStr);
		} catch (NumberFormatException nfe) {
			sc.log("Session ID " + sessionIdStr + " could not be parsed as number ID: ", nfe);
			return Response.status(Status.BAD_REQUEST).build();
		}
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		EntityTransaction tx = em.getTransaction();
		tx.begin();
		Session session = em.find(Session.class, sessionId);
		if (session == null) {
			rollBackAndClose(em);
			return Response.status(Status.NOT_FOUND).build();
		}
		try {
			securityManager.checkReadPermission(req, session);
		} catch (PermissionDeniedException e3) {
			sc.log("No read permission for session ID: " + sessionId + " ", e3);
			rollBackAndClose(em);
			return Response.status(Status.FORBIDDEN).build();
		}

		Set<RecordingFile> recFiles = session.getRecordingFiles();

		return Response.noContent().header("X-Total-Count", recFiles.size()).build();
	}
	

	@GET
	@Path("/{itemcode}/{version}")	
	@Produces("audio/wav")
	public Response getRecFile(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, @PathParam("itemcode") String itemCode,
			@PathParam("version") String versionStr,@QueryParam("startFrame") Long startFrame,@QueryParam("frameLength") Long frameLength) {
		int sessionId;
		try {
			sessionId = Integer.parseInt(sessionIdStr);
		} catch (NumberFormatException nfe) {
			sc.log("Session ID " + sessionIdStr + " could not be parsed as number ID: ", nfe);
			return Response.status(Status.BAD_REQUEST).build();
		}
		int version;
		try {
			version = Integer.parseInt(versionStr);
		} catch (NumberFormatException nfe) {
			sc.log("Recording file version " + versionStr + " could not be parsed as number: ", nfe);
			return Response.status(Status.BAD_REQUEST).build();
		}

		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});
		try {
			EntityTransaction tx = em.getTransaction();
			tx.begin();
			Session session = em.find(Session.class, sessionId);
			if (session == null) {
				rollBackAndClose(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			try {
				securityManager.checkReadPermission(req, session);
			} catch (PermissionDeniedException e3) {
				sc.log("No read permission for session ID: " + sessionId + " ", e3);
				rollBackAndClose(em);
				return Response.status(Status.FORBIDDEN).build();
			}

			Set<RecordingFile> recFiles = session.getRecordingFiles();
			for (RecordingFile rf : recFiles) {
				Recording r = rf.getRecording();
				if (itemCode.equals(r.getItemcode())) {
					Integer rfVers = rf.getVersion();
					if (rfVers != null && rfVers == version && "WAVE".equals(rf.getFormat())) {
						// Found
						return recordingFileStreamResponse(em,rf,startFrame,frameLength);
					}
				}
			}
			return Response.status(Status.NOT_FOUND).build();
		} finally {
			close(em);
		}
	}
	
	@POST
	@Path("/{itemcodeOrUUID}")
	@Consumes({"audio/wav","audio/x-wav"})
	@Produces(MediaType.APPLICATION_JSON)
	public Response postRecfile(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, InputStream inputStream, @PathParam("itemcodeOrUUID") String itemCodeOrUUID) {
		Integer sessionId = null;
		UUID sessionUUID = null;
		try {
			// Try database ID
			sessionId = Integer.parseInt(sessionIdStr);
		} catch (NumberFormatException nfe) {
			// Try UUID
			try{
				sessionUUID=UUID.fromString(sessionIdStr);
			}catch(IllegalArgumentException iae){
				sc.log("Session ID " + sessionIdStr + " could not be parsed as DB number ID or UUID: ", nfe);
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		EntityTransaction tx = em.getTransaction();
		tx.begin();
		Session session=null;
		if(sessionId!=null){
			session = em.find(Session.class, sessionId);
		}else if(sessionUUID!=null){
			List<Session> sessionsListByUUID=sessionsByUUID(em, sessionUUID);
			int sessionsListByUUIDSize=sessionsListByUUID.size();
			if(sessionsListByUUIDSize==1){
				// OK found exactly one
				session=sessionsListByUUID.get(0);
			}else if(sessionsListByUUIDSize>1){
				// ambiguous
				rollBackAndClose(em);
				return Response.status(Status.INTERNAL_SERVER_ERROR).build();
			}
		}
		if (session == null) {
			// Session not found
			rollBackAndClose(em);
			return Response.status(Status.NOT_FOUND).build();
		}
		try {
			securityManager.checkMergePermission(req, session);
		} catch (PermissionDeniedException e3) {
			sc.log("No merge permission for session ID: " + sessionId + " ", e3);
			rollBackAndClose(em);
			return Response.status(Status.FORBIDDEN).build();
		}

		boolean typeSpeechRecorder=(session.getScript()!=null);
		String sessionDir = session.getStorageDirectoryURL();
		if (sessionDir == null) {
			
			String recsDir = sc.getInitParameter("recsDir");
			String recsDirURL = "file:" + recsDir;
			sessionDir = SessionController.createSessionFilePath(recsDirURL, session.getSessionId());
			session.setStorageDirectoryURL(sessionDir);
		}
		tx.commit();

		String speakerCode = "";
		Set<Speaker> spks = session.getSpeakers();
		if (!spks.isEmpty()) {
			speakerCode = spks.iterator().next().getCode();
		}

		// Integer recVersion=null;
		boolean overwrite = false; // default

		UUID rfUUId;
		
		Script recScript=session.getScript();
		if(recScript==null) {
			try {
				rfUUId=UUID.fromString(itemCodeOrUUID);
			}catch(IllegalArgumentException iae) {
				sc.log(iae.getMessage());
				return Response.status(Status.BAD_REQUEST).build();
			}
		}else {
			rfUUId=UUID.randomUUID();
		}
		String tempFilename = getClass().getName() + "_" + session.getSessionId() + "_" + itemCodeOrUUID + "_"
				+ rfUUId + ".wav";
		File tempAudioFile = new File(tempDir, tempFilename);

		try {
			StreamCopy.copy(inputStream, tempAudioFile, true);
		} catch (IOException e2) {
			sc.log("Could not copy POST input stream to temp audio file: ", e2);
			return Response.serverError().build();
		}

		AudioFileFormat aff = null;
		try {
			aff = ThreadSafeAudioSystem.getAudioFileFormat(tempAudioFile);
		} catch (UnsupportedAudioFileException | IOException e) {
			sc.log(e.getMessage());

			tempAudioFile.deleteOnExit();
			// hold the temporary files for debug purposes
			// tmpFile.delete();
			sc.log(e.getMessage());

			return Response.serverError().build();
		}

		// store meta data to DB

		if (aff != null) {
			sc.log("detected audio file format: " + aff);
		} else {
			sc.log("could not detect audio file format !");
		}
		RecordingFile recFile = new RecordingFile();
		recFile.setUuid(rfUUId.toString());
		recFile.setStatus(RecordingFile.Status.REGISTERED);
		recFile.setDate(new Date());
		// recFile.setSignalFile("file:" + f.getCanonicalPath());
		tempAudioFile.deleteOnExit();
		tx = em.getTransaction();
		tx.begin();
		
		Recording recording = null;
		if(recScript!=null) {
			Query q = em.createQuery(
					"SELECT r FROM Recording AS r WHERE  r.group.section.script = :recScript AND r.itemcode = :itemcode");
			q.setParameter("recScript", recScript);
			q.setParameter("itemcode", itemCodeOrUUID);
			Object srObj = q.getSingleResult();
			if (srObj == null) {
				throw new PersistenceException("Could not retrieve associated recording script element!");
			}
			if (!(srObj instanceof Recording)) {
				throw new PersistenceException("Expected type Recording!");
			}
			recording = (Recording) srObj;
			recFile.setRecording(recording);
		}
		recFile.setSession(session);

		recFile.setDate(new Date());
		String extension = "unknown";
		if (aff != null) {
			AudioFileFormat.Type affType = aff.getType();
			extension = affType.getExtension();
			applyAudioFileFormat(recFile, aff);
		}

		List<RecordingFile> rfs = null;
		if (recording != null) {
			TypedQuery<RecordingFile> q = em.createQuery(
					"SELECT rf FROM RecordingFile AS rf,Recording AS r WHERE r = :recording AND rf MEMBER OF r.recordingFiles AND rf.session = :session",RecordingFile.class);
			q.setParameter("recording", recording);
			q.setParameter("session", session);
			rfs = q.getResultList();
		}
		// String signalFile=null;
		
		if(!BasicRecordingfileResource.speakerCodeValid(speakerCode)) {
			rollBackAndClose(em);
			sc.log("Invalid input");
			return Response.serverError().build();
		}

		if(!BasicRecordingfileResource.itemCodeValid(itemCodeOrUUID)) {
			rollBackAndClose(em);
			sc.log("Invalid input");
			return Response.serverError().build();
		}

		//try {
		if (overwrite) {
			// overwrite
			if (rfs == null || rfs.size() == 0) {

				String signalFilePathStr= createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode, itemCodeOrUUID, extension, recFile, overwrite);
				recFile.setSignalFile(signalFilePathStr);
				try {
					securityManager.checkPersistPermission(req, recFile);
				} catch (PermissionDeniedException e) {
					sc.log("Permission denied to persist recording file entity: ", e);
					rollBackAndClose(em);
					return Response.status(Status.FORBIDDEN).build();
				}
				em.persist(recFile);
				recFile.setSession(session);
				session.getRecordingFiles().add(recFile);
			} else {
				// Hmm. Not really correct.
				RecordingFile rf = (RecordingFile) rfs.get(0);
				Integer version = rf.getVersion();
				if (version != null) {
					recFile.setVersion(version++);
				} else {
					recFile.setVersion(0);
				}
				recFile.setRecordingFileId(rf.getRecordingFileId());

				recFile.setSignalFile(
						createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode, itemCodeOrUUID, extension, recFile, overwrite));
				// overwrite by merge
				em.merge(recFile);
			}
		} else {
			int version = 0;
			if (rfs != null) {
				for (RecordingFile rf : rfs) {
					Integer sVersion = rf.getVersion();
					if (sVersion != null) {
						if (sVersion >= version)
							version = sVersion + 1;
					}
				}
			}
			recFile.setVersion(version);
			recFile.setSignalFile(
					createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode, itemCodeOrUUID, extension, recFile, overwrite));
			em.persist(recFile);
			recFile.setSession(session);
			session.getRecordingFiles().add(recFile);
		}

//		} catch (InvalidInputException iie) {
//			rollBackAndClose(em);
//			sc.log("Invalid input exception " + iie.getMessage());
//			return Response.serverError().build();
//		}

		
		

//		if (recFile == null) {
//			rollBackAndClose(em);
//			tempAudioFile.delete();
//			sc.log("Could not generate database entry !");
//			return Response.serverError().build();
//		}
		URL signalFile = null;
		try {
			signalFile = new URL(recFile.getSignalFile());
		} catch (MalformedURLException e1) {
			sc.log("Malformed URL: ", e1);
		}

		File storeFile = null;
		if (signalFile.getProtocol().equalsIgnoreCase("file")) {
			// copy File
			FileInputStream fis = null;
			try {
				storeFile = new File(signalFile.getFile());
				fis = new FileInputStream(tempAudioFile);
				storeToFile(sc, fis, storeFile, false);
				fis.close();
				recFile.setStatus(RecordingFile.Status.RECORDED);
				// tempAudioFile.delete();
			} catch (IOException ioe) {
				rollBackAndClose(em);
				// throw ioe;
				sc.log("IO Exception storing audio file: " + ioe);
				return Response.serverError().build();

			} finally {
				if (fis != null) {
					try {
						fis.close();
					} catch (IOException e) {
						rollBackAndClose(em);
						sc.log("Could not close audio file !");
						return Response.serverError().build();
					}
				}
			}

			// DSPProcessor.notifyThread();
		} else {
			// tempAudioFile.delete();
			rollBackAndClose(em);
			sc.log("Cannot store URL " + signalFile + "\nOnly protocol file: is supported");
			return Response.serverError().build();
		}
		try {
			// recFileContr.commit();
			tx.commit();
			
			// Invalidate missing recording items cache
			if(sessionId!=null) {
				SessionController.invalidateCachedValues(sc, sessionId);
			}
			
			// notify the DSP processor about new recording file
			DSPProcessor.notifyThread();
			// JPA/DB operations finished
			em.close();
			// Cleanup temp file
			tempAudioFile.delete();
			// empty JSON to make Angular HttpClient happy
			return Response.ok("{}").build();
		} catch (Exception e) {
			rollBackAndClose(em);
			sc.log("Could not store recording file metadata to database !");
			return Response.serverError().build();
		} finally {

		}

	}
	
	
//	private static synchronized java.nio.file.Path chunksBaseDirectory(HttpServletRequest req) throws IOException {
//		ServletContext ctx=req.getServletContext();
//		Object chunksTempDirPathObj=ctx.getAttribute(CHUNKS_TEMP_DIRECTORY_NAME_KEY);
//		java.nio.file.Path chunksTempDirPath=null;
//		if(chunksTempDirPathObj==null) {
//			chunksTempDirPath=Files.createTempDirectory(CHUNKS_TEMP_DIRECTORY_NAME);
//			ctx.setAttribute(CHUNKS_TEMP_DIRECTORY_NAME_KEY, chunksTempDirPath);
//		}else {
//			if(chunksTempDirPathObj instanceof java.nio.file.Path) {
//				chunksTempDirPath=(java.nio.file.Path)chunksTempDirPathObj;
//			}else {
//				String errMsg="ERROR: Application attribute "+CHUNKS_TEMP_DIRECTORY_NAME_KEY+" not of expected type "+java.nio.file.Path.class.getName();
//				throw new IOException(errMsg);
//			}
//		}
//
//		return chunksTempDirPath;
//
//	}
	
	private java.nio.file.Path chunksBaseDirectoryPath() {
		java.nio.file.Path chunksDirPath;
		if(USE_SYSTEM_TMP_DIR_FOR_CHUNKS) {
			chunksDirPath=tempDirPath.resolve(CHUNKS_TEMP_DIRECTORY_NAME);
		}else {
			File appBaseDir=ipsk.webapps.db.servlets.FileServer.getBaseDir();
			java.nio.file.Path appBaseDirPath= appBaseDir.toPath();
			java.nio.file.Path appBaseTmpDirPath=appBaseDirPath.resolve(ipsk.webapps.db.servlets.FileServer.TMP_DIR);
			chunksDirPath=appBaseTmpDirPath.resolve(CHUNKS_TEMP_DIRECTORY_NAME);
		}
		return chunksDirPath;
	}
	
	
	private void applyAudioFileFormat(RecordingFile recFile,AudioFileFormat aff){
		AudioFormat af = aff.getFormat();
		AudioFileFormat.Type affType = aff.getType();
		recFile.setFormat(affType.toString());
		// TODO long method versions in JDK 1.5 ?
		recFile.setBytes((long) aff.getByteLength());
		long frameLength = aff.getFrameLength();
		Long recFileFl = null;
		if (frameLength != AudioSystem.NOT_SPECIFIED) {
			recFileFl = (Long) frameLength;
		}
		recFile.setFrames(recFileFl);
		recFile.setEncoding(af.getEncoding().toString());

		recFile.setChannels(af.getChannels());
		recFile.setQuantisation(af.getSampleSizeInBits());
		recFile.setSamplerate((double)af.getSampleRate());
		recFile.setBigendian(af.isBigEndian());
	}
	
	@POST
	@Path("/{UUID}/{chunkIdx}")
	@Consumes({"audio/wav","audio/x-wav"})
	@Produces(MediaType.APPLICATION_JSON)
	public Response postArRecfileChunk(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, InputStream inputStream, @PathParam("UUID") String uuid,@PathParam("chunkIdx") int chunkIdx) {
		return postRecfileChunk(sc, sec, req, inputStream, null,uuid, chunkIdx);
	}
	
	@POST
	@Path("/{itemcode}/{UUID}/{chunkIdx}")
	@Consumes({"audio/wav","audio/x-wav"})
	@Produces(MediaType.APPLICATION_JSON)
	public Response postSprRecfileChunk(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, InputStream inputStream, @PathParam("itemcode") String itemCode,@PathParam("UUID") String uuid,@PathParam("chunkIdx") int chunkIdx) {
		return postRecfileChunk(sc, sec, req, inputStream, itemCode, uuid,chunkIdx);
	}
		
	public Response postRecfileChunk(ServletContext sc, SecurityContext sec,HttpServletRequest req, InputStream inputStream, String itemCode,String uuid,int chunkIdx) {
			
		Response response=null;
		
		Integer sessionId = null;
		UUID sessionUUID = null;
		try {
			// Try database ID
			sessionId = Integer.parseInt(sessionIdStr);
		} catch (NumberFormatException nfe) {
			// Try UUID
			try{
				sessionUUID=UUID.fromString(sessionIdStr);
			}catch(IllegalArgumentException iae){
				sc.log("Session ID " + sessionIdStr + " could not be parsed as DB number ID or UUID: ", nfe);
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		EntityTransaction tx = em.getTransaction();
		tx.begin();
		Session session=null;
		if(sessionId!=null){
			session = em.find(Session.class, sessionId);
		}else if(sessionUUID!=null){
			List<Session> sessionsListByUUID=sessionsByUUID(em, sessionUUID);
			int sessionsListByUUIDSize=sessionsListByUUID.size();
			if(sessionsListByUUIDSize==1){
				// OK found exactly one
				session=sessionsListByUUID.get(0);
			}else if(sessionsListByUUIDSize>1){
				// ambiguous
				rollBackAndClose(em);
				return Response.status(Status.INTERNAL_SERVER_ERROR).build();
			}
		}
		if (session == null) {
			// Session not found
			rollBackAndClose(em);
			return Response.status(Status.NOT_FOUND).build();
		}
		try {
			securityManager.checkMergePermission(req, session);
		} catch (PermissionDeniedException e3) {
			sc.log("No merge permission for session ID: " + sessionId + " ", e3);
			rollBackAndClose(em);
			response= Response.status(Status.FORBIDDEN).build();
			return response;
		}

		boolean typeSpeechRecorder=(session.getScript()!=null);
		String sessionDir = session.getStorageDirectoryURL();
		if (sessionDir == null) {
			
			String recsDir = sc.getInitParameter("recsDir");
			String recsDirURL = "file:" + recsDir;
			sessionDir = SessionController.createSessionFilePath(recsDirURL, session.getSessionId());
			session.setStorageDirectoryURL(sessionDir);
		}
		tx.commit();

		String speakerCode = "";
		Set<Speaker> spks = session.getSpeakers();
		if (!spks.isEmpty()) {
			speakerCode = spks.iterator().next().getCode();
		}

		// Integer recVersion=null;
		boolean overwrite = false; // default

		UUID rfUUId;
		
		
			try {
				rfUUId=UUID.fromString(uuid);
			}catch(IllegalArgumentException iae) {
				sc.log(iae.getMessage());
				rollBackAndClose(em);
				return Response.status(Status.BAD_REQUEST).build();
			}
		
		// Lock
		ChunkedRecordingState crs;
		synchronized(ChunkedRecordingStates.recordingFileLocks) {
			crs=ChunkedRecordingStates.recordingFileLocks.get(rfUUId);
			if(crs==null) {
				if(chunkIdx==0) {
					// Create recording state only if first chunk
					crs=new ChunkedRecordingState();
					ChunkedRecordingStates.recordingFileLocks.put(rfUUId, crs);
				}else {
					sc.log("Warning : No recording state object found. Proceeding without. (Server restarted?)");
				}
			}
		}
		ReentrantLock lock=null;
		boolean locked=false;
		if(crs!=null) {
			lock=crs.getReentrantLock();
			int timeoutMinutes=CHUNK_LOCK_TIMEOUT_MINUTES;
			try {
				locked=lock.tryLock(timeoutMinutes, TimeUnit.MINUTES);
			} catch (InterruptedException e3) {
				e3.printStackTrace();
				rollBackAndClose(em);
				return Response.status(Status.SERVICE_UNAVAILABLE).build();
			}

			if(!locked) {
				sc.log("Could not get lock for recording file "+rfUUId+" in "+timeoutMinutes+" minutes");
				rollBackAndClose(em);
				return Response.status(Status.SERVICE_UNAVAILABLE).build();
			}
		}
		
		Script recScript=session.getScript();
		String filePathId;
		if(itemCode==null) {
			filePathId=rfUUId.toString();
		}else {
			filePathId=itemCode;
		}

		
		
		java.nio.file.Path tempAudioFilePath= chunkAudioFilePath(session, rfUUId, chunkIdx);
		java.nio.file.Path tempAudioFileParentPath=tempAudioFilePath.getParent();
		try {
			Files.createDirectories(tempAudioFileParentPath);
		}catch(IOException e3) {
			sc.log("Could not create directory: "+tempAudioFileParentPath, e3);
			rollBackAndClose(em);
			if(lock!=null) {
				lock.unlock();
			}
			return Response.serverError().build();
		}
		try {
			
			StreamCopy.copy(inputStream, tempAudioFilePath, true,true);
			sc.log("Copied (locked) chunk temp audio file: "+ tempAudioFilePath);
		} catch (IOException e2) {
			sc.log("Could not copy POST input stream to temp audio file: ", e2);
			rollBackAndClose(em);
			if(lock!=null) {
				lock.unlock();
			}
			return Response.serverError().build();
		}

		AudioFileFormat aff = null;
		try {
			File tempAudioFile=tempAudioFilePath.toFile();
			aff = ThreadSafeAudioSystem.getAudioFileFormat(tempAudioFile);
			//tempAudioFile.deleteOnExit();
		} catch (UnsupportedAudioFileException | IOException e) {
			sc.log(e.getMessage());
			// hold the temporary file
			rollBackAndClose(em);
			if(lock!=null) {
				lock.unlock();
			}
			return Response.serverError().build();
		}

		// store meta data to DB

		if (aff != null) {
			sc.log("detected audio file format: " + aff);
		} else {
			sc.log("could not detect audio file format !");
		}
		
		tx = em.getTransaction();
		tx.begin();
		
		if(chunkIdx==0) {
			// Create metadata in database
			RecordingFile recFile = new RecordingFile();
			recFile.setUuid(rfUUId.toString());
			recFile.setStatus(RecordingFile.Status.REGISTERED);
			if(crs!=null) {
				Date startedDate=crs.getStartedDate();
				if(startedDate!=null) {
					recFile.setStartedDate(startedDate);
				}
			}
			recFile.setDate(new Date());
			
			Recording recording = null;
			if(recScript!=null && itemCode!=null) {
				Query q = em.createQuery(
						"SELECT r FROM Recording AS r WHERE  r.group.section.script = :recScript AND r.itemcode = :itemcode");
				q.setParameter("recScript", recScript);
				q.setParameter("itemcode", itemCode);
				Object srObj = q.getSingleResult();
				if (srObj == null) {
					throw new PersistenceException("Could not retrieve associated recording script element!");
				}
				if (!(srObj instanceof Recording)) {
					throw new PersistenceException("Expected type Recording!");
				}
				recording = (Recording) srObj;
				recFile.setRecording(recording);
			}
			recFile.setSession(session);

			// recFile.setVersion(overwrite);
			recFile.setDate(new Date());
			String extension = "unknown";
			if (aff != null) {
				AudioFileFormat.Type affType = aff.getType();
				extension = affType.getExtension();
				applyAudioFileFormat(recFile, aff);
			}

			List<RecordingFile> rfs = null;
			if (recording != null) {
				rfs = recordingFilesOfSessionOfRecording(em, session, recording);
			}
			
			if(!BasicRecordingfileResource.speakerCodeValid(speakerCode)) {
				rollBackAndClose(em);
				sc.log("Invalid input");
				return Response.serverError().build();
			}

			if (overwrite) {
				// overwrite
				if (rfs == null || rfs.size() == 0) {
					recFile.setSignalFile(
							createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode, filePathId, extension, recFile, overwrite));
					try {
						securityManager.checkPersistPermission(req, recFile);
					} catch (PermissionDeniedException e) {
						sc.log("Permission denied to persist recording file entity: ", e);
						rollBackAndClose(em);
						if(lock!=null) {
							lock.unlock();
						}
						return Response.status(Status.FORBIDDEN).build();
					}
					em.persist(recFile);
					recFile.setSession(session);
					session.getRecordingFiles().add(recFile);
				} else {
					// Hmm. Not really correct.
					RecordingFile rf = (RecordingFile) rfs.get(0);
					Integer version = rf.getVersion();
					if (version != null) {
						recFile.setVersion(version++);
					} else {
						recFile.setVersion(0);
					}
					recFile.setRecordingFileId(rf.getRecordingFileId());

					recFile.setSignalFile(
							createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode,filePathId, extension, recFile, overwrite));
					// overwrite by merge
					em.merge(recFile);
				}
			} else {
				int version = 0;
				if (rfs != null) {
					for (RecordingFile rf : rfs) {
						Integer sVersion = rf.getVersion();
						if (sVersion != null) {
							if (sVersion >= version)
								version = sVersion + 1;
						}
					}
				}
				recFile.setVersion(version);
				recFile.setSignalFile(
						createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode, filePathId, extension, recFile, overwrite));
				em.persist(recFile);
				recFile.setSession(session);
				session.getRecordingFiles().add(recFile);

			}

			crs.addChunkIndex(chunkIdx);
			
			String fileURLStr=recFile.getSignalFile();
			
			java.nio.file.Path raFile=null;
			try {
				URL fileURL = new URL(fileURLStr);
				raFile=Utils.pathFromDecodedURL(fileURL);
			} catch (MalformedURLException | UnsupportedEncodingException e1) {
				e1.printStackTrace();
				rollBackAndClose(em);
				if(lock!=null) {
					lock.unlock();
				}
				return Response.serverError().build();
			}
			
			try {
				System.out.println("Copying first chunk to file: "+raFile);
				StreamCopy.copy(tempAudioFilePath, raFile,true);
				//Files.createDirectories(raFile.getParent());
				//Files.copy(tempAudioFilePath, raFile);
				
				crs.setConcatenatedChunkCount(1);
			} catch (IOException e) {
				e.printStackTrace();
				rollBackAndClose(em);
				if(lock!=null) {
					lock.unlock();
				}
				return Response.serverError().build();
			}
		}else {
			if(crs!=null) {
				crs.addChunkIndex(chunkIdx);
				int availSeqChunks=crs.availableSequencedChunks();
				int concatenatedChunks=crs.getConcatenatedChunkCount();
				// Build recording file from available chunks
				if((availSeqChunks-concatenatedChunks)>0) {
					try {
						Date startedDate=crs.getStartedDate();
						concatChunks(em, session, rfUUId,startedDate,availSeqChunks,false);
						crs.setConcatenatedChunkCount(availSeqChunks);
					} catch (IOException | AudioFormatNotSupportedException | AudioSourceException e) {
						e.printStackTrace();
						rollBackAndClose(em);
						if(lock!=null) {
							lock.unlock();
						}
						return Response.serverError().build();
					}
				}
			}else {
				sc.log("Warning: received audio chunk upload, but no state object found.");
			}
		}
		
		// In most cases the final concatenation chunks request is received _before_ the last audio chunk because of asynchronicity of float array to Wave file conversion by the recording client
		// So check after each audio chunk upload
		boolean comp;
		try {
			comp = concatChunksIfComplete(em, session, rfUUId,crs);
			response=Response.ok().build();
		} catch (IOException | AudioFormatNotSupportedException | AudioSourceException e1) {
			e1.printStackTrace();
			rollBackAndClose(em);
			if(lock!=null) {
				lock.unlock();
			}
			return Response.serverError().build();
		}
		
		try {
			tx.commit();
			em.close();
		} catch (Exception e) {
			rollBackAndClose(em);
			sc.log("Could not store recording file chunk metadata!");
			response=Response.serverError().build();
		}finally {			
			if(lock!=null) {
				lock.unlock();
			}
		}
		
		if(comp) {
			ChunkedRecordingStates.recordingFileLocks.remove(rfUUId);
			updateDSP(sc, sessionId);
		}
		
		
		return response;
	}
	
	private java.nio.file.Path chunksDirectoryPath(Session session,UUID rfUUID){
		return chunksBaseDirectoryPath().resolve(rfUUID.toString());
	}
	
	private String chunksFilenamebody(Session session,UUID rfUUID) {
		return getClass().getName() + "_" + session.getSessionId() + "_" + rfUUID;
	}
	
	private java.nio.file.Path chunkAudioFilePath(Session session,UUID rfUUID,int chunkIdx){
		java.nio.file.Path  chunksDirPath=chunksDirectoryPath(session,rfUUID);
		String chkAudioFilename = chunksFilenamebody(session,rfUUID)+ "_"+chunkIdx+".wav";
		return chunksDirPath.resolve(chkAudioFilename);
	}
	
	private java.nio.file.Path concatAudioFilePath(Session session,UUID rfUUID,int chunkCount){
		java.nio.file.Path  chunksDirPath=chunksDirectoryPath(session,rfUUID);
		String chkAudioFilename = chunksFilenamebody(session,rfUUID)+ "_concat_parts_"+chunkCount+".wav";
		return chunksDirPath.resolve(chkAudioFilename);
	}

	private java.nio.file.Path chunksPrepareRequestFilePath(Session session,UUID rfUUID){
		java.nio.file.Path  chunksDirPath=chunksDirectoryPath(session,rfUUID);
		String prepareChksReqFile = chunksFilenamebody(session,rfUUID)+ "_prepareChunksRequest.txt";
		return chunksDirPath.resolve(prepareChksReqFile);
	}
	
	private java.nio.file.Path chunksConcatRequestFilePath(Session session,UUID rfUUID){
		java.nio.file.Path  chunksDirPath=chunksDirectoryPath(session,rfUUID);
		String concatChksReqFile = chunksFilenamebody(session,rfUUID)+ "_concatChunksRequest.txt";
		return chunksDirPath.resolve(concatChksReqFile);
	}
	
	
	private boolean checkExistenceOfChunkFiles(Session session,UUID rfUUID, int chunkCnt) {
		for(int ci=0;ci<chunkCnt;ci++) {
			java.nio.file.Path cfp= chunkAudioFilePath(session,rfUUID,ci);
			if(!Files.exists(cfp)) {
				return false;
			}
		}
		return true;
	}
	
	private void concatChunks(EntityManager em, Session session,UUID rfUUID, Date startedDate,int chunkCount,boolean lastChunk) throws IOException, AudioFormatNotSupportedException, AudioSourceException{

		java.nio.file.Path cfp0= chunkAudioFilePath(session,rfUUID,0);
		
		AudioSource as=new FileAudioSource(cfp0.toFile());
		
		PluginChain pch=new PluginChain(as);

		for(int ci=1;ci<chunkCount;ci++) {
			java.nio.file.Path cfpi= chunkAudioFilePath(session,rfUUID,ci);
			if(!Files.exists(cfpi)) {
				throw new IOException("A chunk file is missing for concatenation.");
			}
			AudioSource asi=new FileAudioSource(cfpi.toFile());
		
			try {
				pch.add(new AppendPlugin(asi));
			} catch (AudioFormatNotSupportedException e) {
				e.printStackTrace();
				throw e;
			}
		}
		RecordingFile rf=recordingFileByUUID(em, rfUUID);
		String fileURLStr=rf.getSignalFile();

		File recFile=null;
		try {
			URL fileURL = new URL(fileURLStr);
			recFile=Utils.fileFromDecodedURL(fileURL);
		} catch (MalformedURLException e1) {
			e1.printStackTrace();
			throw e1;
		}

		File parentDir=recFile.getParentFile();
		if(!parentDir.isDirectory()) {
			parentDir.mkdirs();
		}
		
		java.nio.file.Path concatPartsPath=concatAudioFilePath(session, rfUUID, chunkCount);
		File concatPartsFile=concatPartsPath.toFile();
		System.out.println("Writing concat file: "+concatPartsFile);
		AudioInputStream pchAis=pch.getAudioInputStream();
		ThreadSafeAudioSystem.write(pchAis,AudioFileFormat.Type.WAVE,concatPartsFile);
		pchAis.close();
		
		System.out.println("Copy to rec file: "+recFile);
		StreamCopy.copy(concatPartsFile,recFile);
		
		// Set started date if not already set
		Date rfStartedDate=rf.getStartedDate();
		if(rfStartedDate==null && startedDate!=null) {
			rf.setStartedDate(startedDate);
		}
		
		// Get frame length of concatenated file
		AudioInputStream concatAis=null;
		try {
			concatAis=ThreadSafeAudioSystem.getAudioInputStream(concatPartsFile);
			long frameLength=concatAis.getFrameLength();
			concatAis.close();
			Long frameLenDb=null;
			if(frameLength!=AudioSystem.NOT_SPECIFIED) {
				frameLenDb=frameLength;
			}
			rf.setFrames(frameLenDb);
		} catch (UnsupportedAudioFileException | IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}finally {
			if(concatAis!=null) {
				concatAis.close();
			}
		}
		
		// Delete temporary (local) concat file
		Files.delete(concatPartsPath);
		
		if(lastChunk) {
			rf.setStatus(RecordingFile.Status.RECORDED);

			// Cleanup temporary chunk directory and files

			java.nio.file.Path cdp=chunksDirectoryPath(session, rfUUID);
			
			try(Stream<java.nio.file.Path> fileListStream=Files.list(cdp)){
				fileListStream.forEach((fp)->{
					try {
						Files.deleteIfExists(fp);
					} catch (IOException e) {
						e.printStackTrace();
						// OK, temporary files only
					}
				});
			}
			
			// Finally delete the directory
			try {
				Files.deleteIfExists(cdp);
			}catch(DirectoryNotEmptyException dne) {
				System.out.println("Error: Could not delete directory: "+cdp+ ": "+dne.getMessage());
			}

		}else {
			rf.setStatus(RecordingFile.Status.PARTIALLY_RECORDED);
		}

		em.merge(rf);

	}
	
	private boolean concatChunksIfComplete(EntityManager em, Session session,UUID rfUUID, ChunkedRecordingState csr) throws IOException, AudioFormatNotSupportedException, AudioSourceException{
		boolean complete=false;
		int chunkCount=0;
		Date startedDate=null;
		if(csr==null) {
			// No chunked recording state available, maybe after a server restart
			// Try to concat the chunk files anyway
			java.nio.file.Path concatReqFile=chunksConcatRequestFilePath(session, rfUUID);
			if(Files.exists(concatReqFile)) {
				// Read the concat request file
				try (FileChannel channel = FileChannel.open(concatReqFile)) {
					try(FileLock fl=channel.lock(0,Long.MAX_VALUE, true)){
						List<String> lines=Files.readAllLines(concatReqFile);
						if(lines.size()==1) {
							String chunkCntStr=lines.get(0);
							chunkCount=Integer.parseInt(chunkCntStr);
							System.out.println("Found concat request file: "+chunkCount);

							// Check existence of all chunk files
							complete=checkExistenceOfChunkFiles(session, rfUUID, chunkCount);
						}
					}
				}
			}
			
			java.nio.file.Path prepareReqFile=chunksPrepareRequestFilePath(session, rfUUID);
			if(Files.exists(prepareReqFile)) {
				// Read the prepare request file
				try (FileChannel channel = FileChannel.open(prepareReqFile)) {
					try(FileLock fl=channel.lock(0,Long.MAX_VALUE, true)){
						List<String> lines=Files.readAllLines(concatReqFile);
						if(lines.size()==1) {
							String startedDateStr=lines.get(0);
							startedDate=jsonDtFmt.parse(startedDateStr);
							System.out.println("Found prepare request file with started date: "+startedDate);
						}
					} catch (ParseException e) {
						e.printStackTrace();
						// OK Giving up: do not set started date
					}
				}
			}
			
		}else {
			Integer finalChunkCount=csr.getFinalChunkCount();
			int availChunkCount=csr.availableSequencedChunks();
			if(finalChunkCount!=null) {
				chunkCount=finalChunkCount;
				complete=(chunkCount==availChunkCount);
			}
			startedDate=csr.getStartedDate();
		}

		if(complete && chunkCount>0) {
			concatChunks(em, session, rfUUID,startedDate, chunkCount,true);
		}

		return complete;
	}

	private void updateDSP(ServletContext sc,Integer sessionId){	
		// Invalidate cached values about the session (missing recording items, level,..)notify the DSP processor about new recording file
		SessionController.invalidateCachedValues(sc, sessionId);
		// Notify the DSP processor about the new recording file for processing 
		DSPProcessor.notifyThread();
	}
	
	@POST
	@Path("/{UUID}/prepareChunksRequest")
	@Consumes({"multipart/form-data"})
	@Produces(MediaType.APPLICATION_JSON)
	public Response postPrepareChunksRequest(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, @FormDataParam("uuid") String uuidParam,@FormDataParam("startedDate") String startedDateStr, @PathParam("UUID") String uuidStr) {	
		//Response res=null;
		Integer sessionId = null;
		UUID sessionUUID = null;
		try {
			// Try database ID
			sessionId = Integer.parseInt(sessionIdStr);
		} catch (NumberFormatException nfe) {
			// Try UUID
			try{
				sessionUUID=UUID.fromString(sessionIdStr);
			}catch(IllegalArgumentException iae){
				sc.log("Session ID " + sessionIdStr + " could not be parsed as DB number ID or UUID: ", nfe);
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		EntityTransaction tx = em.getTransaction();
		tx.begin();
		Session session=null;
		if(sessionId!=null){
			session = em.find(Session.class, sessionId);
		}else if(sessionUUID!=null){
			List<Session> sessionsListByUUID=sessionsByUUID(em, sessionUUID);
			int sessionsListByUUIDSize=sessionsListByUUID.size();
			if(sessionsListByUUIDSize==1){
				// OK found exactly one
				session=sessionsListByUUID.get(0);
			}else if(sessionsListByUUIDSize>1){
				// ambiguous
				rollBackAndClose(em);
				sc.log("Found ambigious sessions by UUID: "+sessionUUID);
				return Response.status(Status.INTERNAL_SERVER_ERROR).build();
			}
		}
		if (session == null) {
			// Session not found
			rollBackAndClose(em);
			return Response.status(Status.NOT_FOUND).build();
		}
		try {
			securityManager.checkMergePermission(req, session);
		} catch (PermissionDeniedException e3) {
			sc.log("No merge permission for session ID: " + sessionId + " ", e3);
			rollBackAndClose(em);
			return Response.status(Status.FORBIDDEN).build();
		}

		String sessionDir = session.getStorageDirectoryURL();
		if (sessionDir == null) {	
			String recsDir = sc.getInitParameter("recsDir");
			String recsDirURL = "file:" + recsDir;
			sessionDir = SessionController.createSessionFilePath(recsDirURL, session.getSessionId());
			session.setStorageDirectoryURL(sessionDir);
		}

		UUID rfUUId;
		
			try {
				rfUUId=UUID.fromString(uuidStr);
				
			}catch(IllegalArgumentException iae) {
				sc.log(iae.getMessage());
				return Response.status(Status.BAD_REQUEST).build();
			}
			
			
			
			// Lock
			ChunkedRecordingState crs;
			synchronized(ChunkedRecordingStates.recordingFileLocks) {
				crs=ChunkedRecordingStates.recordingFileLocks.get(rfUUId);
				if(crs==null) {
					crs=new ChunkedRecordingState();
					ChunkedRecordingStates.recordingFileLocks.put(rfUUId, crs);
				}
			}

			ReentrantLock lock=crs.getReentrantLock();
			lock.lock();
			
			if(startedDateStr!=null) {
				//DateFormat jsonDtFmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
				Date startedDate;
				try {
					startedDate = jsonDtFmt.parse(startedDateStr);
					crs.setStartedDate(startedDate);
				} catch (ParseException e) {
					// Log warning and ignore
					sc.log("Warning: Could not parse JSON date string: "+startedDateStr);
				}
			}
			
			java.nio.file.Path prepareReqPath=chunksPrepareRequestFilePath(session, rfUUId);
			java.nio.file.Path prepareReqParentPath=prepareReqPath.getParent();
			try {
				Files.createDirectories(prepareReqParentPath);
				sc.log("Created chunk upload directory: "+prepareReqParentPath);
			} catch (IOException e2) {
				e2.printStackTrace();
				rollBackAndClose(em);
				lock.unlock();
				sc.log("Error creating chunked recording directory: "+e2.getMessage());
				return Response.serverError().build();
			}
			
			try(
					FileOutputStream fos=new FileOutputStream(prepareReqPath.toFile());
					FileChannel ch=fos.getChannel();
					PrintWriter pw=new PrintWriter(fos);
					FileLock fl=ch.lock();
					// NOTE: The order of the resources in the try-with-resources clause is important
					// The print writer must be after the channel, to be closed before the channel is closed because the channel seems to close the out stream too.
					){
				pw.write(startedDateStr);
				pw.flush(); // Flush to write the data before the lock is released
				sc.log("Written started date to prepare request file: "+startedDateStr);
			} catch (IOException e1) {
				e1.printStackTrace();
				rollBackAndClose(em);
				if(lock!=null) {
					lock.unlock();
				}
				sc.log("Error writing prepare request file: "+e1.getMessage());
				return Response.serverError().build();
			}
			
			
			boolean compl;
			try {
				compl = concatChunksIfComplete(em, session, rfUUId,crs);
				sc.log("Received prepare chunks request for recording file "+rfUUId+" in session ID: " + sessionId);
			} catch (IOException | AudioFormatNotSupportedException | AudioSourceException e) {
				rollBackAndClose(em);
				e.printStackTrace();
				lock.unlock();
				return Response.serverError().build();
			}
			
			try {
				tx.commit();
				em.close();
			} catch (Exception e) {
				rollBackAndClose(em);
				sc.log("Could not store recording file chunk metadata!");
				return Response.serverError().build();
			}finally {			
				lock.unlock();
			}
			
			if(compl) {
				ChunkedRecordingStates.recordingFileLocks.remove(rfUUId);
				updateDSP(sc, sessionId);
			}
			
		return Response.ok("{}").build();
	}
		
		@POST
		@Path("/{UUID}/concatChunksRequest")
		@Consumes({"multipart/form-data"})
		@Produces(MediaType.APPLICATION_JSON)
		public Response postConcatRequest(@Context ServletContext sc, @Context SecurityContext sec,
				@Context HttpServletRequest req, @FormDataParam("uuid") String uuid,@FormDataParam("chunkCount") int chunkCount, @PathParam("UUID") String itemCodeOrUUID) {	
			//Response res=null;
			Integer sessionId = null;
			UUID sessionUUID = null;
			try {
				// Try database ID
				sessionId = Integer.parseInt(sessionIdStr);
			} catch (NumberFormatException nfe) {
				// Try UUID
				try{
					sessionUUID=UUID.fromString(sessionIdStr);
				}catch(IllegalArgumentException iae){
					sc.log("Session ID " + sessionIdStr + " could not be parsed as DB number ID or UUID: ", nfe);
					return Response.status(Status.BAD_REQUEST).build();
				}
			}
			final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
			WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
				@Override
				public EntityManager getThreadEntityManager() {
					return em;
				}
			});

			EntityTransaction tx = em.getTransaction();
			tx.begin();
			Session session=null;
			if(sessionId!=null){
				session = em.find(Session.class, sessionId);
			}else if(sessionUUID!=null){
				List<Session> sessionsListByUUID=sessionsByUUID(em, sessionUUID);
				int sessionsListByUUIDSize=sessionsListByUUID.size();
				if(sessionsListByUUIDSize==1){
					// OK found exactly one
					session=sessionsListByUUID.get(0);
				}else if(sessionsListByUUIDSize>1){
					// ambiguous
					rollBackAndClose(em);
					return Response.status(Status.INTERNAL_SERVER_ERROR).build();
				}
			}
			if (session == null) {
				// Session not found
				rollBackAndClose(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			try {
				securityManager.checkMergePermission(req, session);
			} catch (PermissionDeniedException e3) {
				sc.log("No merge permission for session ID: " + sessionId + " ", e3);
				rollBackAndClose(em);
				return Response.status(Status.FORBIDDEN).build();
			}

			String sessionDir = session.getStorageDirectoryURL();
			if (sessionDir == null) {	
				String recsDir = sc.getInitParameter("recsDir");
				String recsDirURL = "file:" + recsDir;
				sessionDir = SessionController.createSessionFilePath(recsDirURL, session.getSessionId());
				session.setStorageDirectoryURL(sessionDir);
			}

			UUID rfUUId;
			
				try {
					rfUUId=UUID.fromString(itemCodeOrUUID);
					
				}catch(IllegalArgumentException iae) {
					sc.log(iae.getMessage());
					return Response.status(Status.BAD_REQUEST).build();
				}
				
				// Lock
				ChunkedRecordingState crs;
				synchronized(ChunkedRecordingStates.recordingFileLocks) {
					crs=ChunkedRecordingStates.recordingFileLocks.get(rfUUId);
					if(crs==null) {
//						crs=new ChunkedRecordingState();
//						recordingFileLocks.put(rfUUId, crs);
						sc.log("Warning: Received concat chunks request, but no recording state object found.");
					}
				}
				ReentrantLock lock=null;
				if(crs!=null) {
					lock=crs.getReentrantLock();
					lock.lock();
				}
				java.nio.file.Path concatReqFile=chunksConcatRequestFilePath(session, rfUUId);
				try {
					Files.createDirectories(concatReqFile.getParent());
				} catch (IOException e2) {
					e2.printStackTrace();
					rollBackAndClose(em);
					if(lock!=null) {
						lock.unlock();
					}
					sc.log("Error creating chunked recording directories: "+e2.getMessage());
					return Response.serverError().build();
				}
				
				try(
					FileOutputStream fos=new FileOutputStream(concatReqFile.toFile());
					FileChannel ch=fos.getChannel();
					FileLock fl=ch.lock();
					) {
						String chunkCntStr=Integer.toString(chunkCount);
						fos.write(chunkCntStr.getBytes());
						fos.flush(); // Flush to write the data before the lock is released
				} catch (IOException e1) {
					e1.printStackTrace();
					rollBackAndClose(em);
					if(lock!=null) {
						lock.unlock();
					}
					sc.log("Error writing concat request file: "+e1.getMessage());
					return Response.serverError().build();
				}
				if(crs!=null) {
					crs.finalChunkCount(chunkCount);
				}
				boolean compl;
				try {
					compl = concatChunksIfComplete(em, session, rfUUId,crs);
					sc.log("Received concat chunks request for recording file "+rfUUId+" in session ID: " + sessionId);
				} catch (IOException | AudioFormatNotSupportedException | AudioSourceException e) {
					rollBackAndClose(em);
					e.printStackTrace();
					sc.log("Error checking completeness of chunked recording upload: "+e.getMessage());
					if(lock!=null) {
						lock.unlock();
					}
					return Response.serverError().build();
				}
				
				try {
					tx.commit();
					em.close();
				} catch (Exception e) {
					rollBackAndClose(em);
					sc.log("Could not process concat chunks request: "+e.getMessage());
					e.printStackTrace();
					return Response.serverError().build();
				}finally {
					// Unlock chunked recording file lock
					if(lock!=null) {
						lock.unlock();
					}
				}
				
				if(compl) {
					ChunkedRecordingStates.recordingFileLocks.remove(rfUUId);
					updateDSP(sc, sessionId);
				}
				
			return Response.ok("{}").build();
		}

	@POST
	@Path("/{itemcodeOrUUID}")
	@Consumes({"multipart/form-data"})
	@Produces(MediaType.APPLICATION_JSON)
	public Response postRecfileAndDescr(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, @FormDataParam("uuid") String uuid,@FormDataParam("startedDate") String startedDateStr,@FormDataParam("audio") InputStream inputStream, @PathParam("itemcodeOrUUID") String itemCodeOrUUID) {
		Integer sessionId = null;
		UUID sessionUUID = null;
		try {
			// Try database ID
			sessionId = Integer.parseInt(sessionIdStr);
		} catch (NumberFormatException nfe) {
			// Try UUID
			try{
				sessionUUID=UUID.fromString(sessionIdStr);
			}catch(IllegalArgumentException iae){
				sc.log("Session ID " + sessionIdStr + " could not be parsed as DB number ID or UUID: ", nfe);
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		
		
		
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		EntityTransaction tx = em.getTransaction();
		tx.begin();
		Session session=null;
		if(sessionId!=null){
			session = em.find(Session.class, sessionId);
		}else if(sessionUUID!=null){
			List<Session> sessionsListByUUID=sessionsByUUID(em, sessionUUID);
			int sessionsListByUUIDSize=sessionsListByUUID.size();
			if(sessionsListByUUIDSize==1){
				// OK found exactly one
				session=sessionsListByUUID.get(0);
			}else if(sessionsListByUUIDSize>1){
				// ambiguous
				rollBackAndClose(em);
				return Response.status(Status.INTERNAL_SERVER_ERROR).build();
			}
		}
		if (session == null) {
			// Session not found
			rollBackAndClose(em);
			return Response.status(Status.NOT_FOUND).build();
		}
		try {
			securityManager.checkMergePermission(req, session);
		} catch (PermissionDeniedException e3) {
			sc.log("No merge permission for session ID: " + sessionId + " ", e3);
			rollBackAndClose(em);
			return Response.status(Status.FORBIDDEN).build();
		}

		boolean typeSpeechRecorder=(session.getScript()!=null);
		
		String sessionDir = session.getStorageDirectoryURL();
		if (sessionDir == null) {
			
			String recsDir = sc.getInitParameter("recsDir");
			String recsDirURL = "file:" + recsDir;
			sessionDir = SessionController.createSessionFilePath(recsDirURL, session.getSessionId());
			session.setStorageDirectoryURL(sessionDir);
		}
		tx.commit();

		String speakerCode = "";
		Set<Speaker> spks = session.getSpeakers();
		if (!spks.isEmpty()) {
			speakerCode = spks.iterator().next().getCode();
		}

		// Integer recVersion=null;
		boolean overwrite = false; // default
		
		UUID rfUUId=null;
		try {
			rfUUId=UUID.fromString(uuid);
		}catch(IllegalArgumentException iae) {
			sc.log(iae.getMessage());
			return Response.status(Status.BAD_REQUEST).build();
		}
		
		Script recScript=session.getScript();
		
		if(rfUUId==null) {
			if(recScript==null) {
				try {
					rfUUId=UUID.fromString(itemCodeOrUUID);
				}catch(IllegalArgumentException iae) {
					sc.log(iae.getMessage());
					return Response.status(Status.BAD_REQUEST).build();
				}
			}else {
				rfUUId=UUID.randomUUID();
			}
		}
		String tempFilename = getClass().getName() + "_" + session.getSessionId() + "_" + itemCodeOrUUID + "_"
				+ rfUUId + ".wav";
		File tempAudioFile = new File(tempDir, tempFilename);

		try {
			StreamCopy.copy(inputStream, tempAudioFile, true);
		} catch (IOException e2) {
			sc.log("Could not copy POST input stream to temp audio file: ", e2);
			return Response.serverError().build();
		}

		AudioFileFormat aff = null;
		try {
			aff = ThreadSafeAudioSystem.getAudioFileFormat(tempAudioFile);
		} catch (UnsupportedAudioFileException | IOException e) {
			sc.log(e.getMessage());

			tempAudioFile.deleteOnExit();
			// hold the temporary files for debug purposes
			// tmpFile.delete();
			sc.log(e.getMessage());

			return Response.serverError().build();
		}

		// store meta data to DB

		if (aff != null) {
			sc.log("detected audio file format: " + aff);
		} else {
			sc.log("could not detect audio file format !");
		}
		
		
		RecordingFile recFile = new RecordingFile();
		recFile.setUuid(rfUUId.toString());
		recFile.setStatus(RecordingFile.Status.REGISTERED);
		recFile.setDate(new Date());
		if(startedDateStr!=null) {
			DateFormat jsonDtFmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
			Date startedDate;
			try {
				startedDate = jsonDtFmt.parse(startedDateStr);
				recFile.setStartedDate(startedDate);
			} catch (ParseException e) {
				// Log warning and ignore
				sc.log("Warning: Could not parse JSON date string: "+startedDateStr);
			}
			
		}

		tempAudioFile.deleteOnExit();
		tx = em.getTransaction();
		tx.begin();
		
		Recording recording = null;
		if(recScript!=null) {
			Query q = em.createQuery(
					"SELECT r FROM Recording AS r WHERE  r.group.section.script = :recScript AND r.itemcode = :itemcode");
			q.setParameter("recScript", recScript);
			q.setParameter("itemcode", itemCodeOrUUID);
			Object srObj = q.getSingleResult();
			if (srObj == null) {
				throw new PersistenceException("Could not retrieve associated recording script element!");
			}
			if (!(srObj instanceof Recording)) {
				throw new PersistenceException("Expected type Recording!");
			}
			recording = (Recording) srObj;
			recFile.setRecording(recording);
		}
		recFile.setSession(session);

		recFile.setDate(new Date());
		String extension = "unknown";
		if (aff != null) {
			AudioFileFormat.Type affType = aff.getType();
			extension = affType.getExtension();
			applyAudioFileFormat(recFile, aff);
		}

		List<RecordingFile> rfs = null;
		if (recording != null) {
			TypedQuery<RecordingFile> q = em.createQuery(
					"SELECT rf FROM RecordingFile AS rf,Recording AS r WHERE r = :recording AND rf MEMBER OF r.recordingFiles AND rf.session = :session",RecordingFile.class);
			q.setParameter("recording", recording);
			q.setParameter("session", session);
			rfs = q.getResultList();
		}
		if(!BasicRecordingfileResource.speakerCodeValid(speakerCode)) {
			rollBackAndClose(em);
			sc.log("Invalid input");
			return Response.serverError().build();
		}

		if(!BasicRecordingfileResource.itemCodeValid(itemCodeOrUUID)) {
			rollBackAndClose(em);
			sc.log("Invalid input");
			return Response.serverError().build();
		}

		// String signalFile=null;

		if (overwrite) {
			// overwrite
			if (rfs == null || rfs.size() == 0) {
				recFile.setSignalFile(
						createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode, itemCodeOrUUID, extension, recFile, overwrite));
				try {
					securityManager.checkPersistPermission(req, recFile);
				} catch (PermissionDeniedException e) {
					sc.log("Permission denied to persist recording file entity: ", e);
					rollBackAndClose(em);
					return Response.status(Status.FORBIDDEN).build();
				}
				em.persist(recFile);
				recFile.setSession(session);
				session.getRecordingFiles().add(recFile);
			} else {
				// Hmm. Not really correct.
				RecordingFile rf = (RecordingFile) rfs.get(0);
				Integer version = rf.getVersion();
				if (version != null) {
					recFile.setVersion(version++);
				} else {
					recFile.setVersion(0);
				}
				recFile.setRecordingFileId(rf.getRecordingFileId());

				recFile.setSignalFile(
						createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode, itemCodeOrUUID, extension, recFile, overwrite));
				// overwrite by merge
				em.merge(recFile);
			}
		} else {
			int version = 0;
			if (rfs != null) {
				for (RecordingFile rf : rfs) {
					Integer sVersion = rf.getVersion();
					if (sVersion != null) {
						if (sVersion >= version)
							version = sVersion + 1;
					}
				}
			}
			recFile.setVersion(version);
			recFile.setSignalFile(
					createSignalFilePath(typeSpeechRecorder,sessionDir, speakerCode, itemCodeOrUUID, extension, recFile, overwrite));
			em.persist(recFile);
			recFile.setSession(session);
			session.getRecordingFiles().add(recFile);
		}


		URL signalFile = null;
		try {
			signalFile = new URL(recFile.getSignalFile());
		} catch (MalformedURLException e1) {
			sc.log("Malformed URL: ", e1);
		}

		File storeFile = null;
		if (signalFile.getProtocol().equalsIgnoreCase("file")) {
			// copy File
			FileInputStream fis = null;
			try {
				storeFile = new File(signalFile.getFile());
				fis = new FileInputStream(tempAudioFile);
				storeToFile(sc, fis, storeFile, false);
				fis.close();
				recFile.setStatus(RecordingFile.Status.RECORDED);
				// tempAudioFile.delete();
			} catch (IOException ioe) {
				rollBackAndClose(em);
				// throw ioe;
				sc.log("IO Exception storing audio file: " + ioe);
				return Response.serverError().build();

			} finally {
				if (fis != null) {
					try {
						fis.close();
					} catch (IOException e) {
						rollBackAndClose(em);
						sc.log("Could not close audio file !");
						return Response.serverError().build();
					}
				}
			}

			// DSPProcessor.notifyThread();
		} else {
			// tempAudioFile.delete();
			rollBackAndClose(em);
			sc.log("Cannot store URL " + signalFile + "\nOnly protocol file: is supported");
			return Response.serverError().build();
		}
		try {
			// recFileContr.commit();
			tx.commit();
			
			// Invalidate missing recording items cache
			if(sessionId!=null) {
				SessionController.invalidateCachedValues(sc, sessionId);
			}
			
			// notify the DSP processor about new recording file
			DSPProcessor.notifyThread();
			// JPA/DB operations finished
			em.close();
			// Cleanup temp file
			tempAudioFile.delete();
			// empty JSON to make Angular HttpClient happy
			return Response.ok("{}").build();
		} catch (Exception e) {
			rollBackAndClose(em);
			sc.log("Could not store recording file metadata to database !");
			return Response.serverError().build();
		} finally {

		}

	}
	
	@GET
	@Path("/{itemcode}/{version}")	
	@Produces({ MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
	public Response recordingFileDescr(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, @PathParam("itemcode") String itemCode,
			@PathParam("version") String versionStr) {
		int sessionId;
		try {
			sessionId = Integer.parseInt(sessionIdStr);
		} catch (NumberFormatException nfe) {
			sc.log("Session ID " + sessionIdStr + " could not be parsed as number ID: ", nfe);
			return Response.status(Status.BAD_REQUEST).build();
		}
		int version;
		try {
			version = Integer.parseInt(versionStr);
		} catch (NumberFormatException nfe) {
			sc.log("Recording file version " + versionStr + " could not be parsed as number: ", nfe);
			return Response.status(Status.BAD_REQUEST).build();
		}

		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});
		try {
			EntityTransaction tx = em.getTransaction();
			tx.begin();
			Session session = em.find(Session.class, sessionId);
			if (session == null) {
				rollBackAndClose(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			try {
				securityManager.checkReadPermission(req, session);
			} catch (PermissionDeniedException e3) {
				sc.log("No read permission for session ID: " + sessionId + " ", e3);
				rollBackAndClose(em);
				return Response.status(Status.FORBIDDEN).build();
			}

			Set<RecordingFile> recFiles = session.getRecordingFiles();
			for (RecordingFile rf : recFiles) {
				Recording r = rf.getRecording();
				if (itemCode.equals(r.getItemcode())) {
					Integer rfVers = rf.getVersion();
					if (rfVers != null && rfVers == version && "WAVE".equals(rf.getFormat())) {
						return Response.ok(rf).build();
					}
				}
			}
			return Response.status(Status.NOT_FOUND).build();
		} finally {
			em.close();
		}

	}
	

}
