package ipsk.webapps.db.speech.ws;

import java.io.StringReader;
import java.util.UUID;

import javax.json.Json;
import javax.json.JsonNumber;
import javax.json.JsonObject;
import javax.json.JsonReader;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.PATCH;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;

import ipsk.db.speech.RecordingFile;
import ipsk.db.speech.Session;
import ipsk.persistence.EntityManagerProvider;
import ipsk.webapps.EntityManagerFactoryInitializer;
import ipsk.webapps.PermissionDeniedException;
import ipsk.webapps.db.speech.WikiSpeechSecurityManager;

public class SubRecordingfileResource extends BasicRecordingfileResource{

	private final static boolean DEBUG = true;
	
	public final static String RECS_DIR_PARAM_NAME="recsDir";
	
	private static final String EDIT_SAMPLERATE_PNAME="editSampleRate";
	private static final String EDIT_START_FRAME_PNAME="editStartFrame";
	private static final String EDIT_END_FRAME_PNAME="editEndFrame";
	
	private static final long serialVersionUID = 1L;

	public SubRecordingfileResource() {
		super(null);
	}
	
	public SubRecordingfileResource(String sessionIdStr) {
		super(sessionIdStr);
	}

	@GET
	@Produces({ MediaType.APPLICATION_JSON })
	public Response getRecordingFileList(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req,@QueryParam("latestVersionsOnly") Boolean latestVersionsOnly) {
		return recordingFileList(sc, sec, req, latestVersionsOnly);
	}


	
	@GET
	@Path("/{recordingFileIdOrUUIDStr}")	
	@Produces("audio/wav")
	public Response getRecFile(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, @PathParam("recordingFileIdOrUUIDStr") String recordingFileIdStr,
			@PathParam("version") String versionStr,@QueryParam("startFrame") Long startFrame,@QueryParam("frameLength") Long frameLength) {
		Integer recordingFileId=null;
		UUID uuid=null;
		try {
			recordingFileId = Integer.parseInt(recordingFileIdStr);
		} catch (NumberFormatException nfe) {
			try {
				uuid=UUID.fromString(recordingFileIdStr);
			}catch(IllegalArgumentException iae) {
				sc.log("Recording file ID " + recordingFileIdStr + " could not neither be parsed as number nor as UUID.");
				return Response.status(Status.BAD_REQUEST).build();
			}
			
		}
		
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});
		try {
			EntityTransaction tx = em.getTransaction();
			tx.begin();
			
			RecordingFile rf=null;
			if(recordingFileId!=null) {
				rf=em.find(RecordingFile.class, recordingFileId);
			}else if(uuid!=null){
				rf=recordingFileByUUID(em, uuid);
			}
			
			if (rf == null) {
				rollBack(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			Session session=rf.getSession();
			if (session == null) {
				rollBack(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			try {
				securityManager.checkReadPermission(req, session);
			} catch (PermissionDeniedException e3) {
				sc.log("No read permission for session ID: " + session+ " ", e3);
				rollBack(em);
				return Response.status(Status.FORBIDDEN).build();
			}
			
			return recordingFileStreamResponse(em,rf,startFrame,frameLength);
		} finally {
			close(em);
		}

	}
	
	@GET
	@Path("/{recordingFileIdOrUUIDStr}")	
	@Produces({ MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
	public Response recordingFile(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, @PathParam("recordingFileIdOrUUIDStr") String recordingFileIdStr) {
		Integer recordingFileId=null;
		UUID uuid=null;
		try {
			recordingFileId = Integer.parseInt(recordingFileIdStr);
		} catch (NumberFormatException nfe) {
			try {
				uuid=UUID.fromString(recordingFileIdStr);
			}catch(IllegalArgumentException iae) {
				sc.log("Recording file ID " + recordingFileIdStr + " could not neither be parsed as number nor as UUID.");
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});
		try {
			EntityTransaction tx = em.getTransaction();
			tx.begin();
			RecordingFile rf=null;
			if(recordingFileId!=null) {
				rf=em.find(RecordingFile.class, recordingFileId);
			}else if(uuid!=null){
				rf=recordingFileByUUID(em, uuid);
			}
			if (rf == null) {
				rollBack(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			Session session=rf.getSession();
			if (session == null) {
				rollBack(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			try {
				securityManager.checkReadPermission(req, session);
			} catch (PermissionDeniedException e3) {
				sc.log("No read permission for session ID: " + session+ " ", e3);
				rollBack(em);
				return Response.status(Status.FORBIDDEN).build();
			}

			
			// Found
			//securityManager.checkReadPermission(req, rf);
			return Response.ok(rf).build();
			
		} finally {
			close(em);
		}

	}
	
	@PATCH
	@Path("/{recordingFileIdStr}")	
	@Consumes({ MediaType.APPLICATION_JSON})
	public Response recordingFilePut(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, @PathParam("recordingFileIdStr") String recordingFileIdStr, String jsonPatch) {
		int recordingFileId;
		try {
			recordingFileId = Integer.parseInt(recordingFileIdStr);
		} catch (NumberFormatException nfe) {
			sc.log("Recording file ID " + recordingFileIdStr + " could not be parsed as number ID: ", nfe);
			return Response.status(Status.BAD_REQUEST).build();
		}
		
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});
		try {
			EntityTransaction tx = em.getTransaction();
			tx.begin();
			RecordingFile rf=em.find(RecordingFile.class, recordingFileId);
			
			if (rf == null) {
				rollBack(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			Session session=rf.getSession();
			if (session == null) {
				rollBack(em);
				return Response.status(Status.NOT_FOUND).build();
			}
			try {
				securityManager.checkReadPermission(req, session);
			} catch (PermissionDeniedException e3) {
				sc.log("No read permission for session ID: " + session+ " ", e3);
				rollBack(em);
				return Response.status(Status.FORBIDDEN).build();
			}

			JsonReader jsonReader = Json.createReader(new StringReader(jsonPatch));
			JsonObject jsonObj=jsonReader.readObject();
			
			Double esr=null;
			if(jsonObj.containsKey(EDIT_SAMPLERATE_PNAME)){
				if(!jsonObj.isNull(EDIT_SAMPLERATE_PNAME)) {
					JsonNumber esrJsN=jsonObj.getJsonNumber(EDIT_SAMPLERATE_PNAME);
					if(esrJsN!=null) {
						esr=esrJsN.doubleValue();
					}
				}
			}
			
			Long esf=null;
			if(jsonObj.containsKey(EDIT_START_FRAME_PNAME)){
				if(!jsonObj.isNull(EDIT_START_FRAME_PNAME)) {
					JsonNumber esfJsN=jsonObj.getJsonNumber(EDIT_START_FRAME_PNAME);
					if(esfJsN!=null) {
						esf=esfJsN.longValue();
					}
				}
			}
			
			Long eef=null;
			if(jsonObj.containsKey(EDIT_END_FRAME_PNAME)){
				if(!jsonObj.isNull(EDIT_END_FRAME_PNAME)) {
					JsonNumber eefJsN=jsonObj.getJsonNumber(EDIT_END_FRAME_PNAME);
					if(eefJsN!=null) {
						eef=eefJsN.longValue();
					}
				}
			}
			rf.setEditSampleRate(esr);
			rf.setEditStartFrame(esf);
			rf.setEditEndFrame(eef);
			em.merge(rf);
			tx.commit();
			sc.log("Applied edit selection to recording file ID " + recordingFileId+ ": "+esf+" to "+eef);
			
			// Found
			//securityManager.checkReadPermission(req, rf);
			return Response.ok(rf).build();
			
		} finally {
			close(em);
		}

	}




}
