package ipsk.db.speech.utils;

import ipsk.awt.WorkerException;
import ipsk.audio.capture.session.info.RecordingSegment;
import ipsk.audio.capture.session.info.RecordingSequence;
import ipsk.audio.capture.session.info.RecordingSession;
import ipsk.awt.Worker.State;
import ipsk.beans.DOMCodec;
import ipsk.beans.DOMCodecException;
import ipsk.db.speech.Project;
import ipsk.db.speech.RecordingFile;
import ipsk.db.speech.Session;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.script.Script;
import ipsk.db.speech.speaker.Speakers;
import ipsk.persistence.EntityManagerWorker;
import ipsk.util.LocalizableMessage;
import ipsk.util.zip.ZipPackerWorker;
import ipsk.webapps.audio.RecordingFileHelper.RecordingFileEdit;
import ipsk.xml.DOMConverter;
import ipsk.xml.DOMConverterException;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.xml.bind.DataBindingException;
import javax.xml.bind.JAXB;

import org.w3c.dom.Document;


public class SessionExporter extends EntityManagerWorker{
	
	public static class RecordingFileList{
		private Set<RecordingFile> recordingFileList=new HashSet<RecordingFile>();

		public Set<RecordingFile> getRecordingFileList() {
			return recordingFileList;
		}

		public void setRecordingFileList(Set<RecordingFile> recordingFileList) {
			this.recordingFileList = recordingFileList;
		}
	}
	public static String PERSISTENCE_UNIT_NAME="WebSpeechDBPU";
	public static final String SESSION_FILE_PREFIX="session_";
	public static final String SCRIPT_FILE_PREFIX="script_";
	public static final String XML_FILE_EXTENSION="xml";
	private final static String RESOURCE_BUNDLE_NAME="Messages";
	
	private SessionExportConfiguration configuration;
	private int sessionId;

	private File outputBaseDir;
	
	private RecordingFileExporter recordingFileExporter=null;
	private File exportArchiveFile;
	private ZipPackerWorker zipPacker;
	
	public SessionExporter(EntityManagerFactory entityManagerFactory,SessionExportConfiguration configuration){
		super(entityManagerFactory);
		this.configuration=configuration;
	}
	public SessionExporter(EntityManager entityManager,SessionExportConfiguration configuration){
		super(entityManager);
		this.configuration=configuration;
	}
	public void doWork() throws WorkerException{
		if(recordingFileExporter==null){
			recordingFileExporter=new RecordingFileExporter(entityManager,configuration);
		}
		recordingFileExporter.setEntityManager(entityManager);
		Session session=entityManager.find(Session.class, sessionId);
		
		Integer exportFormatVersion=null;
		Project prj=session.getProject();
		if(prj!=null) {
			exportFormatVersion=prj.getExportFormatVersion();
		}
		// create dir if necessary
		if (outputBaseDir==null){
			throw new ExportException("No output directory given!");
		}
		File outputSessionDir=new File(outputBaseDir,Integer.toString(session.getSessionId()));
		if(outputSessionDir.exists()){
			throw new ExportException("Output directory already exists!");
		}	
		
		int recFilesProgressLength=98;
		int zipPackLength=0;
		if(configuration.isCreateSessionZIPArchive()){
			recFilesProgressLength=32;
			zipPackLength=66;
		}
		
		boolean created=outputSessionDir.mkdirs();
		if(!created){
			throw new ExportException("Could not create session export dir "+outputSessionDir);
		}else{
			progressStatus.setProgress(1);
		}
		recordingFileExporter.setOutputDir(outputSessionDir);
		recordingFileExporter.setRunningWithParentWorker();
		
		
		// export session XML file
		Package basePack = Session.class.getPackage();
		DOMCodec ph;
		DOMConverter dc;
		try {
			ph = new DOMCodec(basePack, false);
			dc = new DOMConverter();
		} catch (DOMCodecException e) {
			e.printStackTrace();
			throw new ExportException(e);
		}
		
		Document sessionXMLdoc;
		Document scriptXMLdoc;
		Script script=session.getScript();
		Set<Speaker> speakers=session.getSpeakers();
		PrintWriter pw=null;
		try {
			File sessionXMLFile=new File(outputSessionDir,session.getSessionId()+"."+XML_FILE_EXTENSION);
			if(exportFormatVersion!=null && exportFormatVersion>0) {
				JAXB.marshal(session, sessionXMLFile);
			}else {
				sessionXMLdoc = ph.createDocument(session);
				pw=new PrintWriter(sessionXMLFile,"UTF-8");
				dc.writeXML(sessionXMLdoc, pw);
				pw.close();
			}
			if(script!=null){
				// SpeechRecorder mode:
				// Add script to export
				
				// (Script cannot be marshaled by JAXB)
				File scriptDir=new File(outputSessionDir,"script");
				scriptDir.mkdirs();
				File scriptXMLFile=new File(scriptDir,script.getScriptId()+"."+XML_FILE_EXTENSION);
				scriptXMLdoc=ph.createDocument(script);
				pw=new PrintWriter(scriptXMLFile,"UTF-8");
				dc.writeXML(scriptXMLdoc, pw);
				pw.close();

			}else {
				// Audio recorder mode:
				// Add recording infos list to export
			}

			if(speakers!=null){
				File spksXMLFile=new File(outputSessionDir,"speakers."+XML_FILE_EXTENSION);
				if(exportFormatVersion!=null &&  exportFormatVersion>0) {
					Speakers spksDb=new Speakers();
					spksDb.setSpeakers(speakers);
					JAXB.marshal(spksDb, spksXMLFile);
				}else {
					pw=new PrintWriter(spksXMLFile,"UTF-8");
					for(Speaker sp:speakers){
						Document spkDoc=ph.createDocument(sp);
						dc.writeXML(spkDoc, pw);
						pw.println();
					}
					pw.close();
				}
			}

		} catch (DOMCodecException | FileNotFoundException | DOMConverterException | 
				UnsupportedEncodingException | DataBindingException e) {
			e.printStackTrace();
			throw new ExportException(e);
		} finally{
			if(pw!=null)pw.close();
		}
		
		
		
		progressStatus.setProgress(2);
		
		// export recordings
		Set<RecordingFile> recFiles=session.getRecordingFiles();
		int recFilesSize=recFiles.size();
		int recFilesCount=0;
		
		List<ipsk.audio.capture.session.info.RecordingSequence> rSeqs=new ArrayList<ipsk.audio.capture.session.info.RecordingSequence>();
		for(RecordingFile rf:recFiles){
			if(hasCancelRequest()) {
				progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME, "session.export.canceled", new Object[]{session}));
				return;
			}
			recordingFileExporter.setRecordingFileId(rf.getRecordingFileId());
			recordingFileExporter.setRunningWithParentWorker();
			recordingFileExporter.doWork();
			recFilesCount++;
			
			if(script==null) {
				// Only in audio recorder mode (for now):
				// Add recording infos
				ipsk.audio.capture.session.info.RecordingFile rfInfo=recordingFileExporter.getRecordingFileInfo();
				if(rfInfo!=null) {
					ipsk.audio.capture.session.info.RecordingSequence rSeq=new RecordingSequence();
					rSeq.getRecordingFileList().add(rfInfo);
					RecordingFileEdit rfe=new RecordingFileEdit(rf);
					RecordingSegment rSeg=new RecordingSegment();
					rSeg.setStartTime(rf.getStartedDate());
					Long rfFrameLength=rfe.frameLength();
					if(rfFrameLength!=null) {
						long frames=rfFrameLength.longValue();
						rSeq.setFrameLength(frames);
						rSeg.setFrameLength(frames);
					}
					rfInfo.getRecordingSegmentList().add(rSeg);
					rSeqs.add(rSeq);
				}
			}
			progressStatus.setProgress(2+((recFilesProgressLength*recFilesCount)/recFilesSize));
		}
		if(script==null) {
			// Audio recorder mode:
			// Add recording infos list to export
			
			File rfsXMLFile=new File(outputSessionDir,RecordingSession.DEFAULT_RECORDING_SESSION_INFO_FILENAME);
			RecordingSession rs=new RecordingSession();
			rs.setRecordingSequenceList(rSeqs);
			JAXB.marshal(rs, rfsXMLFile);
		}
		progressStatus.setProgress(2+recFilesProgressLength);
		if(configuration.isCreateSessionZIPArchive()){
			progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME, "zip.creating"));
			FileOutputStream exportArchivestream;
			try {
				exportArchivestream = new FileOutputStream(exportArchiveFile);
			} catch (FileNotFoundException e) {
				throw new ExportException(e);
			}
			zipPacker = new ZipPackerWorker();
//			zipPacker.setLimitTo32bitSizes(true);
			zipPacker.setSrcDir(outputSessionDir);
			zipPacker.setPackRecusive(true);
			zipPacker.setOutputStream(exportArchivestream);
			
			progressStatus.setSubStatus(zipPacker.getProgressStatus(),zipPackLength);
			zipPacker.open();
			zipPacker.setRunningWithParentWorker();
			zipPacker.doWork();
			zipPacker.close();
			
			if(hasCancelRequest()) {
				progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME, "session.export.canceled", new Object[]{session}));
				return;
			}
			
		}
		progressStatus.setSubStatus(null, 0);
		//progressStatus.setProgress(100);
		progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME, "session.export.done", new Object[]{session}));
	}

	public void cancel(){
		if(recordingFileExporter!=null){
			recordingFileExporter.cancel();
		}
		if(zipPacker!=null){
			zipPacker.cancel();
		}
		super.cancel();
	}
	
	public void close() throws WorkerException{
		if(recordingFileExporter!=null){
			recordingFileExporter.close();
		}
		super.close();
	}
	public File getOutputBaseDir() {
		return outputBaseDir;
	}

	public void setOutputBaseDir(File outputBaseDir) {
		this.outputBaseDir = outputBaseDir;
	}

	public int getSessionId() {
		return sessionId;
	}

	public void setSessionId(int sessionId) {
		this.sessionId = sessionId;
	}
	
	public File getExportArchiveFile() {
		return exportArchiveFile;
	}

	public void setExportArchiveFile(File exportArchiveFile) {
		this.exportArchiveFile = exportArchiveFile;
	}
	
}
