package ipsk.webapps.db.speech;

import java.beans.PropertyDescriptor;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Principal;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TimeZone;
import java.util.UUID;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.naming.NamingException;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.servlet.http.HttpServletRequest;

import ips.security.jaas.InetOrgPersonPrincipal;
import ipsk.beans.BeanModel;
import ipsk.beans.PropertyValidationResult;
import ipsk.beans.validation.ValidationException;
import ipsk.beans.validation.ValidationResult;
import ipsk.db.speech.Account;
import ipsk.db.speech.Organisation;
import ipsk.db.speech.Person;
import ipsk.db.speech.Project;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.UserRole;
import ipsk.db.speech.UserRoleId;
import ipsk.db.speech.UserRoleId.RoleName;
import ipsk.db.speech.account.AccountRequest;
import ipsk.db.speech.account.InvitationRequest;
import ipsk.net.SendMail;
import ipsk.net.SendMailException;
import ipsk.persistence.ParameterizedQuery;
import ipsk.persistence.QueryParam;
import ipsk.util.LocalizableMessage;
import ipsk.util.PasswordGenerator;
import ipsk.util.RadixConverters;
import ipsk.webapps.ControllerException;
import ipsk.webapps.PermissionDeniedException;
import ipsk.webapps.ProcessResult;
import ipsk.webapps.ProcessResult.Type;


public class AccountController extends BasicWikiSpeechController<Account> {
	public final static String DEFAULT_ADMIN_LOGIN="admin";
	public final static UserRoleId.RoleName DEFAULT_ADMIN_ROLE=UserRoleId.RoleName.ADMIN;
	public final static String DEFAULT_ADMIN_PASSWORD="corpus";
	
	public final static String PASSWORD_RESET_REQUEST_LIST_ATTR_NAME="passwordRequestManager";
	
	public static final Calendar INTRODUCED_LAST_LOGGED_IN_DATE;
	static {
		INTRODUCED_LAST_LOGGED_IN_DATE=Calendar.getInstance(TimeZone.getTimeZone("GMT +0100"));
		INTRODUCED_LAST_LOGGED_IN_DATE.set(2023,Calendar.MARCH,13);
	}
	
	private Account account;
	
	private AccountRequest accountRequest=null;
	
	private AccountUtils accountUtils;
	
	public AccountController() {
		super("WebSpeechDBPU", Account.class, "account");
		securityManager=new WikiSpeechSecurityManager(this);
		accountUtils=new AccountUtils();
		
	}
	
	public void processRequest(HttpServletRequest req) throws ControllerException {
		super.processRequest(req);
	}

	public Account getById(String login) {
		return (Account)super.getById(login);
	}
	
	public boolean removable(Object bean) throws ControllerException {
		if(currentRequest !=null && bean instanceof Account) {
			Account oAcc=(Account)bean;
			Account reqAcc=getAccountByRequest(currentRequest);
			if(reqAcc!=null) {
				// protect own account for now
				String reqLogin=reqAcc.getLogin();
				String oAccLogin=oAcc.getLogin();
				if(!reqLogin.equals(oAccLogin)) {
					return super.removable(bean);
				}
			}
		}
		return false;
	}
	
	public void initializeAdminAccount(){
		// check if accounts are available
		EntityManager em = getThreadEntityManager();
		Query q = em.createQuery("SELECT object(o) FROM Account o");
		q.setMaxResults(1);
		@SuppressWarnings("unchecked")
		List<ipsk.db.speech.Account> list = q.getResultList();
			
		if(list.size()==0){
			// create a default admin account if account list is empty
			Account initialAdminAccount=new Account(DEFAULT_ADMIN_LOGIN);
			initialAdminAccount.setPassword(DEFAULT_ADMIN_PASSWORD);
			UserRole adminRole=new UserRole(new UserRoleId(initialAdminAccount.getLogin(),DEFAULT_ADMIN_ROLE),initialAdminAccount);
			em.persist(initialAdminAccount);
			em.persist(adminRole);
		}
		
	}
	
	public void  projectAdminAccounts(HttpServletRequest req) throws ControllerException {
		
		boolean isAdmin=req.isUserInRole(UserRoleId.RoleName.ADMIN.name());

		if (isAdmin) {
			createEmptyBeanTableModel(req);
			//String cmd=processCommand(req);
			ParameterizedQuery pq = new ParameterizedQuery(queryType);

			String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;
			pq.setWhereClause("("+jse+".adminOfProjects IS NOT EMPTY)");

			super.processRequest(req,pq);
		}else {
			throw new PermissionDeniedException();
		}
	}
	
	public void applyDigestPassword(Account a,String plainTextPassword){
		
		String rfcPasswd = accountUtils.getRfcPasswordEncryptor().encryptPassword(plainTextPassword);
		a.setRfc2307Password(rfcPasswd);

		MessageDigest sha5Digest=accountUtils.getSha5Digest();
		if(sha5Digest!=null){
			byte[] ba=plainTextPassword.getBytes();
			String digestedPassw=RadixConverters.bytesToHex(sha5Digest.digest(ba));
			a.setSha5HexPassword(digestedPassw);
		}

		String strongPasswd = accountUtils.getStrongPasswordEncryptor()
				.encryptPassword(plainTextPassword);
		a.setStrongPassword(strongPasswd);
	}
	
//	public void digestAndStorePassword(Account a,String plainTextPassword){
//		EntityManager em = getThreadEntityManager();
//
//		applyDigestPassword(a, plainTextPassword);
//		em.merge(a);
//
//	}
	
	
	private void digestPassword(Account a){
		EntityManager em = getThreadEntityManager();
		String plainTextPassword = a.getPassword();
		if (plainTextPassword != null && ! "".equals(plainTextPassword)) {
			String aRfcPw = a.getRfc2307Password();
			if (aRfcPw == null) {
				String rfcPasswd = accountUtils.getRfcPasswordEncryptor().encryptPassword(plainTextPassword);
				a.setRfc2307Password(rfcPasswd);
				em.merge(a);
			}
			String sha5Password=a.getSha5HexPassword();
			MessageDigest sha5Digest=accountUtils.getSha5Digest();
			if(sha5Digest!=null && sha5Password==null){
				byte[] ba=plainTextPassword.getBytes();
				String digestedPassw=RadixConverters.bytesToHex(sha5Digest.digest(ba));
				a.setSha5HexPassword(digestedPassw);
				em.merge(a);
			}
			String aStrongPw = a.getStrongPassword();
			if (aStrongPw == null) {
				String strongPasswd = accountUtils.getStrongPasswordEncryptor()
						.encryptPassword(plainTextPassword);
				a.setStrongPassword(strongPasswd);
				em.merge(a);
			}
		}

	}
	
	public void digestPasswords() {
		EntityManager em = getThreadEntityManager();
		Query q = em.createQuery("SELECT object(o) FROM Account o");
//		q.setMaxResults(1);
		@SuppressWarnings("unchecked")
		List<ipsk.db.speech.Account> list = q.getResultList();
		for (Account a : list) {
			digestPassword(a);
		}
	}
	
	public static synchronized void expireInvalidAccountRequests(EntityManager em) {
		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<AccountRequest> cq = cb.createQuery(AccountRequest.class);
		Root<AccountRequest> qr = cq.from(AccountRequest.class);
		cq.select(qr);
		TypedQuery<AccountRequest> aq = em.createQuery(cq);
		List<AccountRequest> arList=aq.getResultList();
		for(AccountRequest ar:arList) {
			if(!ar.isValid()) {
				if(ar instanceof InvitationRequest) {
					// disconnect first
					InvitationRequest ir=(InvitationRequest)ar;
					List<Project> irPrjs=ir.getProjects();
					for(Project irPrj:irPrjs) {
						irPrj.getInvitationRequests().remove(ir);
						em.merge(irPrj);
					}
					ir.getProjects().clear();
					em.merge(ir);
				}
				em.remove(ar);
			}
		}
	}
	
	public void sendInvitation(HttpServletRequest req) throws ControllerException{
		currentRequest=req;
//		String emailParam=req.getParameter("email");
//		if(emailParam==null){
//			throw new ControllerException("No e-mail address given.");
//		}
//		String projectParam=req.getParameter("project.name");
		
		String loginParam=req.getParameter("login");
		EntityManager em=getThreadEntityManager();
		Account acc=em.find(Account.class,loginParam);
		String email=acc.getEmail();
		if(email==null){
			Person p=acc.getPerson();
			if(p!=null) {
				email=p.getEmail();
			}
		}
		if(email!=null){
			
			SendMail sm=new SendMail();
			if(!sm.isAddressValid(email)){
				throw new ControllerException("Invalid e-mail address.");
			}else{
				// create a password reset request
				UUID uuid=UUID.randomUUID();
				Date requestDate=new Date();
				Date validUntil=new Date(requestDate.getTime()+(1000*60*10)); // TODO Example 10 minutes 
				AccountRequest prr=new AccountRequest(uuid, acc.getLogin(), email,requestDate, validUntil);
			
//				// and store in the servlet container context
//				String attribute=getPasswordRequestManagerAttributeName();
//				AccountRequestsManager pwrm=(AccountRequestsManager) servletContext.getAttribute(attribute);
//				if(pwrm==null){
//					// create a new password reset manager if we do not have one
//					pwrm=new AccountRequestsManager();
//					servletContext.setAttribute(attribute, pwrm);
//				}
//				// expire invalid requests
//				// not sure if this a good place for this
//				pwrm.expireInvalidRequests();
//				
//				// add request
//				pwrm.add(prr);
				
				expireInvalidAccountRequests(em);
				em.persist(prr);
				
				// prepare E-Mail:
				
				// create the URL for te user to set the new password
				String url=req.getScheme()+"://"+req.getServerName()+":"+req.getServerPort()+"/"+req.getContextPath()+"/account/password_form.jsp?uuid="+uuid;
//				String content=url.toString();
			//	Locale loc=req.getLocale();
				ResourceBundle rb=ResourceBundle.getBundle("Messages");
				StringBuffer cb=new StringBuffer();
				String wikispeechSaluTation=rb.getString("wikispeech.salutation");
				cb.append(wikispeechSaluTation);
				cb.append(',');
				cb.append("\n\n");
//				String contentMsg=rb.getString("account.create.by_invitation.message.content");
//				cb.append(contentMsg);
				cb.append("\n\n");
				cb.append(url);
				cb.append("\n\n");
				String greeting=rb.getString("greeting");
				cb.append(greeting);
				cb.append("\n");
				String content=cb.toString();
				String wikiSpeechEmail=servletContext.getInitParameter("wikispeech.email_address");
				if(wikiSpeechEmail==null){
					String errMsg="Could not get WikiSpeech e-mail (sender) address (\"wikispeech.email_address\" in web.xml)";
					servletContext.log(errMsg);
					throw new ControllerException(errMsg);
				}
				try {
					sm.send(wikiSpeechEmail, email, "IPS Wikispeech Password Reset Request", content);
					servletContext.log("Sent password reset request E-Mail from "+wikiSpeechEmail+" to "+email);
				} catch (SendMailException e) {
					servletContext.log("Could not send password reset request E-Mail",e);
					return;
				}
				
			}
		}
		
	}

	public void delete(String login) {

		EntityManager em = getEntityManager();
		EntityTransaction tx = null;
		try {
			tx = em.getTransaction();
			tx.begin();
			Account o = em.find(Account.class, login);
			em.remove(o);
			tx.commit();
		} catch (Exception e) {
			e.printStackTrace();
			if (tx != null && tx.isActive())
				tx.rollback();
		} finally {

			em.close();
		}
		// super.delete(Organisation.class,id);
	}
	
	public String suggestLoginForOrganisation() {
		// TODO check if transaction required here
		String sugg=null;
		EntityManager em = getEntityManager();
		EntityTransaction tx = null;
		try {
			tx = em.getTransaction();
			tx.begin();
			sugg=suggestLoginForOrganisation(em);
			tx.commit();

		} catch (RuntimeException e) {
			e.printStackTrace();

		} finally {

			em.close();
		}
		return sugg;
	}

	@SuppressWarnings("unchecked")
	public static String suggestLoginForOrganisation(EntityManager em) {
		// find highest account value

		int lastIndex = 0;

		Query q = em
				.createQuery("SELECT a.login FROM Account a WHERE a.login LIKE '"
						+ OrganisationController.ACCOUNT_PREFIX
						+ "%' ORDER BY a.login DESC");
		List<String> list = q.getResultList();
		for (String l : list) {

			try {
				lastIndex = OrganisationController.ACCOUNT_FORMAT.parse(l)
						.intValue();
				break;
			} catch (ParseException e) {
				continue;
			}

		}

		return OrganisationController.ACCOUNT_FORMAT.format(lastIndex + 1);
	}

	public String suggestPassword() {
		PasswordGenerator passwdGen = new PasswordGenerator();
		return passwdGen.generateRandom();
	}
	
	public void updateAccountLastLoggedIn(EntityManager em,Account acc) {
		Date currentLastLoggedIn=acc.getLastLoggedIn();
		Date now=new Date();
		if(currentLastLoggedIn==null || currentLastLoggedIn.compareTo(now)<0) {
			account.setLastLoggedIn(now);
		}

		// Set last visited of every project of which this account is an administrator
		Set<Project> admProjs = acc.getAdminOfProjects();
		for(Project admProj:admProjs) {
			Date currLastVisit=admProj.getLastVisitOfProjectAdmin();
			if(currLastVisit==null || currLastVisit.compareTo(now)<0) {
				admProj.setLastVisitOfProjectAdmin(now);
			}
		}
	}
	
	public void login(HttpServletRequest req){
		currentRequest=req;
		String userPrincipalName=req.getUserPrincipal().getName();
		EntityManager em=getThreadEntityManager();
		account=em.find(Account.class,userPrincipalName);
		Account inetOrgAcc=getOrCreateInetOrgPerson();
		if(inetOrgAcc!=null){
			account=inetOrgAcc;
		}
		updateAccountLastLoggedIn(em, account);
	}
	
	public InetOrgPersonPrincipal getInetOrgPersonPrincipal(){
		if(currentRequest!=null){
			Principal up=currentRequest.getUserPrincipal();
			if(up instanceof ips.security.jaas.InetOrgPersonPrincipal){
				return (InetOrgPersonPrincipal)up;
			}
		}
		return null;
	}
	
	public static InetOrgPersonPrincipal getInetOrgPersonPrincipal(HttpServletRequest req){
		if(req!=null){
			Principal up=req.getUserPrincipal();
			if(up instanceof ips.security.jaas.InetOrgPersonPrincipal){
				return (InetOrgPersonPrincipal)up;
			}
		}
		return null;
	}
	
	public static Account getAccountInetOrgPerson(HttpServletRequest req,EntityManager em){
		
		// TODO duplicate code  getOrCreateInetOrgperson
		Account cAcc=null;
		InetOrgPersonPrincipal inetOrgPersonPrinc=getInetOrgPersonPrincipal(req);
		
		
		if(inetOrgPersonPrinc!=null){
			
			//  lookup corresponding JPA/SQL DB account
			String distinguishedName=inetOrgPersonPrinc.getName();
			
			// Fix for Bug ID WSP-0002 2019-02-22 closed
			// "Distinguished names of LDAP accounts were handled case sensitive but they should not"
			String distinguishedNameLc=distinguishedName.toLowerCase(Locale.ENGLISH);
			
			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<Account> cq = cb.createQuery(Account.class);
			Root<Account> qr = cq.from(Account.class);
			cq.select(qr);
			Expression<String> loginLc=cb.lower(qr.<String>get("login"));
			Predicate loginEqualsCaseInsensitive=cb.equal(loginLc, distinguishedNameLc);
			cq.where(loginEqualsCaseInsensitive);
			TypedQuery<Account> aq = em.createQuery(cq);
			List<Account> resultList = aq.getResultList();

			
			int accsSize=resultList.size();
			if(resultList.isEmpty() || accsSize==0){
				
			}else {
				cAcc=resultList.get(0);
//				if(accsSize>1) {
//					getServletContext().log("WARNING: "+accsSize+" account entries found for case insensitive query "+distinguishedName);
//					getServletContext().log("WARNING: Using first account entry: "+cAcc.getLogin());
//				}
				
			}
			
		}
		return cAcc;
	}

	
	
	public boolean copyOfInetOrgPersonRequired(){
		InetOrgPersonPrincipal inetOrgPersonPrinc=getInetOrgPersonPrincipal();
		if(inetOrgPersonPrinc!=null){
			//  lookup corresponding JPA/SQL DB account
			String distinguishedName=inetOrgPersonPrinc.getName();
			EntityManager em = getThreadEntityManager();
			
			Account cAcc=em.find(Account.class,distinguishedName);
			return(! (cAcc!=null));
		}
		return false;
	}
	
	public Account getOrCreateInetOrgPerson(){
		//Account newAcc=null;
		Account cAcc=null;
		InetOrgPersonPrincipal inetOrgPersonPrinc=getInetOrgPersonPrincipal();
		if(inetOrgPersonPrinc!=null){
			
			//  lookup corresponding JPA/SQL DB account
			String distinguishedName=inetOrgPersonPrinc.getName();
//			EntityManager em = getThreadEntityManager();
//			
//			Account cAcc=em.find(Account.class,distinguishedName);
//			if(cAcc==null){
//				cAcc=new Account(distinguishedName);
//				cAcc.setEditable(false);
//				em.persist(cAcc);
//				newAcc=cAcc;
//			}
//			
			// Fix for Bug ID WSP-0002 2019-02-22 closed
			// "Distinguished names of LDAP accounts were handled case sensitive but they should not"
			String distinguishedNameLc=distinguishedName.toLowerCase(Locale.ENGLISH);
			EntityManager em = getThreadEntityManager();

			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<Account> cq = cb.createQuery(Account.class);
			Root<Account> qr = cq.from(Account.class);
			cq.select(qr);
			Expression<String> loginLc=cb.lower(qr.<String>get("login"));
			Predicate loginEqualsCaseInsensitive=cb.equal(loginLc, distinguishedNameLc);
			cq.where(loginEqualsCaseInsensitive);
			TypedQuery<Account> aq = em.createQuery(cq);
			List<Account> resultList = aq.getResultList();

			
			int accsSize=resultList.size();
			if(resultList.isEmpty() || accsSize==0){

				cAcc=new Account(distinguishedNameLc);
				cAcc.setEditable(false);
				try {
					String emailAddrAttr=inetOrgPersonPrinc.getAttrMail();
					if(emailAddrAttr!=null) {
						String emailAddr=emailAddrAttr.trim();
						cAcc.setEmail(emailAddr);
					}
				} catch (NamingException e) {
					e.printStackTrace();
				}
				em.persist(cAcc);
				//newAcc=cAcc;
			}else {
				cAcc=resultList.get(0);
				// Check email
				String cAccEmail=cAcc.getEmail();
				if(cAccEmail==null || cAccEmail.isBlank()) {
					// email not set, try to retrieve from InetOrg principal (from LDAP)
					try {
						String emailAddrAttr=inetOrgPersonPrinc.getAttrMail();
						if(emailAddrAttr!=null) {
							// Email found, set to account
							String emailAddr=emailAddrAttr.trim();
							cAcc.setEmail(emailAddr);
							em.merge(cAcc);
						}
					} catch (NamingException e) {
						e.printStackTrace();
					}
					
				}
				
				if(accsSize>1) {
					getServletContext().log("WARNING: "+accsSize+" account entries found for case insensitive query "+distinguishedName);
					getServletContext().log("WARNING: Using first account entry: "+cAcc.getLogin());
				}

			}


//			Person ap=cAcc.getPerson();
//			if(ap==null){
//
//				Speaker p=new Speaker();
//
//				String foreName;
//				try {
//					foreName = inetOrgPersonPrinc.getAttrGivenname();
//					p.setForename(foreName);
//					String lastName=inetOrgPersonPrinc.getAttrSurname();
//					p.setName(lastName);
//				} catch (NamingException e) {
//					// TODO Auto-generated catch block
//					e.printStackTrace();
//				}
//				em.persist(p);
//
//				cAcc.setPerson(p);
//				em.merge(cAcc);
//
//				// Query all orgas with userDn ends with baseDn of Orga
//				// e.g. 'uid=test,ou=People,dc=example,dc=org' ends with 'dc=example,dc=org';
//				// -> LOCATE returns 20  = 54-34 -> match
//				Query q=em.createQuery("SELECT orga FROM Organisation orga WHERE orga.baseDN IS NOT NULL AND LOCATE(orga.baseDN,'"+distinguishedName+"') = (LENGTH('"+distinguishedName+"') - LENGTH(orga.baseDN) +1 )");
//
//				@SuppressWarnings("unchecked")
//				List<Organisation> maOrgas=q.getResultList();
//
//				//				CriteriaBuilder cb = em.getCriteriaBuilder();
//				//				Metamodel m = em.getMetamodel();
//				//
//				//				//						EntityType<Organisation> Organisation_ = m.entity(Organisation.class);
//				//				CriteriaQuery<Organisation> cq = cb.createQuery(Organisation.class);
//				//
//				//				Root<Organisation> orga = cq.from(Organisation.class);
//				//				//						EntityType<Organisation> Organisation_ = orga.getModel();
//				//				cq.select(orga);
//				//				cq.where(orga.get(Organisation_.baseDN).isNotNull());
//				//				cq.where(cb.like(orga.get(Organisation_.baseDN),"%"+distinguishedName));
//				//
//				//				List<Organisation> maOrgas = em.createQuery(cq).getResultList();
//				for(Organisation mOrga:maOrgas){
//					p.getOrganisations().add(mOrga);
//					mOrga.getPersons().add(p);
//					em.merge(p);
//					em.merge(mOrga);
//				}
//				//em.merge(cAcc);
//			}
			
		}
		return cAcc;
	}

	@SuppressWarnings("unchecked")
	public List<ipsk.db.speech.Account> getAccounts() {
		EntityManager em = getThreadEntityManager();
		
		List<ipsk.db.speech.Account> list = null;
		try {
		
			Query q = em.createQuery("select object(o) from Account as o");
			q.setMaxResults(batchSize);
			q.setFirstResult(firstItem);
			list = q.getResultList();
			
		} catch (RuntimeException e) {
			e.printStackTrace();
			close();
			throw e;
		} finally {

			// em.close();
		}

		return list;
	}

	protected void setPropertiesOnNew(Object o) {
		if (o instanceof Account) {
			Account a = (Account) o;
			if (a.getOrganisation() != null) {
				// suggest login/paasword for organisations
				String suggestedLogin = suggestLoginForOrganisation();
				String suggestedPasswd = suggestPassword();
				a.setLogin(suggestedLogin);
				a.setPassword(suggestedPasswd);
			}
		}
	}

	protected void setPropertiesOnCreate(Object o) {
		if (o instanceof Account) {
			Account account = (Account) o;
			account.setCreated(new Date());
		}
	}

	public void newAccount() {
		this.account = new Account();

	}
	
	
	
	public ValidationResult validate(Object o,ValidationResult validationResult) throws ValidationException{
		ValidationResult vr=super.validate(o, validationResult);
		if(o instanceof Account) {
			Account acc=(Account)o;
			String login=acc.getLogin();
			LocalizableMessage errMsgUsername=AccountUtils.validateUsername(login);
			
			if(errMsgUsername!=null) {
				vr.putPropertyValidationResult("login",new PropertyValidationResult(PropertyValidationResult.Type.ERROR,errMsgUsername));
				vr.setType(ValidationResult.Type.ERRORS);
			}
			
			LocalizableMessage errMsgEmail=null;
			String email=acc.getEmail();
			if(email!=null && !email.isBlank()) {
				try {
					InternetAddress[] testIas = InternetAddress.parse(email);

					if(testIas==null) {
						errMsgEmail=new LocalizableMessage("Could not parse e-mail address");
					}else if(testIas.length==0){
						errMsgEmail=new LocalizableMessage("Could not parse e-mail address");
					}else if(testIas.length>1){
						errMsgEmail=new LocalizableMessage("Multiple e-mail addresses not allowed");
					}else{
						InternetAddress ias=testIas[0];
						ias.validate();
					}
				} catch (AddressException e1) {
					errMsgEmail=new LocalizableMessage("Invalid e-mail address: "+e1.getMessage());
				}
				//				try {
				//					new InternetAddress(email,true);
				//				} catch (AddressException e) {
				//					errMsg=new LocalizableMessage("Invalid e-mail address: "+e.getMessage());
				//					vr.putPropertyValidationResult("email",new PropertyValidationResult(PropertyValidationResult.Type.ERROR,errMsg));
				//					vr.setType(ValidationResult.Type.ERRORS);
				//				}
			}
			if(errMsgEmail!=null) {
				vr.putPropertyValidationResult("email",new PropertyValidationResult(PropertyValidationResult.Type.ERROR,errMsgEmail));
				vr.setType(ValidationResult.Type.ERRORS);
			}

		}
		return vr;
	}
	

	// public void processRequest(HttpServletRequest request) throws
	// ControllerException {
	// if (request.getCharacterEncoding() == null) {
	// try {
	// System.out.println("Set encoding to UTF 8");
	// request.setCharacterEncoding("UTF-8");
	// } catch (UnsupportedEncodingException e) {
	// // TODO Auto-generated catch block
	// e.printStackTrace();
	// }
	// }
	// String command = request.getParameter("_cmd");
	// System.out.println("Cmd: " + command);
	// if (command != null && request.getParameter("_cancel") == null) {
	// // if (command.equals("add")) {
	// //
	// //
	// // Map map = request.getParameterMap();
	// //
	// // newAccount();
	// // MapConverter mapConverter = new MapConverter();
	// //
	// //
	// // account.setCreated(new Date());
	// //
	// // //Set<UserRoles> urs=account.getUserRoleses();
	// // //urs.add(userRole);
	// // //account.setUserRoleses(urs);
	// //
	// // try {
	// // mapConverter.setBeanProperties(account, map);
	// //
	// // } catch (MapConverterException e) {
	// // // TODO Auto-generated catch block
	// // e.printStackTrace();
	// // }
	// // if (command.equals("add")){
	// // String
	// orgaIdStr=request.getParameter("organisation."+MapConverter.OBJECT_ID);
	// // String
	// personIdStr=request.getParameter("person."+MapConverter.OBJECT_ID);
	// //
	// //
	// // EntityManager em=getThreadEntityManager();
	// // EntityTransaction tx=null;
	// // try{
	// // tx=em.getTransaction();
	// // tx.begin();
	// // if(personIdStr!=null){
	// // Person p=em.find(Person.class, Integer.parseInt(personIdStr));
	// // account.setPerson(p);
	// // }
	// // if (orgaIdStr!=null){
	// // Organisation o=em.find(Organisation.class,
	// Integer.parseInt(orgaIdStr));
	// // account.setOrganisation(o);
	// // }
	// //
	// // UserRoleId urId=new UserRoleId(account.getLogin(),"organisation");
	// // UserRole userRole=(UserRole)em.find(UserRole.class, urId);
	// // if(userRole==null)userRole=new UserRole(urId,account);
	// // Set<UserRole> urs=account.getUserRoles();
	// // urs.add(userRole);
	// // account.setUserRoles(urs);
	// // account.setCreated(new Date());
	// // em.persist(account);
	// // tx.commit();
	// // }catch(PersistenceException e){
	// // e.printStackTrace();
	// // if(tx!=null && tx.isActive())tx.rollback();
	// // throw e;
	// //
	// // }finally{
	// // close();
	// // }
	// // }
	// //// }else{
	// //// merge(account);
	// //// }
	// //
	// //
	// //}else
	// // if (command.equals("new")){
	// // account=new Account();
	// //
	// // String personIdParam=request.getParameter("person_id");
	// // if (personIdParam!=null){
	// // account.setPerson(getPersonById(Integer.parseInt(personIdParam)));
	// // }else{
	// // String orgaIdParam=request.getParameter("organisation_id");
	// // if (orgaIdParam!=null){
	// //
	// account.setOrganisation(getOrganisationById(Integer.parseInt(orgaIdParam)));
	// //
	// // }
	// // }
	// //
	// // }else
	// if (command.equals("delete")){
	// String id=request.getParameter("id");
	// delete(id);
	// }else{
	// super.processRequest(request);
	// }
	// } else {
	// super.processRequest(request);
	// }
	// }

	public Object getEditObject() {
		return account;
	}

	public Account getAccount() {
		return account;
	}

	public void setAccount(Account account) {
		this.account = account;
	}

//	public void accountsByAccount(HttpServletRequest req)
//	throws ControllerException {
//		EntityManager em = getThreadEntityManager();
//		Account acc = AccountController.getAccountByRequest(req, em);
//		Project selProject=getSelectedProject(req);
//		setDropInSecureItems(true);
//		createEmptyBeanTableModel(req);
//		//beanTableModel = new BeanTableModel();
//		if (acc == null)
//			return;
//
//		ParameterizedQuery pq = new ParameterizedQuery(queryType);
//		String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;
//		if(selProject==null){
//			pq.setWhereClause("( "+jse+" = :account ) OR (EXISTS (SELECT pr FROM Project pr WHERE :account MEMBER OF pr.adminAccounts AND "
//					+ jse + ".organisation MEMBER OF pr.organisations))");
//
//			pq.setQueryParams(new QueryParam[] { new QueryParam("account", acc) });
//		}else{
////			pq.setWhereClause("( "+jse+" = :account ) OR (EXISTS (SELECT pr FROM Project pr WHERE :account MEMBER OF pr.adminAccounts AND "
////					+ jse + ".organisation MEMBER OF pr.organisations AND pr.name = :selProjectName))");
////			
//			// problem with the above query:
//			// In WikiSpeech the OneToOne relationship is violated by the accounts 'lusigrav','lusiadmin'
//			// they have both 'IPS' as organisation but account 'draxler' is the account of 'IPS'
//			// the query above works but is not compatible with the Account check of the WikiSpeechSecurityManager
//			// The query below uses the same "direction" as the security manager.
//			pq.setSelectDistinct(true);
//			pq.setAdditionalFromDeclarations(new ParameterizedQuery.FromDeclaration[]{new ParameterizedQuery.FromDeclaration("Project","pr"),new ParameterizedQuery.FromDeclaration("Organisation","orga")});
//			pq.setWhereClause("( "+jse+" = :account ) OR ( pr.name = :selProjectName AND :account MEMBER OF pr.adminAccounts AND "+jse+" = orga.account AND orga MEMBER OF pr.organisations)");
//			pq.setQueryParams(new QueryParam[] { new QueryParam("account", acc),new QueryParam("selProjectName",selProject.getName()) });
//		}
//		setParameterizedQuery(pq);
//		processListRequest(req);
//		setDropInSecureItems(false);
//	}
//	

	
//	public void allAccountsByAccount(HttpServletRequest req)
//			throws ControllerException {
//				Account acc = getAccountByRequest(req);
//				createEmptyBeanTableModel(req);
//				if (acc == null)
//					return;
//
//				ParameterizedQuery pq = new ParameterizedQuery(queryType);
//				String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;
//				pq.setWhereClause("( "+jse+" = :account ) OR (EXISTS (SELECT pr FROM Project pr WHERE :account MEMBER OF pr.adminAccounts AND "
//							+ jse + ".organisation MEMBER OF pr.organisations))");
//				pq.setQueryParams(new QueryParam[] { new QueryParam("account", acc) });
//				
//				setParameterizedQuery(pq);
//				processListRequest(req);
//			}
	
	public void addDirectoryServiceAccount(HttpServletRequest request) throws ControllerException{
		currentRequest=request;
		clear();
		String command=processCommand(request);

		if(CMD_ADD.equals(command)){
			String loginParam=request.getParameter("login");
			String loginParamLc=loginParam.toLowerCase(Locale.ENGLISH);
			EntityManager em=getThreadEntityManager();
			if(AccountUtils.accountExists(em, loginParamLc)) {
				ValidationResult vr=new ValidationResult();
				PropertyValidationResult pvr=new PropertyValidationResult(PropertyValidationResult.Type.ERROR, new LocalizableMessage("Account "+loginParamLc+" already exists!"));
				vr.putPropertyValidationResult("login", pvr);
				processResult=new ProcessResult(vr);
			}else {
				Account newDsAcc=new Account(loginParamLc);
				newDsAcc.setEditable(false);
				setPropertiesOnCreate(newDsAcc);
//				String dsServicePattern=request.getServletContext().getInitParameter("directoryServiceUserDnPattern");
//				String dsEmailPattern=request.getServletContext().getInitParameter("directoryServiceEmailPattern");
//				if(dsServicePattern!=null && dsEmailPattern!=null) {
//					String dsRegexPattStr=dsServicePattern.replace("{0}", "(.*)");
//					Pattern dsRegexPattern = Pattern.compile(dsRegexPattStr);
//					
//					Matcher userMatch = dsRegexPattern.matcher(loginParam);
//					if(userMatch.matches()) {
//						int gc=userMatch.groupCount();
//						if(gc==1) {
//							String username=userMatch.group(1);
//							String email=dsEmailPattern.replace("{0}", username);
//							newDsAcc.setEmail(email);
//							//System.out.println(userDnStr+": Username: "+userMatch.find()+" "+gc+" "+username);
//						}
//					}
//				}
				em.persist(newDsAcc);
				processResult=new ProcessResult(ProcessResult.Type.SUCCESS);
			}
		}else {
			super.processRequest(request);
		}

	}
	
	public void allAccountsByAccount(HttpServletRequest req)
			throws ControllerException {
				Account acc = getAccountByRequest(req);
				createEmptyBeanTableModel(req);
				if (acc == null)
					return;

				ParameterizedQuery pq = new ParameterizedQuery(queryType);
				String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;
//				pq.setWhereClause("( "+jse+" = :account ) OR "+
//								  "(EXISTS (SELECT pr FROM Project pr WHERE :account MEMBER OF pr.adminAccounts AND  "+ jse + ".organisation MEMBER OF pr.organisations)) OR "+
//							      "(( "+jse+".person IS NOT NULL) AND (EXISTS (SELECT pr FROM Project pr WHERE :account MEMBER OF pr.adminAccounts AND pr MEMBER OF "+jse+ ".projects)))");
//				
				// Fix for WSP-0043 2020-08-11 ? 
				// "Project admin does not see user accounts which have no associated speaker or organization, but are members of the project."
				
				pq.setWhereClause("( "+jse+" = :account ) OR "+
						  "(EXISTS (SELECT pr FROM Project pr WHERE :account MEMBER OF pr.adminAccounts AND  "+ jse + ".organisation MEMBER OF pr.organisations)) OR "+
					      "(EXISTS (SELECT pr FROM Project pr,UserRole ur WHERE :account MEMBER OF pr.adminAccounts AND pr MEMBER OF "+jse+ ".projects))");
				pq.setQueryParams(new QueryParam[] { new QueryParam("account", acc) });
				pq.setSelectDistinct(true);
				setParameterizedQuery(pq);
				processListRequest(req);
			}
	

	
	public void accountsByAccount(HttpServletRequest req)
			throws ControllerException {

		Project selProject=getSelectedProject(req);
		if(selProject==null){
			allAccountsByAccount(req);
		}else {
			Account acc = getAccountByRequest(req);
			createEmptyBeanTableModel(req);
			if (acc == null)
				return;

			ParameterizedQuery pq = new ParameterizedQuery(queryType);
			String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;

			pq.setSelectDistinct(true);
			pq.setAdditionalFromDeclarations(new ParameterizedQuery.FromDeclaration[]{new ParameterizedQuery.FromDeclaration("Project","pr"),new ParameterizedQuery.FromDeclaration("Organisation","orga")});
			pq.setWhereClause("( "+jse+" = :account ) OR ( pr.name = :selProjectName AND :account MEMBER OF pr.adminAccounts AND "+jse+" MEMBER OF orga.accounts AND orga MEMBER OF pr.organisations)");
			pq.setQueryParams(new QueryParam[] { new QueryParam("account", acc),new QueryParam("selProjectName",selProject.getName()) });

			setParameterizedQuery(pq);
			processListRequest(req);
			
		}
	}
	
	
	
	public static Account getAccountByRequest(HttpServletRequest req,EntityManager em){

		Account acc=null;
		if(req!=null){
			String accName=req.getRemoteUser();
			Principal userPrincipal=req.getUserPrincipal();
			if(userPrincipal!=null){
				String upNm=userPrincipal.getName();
				if(upNm!=null){
					accName=upNm;
				}
			}
			if(accName!=null){
				acc= em.find(Account.class,accName);
				//acc=AccountController.accountByUsername(em, accName);
				if(acc==null) {
					acc=getAccountInetOrgPerson(req, em);
				}
				if(acc!=null) {
					em.refresh(acc);
				}
			}
		}
		return acc;
	}
	
	public static String getPasswordRequestManagerAttributeName(){
		return AccountController.class.getName()+"."+PASSWORD_RESET_REQUEST_LIST_ATTR_NAME;
	}

	public void passwordResetRequest(HttpServletRequest req,String reqLogin){
		
		// Fix WSP-0056
		// "Reset password request does not work for case insensitive user names when the given user name is not equal to the stored one."
		//account=getById(reqLogin);
		EntityManager em=getThreadEntityManager();
		account=accountByUsername(em, reqLogin);
		if(account==null){
			servletContext.log("Password reset request for unkown login: "+reqLogin);
//			ValidationResult vr=new ValidationResult(ValidationResult.Type.ERRORS);
//			LocalizableMessage locMsg=new Loc
//			PropertyValidationResult pvr=new PropertyValidationResult(PropertyValidationResult.Type.ERROR,locMsg);
//			vr.putPropertyValidationResult("login", null);
			processResult=new ProcessResult(ProcessResult.Type.ERROR);
			return;
		}
		String email=account.contactEmail();
		if(email==null || "".equals(email)){
			servletContext.log("Password reset request login: "+reqLogin+": no E-Mail address found!");
			processResult=new ProcessResult(ProcessResult.Type.ERROR);
		}else{
			SendMail sm=new SendMail();
			if(sm.isAddressValid(email)){
				// create a password reset request
				UUID uuid=UUID.randomUUID();
				Date requestDate=new Date();
				Date validUntil=new Date(requestDate.getTime()+(1000*60*10)); // TODO Example 10 minutes 
				AccountRequest prr=new AccountRequest(uuid, account.getLogin(),email, requestDate, validUntil);
				
//				// and store in the servlet container context
//				String attribute=getPasswordRequestManagerAttributeName();
//				AccountRequestsManager pwrm=(AccountRequestsManager) servletContext.getAttribute(attribute);
//				if(pwrm==null){
//					// create a new password reset manager if we do not have one
//					pwrm=new AccountRequestsManager();
//					servletContext.setAttribute(attribute, pwrm);
//				}
//				// expire invalid requests
//				// not sure if this a good place for this
//				pwrm.expireInvalidRequests();
//				
//				// add request
//				pwrm.add(prr);
				
				
				expireInvalidAccountRequests(em);
				em.persist(prr);
				// prepare E-Mail:
				
				// create the URL for te user to set the new password
				String url=req.getScheme()+"://"+req.getServerName()+":"+req.getServerPort()+req.getContextPath()+"/account/password_form.jsp?uuid="+uuid;
//				String content=url.toString();
				Locale loc=Locale.ENGLISH;
				Locale reqLoc=req.getLocale();
				if(reqLoc!=null) {
					loc=reqLoc;
				}
				
				ResourceBundle rb=ResourceBundle.getBundle("Messages",loc);
				StringBuffer cb=new StringBuffer();
				String wikispeechSaluTation=rb.getString("wikispeech.salutation");
				cb.append(wikispeechSaluTation);
				cb.append(',');
				cb.append("\n\n");
				String contentMsg=rb.getString("password.reset.request.message_content");
				cb.append(contentMsg);
				cb.append("\n\n");
				cb.append(url);
				cb.append("\n\n");
				String greeting=rb.getString("greeting");
				cb.append(greeting);
				cb.append("\n");
				String content=cb.toString();
				String wikiSpeechEmail=servletContext.getInitParameter("wikispeech.email_address");
				if(wikiSpeechEmail==null){
					servletContext.log("Could not get WikiSpeech E-Mail (sender) address (\"wikispeech.email_address\" in web.xml)");
					processResult=new ProcessResult(ProcessResult.Type.ERROR);
					return;
				}
				try {
					sm.send(wikiSpeechEmail, email, "IPS Wikispeech Password Reset Request", content);
					servletContext.log("Sent password reset request E-Mail from "+wikiSpeechEmail+" to "+email);
					processResult=new ProcessResult(ProcessResult.Type.SUCCESS);
				} catch (SendMailException e) {
					servletContext.log("Could not send password reset request E-Mail",e);
					processResult=new ProcessResult(ProcessResult.Type.ERROR);
					return;
				}
				
			
			}
		}
	}

	public void passwordResetRequest(HttpServletRequest req){
		// TODO handling of constants (page URLs, EMail sender address,...)
		String reqLogin=req.getParameter("login");
		passwordResetRequest(req, reqLogin);
	}
	
//	public String getInvitationEmail(){
//		Account acc;
//		try {
//			acc = getItem();
//		} catch (ControllerException e) {
//			return null;
//		}
//		Person p=acc.getPerson();
//		if(p!=null){
//			String email=p.getEmail();
//			if(email!=null && ! "".equals(email)){
//				return email;
//			}
//		}
//		// TODO same for organisations ? 
//		return null;
//	}
	
	/**
	 * Sets encrypted account password.
	 * @param req the HTTP request, 
	 * @throws NoSuchAlgorithmException
	 */
	public void setPassword(HttpServletRequest req) throws ControllerException{
		this.currentRequest=req;
		
		ValidationResult vr=new ValidationResult();
		setAccount(null);
		String passw1=req.getParameter("password1");
		String passw2=req.getParameter("password2");
		String login=req.getParameter("login");
		if(login==null){
			vr=new ValidationResult(ValidationResult.Type.ERRORS);
			processResult=new ProcessResult(vr);
		}else{
			EntityManager em=getThreadEntityManager();
			account=em.find(Account.class,login);
			securityManager.checkMergePermission(req, account);
			beanModel=new BeanModel<Account>(account);
			String cmd=processCommand(req,new String[]{"submit"});
			if("submit".equals(cmd)){
				secureRequestTokenProvider.checkSecureRequestToken(req);
				LocalizableMessage pwCheckMsg=accountUtils.checkPassword(passw1, passw2);
				if(pwCheckMsg==null) {
					applyDigestPassword(account, passw1);
					em.merge(account);
					processResult=new ProcessResult(ProcessResult.Type.SUCCESS);
				}else {
					processResult=new ProcessResult(Type.ERROR);
					processResult.setResultKey("failed");
					processResult.setLocalizableMessage(pwCheckMsg);
					processResult.setValidationResult(new ValidationResult(ValidationResult.Type.ERRORS));
				}
			}else {
			
				
				processResult=null;
			}
		}
	}
	
	public Account getItem() throws ControllerException {
		Account a=super.getItem();
		return a;
	}
	
	
	public boolean getShowAccountName(){
		if(accountRequest!=null){
			return(accountRequest instanceof InvitationRequest);
		}
		return false;
	}
	public void passwordRequest(HttpServletRequest req) throws NoSuchAlgorithmException, ControllerException{
		String cmd=processCommand(req,new String[]{"submit"});
		if("submit".equals(cmd)){
			passwordSetRequest(req);
		}else{
			accountRequest=null;
			setAccount(null);
			String uuidString=req.getParameter("uuid");
			if(uuidString!=null && ! uuidString.equals("")){
				UUID uuid=UUID.fromString(uuidString);
				//AccountRequestsManager pwrm=(AccountRequestsManager)servletContext.getAttribute(getPasswordRequestManagerAttributeName());
				EntityManager em=getThreadEntityManager();
				if(em!=null){

					AccountRequest pwrr=accountUtils.accountRequestByUUID(em, uuid);
					if(pwrr!=null){
						accountRequest=pwrr;
						String reqLogin=pwrr.getLogin();
						Account acc=getById(reqLogin);
						if(acc!=null){
							if(pwrr.isValid()){
								// OK

								account=acc;
							}else{
								servletContext.log("Expired password request for \""+account+"\" !");
								processResult=new ProcessResult(Type.ERROR);
								processResult.setResultKey("failed");
							}
						}else{
							servletContext.log("Password request: Corresponding account for login\""+reqLogin+"\" not found !");
							processResult=new ProcessResult(Type.ERROR);
							processResult.setResultKey("failed");
						}
					}else{
						servletContext.log("Password request: matching request not found!");
						processResult=new ProcessResult(Type.ERROR);
						processResult.setResultKey("failed");
					}
				}else{
					servletContext.log("Password request: matching request not found!");
					processResult=new ProcessResult(Type.ERROR);
					processResult.setResultKey("failed");
				}
			}
			if(account !=null){
				servletContext.log("Valid password request for \""+account+"\"");
			}
		}
	}
	
	public static boolean accountHasUserRole(Account acc,UserRoleId.RoleName userRoleName) {
		Set<UserRole> urs=acc.getUserRoles();
		for(UserRole ur:urs) {
			if(userRoleName.equals(ur.getId().getRoleName())) {
				return true;
			}
		}
		return false;
	}
	
	private void applyAccountEMail(Account a,String email){
		a.setEmail(email);
		if(a.getLogin().equalsIgnoreCase(a.getEmail())) {
			a.setLoginCaseInsensitiv(true);
		}
	}
	
	
	public static Account accountByUsername(EntityManager em,String username) {
		String usernameLc=username.toLowerCase(Locale.ENGLISH);
		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<Account> cq = cb.createQuery(Account.class);
		Root<Account> qr = cq.from(Account.class);
		cq.select(qr);
		
		// caseInsensitive login
		Predicate loginCaseInsensitive=cb.equal(qr.<Boolean>get("loginCaseInsensitiv"),true);
		Expression<String> loginLc=cb.lower(qr.<String>get("login"));
		Predicate loginEqualsCaseInsensitive=cb.equal(loginLc,usernameLc);
		Predicate caseInsensitiveMatch=cb.and(loginCaseInsensitive,loginEqualsCaseInsensitive);
		
		// case sensitive login
		Predicate loginEquals=cb.equal(qr.<String>get("login"),username);
		
		Predicate accountMatch=cb.or(loginEquals,caseInsensitiveMatch);
		
		cq.where(accountMatch);
		TypedQuery<Account> aq = em.createQuery(cq);
		List<Account> accsList=aq.getResultList();
		if(accsList.size()==1) {
			return accsList.get(0);
		}else {
			return null;
		}
	}
	
	public static List<Account> accountsByLoginCaseInsensitive(EntityManager em,String login) {
		String reqLoginLc=login.toLowerCase(Locale.ENGLISH);
		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<Account> cq = cb.createQuery(Account.class);
		Root<Account> qr = cq.from(Account.class);
		cq.select(qr);
		Expression<String> loginLc=cb.lower(qr.<String>get("login"));
		Predicate loginEqualsCaseInsensitive=cb.equal(loginLc,reqLoginLc);
		cq.where(loginEqualsCaseInsensitive);
		TypedQuery<Account> aq = em.createQuery(cq);
		List<Account> accsList=aq.getResultList();
		return accsList;
	}
	

	protected ValidationResult validateAlreadyExisting(EntityManager em,PropertyDescriptor idPd,Account mergeItem) throws ControllerException {
		ValidationResult validationResult=null;
		String login=mergeItem.getLogin();
		
		if(AccountUtils.accountExists(getThreadEntityManager(), login,true)) {
			LocalizableMessage errMsg=new LocalizableMessage("Account already exists!");
			validationResult=new ValidationResult();
			validationResult.putPropertyValidationResult("login",new PropertyValidationResult(PropertyValidationResult.Type.ERROR,errMsg));
			validationResult.setType(ValidationResult.Type.ERRORS);
		}
		
		return validationResult;
	}
	
	public void newAccountRequest(HttpServletRequest req) throws NoSuchAlgorithmException, ControllerException{

		String cmd=processCommand(req,new String[]{"submit"});

		String uuidString=req.getParameter("uuid");
		servletContext.log("New account request UUID: "+uuidString);
		if(uuidString!=null && ! uuidString.equals("")){
			UUID uuid=UUID.fromString(uuidString);
			//AccountRequestsManager pwrm=(AccountRequestsManager)servletContext.getAttribute(getPasswordRequestManagerAttributeName());
			EntityManager em=getThreadEntityManager();
			if(em!=null){

				AccountRequest pwrr=accountUtils.accountRequestByUUID(em, uuid);

				if(pwrr!=null && pwrr instanceof InvitationRequest){
					InvitationRequest invitationRequest=(InvitationRequest)pwrr;

					accountRequest=invitationRequest;
					String reqLogin=invitationRequest.getLogin();
					String email=invitationRequest.getEmail();

					if(!AccountUtils.accountExists(em, reqLogin,true)){
						if(invitationRequest.isValid()){
							servletContext.log("Valid matching invitation found.");
							if("submit".equals(cmd)){
								// form submit
								account=new Account(reqLogin);
								//								createAccountRequest(req);
								String passw1=req.getParameter("password1");
								String passw2=req.getParameter("password2");
								if(passw1 != null && passw2 !=null){
									LocalizableMessage pwCheckMsg=accountUtils.checkPassword(passw1, passw2);
									if (pwCheckMsg==null) {
										// password check OK
										servletContext.log("Create account request for login \""+account.getLogin()+"\"...");
										applyDigestPassword(account, passw1);
										em.persist(account);

										Set<String> userRoles=invitationRequest.getUserRoles();
										// set user roles

										for(String urStr:userRoles){
											RoleName rn=RoleName.parse(urStr);
											if(rn!=null){
												UserRoleId urId=new UserRoleId(reqLogin,rn);
												UserRole ur=new UserRole(urId);
												em.persist(ur);
												account.getUserRoles().add(ur);
												em.merge(ur);
												em.merge(account);
												servletContext.log("Added user role "+ ur+" for login \""+account.getLogin()+"\"");
											}
										}

										// Set account E-Mail and case insensitive login if login equals email
										if(email!=null){
											applyAccountEMail(account, email);
										}

										Organisation orga=invitationRequest.getOrganisation();
										if(orga!=null) {

											// create account for organisation
											//Organisation orga=em.find(Organisation.class,orgaId);
											//													if(orga==null) {
											//														servletContext.log("Organisation for account request "+invitationRequest+" not found!");
											//														throw new ControllerException("Organisation with ID "+orgaId+ " not found");
											//													}
											account.setOrganisation(orga);
											orga.getAccounts().add(account);

											// Double check organisation role
											if(!AccountController.accountHasUserRole(account, UserRoleId.RoleName.ORGANISATION)) {
												UserRoleId urId=new UserRoleId(reqLogin,UserRoleId.RoleName.ORGANISATION);
												UserRole ur=new UserRole(urId);
												em.persist(ur);
												account.getUserRoles().add(ur);
												em.merge(ur);
											}

											em.merge(account);
											em.merge(orga);

										}else {

											if(AccountController.accountHasUserRole(account, UserRoleId.RoleName.SUBJECT)) {
												servletContext.log("Login \""+account.getLogin()+"\" has role SUBJECT");
//												// create speaker/person object
//												Speaker spk=new Speaker();
//
//												// Fix: WSP-0044 2020-09-08
//												// "Privacy: E-mail address is stored to person when a subject is invited per E-Mail. Speaker/person data should be anonymous after removing the account."
//												//												if(email!=null){
//												//													spk.setEmail(email);
//												//												}
//												em.persist(spk);
//
//												// associate account with the speaker
//												account.setPerson(spk);
//												spk.setAccount(account);
//												servletContext.log("Persisted speaker entity "+spk.getPersonId()+ " and set as person to login \""+account.getLogin()+"\"");
												// associate with project(s) from invitation request
												List<Project> projects=invitationRequest.getProjects();
												for(Project pr:projects){
													//Project pr=em.find(Project.class, prName);
													pr.getAccounts().add(account);
													account.getProjects().add(pr);
													servletContext.log("Added account \""+account.getLogin()+"\" to project \""+pr.getName()+"\"");
												}
												em.merge(account);
												//em.merge(spk);
											}

										}

										if(accountHasUserRole(account, UserRoleId.RoleName.PROJECT_ADMIN)) {
											servletContext.log("Login \""+account.getLogin()+"\" has role PROJECT_ADMIN");
											// grant project admin rights to project(s) from invitation request 
											List<Project> projects=invitationRequest.getProjects();
											for(Project pr:projects){
												//														Project pr=em.find(Project.class, prName);
												//														if(pr!=null) {
												pr.getAdminAccounts().add(account);
												account.getAdminOfProjects().add(pr);
												em.merge(pr);
												em.merge(account);
												servletContext.log("Added account \""+account.getLogin()+"\" as admin to project \""+pr.getName()+"\"");
												//														}
											}
										}

										// remove the request
										List<Project> prjs=invitationRequest.getProjects();
										for(Project prj:prjs) {
											prj.getInvitationRequests().remove(invitationRequest);
											em.merge(prj);
											//													invitationRequest.getProjects().remove(prj);
											//													em.merge(invitationRequest);
										}
										em.remove(invitationRequest);
										servletContext.log("Created account \""+account.getLogin()+"\".");
										processResult=new ProcessResult(Type.SUCCESS);
									} else {

										processResult=new ProcessResult(Type.ERROR);
										processResult.setLocalizableMessage(pwCheckMsg);
										processResult.setValidationResult(new ValidationResult(ValidationResult.Type.ERRORS));
										processResult.setResultKey("failed");
									}
								}
							}else{
								accountRequest=null;
								// create new bean for form
								account=new Account();
								account.setLogin(reqLogin);

								// show form
								processResult=null;
							}
						}else{
							// request expired (invalid)
							setAccount(null);
							servletContext.log("Expired account request for \""+account+"\" !");
							ProcessResult pr=new ProcessResult(Type.ERROR);
							pr.setResultKey("failed");
						}
					}else{
						// account already exists
						servletContext.log("Account create request: Corresponding account for login\""+reqLogin+"\" already exists !");
						processResult=new ProcessResult(Type.ERROR);
						LocalizableMessage lm=new LocalizableMessage("Messages", "account.already_exists");
						processResult.setLocalizableMessage(lm);
						processResult.setValidationResult(new ValidationResult(ValidationResult.Type.ERRORS));
					}
				}else{
					// request not found
					setAccount(null);
					servletContext.log("Account request: matching request not found!");
					processResult=new ProcessResult(Type.ERROR);
					processResult.setResultKey("failed");
				}
			}else{
				// No request manager yet.
				// Happens if WikiSpeech was restarted.
				// The request are not persisted and lost on restart
				servletContext.log("No account request manager found \""+account+"\" !");
				processResult=new ProcessResult(Type.ERROR);
				processResult.setResultKey("failed");
			}
		}
	}

	public void passwordSetRequest(HttpServletRequest req) throws ControllerException{
		processResult=null;
		accountRequest=null;
		setAccount(null);
		String uuidString=req.getParameter("uuid");
		if(uuidString!=null && ! uuidString.equals("")){
			UUID uuid=UUID.fromString(uuidString);
			String passw1=req.getParameter("password1");
			String passw2=req.getParameter("password2");

			//AccountRequestsManager pwrm=(AccountRequestsManager)servletContext.getAttribute(getPasswordRequestManagerAttributeName());
			EntityManager em=getThreadEntityManager();
			if(em!=null){
				AccountRequest pwrr=accountUtils.accountRequestByUUID(em, uuid);
				if(pwrr!=null){
					accountRequest=pwrr;
					Account acc=getById(pwrr.getLogin());
					if(acc!=null){
						LocalizableMessage pwCheckMsg=accountUtils.checkPassword(passw1, passw2);
						if(pwCheckMsg==null){
							applyDigestPassword(acc,passw1);
							em.merge(acc);
							this.account=acc;
							processResult=new ProcessResult(Type.SUCCESS);
							servletContext.log("Remove password request "+pwrr+" for login "+"\""+pwrr.getLogin()+"\".");
							em.remove(pwrr);
						}else{
							processResult=new ProcessResult(Type.ERROR);
							processResult.setResultKey("failed");
							processResult.setLocalizableMessage(pwCheckMsg);
							processResult.setValidationResult(new ValidationResult(ValidationResult.Type.ERRORS));
						}
					}else{
						processResult=new ProcessResult(Type.ERROR);
						processResult.setResultKey("failed");
						processResult.setLocalizableMessage(new LocalizableMessage("Could not find corresponding account!"));
					}
					
				}
			}
		}
	}
	
	public void loginSendRequest(HttpServletRequest req){
		String eMailParam=req.getParameter("email");
		if(eMailParam!=null && !eMailParam.isBlank()){
			String eMailTr=eMailParam.trim();
			SendMail sm=new SendMail();
			if(sm.isAddressValid(eMailTr)){
				URI emailUri=null;
				try{
					emailUri=new URI(eMailTr);
				}catch(URISyntaxException use){
					servletContext.log("Login send request: E-Mail address \""+eMailParam+"\" malformed!", use);
					return;
				}
				if(emailUri!=null){
					List<Account> acclist=new ArrayList<Account>();
					EntityManager em=getThreadEntityManager();
				 
					//"SELECT ac FROM Account ac"
					CriteriaBuilder queryBuilder = em.getCriteriaBuilder();
					CriteriaQuery<Account> qdef = queryBuilder.createQuery(Account.class);
					Root<Account> accRoot = qdef.from(Account.class);
					qdef.select(accRoot);
					TypedQuery<Account> tq=em.createQuery(qdef);
					
					List<Account> allAccs=tq.getResultList();
					for(Account a:allAccs){
						String aEmail=a.contactEmail();
						if(aEmail!=null){
							if(eMailTr.equalsIgnoreCase(aEmail)) {
								acclist.add(a);
							}
						}
					}

					if(acclist.size()>0){
						// prepare E-Mail:
						// TODO localize!!
						StringBuffer contentBuffer=new StringBuffer("Dear WikiSpeech user!\n\nYour E-mail address is associated with the following accounts:\n\n");
						for(Account a:acclist){
							contentBuffer.append(a.getLogin()+"\n");
						}
						contentBuffer.append("\nPlease try to login:");
						// create the URL for the user to login
						String url=req.getScheme()+"://"+req.getServerName()+":"+req.getServerPort()+req.getContextPath()+"/login.jsp";
						contentBuffer.append(url.toString());
						contentBuffer.append("\n\nBest regards\n");
						String wikiSpeechEmail=servletContext.getInitParameter("wikispeech.email_address");
						if(wikiSpeechEmail==null){
							servletContext.log("Could not get WikiSpeech E-Mail (sender) address (\"wikispeech.email_address\" in web.xml)");
							return;
						}
						try {
							sm.send(wikiSpeechEmail, emailUri.toString(), "IPS Wikispeech Request", contentBuffer.toString());
							servletContext.log("Sent login request E-Mail from "+wikiSpeechEmail+" to "+emailUri.toString());
						} catch (SendMailException e) {
							servletContext.log("Could not send login request E-Mail",e);
							return;
						}
					}else{
						servletContext.log("Login send request: No account found for E-Mail address \""+eMailParam+"\"!");
					}
				}
			}
		}
	}
	
	@Override
	protected void add(HttpServletRequest request, String command)throws ControllerException {
		// store ...
		super.add(request,command);
		// ... and digest password
		BeanModel<Account> bm=getBeanModel();
		Account currAcc=bm.getBean();
		digestPassword(currAcc);
		
		
	}

//	public String getEmailSent(){
//		Account a=getAccount();
//		if(a==null)return null;
//		return a.contactEmail();
//	}
//	
	
	public void deleteAccount(HttpServletRequest req) throws ControllerException {
		beanModel=null;
		String cmd = processCommand(req, new String[] { CMD_CANCEL, CMD_DELETE });
		if (CMD_DELETE.equals(cmd)) {
			if(checkSecureRequestToken) {
				secureRequestTokenProvider.checkSecureRequestToken(req);
			}
			String idStr = req.getParameter(beanInfo
					.getIdPropertyDescriptor().getName());
			try {
				setId(beanInfo.createIdValueByString(idStr));
			} catch (Exception e) {
				e.printStackTrace();
				throw new ControllerException(e);
			}
			EntityManager em=getThreadEntityManager();
			Account accToDelete= em.find(queryType, id);
			em.refresh(accToDelete);
			beanModel =new BeanModel<Account>(accToDelete);
			
			if(accToDelete!=null) {
				securityManager.checkRemovePermission(req, accToDelete);
				accountUtils.deleteAccount(em, accToDelete);
				processResult = new ProcessResult(ProcessResult.Type.SUCCESS);
			}else {
				processResult=new ProcessResult(ProcessResult.Type.ERROR);
			}
		}else if(CMD_CANCEL.equals(cmd)) {
			processResult=new ProcessResult(ProcessResult.Type.CANCELLED);
		}else {
			super.processRequest(req);
		}
	}
	
	public void deleteOwnAccount(HttpServletRequest req) throws ControllerException {

		String cmd = processCommand(req, new String[] { CMD_CANCEL, CMD_DELETE });
		if (CMD_DELETE.equals(cmd)) {
			if(checkSecureRequestToken) {
				secureRequestTokenProvider.checkSecureRequestToken(req);
			}
			EntityManager em = getThreadEntityManager();
			Account acc = AccountController.getAccountByRequest(req, em);
			securityManager.checkRemovePermission(req, acc);
			accountUtils.deleteAccount(em, acc);
			processResult = new ProcessResult(ProcessResult.Type.SUCCESS);
		}else if(CMD_CANCEL.equals(cmd)) {
			processResult=new ProcessResult(ProcessResult.Type.CANCELLED);
		}
	}
	
	
	
}
