package ipsk.webapps.db.speech;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.persistence.EntityManager;
import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.annotation.WebListener;
import javax.servlet.http.HttpServletRequest;

import ipsk.beans.BeanModel;
import ipsk.beans.MapConverter;
import ipsk.beans.PropertyValidationResult;
import ipsk.beans.PropertyValidationResult.Type;
import ipsk.beans.validation.ValidationException;
import ipsk.beans.validation.ValidationResult;
import ipsk.db.speech.Account;
import ipsk.db.speech.AudioDevice;
import ipsk.db.speech.AudioFormat;
import ipsk.db.speech.DialectRegion;
import ipsk.db.speech.FormConfiguration;
import ipsk.db.speech.InformedConsent;
import ipsk.db.speech.Message;
import ipsk.db.speech.Organisation;
import ipsk.db.speech.Person;
import ipsk.db.speech.Project;
import ipsk.db.speech.Project.AudioStorageType;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.UserRoleId;
import ipsk.db.speech.account.InvitationRequest;
import ipsk.db.speech.project.MediaCaptureFormat;
import ipsk.db.speech.project.MediaStorageFormat;
import ipsk.io.FileUtils;
import ipsk.persistence.OrderBy;
import ipsk.persistence.ParameterizedQuery;
import ipsk.persistence.QueryParam;
import ipsk.persistence.SecurityManager;
import ipsk.sql.OrderByClause;
import ipsk.util.LocalizableMessage;
import ipsk.webapps.ControllerException;
import ipsk.webapps.ProcessResult;
import ipsk.webapps.db.speech.project.ProjectHelper;

@WebListener
public class ProjectController extends BasicWikiSpeechController<Project> implements ServletContextListener{
	
	
	// Format of XML exports:
	// 0 or null in the DB: Initial WikiSpeech format
	// 1: Session and Speaker converted by JAXB, Speaker representation contains informed consents
	public static final int DEFAULT_EXPORT_FORMAT_VERSION=1;
	
	private DateTimeFormatter maintWarnApiDateTimeFormatter;
	
	protected static ProjectHelper helper;
	
	protected Set<Organisation> deletableOrgas=new HashSet<>();
	protected Set<Organisation> sharedOrgas=new HashSet<>();
	
	public static class FilesystemCounts{
		long numberOfDirectories;
		long numberOfRegularFiles;
		public long getNumberOfRegularFiles() {
			return numberOfRegularFiles;
		}
		
		public FilesystemCounts(long numberOfDirectories, long numberOfRegularFiles) {
			super();
			this.numberOfDirectories = numberOfDirectories;
			this.numberOfRegularFiles = numberOfRegularFiles;
		}
		public FilesystemCounts() {
			this(0,0);
		}
		public FilesystemCounts(Path p) {
			this();
			if(Files.isDirectory(p)){
				this.numberOfDirectories =1;
			}else if(Files.isRegularFile(p)){
				this.numberOfRegularFiles = 1;
			}
		}
		public long getNumberOfDirectories() {
			return numberOfDirectories;
		}
		
		public boolean isEmpty() {
			return numberOfDirectories==0 && numberOfRegularFiles==0;
		}
	
	}
	
	public ProjectController() {
		super("WebSpeechDBPU",Project.class);
		securityManager=new WikiSpeechSecurityManager(this);
		maintWarnApiDateTimeFormatter = new DateTimeFormatterBuilder()
			    // date/time
			    .append(DateTimeFormatter.ISO_LOCAL_DATE_TIME)
			    // offset (hh:mm - "+00:00" when it's zero)
			    .optionalStart().appendOffset("+HH:MM", "+00:00").optionalEnd()
			    // offset (hhmm - "+0000" when it's zero)
			    .optionalStart().appendOffset("+HHMM", "+0000").optionalEnd()
			    // offset (hh - "Z" when it's zero)
			    .optionalStart().appendOffset("+HH", "Z").optionalEnd()
			    // create formatter
			    .toFormatter();
	}
//	public ProjectController(String persistenceUnit, Class queryType, String jsfIdentifier) {
//		super(persistenceUnit, queryType, jsfIdentifier);
//		securityManager=new WikiSpeechSecurityManager(this);
//		
//	}
	

	protected void setPropertiesOnNew(Object bean) {
		super.setPropertiesOnNew(bean);
		if(bean instanceof Project) {
			Project project=(Project)bean;
			// Set a random unique UUID for the project
			project.setUuid(UUID.randomUUID().toString());

			// Set current export format version
			project.setExportFormatVersion(DEFAULT_EXPORT_FORMAT_VERSION);
		}
	}
	
	protected void setPropertiesOnCreate(HttpServletRequest request,EntityManager em,Object bean) {
		Project project=(Project)bean;
		// set date registered
		project.setRegistered(new Date());
		
		// Set account who registered the new project
		Account regAcc=project.getRegisteredByAccount();
		if(regAcc==null){
			Account acc=getAccountByRequest(request);
			project.setRegisteredByAccount(acc);
		}
		
		// Comment the code below in the next major version to set new defaults: Mono recording and use PCM float encoding
//		// New default is one channel (mono) recording
//		MediaCaptureFormat mcf=new MediaCaptureFormat();
//		mcf.setProject(project);
//		mcf.setAudioChannelCount(1);
//		project.setMediaCaptureFormat(mcf);
//
//		// Use float encoded audio for new projects
//		MediaStorageFormat msf=new MediaStorageFormat();
//		msf.setProject(project);
//		msf.setAudioEncoding(MediaStorageFormat.AudioEncoding.PCM_FLOAT);
//		project.setMediaStorageFormat(msf);
	}
	
	public List<String> checkMaintenanceWarning(String maintUrlStr) throws ControllerException {

		List<String> lines=null;


		//maintUrlStr="https://www.phonetik.uni-muenchen.de/admin/public/api/availability/next_outage.php?hours_before=7200";
		//String maintUrlStr="https://www.phonetik.uni-muenchen.de/admin/public/api/availability/next_outage.php";


		//String maintUrlStr=req.getServletContext().getInitParameter("maintenanceWarningApiURL");
		//String maintUrlStr="https://www.phonetik.uni-muenchen.de/admin/public/api/availability/next_outage_text.php?hours_before=720";
		if(maintUrlStr!=null && ! maintUrlStr.trim().isEmpty() && !maintUrlStr.startsWith("$")) {
			
				URL maintWarnApiURL;

				try {
					maintWarnApiURL = new URL(maintUrlStr);
					URLConnection maintWarnApiConn;
					maintWarnApiConn = maintWarnApiURL.openConnection();
					maintWarnApiConn.setConnectTimeout(30000);
					maintWarnApiConn.setReadTimeout(40000);
					
					if(maintWarnApiConn instanceof HttpURLConnection) {
						HttpURLConnection maintWarnApiHttpConn=(HttpURLConnection)maintWarnApiConn;
						maintWarnApiHttpConn.setInstanceFollowRedirects(true);
						maintWarnApiHttpConn.connect();
						int respCode=maintWarnApiHttpConn.getResponseCode();

						if(respCode!=204) {
							InputStream is=maintWarnApiHttpConn.getInputStream();
							InputStreamReader isr=new InputStreamReader(is);
							LineNumberReader lnr=new LineNumberReader(isr);
							lines=lnr.lines().collect(Collectors.toList());
							lnr.close();
						}
						maintWarnApiHttpConn.disconnect();
					}
				} catch (MalformedURLException e) {
					e.printStackTrace();
				} catch (IOException e) {
					e.printStackTrace();
				}

//			HttpRequest request = HttpRequest.newBuilder()
//					.uri(URI.create(maintUrlStr))
//					.timeout(Duration.ofSeconds(10))
//					//.header("Accept", "application/json")
//					.GET()
//					.build();
//
//			ExecutorService httpClientExecutor = Executors.newSingleThreadExecutor();
//			HttpClient hc=HttpClient.newBuilder().executor(httpClientExecutor).build();
//			// Tomcat complains on redeploy:
////			WARNUNG [https-openssl-nio-443-exec-1729] org.apache.catalina.loader.WebappClassLoaderBase.clearReferencesThreads The web application [wikispeech] appea
////			rs to have started a thread named [HttpClient-3379-SelectorManager] but has failed to stop it. This is very likely to create a memory leak. Stack trace of thread:
////				 java.base@11.0.10/sun.nio.ch.EPoll.wait(Native Method)
////				 java.base@11.0.10/sun.nio.ch.EPollSelectorImpl.doSelect(EPollSelectorImpl.java:120)
////				 java.base@11.0.10/sun.nio.ch.SelectorImpl.lockAndDoSelect(SelectorImpl.java:124)
////				 java.base@11.0.10/sun.nio.ch.SelectorImpl.select(SelectorImpl.java:136)
////				 platform/java.net.http@11.0.10/jdk.internal.net.http.HttpClientImpl$SelectorManager.run(HttpClientImpl.java:867)
//
//			// https://stackoverflow.com/questions/53919721/close-java-http-client
//			HttpResponse<String> response;
//			try {
//				response = hc.send(request, BodyHandlers.ofString());
//			} catch (IOException | InterruptedException e) {
//				e.printStackTrace();
//				return null;
//			}
//			int respCode=response.statusCode();
//			String rspBody=null;
//			if(respCode!=204) {
//				rspBody=response.body();
//			}
//			httpClientExecutor.shutdownNow();
//			try {
//				httpClientExecutor.awaitTermination(30,TimeUnit.SECONDS);
//			} catch (InterruptedException e) {
//				// TODO Auto-generated catch block
//				e.printStackTrace();
//			}
			
		

			//			StringReader sr=new StringReader(response.body());
			//			JsonParser jsonP=Json.createParser(sr);
			//			while (jsonP.hasNext()) {
			//			     jsonP.next(); // advance parser state
			//			     JsonValue value = jsonP.getValue();
			//			     JsonObject warnObj=value.asJsonObject();
			//			     String begStr=warnObj.getString("begin");
			//			     OffsetDateTime begDt=OffsetDateTime.parse(begStr, maintWarnApiDateTimeFormatter);
			//			     String fmtBeg=begDt.format(DateTimeFormatter.RFC_1123_DATE_TIME);
			//					String endStr=warnObj.getString("end");
			//					OffsetDateTime endDt=OffsetDateTime.parse(endStr, maintWarnApiDateTimeFormatter);
			//					String fmtEnd=endDt.format(DateTimeFormatter.RFC_1123_DATE_TIME);
			//					return fmtBeg+" to "+fmtEnd;
			//			 }

		}
		return lines;
	}

	
	/**
	 * Creates list of all projects.
	 * Warning! Access is not checked by the security manager!
	 * The JSP page is responsible to show only property which are public. 
	 * @param req
	 * @throws ControllerException
	 */
	public void projectsUncheckedSortedByName(HttpServletRequest req)
	throws ControllerException {
		
		//createEmptyBeanTableModel(req);
		ParameterizedQuery pq = new ParameterizedQuery(queryType);
		String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;
		setOrderByClause(new OrderByClause(new OrderBy[]{new OrderBy("name")}));
		setParameterizedQuery(pq);
		// temporary disable security manager
		 SecurityManager currSecManager = getSecurityManager();
		 setSecurityManager(null);
		 setBatchSize(Integer.MAX_VALUE);
		processListRequest(req);
		setSecurityManager(currSecManager);
		
	}
	
	public boolean getAdmin(){
		return getAdmin(currentRequest);
	}
	
	public boolean getAdmin(HttpServletRequest req){			
		return (req!=null && req.isUserInRole(UserRoleId.RoleName.ADMIN.name()));
	}
	
	public boolean getProjectExperimenter(){
		return getProjectExperimenter(currentRequest);
	}
	
	public boolean getProjectExperimenter(HttpServletRequest req){	
		
		boolean isProjectAdm=false;
		boolean isOrga=false;
		if(req!=null){
			isProjectAdm=req.isUserInRole(UserRoleId.RoleName.PROJECT_ADMIN.name());
			isOrga=req.isUserInRole(UserRoleId.RoleName.ORGANISATION.name());
		
			EntityManager em = getThreadEntityManager();

			Account acc=AccountController.getAccountByRequest(req, em);
			if(acc!=null){

				Project p=getSelectedProject(req);
				if(p!=null){
					if(isProjectAdm){
						Set<Account> admAccs=p.getAdminAccounts();
						if(admAccs.contains(acc)){
							return true;
						}
					}
					if(isOrga){
						Organisation accOrga=acc.getOrganisation();
						if(p.getOrganisations().contains(accOrga)){
							return true;
						}
					}
				}
			}
		}
		return false;
	}

	
	public boolean getProjectAdmin(){
		return getProjectAdmin(currentRequest);
	}
	
	public boolean getProjectAdmin(HttpServletRequest req){	
		
		if(req!=null && req.isUserInRole(UserRoleId.RoleName.PROJECT_ADMIN.name())){
			EntityManager em = getThreadEntityManager();

			Account acc=AccountController.getAccountByRequest(req, em);
			if(acc!=null){

//				Project p=getSelectedProject(req);
//				if(p!=null){
//					Set<Account> admAccs=p.getAdminAccounts();
//					if(!admAccs.contains(acc)){
//						return true;
//					}
//				}
				return(acc.getAdminOfProjects().size()>0);
			}
		}
		return false;
	}
	public boolean getProjectSubject(){
		return getProjectSubject(currentRequest);
	}
	
	public boolean getProjectSubject(HttpServletRequest req){
		// TODO do we need to check if account is assigned to project??
		if(req!=null && (req.isUserInRole(UserRoleId.RoleName.SUBJECT.name()) || req.isUserInRole(UserRoleId.RoleName.ORGANISATION.name()))){
			Project p=getSelectedProject(req);
			return(p!=null);	
			
		}
		return false;
	}
	
	public boolean getSubjectHasInformedConsentForSelectedProject() {
		boolean isSubject=getProjectSubject();
		
		Project selPrj=getSelectedProject();
		if(selPrj!=null && isSubject) {
			EntityManager em=getThreadEntityManager();
			Account acc=AccountController.getAccountByRequest(currentRequest, em);
			Speaker spk=getSpeaker(acc);
			if(spk!=null) {
				Set<InformedConsent> infCnsts=spk.getInformedConsents();
				for(InformedConsent ic:infCnsts) {
					if(selPrj.equals(ic.getProject())) {
						return true;
					}
				}
			}
		}
		return false;
	}
	
	
	public void projectsByAccount(HttpServletRequest req)
	throws ControllerException {
		EntityManager em = getThreadEntityManager();
		Account acc = AccountController.getAccountByRequest(req, em);
		createEmptyBeanTableModel(req);
		if (acc == null)
			return;
		if(req.isUserInRole(UserRoleId.RoleName.PROJECT_ADMIN.name())){
			ParameterizedQuery pq = new ParameterizedQuery(queryType);
			String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;
			pq.setWhereClause(":account MEMBER OF "+jse+".adminAccounts");
			pq.setQueryParams(new QueryParam[] { new QueryParam("account", acc) });
			setParameterizedQuery(pq);
			processListRequest(req);
		}
	}
	
	private List<Project> _getAssociatedProjects(EntityManager em,Account acc){
		HashSet<Project> projects=new HashSet<Project>();
		if(acc!=null){
			Person person=acc.getPerson();
			Set<Speaker> spkDatas=acc.getSpeakerData();
			Organisation pOrga=acc.getOrganisation();
			Set<Project> accProjs=acc.getProjects();
			
			// add administration projects 
			projects.addAll(acc.getAdminOfProjects());
			
			// add account associated projects
			if(accProjs!=null && accProjs.size()>0){
				projects.addAll(accProjs);
			}
			if(person !=null){
				
				// add projects of organisation the person belongs to 
				Set<Organisation> orgas= person.getOrganisations();
				for(Organisation o:orgas){
					Set<Project> orgaProjs=o.getProjects();
					if(orgaProjs!=null)projects.addAll(orgaProjs);
				}	
			}
			for(Speaker spkData:spkDatas) {
				// add projects of organisation the speaker belongs to 
				Set<Organisation> orgas= spkData.getOrganisations();
				for(Organisation o:orgas){
					Set<Project> orgaProjs=o.getProjects();
					if(orgaProjs!=null)projects.addAll(orgaProjs);
				}	
			}
			if(pOrga!=null){
				// add projects of organisation the account belongs to 
				projects.addAll(pOrga.getProjects());
			}
		}
		return Arrays.asList(projects.toArray(new Project[0]));
	}
	
	public List<Project> getAssociatedProjects(Account acc){
		List<Project> projects=new ArrayList<Project>();
		EntityManager em = getThreadEntityManager();
		projects=_getAssociatedProjects(em, acc);

		return projects;
	}
	
	
	
	
	public List<Project> getAssociatedProjects(HttpServletRequest req){
		currentRequest=req;
		List<Project> projects=new ArrayList<Project>();
		if(req!=null){
			EntityManager em = getThreadEntityManager();
			Account acc=AccountController.getAccountByRequest(req, em);
			projects=_getAssociatedProjects(em, acc);
		}
		return projects;
	}
	
	
	public boolean hasProjectSubjectAccounts() throws ControllerException {
		Project prj=getItem();
		int accsSize=0;
		if(prj!=null) {
			Set<Account> prjAccs=prj.getAccounts();
			List<Account> prjAccsAsList=new ArrayList<>();
			prjAccsAsList.addAll(prjAccs);
			accsSize=ipsk.webapps.db.speech.project.AccountController.filterSubjectAccounts(prj,prjAccsAsList).size();
		}
		return (accsSize>0);
	}
	
	public FilesystemCounts resourceFilesystemCounts() throws ControllerException {
		Project prj=getItem();
		FilesystemCounts fsc=null;
		
		if(prj!=null && helper!=null) {
			Path prjResPath=helper.resourcesDirPath(prj.getName());
			if(Files.exists(prjResPath)) {
				try {
					
					// TODO Move code to Java Utils package. Might be useful for other projects/packages as well
					
					// One of my first map/reduce operations in Java, doing it step by step ;)
					// Get filesystem walk stream
					Stream<Path> wkStr=Files.walk(prjResPath);
					// Map to count object depending if it is a directory or regular file
					Stream<FilesystemCounts> fsCntsStr=wkStr.map(p->p.equals(prjResPath)?new FilesystemCounts():new FilesystemCounts(p));
					// Reduce the count objects stream by summing up the counts
					Optional<FilesystemCounts> oFsc=fsCntsStr.reduce((fsc1,fsc2)->new FilesystemCounts(fsc1.numberOfDirectories+fsc2.numberOfDirectories, fsc1.numberOfRegularFiles+fsc2.numberOfRegularFiles));
					// Not sure if JSP EL is capable of Optional, so use classic nullable object 
					fsc=oFsc.get();
				} catch (IOException e) {
					e.printStackTrace();
					throw new ControllerException("Could not count project resource directories in "+prjResPath+": "+e.getMessage());
				}
			}
		}else {
			throw new ControllerException("Could not count project resource directories");
		}
		return fsc;
	}
	
	public long resourceDirCount() throws ControllerException {
		Project prj=getItem();
		long resDirsCnt=0;
		if(prj!=null && helper!=null) {
			Path prjResPath=helper.resourcesDirPath(prj.getName());
			if(Files.exists(prjResPath)) {
				try {
					resDirsCnt=Files.walk(prjResPath).filter(p->Files.isDirectory(p)).count();
				} catch (IOException e) {
					e.printStackTrace();
					throw new ControllerException("Could not count project resource directories in "+prjResPath+": "+e.getMessage());
				}
			}
		}else {
			throw new ControllerException("Could not count project resource directories");
		}
		return resDirsCnt;
	}
	
	// Note: For large rsource filesystems this must be done async
	public long resourceFileCount() throws ControllerException {
		Project prj=getItem();
		long resFilesCnt=0;
		if(prj!=null && helper!=null) {
			Path prjResPath=helper.resourcesDirPath(prj.getName());
			if(Files.exists(prjResPath)) {
				try {
					resFilesCnt=Files.walk(prjResPath).filter(p->Files.isRegularFile(p)).count();
				} catch (IOException e) {
					e.printStackTrace();
					throw new ControllerException("Could not count project resource files in "+prjResPath+": "+e.getMessage());
				}
			}
		}else {
			throw new ControllerException("Could not count project resource files");
		}
		return resFilesCnt;
	}
	
	private void updateOrgas(Project prj) {
		deletableOrgas.clear();
		sharedOrgas.clear();

		Set<Organisation> prjOrgas=prj.getOrganisations();
		//int orgasSize=prjOrgas.size();

		// Check if this is the only project connected to this organisation
		for(Organisation prjOrga:prjOrgas) {
			Set<Project> orgaPrjs=prjOrga.getProjects();
			boolean shared=false;
			for(Project orgaPrj:orgaPrjs) {
				if(!orgaPrj.equals(prj)) {
					shared=true;
					break;
				}
			}
			if(shared) {
				sharedOrgas.add(prjOrga);
			}else {
				deletableOrgas.add(prjOrga);
			}
		}

	}
	
	public int deletableOrganisationCount() throws ControllerException {
		Project prj=getItem();
		updateOrgas(prj);
		return deletableOrgas.size();
	}
	
	public int sharedOrganisationCount() throws ControllerException {
		Project prj=getItem();
		updateOrgas(prj);
		return sharedOrgas.size();
	}
	
	public boolean projectDeletable() throws ControllerException {
		
		Project prj=getItem();
		updateOrgas(prj);
		if(prj!=null) {			
			int deletableOragsSize=deletableOrgas.size();
			int sesssSize=prj.getSessions().size();
			FilesystemCounts fsc=resourceFilesystemCounts();
			if(!hasProjectSubjectAccounts() && deletableOragsSize==0 && sesssSize==0 && (fsc==null || fsc.isEmpty())) {
				return true;
			}
		}
		return false;
	}
	
	public void deleteProject(HttpServletRequest req) throws ControllerException {
		beanModel=null;
		String cmd=processCommand(req);
		
		if(CMD_DELETE.equals(cmd)) {
			if(checkSecureRequestToken) {
				secureRequestTokenProvider.checkSecureRequestToken(req);
			}
			String idStr = req.getParameter(beanInfo
					.getIdPropertyDescriptor().getName());
			try {
				setId(beanInfo.createIdValueByString(idStr));
			} catch (Exception e) {

				e.printStackTrace();
				throw new ControllerException(e);
			}
			EntityManager em=getThreadEntityManager();
			Project prj= em.find(queryType, id);
			em.refresh(prj);
			beanModel =new BeanModel<Project>(prj);
			
			if(prj!=null && projectDeletable()) {
				
				Account acc=AccountController.getAccountByRequest(req, em);
				Set<Account> admAccs=prj.getAdminAccounts();
				
				if(!req.isUserInRole(UserRoleId.RoleName.ADMIN.name()) && !admAccs.contains(acc)) {
					throw new ControllerException("Permission denied.");
				}

				Set<DialectRegion> drs=prj.getDialectRegions();
				for(DialectRegion dr:drs) {
					dr.getProjects().remove(prj);
				}

				AudioFormat af=prj.getAudioFormat();
				if(af!=null) {
					af.getProjects().remove(prj);
					em.merge(af);
				}
				List<AudioDevice> ads=prj.getAudioDevices();
				for(AudioDevice ad:ads) {
					ad.getProjects().remove(prj);
					em.merge(ad);
				}

				FormConfiguration fc=prj.getSpeakerFormConfiguration();
				if(fc!=null) {
					fc.getSpeakerFormProjects().remove(prj);
					em.merge(fc);
				}
				Set<InvitationRequest> invReqs=prj.getInvitationRequests();
				for(InvitationRequest ir:invReqs) {
					ir.getProjects().remove(prj);
					em.merge(ir);
				}

				for(Account admAcc:admAccs) {
					admAcc.getAdminOfProjects();
					em.merge(admAcc);
				}
				
				for(Account prjAcc:prj.getAccounts()) {
					prjAcc.getProjects().remove(prj);
					em.merge(prjAcc);
				}
				
				if(helper!=null) {
					ServletContext ctx=getServletContext();
					String prjNm=prj.getName();
					Path prjResPath=helper.resourcesDirPath(prjNm);
					try {
						boolean del=Files.deleteIfExists(prjResPath);
						if(del && ctx!=null) {
							ctx.log("Deleted project resources directory "+prjResPath);
						}
					} catch (IOException e) {
						e.printStackTrace();
						String errMsg="Could not delete project resources directory (expected to be empty) "+prjResPath+": "+e.getLocalizedMessage();
						if(ctx!=null) {
							ctx.log(errMsg);
						}
						throw new ControllerException(errMsg, e);
					}
					
					// Project dir should be empty now
					Path prjPath=helper.dirPath(prjNm);
					try {
						boolean del=Files.deleteIfExists(prjPath);
						if(del && ctx!=null) {
							ctx.log("Deleted project directory "+prjPath);
						}
					} catch (IOException e) {
						e.printStackTrace();
						String errMsg="Could not delete project directory (expected to be empty) "+prjResPath+": "+e.getLocalizedMessage();
						if(ctx!=null) {
							ctx.log(errMsg);
						}
						throw new ControllerException(errMsg, e);
					}
				}

				em.remove(prj);
				processResult=new ProcessResult(ProcessResult.Type.SUCCESS);
			}else {
				processResult=new ProcessResult(ProcessResult.Type.ERROR);
			}
		}else {
			super.processRequest(req);
		}
	}
	
	public ValidationResult validate(Object o, ValidationResult validationResult)
			throws ValidationException {
		ValidationResult vr = super.validate(o, validationResult);
		if (!vr.isValid())
			return vr;
		
		// Check informed consent
		Project p = (Project) o;
		
		// Check project name
		String prjName=p.getName();
		
		String trimmedPrjNm=prjName.trim();
		
		var propVrMap=new Hashtable<String, PropertyValidationResult>();
		
		if(prjName.isBlank()) {
			vr.setType(ValidationResult.Type.ERRORS);
			var msg=new LocalizableMessage("Project name must not be blank");
			PropertyValidationResult namePropVr=new PropertyValidationResult(Type.ERROR,msg);
			propVrMap.put("name", namePropVr);
			vr.setPropertyValidationResults(propVrMap);
		}else if(!prjName.equals(trimmedPrjNm)) {
			vr.setType(ValidationResult.Type.ERRORS);
			var msg=new LocalizableMessage("Project name must not start or end with blank characters");
			PropertyValidationResult namePropVr=new PropertyValidationResult(Type.ERROR,msg);
			propVrMap.put("name", namePropVr);
			vr.setPropertyValidationResults(propVrMap);
		}else {

			// paper form, everything OK
			if(!p.isInformedConsentPaperForm()) {
				// not paperform
				// check either default informed consent which requires default research purpose
				String defRp=p.getDefaultResearchPurpose();
				if(defRp==null || "".equals(defRp)) {

					// default research purpose not set , check custom informed consent
					String defIc=p.getDefaultInformedConsentText();
					if(defIc==null || "".equals(defIc)) {
						// No informed consent possible !

						vr.setType(ValidationResult.Type.ERRORS);
						LocalizableMessage lm=new LocalizableMessage("Messages", "project.config.consent.informed.validation.error");
						vr.setMessage(lm);
					}

				}
			}
		}
		return vr;

	}
	
	
//	protected void deleteRelatedObject(EntityManager em, Object relObj){
//		if(relObj instanceof Organisation) {
//			Organisation relOrga=(Organisation)relObj;
//			Set<Account> orgaAccs=relOrga.getAccounts();
//			for(Account orgaAcc:orgaAccs) {
//				Set<Message> fromMsgs=orgaAcc.getMessagesForFromLogin();
//				for(Message fromMsg:fromMsgs) {
//					// Avoid foreign key constraint violation on deletion of Message (Account) Organisation
//					fromMsg.setReplyOf(null);
//					em.merge(fromMsg);
//				}
//			}
//			super.deleteRelatedObject(em,relObj);
//		}else {
//			super.deleteRelatedObject(em,relObj);
//		}
//		
//	}
	
	
	
	@Override
	public void contextInitialized(ServletContextEvent sce) {
		setServletContext(sce.getServletContext());
		helper=new ProjectHelper();
		helper.applyResourcesDir(sce);
		
	}
	@Override
	public void contextDestroyed(ServletContextEvent sce) {
		// Not needed
	}
}
