package ipsk.webapps.db.speech.ws;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.ResponseBuilder;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.core.StreamingOutput;

import ipsk.audio.AudioPluginException;
import ipsk.audio.ThreadSafeAudioSystem;
import ipsk.audio.arr.Selection;
import ipsk.audio.plugins.EditPlugin;
import ipsk.beans.validation.ValidationResult;
import ipsk.db.speech.RecordingFile;
import ipsk.db.speech.Session;
import ipsk.db.speech.script.Recording;
import ipsk.io.EditInputStream;
import ipsk.io.FilenameValidator;
import ipsk.io.StreamCopy;
import ipsk.net.URLContext;
import ipsk.net.Utils;
import ipsk.persistence.EntityManagerProvider;
import ipsk.text.InvalidInputException;
import ipsk.util.SystemHelper;
import ipsk.webapps.EntityManagerFactoryInitializer;
import ipsk.webapps.PermissionDeniedException;
import ipsk.webapps.audio.RecordingFileHelper.RecordingFileEdit;
import ipsk.webapps.db.speech.ChunkedRecordingStates;
import ipsk.webapps.db.speech.RecordingFileController;
import ipsk.webapps.db.speech.SessionController;
import ipsk.webapps.db.speech.SpeakerController;
import ipsk.webapps.db.speech.SpeakerInformedConsentController;
import ipsk.webapps.db.speech.WikiSpeechSecurityManager;
import ipsk.webapps.db.speech.ChunkedRecordingStates.ChunkedRecordingState;
import ipsk.webapps.db.speech.InformedConsentSpeakerController;
import ipsk.webapps.db.speech.ws.pub.WavFileEntity;

public class BasicRecordingfileResource {

	private final static boolean DEBUG = true;

	private static final long serialVersionUID = 1L;

	protected String sessionIdStr = null;

	protected Path tempDirPath = null;
	protected File tempDir = null;

	protected final static int DEFAULT_BUFSIZE = 2048;

	protected int bufSize = DEFAULT_BUFSIZE;

	// TODO build a generic and a WikiSpeech specific service class

	/**
	 * @see HttpServlet#HttpServlet()
	 */
	public BasicRecordingfileResource() {
		super();
		tempDirPath=Path.of(System.getProperty("java.io.tmpdir"));
		tempDir =tempDirPath.toFile();
	}

	public BasicRecordingfileResource(String sessionIdStr) {
		this();
		this.sessionIdStr = sessionIdStr;

	}
	
	protected RecordingFile recordingFileByUUID(EntityManager em,UUID rfUUID) {
		RecordingFile rf=null;

		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<RecordingFile> cq = cb.createQuery(RecordingFile.class);
		Root<RecordingFile> qr = cq.from(RecordingFile.class);
		cq.select(qr);
		Predicate eqUuid=cb.equal(qr.get("uuid"),rfUUID);
		cq.where(eqUuid);
		TypedQuery<RecordingFile> aq = em.createQuery(cq);
		List<RecordingFile> arList=aq.getResultList();
		// should be unique
		if(arList.size()>0) {
			rf=arList.get(0);
		}
		return rf;
	}
	
	public static boolean speakerCodeValid(String speakerCode) {
		if(speakerCode==null) {
			return true;
		}
		var	vr= FilenameValidator.validatePlatformFileNameCharacters(speakerCode);
		
		return vr.isValid();
	}
	
	public static boolean itemCodeValid(String itemCode) {
		var vr= FilenameValidator.validateFileNameCharacters(itemCode);
		return vr.isValid();
	}


	protected String createSignalFilePath(boolean typeSpeechRecorder,String sessionPath, String speakerCode, String itemCodeOrUUID, String extension,
			int sessionId, int version, boolean overwrite) {
		String speakerCodeStr = "";
		String spkCodeSessionSeparator="";
		if (speakerCode != null) {
			SpeakerInformedConsentController.validateSpeakerCode(null, speakerCode);
//			if(!speakerCode.matches("^[\\w\\s\\d-_,\\[\\]\\(|\\)]*$")){
//				throw new InvalidInputException("Invalid speaker code: "+speakerCode);
//			}
			speakerCodeStr = speakerCode;
			if(!typeSpeechRecorder) {
				spkCodeSessionSeparator="_";
			}
		}
		if(itemCodeOrUUID!=null) {
//			if(!itemCodeOrUUID.matches("^[\\w\\s\\d-_,\\[\\]\\(|\\)]*$")){
//				throw new InvalidInputException("Invalid item code: "+itemCodeOrUUID);
//			}
		}
		
		String session = SessionController.sessionIDFormat.format(sessionId);
		String uuidSeparator=(typeSpeechRecorder)?"":"_";
		String versionStr = "";
		if (typeSpeechRecorder && !overwrite) {
			versionStr = RecordingFileController.recversionFormatter.format(version);
		}

		return sessionPath + "/" + speakerCodeStr + spkCodeSessionSeparator+  session + uuidSeparator + itemCodeOrUUID + versionStr + "." + extension;
	}

	protected String createSignalFilePath(boolean typeSpeechRecorder,String sessionPath, String speakerCode, String itemCodeOrUUID, String extension,
			RecordingFile rf, boolean overwrite) {
		return createSignalFilePath(typeSpeechRecorder,sessionPath, speakerCode, itemCodeOrUUID, extension, rf.getSession().getSessionId(),
				rf.getVersion(), overwrite);
	}

	protected void storeToFile(ServletContext sc, InputStream is, File f, boolean append) throws IOException {
		byte[] buf = new byte[bufSize];
		FileOutputStream fos;
		File pd = f.getParentFile();
		if (DEBUG)
			sc.log("Parent dir: " + pd.getPath());
		if (DEBUG)
			sc.log("Saving:" + f.getAbsolutePath());
		try {
			if (DEBUG)
				sc.log("Path:" + f.getCanonicalPath());
		} catch (IOException e) {
			sc.log("Cannot get canonical path.", e);
			throw e;
		}
		boolean created = false;
		if (!pd.exists()) {
			if (DEBUG)
				sc.log("Creating dir ...");
			created = pd.mkdirs();
		}
		if (created)
			sc.log("Directory " + pd.getName() + " created.");

		// save the content
		fos = new FileOutputStream(f, append);
		int read = 0;
		long size = 0;
		// try to lock file
		try {
			fos.getChannel().lock();
		} catch (Exception locke) {
			sc.log("could not lock file " + locke.getMessage());
		}
		try {
			do {
				read = is.read(buf, 0, bufSize);

				if (read > 0) {
					size += read;
					fos.write(buf, 0, read);
				}
			} while (read >= 0);

			sc.log("File '" + f.getAbsolutePath() + "' (" + size + " bytes) written.");
		} catch (IOException e) {
			throw e;
		} finally {

			if (fos != null) {
				fos.getChannel().close();
				fos.close();
			}
		}

	}
	
	
	public Response recordingFileList(ServletContext sc,SecurityContext sec,HttpServletRequest req,Boolean latestVersionsOnly) {
		int sessionId;
		try {
			sessionId = Integer.parseInt(sessionIdStr);
		} catch (NumberFormatException nfe) {
			sc.log("Session ID " + sessionIdStr + " could not be parsed as number ID: ", nfe);
			return Response.status(Status.BAD_REQUEST).build();
		}
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		Session session = em.find(Session.class, sessionId);
		if (session == null) {
			rollBackAndClose(em);
			return Response.status(Status.NOT_FOUND).build();
		}
		em.refresh(session);
		try {
			securityManager.checkReadPermission(req, session);
		} catch (PermissionDeniedException e3) {
			sc.log("No read permission for session ID: " + sessionId + " ", e3);
			rollBackAndClose(em);
			return Response.status(Status.FORBIDDEN).build();
		}

		Set<RecordingFile> recFiles = session.getRecordingFiles();
		Set<RecordingFile> resRecFiles=new HashSet<RecordingFile>();
		if(latestVersionsOnly==null || !latestVersionsOnly) {
			resRecFiles.addAll(recFiles);
		}else {
			// latest versions only
			Set<Recording> rs=new HashSet<Recording>();
			for(RecordingFile recFile:recFiles) {
				Recording r=recFile.getRecording();
				rs.add(r);
			}
			
			for(Recording r:rs) {
				Set<RecordingFile> rRfs=r.getRecordingFiles();
				
				RecordingFile latestRf=null;
				for(RecordingFile rRf:rRfs) {
					if(rRf.getSession().equals(session)) {
						int rRfVers=rRf.getVersion();
						if(latestRf==null) {
							latestRf=rRf;
						}else {
							if(rRfVers>latestRf.getVersion()) {
								latestRf=rRf;
							}
						}
					}
				}
				if(latestRf!=null) {
					resRecFiles.add(latestRf);
				}
			}
		}
		
		final GenericEntity<Set<RecordingFile>> rfSetEntity = new GenericEntity<Set<RecordingFile>>(resRecFiles) {};
		
		return Response.ok(rfSetEntity).build();
		
	}

	protected Response streamRecFile(RecordingFile rf,Long startFrame,Long frameLength){

		String fileURLStr = rf.getSignalFile().trim();

		try {
			URL fileURL = new URL(fileURLStr);
			File recFile=Utils.fileFromDecodedURL(fileURL);
			//final InputStream rfStream = fileURL.openStream();
			//final InputStream serveStream;
			WavFileEntity wfe;
			if(startFrame==null && frameLength==null) {
				wfe=new WavFileEntity(recFile.toPath());
			}else {
				long editFrameStart=0;
				if(startFrame!=null) {
					editFrameStart=startFrame;
				}
				long editFrameLen=-1;   // default: to the end of the file
				if(frameLength!=null) {
					editFrameLen=frameLength;
				}
				AudioInputStream ais=ThreadSafeAudioSystem.getAudioInputStream(recFile);
				if(startFrame>=ais.getFrameLength()) {
					return Response.status(Status.NOT_FOUND).build();
				}
				final java.nio.file.Path tmpEditedAudioFile;
				AudioInputStream orgAs=AudioSystem.getAudioInputStream(recFile);
				
				EditPlugin ep=new EditPlugin(editFrameStart,editFrameLen);
				AudioInputStream eAis=ep.getAudioInputStream(orgAs);
				// Files may be large. Use a temporary file instead of memory here.
				tmpEditedAudioFile=Files.createTempFile(getClass().getName(),".wav");
				AudioSystem.write(eAis, AudioFileFormat.Type.WAVE,tmpEditedAudioFile.toFile());
				wfe=new WavFileEntity(tmpEditedAudioFile, true);
			}
			ResponseBuilder rb=Response.ok(wfe,"audio/wav");
			return rb.build();

		} catch (IOException | UnsupportedAudioFileException | AudioPluginException mue) {
			return Response.serverError().build();
		}

	}
	

	protected Response streamRecFile(RecordingFile rf){

		String fileURLStr = rf.getSignalFile().trim();

		try {
			URL fileURL = new URL(fileURLStr);
			final InputStream rfStream = fileURL.openStream();
			StreamingOutput streamOutput = new StreamingOutput() {

				@Override
				public void write(OutputStream outputStream)
						throws IOException, WebApplicationException {
					StreamCopy.copy(rfStream, outputStream);
				}
			};
			return Response.ok(streamOutput).build();

		} catch (IOException mue) {
			return Response.serverError().build();
		}

	}
	
	
	protected Response streamFile(File f){
		StreamingOutput streamOutput = new StreamingOutput() {

			@Override
			public void write(OutputStream outputStream)
					throws IOException, WebApplicationException {
				StreamCopy.copy(f,outputStream);
			}
		};
		return Response.ok(streamOutput).build();
	}
	
	
	protected Response recordingFileStreamResponse(EntityManager em,RecordingFile rf,Long startFrame,Long frameLength) {
		ReentrantLock lock=null;
		boolean locked=false;
		RecordingFile.Status rfStatus=rf.getStatus();
		if(! (RecordingFile.Status.PROCESSED.equals(rfStatus) || RecordingFile.Status.RECORDED.equals(rfStatus) || RecordingFile.Status.PARTIALLY_RECORDED.equals(rfStatus))){
			rollBackAndClose(em);
			return Response.status(Status.NOT_FOUND).build();
		}

		if(RecordingFile.Status.PARTIALLY_RECORDED.equals(rfStatus)){
			// Get lock for partially recorded files in chunked upload mode

			String rfUUIDStr=rf.getUuid();
			if(rfUUIDStr!=null) {
				UUID rfUUID=UUID.fromString(rfUUIDStr);
				if(rfUUID!=null) {
					ChunkedRecordingState crs;
					synchronized(ChunkedRecordingStates.recordingFileLocks) {
						crs=ChunkedRecordingStates.recordingFileLocks.get(rfUUID);
						if(crs==null) {
							crs=new ChunkedRecordingState();
							ChunkedRecordingStates.recordingFileLocks.put(rfUUID, crs);
						}
					}

					lock=crs.getReentrantLock();
					int timeoutMinutes=2;
					try {
						locked=lock.tryLock(timeoutMinutes, TimeUnit.MINUTES);
						if(!locked) {
							// Waited two minutes, return locked state
							rollBackAndClose(em);
							return Response.status(423).build();
						}

						String fileURLStr = rf.getSignalFile().trim();
						try {
							// Fetch file content to buffer while rec file is locked
							URL fileURL = new URL(fileURLStr);
							File recFile=Utils.fileFromDecodedURL(fileURL);
							java.nio.file.Path tmpAudioFilePath=Files.createTempFile(tempDir.toPath(), "_tmp_partially_recorded_","audio");
							File tmpAudioFile=tmpAudioFilePath.toFile();
							StreamCopy.copy(recFile,tmpAudioFile);
							// Already unlock here to avoid blocking further concatenation of uploaded chunks
							if(lock!=null && locked) {
								lock.unlock();
							}

							// Stream the temp file. The temp file acts as a snapshot and needs not be locked
							// TODO support 
							return streamFile(tmpAudioFile);
						}catch(IOException ioe) {
							if(lock!=null && locked) {
								lock.unlock();
							}
							rollBackAndClose(em);
							return Response.serverError().build();
						}
					} catch (InterruptedException e3) {
						e3.printStackTrace();
						rollBackAndClose(em);
						// During waiting for the lock this thread should never be interrupted
						return Response.serverError().build();
					}

				}
			}
		}else {
			// Stream rec file without locking
			return streamRecFile(rf,startFrame,frameLength);

		}
		return Response.status(Status.NOT_FOUND).build();
	}


	protected List<Session>  sessionsByUUID(EntityManager em,UUID sessionUUID){
		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<Session> cq = cb.createQuery(Session.class);
		Root<Session> rt = cq.from(Session.class);
		cq.select(rt);
		cq.where(cb.equal(rt.get("uuid"), sessionUUID.toString()));

		TypedQuery<Session> q = em.createQuery(cq);
		List<Session> sessionsList=q.getResultList();
		return sessionsList;
	
	}

	protected void rollBack(EntityManager em) {
		if (em != null) {
			EntityTransaction tr = em.getTransaction();
			if (tr != null && tr.isActive()) {
				tr.rollback();
			}
			if (em.isOpen()) {
				em.close();
			}
		}
	}
	
	protected void close(EntityManager em) {
		if (em != null) {
			if (em.isOpen()) {
				em.close();
			}
		}
	}

	protected void rollBackAndClose(EntityManager em) {
		if (em != null) {
			EntityTransaction tr = em.getTransaction();
			if (tr != null && tr.isActive()) {
				tr.rollback();
			}
			if (em.isOpen()) {
				em.close();
			}
		}
	}

}
