package ipsk.webapps.db.speech.ws;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Root;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;

import ipsk.db.speech.Account;
import ipsk.db.speech.Person;
import ipsk.db.speech.Project;
import ipsk.db.speech.Session;
import ipsk.db.speech.SessionDTO;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.UserRoleId;
import ipsk.db.speech.script.Script;
import ipsk.persistence.EntityManagerProvider;
import ipsk.webapps.EntityManagerFactoryInitializer;
import ipsk.webapps.db.speech.WikiSpeechSecurityManager;

@Path("/speaker")
public class SpeakerResource extends WikispeechBasicResource<Speaker>{

	public SpeakerResource() {
		super(Speaker.class);
		
	}
	
	private Speaker byUUID(EntityManager em,UUID speakerUUID){
		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<Speaker> cq = cb.createQuery(Speaker.class);
		Root<Speaker> rt = cq.from(Speaker.class);
		cq.select(rt);
		cq.where(cb.equal(rt.get("uuid"), speakerUUID.toString()));

		TypedQuery<Speaker> q = em.createQuery(cq);
		List<Speaker> spksList=q.getResultList();
		if(spksList.isEmpty()){
			return null;
		}else{
			return spksList.get(0);
		}
	
	}
	
	@GET
	@Produces({MediaType.APPLICATION_JSON,MediaType.APPLICATION_XML})
	@Path("/{speakerIdOrUUID}")
	public Response getSpeaker(@Context HttpServletRequest req,@PathParam("speakerIdOrUUID") String speakerIdStr){
		
		Integer speakerId=null;
		UUID speakerUUID=null;
		try{
			speakerId=Integer.parseInt(speakerIdStr);
		}catch(NumberFormatException nfe){
			speakerId=null;
			try{
				speakerUUID=UUID.fromString(speakerIdStr);
			}catch(IllegalArgumentException iae){
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		Speaker s=null;
		final EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		try{
			if(speakerId!=null){
				s=em.find(Speaker.class, speakerId);
			}else if(speakerUUID!=null){
				s=byUUID(em, speakerUUID);
			}
			if(s==null){
				return Response.status(Status.NOT_FOUND).build();
			}

			WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
				@Override
				public EntityManager getThreadEntityManager() {
					return em;
				}
			});
			if(!securityManager.getReadPermission(req, s)){
				return Response.status(Status.FORBIDDEN).build();
			}

			return Response.ok(s).build();
		}finally{
			em.close();
		}
	}
	
	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Path("/{speakerUUID}")
	public Response postSpeaker(@PathParam("speakerUUID") String speakerUUIDStr, @Context HttpServletRequest req, @Context UriInfo uriInfo,Speaker speakerDto){
		
		try{
			
			String speakerDtoUuid=speakerDto.getUuid();
			
			UUID speakerUUID=null;
			// try to get from path
			if(speakerUUIDStr!=null){
				speakerUUID=UUID.fromString(speakerUUIDStr);
				// if UUID is given in data object as well the UUIDs must match
				if(speakerDtoUuid!=null && !speakerDtoUuid.equals(speakerUUID.toString())){
					// Mismatch of UUIDs 
					return Response.status(Status.BAD_REQUEST).build();
				}
			}else{
				if(speakerDtoUuid!=null){
					// try to get from DTO
					speakerUUID=UUID.fromString(speakerDtoUuid);
				}else{
					// no UUID transmitted, create one
					speakerUUID=UUID.randomUUID();
				}
			}
			
			if(speakerDtoUuid==null){
				// set UUID to object
				speakerDto.setUuid(speakerUUID.toString());
			}
			
			
			final EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
			Account acc=getAccountByRequest(req, em);
			// Set registered account
			
			Account regAcc=speakerDto.getRegisteredByAccount();
			if(regAcc !=null && !regAcc.equals(acc)){
				return Response.status(Status.FORBIDDEN).build();
			}
			speakerDto.setRegisteredByAccount(acc);
			
			try{
			WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
				@Override
				public EntityManager getThreadEntityManager() {
					return em;
				}
			});
			
			// Check if exists
			Speaker existingSpk=byUUID(em, speakerUUID);
			if(existingSpk!=null){
				return Response.status(Status.CONFLICT).build();
			}
			
			if(!securityManager.getPersistPermission(req, speakerDto)){
				return Response.status(Status.FORBIDDEN).build();
			}

			
				EntityTransaction tr=em.getTransaction();
				tr.begin();
				
				em.persist(speakerDto);
				em.merge(acc);
				tr.commit();
				 
				UriBuilder builder = uriInfo.getAbsolutePathBuilder();
			     // builder.path(speakerUUID.toString());
				return Response.created(builder.build()).entity(speakerDto).build();
				
		
			}finally{
				em.close();
			}
		}catch(IllegalArgumentException iae){
			return Response.status(Status.BAD_REQUEST).build();
		}
			
	
	}
	
	
	@PUT
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Path("/{speakerUUID}")
	public Response putSpeaker(@PathParam("speakerUUID") String speakerUUIDStr, @Context HttpServletRequest req,Speaker speakerDto){
		
		try{
			
			String speakerDtoUuid=speakerDto.getUuid();
			
			UUID speakerUUID=null;
			// try to get from path
			if(speakerUUIDStr!=null){
				speakerUUID=UUID.fromString(speakerUUIDStr);
				// if UUID is given in data object as well the UUIDs must match
				if(speakerDtoUuid!=null && !speakerDtoUuid.equals(speakerUUID.toString())){
					// Mismatch of UUIDs 
					return Response.status(Status.BAD_REQUEST).build();
				}
			}else{
				if(speakerDtoUuid!=null){
					// try to get from DTO
					speakerUUID=UUID.fromString(speakerDtoUuid);
				}else{
					// no UUID transmitted, create one
					speakerUUID=UUID.randomUUID();
				}
			}
			
			if(speakerDtoUuid==null){
				// set UUID to object
				speakerDto.setUuid(speakerUUID.toString());
			}
			
			final EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
			
			try{
			WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
				@Override
				public EntityManager getThreadEntityManager() {
					return em;
				}
			});
			
			
			
			// Check if exists
			Speaker existingSpk=byUUID(em, speakerUUID);
			if(existingSpk==null){
				return Response.status(Status.NOT_FOUND).build();
			}
			
			// REST uses UUIDs only  
			speakerDto.setPersonId(existingSpk.getPersonId());
			
			// Set registered account
			Account acc=getAccountByRequest(req, em);
			
			Account regAcc=speakerDto.getRegisteredByAccount();
			if(regAcc !=null && !regAcc.equals(acc)){
				return Response.status(Status.FORBIDDEN).build();
			}
			speakerDto.setRegisteredByAccount(acc);
			
			if(!securityManager.getMergePermission(req, speakerDto,null)){
				return Response.status(Status.FORBIDDEN).build();
			}

			
				EntityTransaction tr=em.getTransaction();
				tr.begin();
				
				em.merge(speakerDto);
				
				tr.commit();
				
				return Response.ok(speakerDto).build();
				
		
			}finally{
				em.close();
			}
		}catch(IllegalArgumentException iae){
			return Response.status(Status.BAD_REQUEST).build();
		}
			
	
	}
	
	@GET
	@Produces({MediaType.APPLICATION_JSON,MediaType.APPLICATION_XML})
	public Response getSpeakers(@Context HttpServletRequest req){
		final EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		try{
			
			Account acc=getAccountByRequest(req, em);
			if(req.isUserInRole(UserRoleId.RoleName.ADMIN.name())){
			//	Return all speakers
				CriteriaBuilder cb = em.getCriteriaBuilder();
				CriteriaQuery<Speaker> cq = cb.createQuery(Speaker.class);
				Root<Speaker> rt = cq.from(Speaker.class);
				cq.select(rt);
				TypedQuery<Speaker> q = em.createQuery(cq);
				List<Speaker> allSpks = q.getResultList();
				final GenericEntity<List<Speaker>> listEntity= new GenericEntity<List<Speaker>>(allSpks){};
				return Response.ok(listEntity).build();
			}else{
				Set<Speaker> visSpks=new HashSet<Speaker>();
				if (req.isUserInRole(UserRoleId.RoleName.PROJECT_ADMIN.name())){
					Set<Project> admProjs=acc.getAdminOfProjects();
					for(Project admProj:admProjs){
						Set<Session> admProjSesss=admProj.getSessions();
						for(Session admProjSess:admProjSesss){
							visSpks.addAll(admProjSess.getSpeakers());
						}
					}
					
				}
				Person pers=acc.getPerson();
				if(pers instanceof Speaker){
					visSpks.add((Speaker)pers);
				}
				Set<Speaker> accSpkDatas=acc.getSpeakerData();
				for(Speaker spkData:accSpkDatas) {
					visSpks.add(spkData);
				}
				
				visSpks.addAll(acc.getRegisteredSpeakers());
				
				final GenericEntity<Set<Speaker>> listEntity= new GenericEntity<Set<Speaker>>(visSpks){};
				return Response.ok(listEntity).build();
			}

//			WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
//				@Override
//				public EntityManager getThreadEntityManager() {
//					return em;
//				}
//			});
//			for (Speaker spk : visibleSpks) {
//				boolean readPerm=securityManager.getReadPermission(req, spk);
//				if(!readPerm){
//					return Response.status(Status.FORBIDDEN).build();
//				}
//			}
			
		}finally{
			em.close();
		}
	}
	
	
}
