#!/usr/bin/perl


# #####################################################
#
# This file is part of the Perl scripts of the MASV System.
# MASV = Munich Automatic Speaker Verification
#
# Copyright 2002-2003, Ulrich Trk
# Institute of Phonetics and Speech Communication
# University of Munich
# tuerk@phonetik.uni-muenchen.de
#
#
#   MASV is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   MASV is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with MASV; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# #####################################################

my $CVS_Version_String = '$Id: run_HCopy.pl,v 1.13 2004/04/28 15:17:53 tuerk Exp $';
my $CVS_Name_String = '$Name: rel-1-4-01 $';


use lib $ENV{"MASV_PERL_ROOT"};

use SR_lib;
use strict;

use File::Copy;
use File::Path;
use Getopt::Long qw( GetOptions );
use Pod::Usage;

use Parallel::ForkManager;

my $exit_state = 0;
my $local_exit_code = 0;
my @ssh_stat = [];

my $nice_factor = $SR_lib::nice_factor;
$nice_factor = $SR_lib::nice_factor; # make compiler happy, no warning

###########################
# Default values for options
my $help = 0;
my $Identify = 0;
my $extraOptions = '';
my $distributedClient = 0;
my $parallelComputation = 0;
###########################
my $extraOptionsString;

my ($hostsNum, @hostList);
my ($exe_script_path, $exe_script_file);

my $prompt_HTK_Command_String = ($SR_lib::verbose_level > 1) ? "-A" : "";


# Processing Options

# flat
GetOptions('dist=s' => \$distributedClient,
		   'xo|extraOptions=s' => \$extraOptions,
           'p|parallelComputation' => \$parallelComputation,
           'help|?' => \$help,
           'version' => \$Identify);

if ($Identify) {
	print STDOUT "$0\n$CVS_Version_String\n$CVS_Name_String\n\n"; 
}

if ((@ARGV < 1) || (@ARGV > 3) || ($help)) {
   pod2usage(1);
   exit 1;
}

if ($SR_lib::verbose_level) {print STDOUT "Running $0 ...\n";}

if ($distributedClient && $parallelComputation) {
	print STDERR "Use only the -p|parallelComputation option!\n\n";
	exit 1;
}

my $pool_name = $ARGV[0];

my $config_file = "C_${pool_name}";
my $script_file = "S_pool_$pool_name";

if (@ARGV >= 2) {
  $script_file = $ARGV[1];
}
if (@ARGV == 3) {
  $config_file = $ARGV[2];
}


my $pool_path = "${SR_lib::paramPool_dir}${pool_name}$SR_lib::sign";
my $config_path = "${pool_path}configs$SR_lib::sign";
my $script_path = "${pool_path}scripts$SR_lib::sign";

$extraOptionsString = ($extraOptions) ? "-xo=\'$extraOptions\'" : "";


if ($parallelComputation) {

	($hostsNum, @hostList) = &SR_lib::par_get_available_hosts;
	
	# split the list to equal parts (num of parts is given by $hostsNum)
	&SR_lib::split_fileList("${script_path}${script_file}", $hostsNum);

}


if ($distributedClient) 
{ # distributed computing -> get list with relevant files to process
	$exe_script_path = &SR_lib::get_distList_dir(($script_path.$script_file));
	$exe_script_file = &SR_lib::get_distList_name($distributedClient);
}
else 
{ # only one host for computing -> leave all as it was.
	$exe_script_path = $script_path;
	$exe_script_file = $script_file;
}


if ($parallelComputation) {

	my $pm = new Parallel::ForkManager($hostsNum);
	$pm->run_on_finish(
		sub { my ($pid, $exit_code, $ident) = @_;
			  $ssh_stat[$ident] = $exit_code;
		}
	);
	
	
	for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
		$pm->start($host_counter) and next;
		
		if ($SR_lib::verbose_level) { print STDOUT ("Starting process on host $hostList[$host_counter]\n");}
		my $exec_string = "ssh -x $hostList[$host_counter] /usr/bin/nice -n $nice_factor \"run_HCopy.pl $extraOptionsString -dist=$host_counter $pool_name $script_file $config_file \"";
		if ($SR_lib::verbose_level) { print STDOUT "$exec_string \n";}
		my $local_exit_code = system($exec_string);
		if ($local_exit_code) {
			$local_exit_code = 11;
		}

		if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $local_exit_code \n";}
		if ($SR_lib::verbose_level) { print STDOUT ("Process on  $hostList[$host_counter] complete.\n");}
		
		$pm->finish($local_exit_code);
	}
	
	$pm->wait_all_children;
	if ($SR_lib::verbose_level) { print STDOUT ("All processes completed.\n");}
	
	if ($SR_lib::verbose_level) { print STDOUT ("Deleting split lists...\n");}
	my $rm_status = rmtree(&SR_lib::get_distList_dir( ($exe_script_path . $script_file)));	
	if ($rm_status == 0) {
		print STDERR "Error while deleting split lists in $exe_script_path . \n";
	}
	if ($SR_lib::verbose_level) { print STDOUT ("Process status summary:\n");}
	
	for (my $host_counter=1; $host_counter <= $hostsNum; $host_counter++) {
		if ($SR_lib::verbose_level) { print STDOUT "Host $hostList[$host_counter], status: $ssh_stat[$host_counter] \n";}
		$exit_state += $ssh_stat[$host_counter];
	}
	if ($SR_lib::verbose_level) { print STDOUT "Done.\n\n";}
	
}
else
{
	my $exec_string = "";
	if ($MASV_db_desc::audio_type eq "raw_alaw") {
		$exec_string = "export HWAVEFILTER='rawal2waval.pl -fs $MASV_db_desc::sampling_freq \$ -'; ";
		#$exec_string = "export HWAVEFILTER='rawal2waval.pl -fs $MASV_db_desc::sampling_freq -min_part_length=110 -cut_const_parts \$ -'; ";
	}
	$exec_string .= "${SR_lib::htk_bin}HCopy $prompt_HTK_Command_String -C ${config_path}$config_file $extraOptions -S ${exe_script_path}$exe_script_file";
	$exit_state = system($exec_string);
}

	
exit $exit_state;


__END__

=head1 NAME

run_HCopy.pl  - run HCopy with script files

=head1 SYNOPSIS

run_HCopy.pl [options] poolname [script_file [config_file]]

Run HCopy for the parameter pool <poolname> with the given script_file and the
given config_file. If one or both are omitted, the file "S_pool_<poolname>" is assumed 
to be the script file and "C_<poolname>" is used as HCopy config file.
Script files are searched for in the "scripts/" directory of the pool; 
config files are searched for in the "configs/" directory.

 Options:
 
 -xo | extraOptions         pass extra options to HCopy

 -p | parallelComputation   use several hosts to run the HCopy command.
                          
 -? | help                  display this message.
 
=cut

